#include "aisexpert_appl.h"
#include "event_log.h"
#include "functions.h"
#include "user_rights.h"
#include "garbage_collector.h"
#include "xgboost/options.h"
#include "database/connect.h"
#include "database/firebird_driver.h"
#include "monitoring/mail_smtp.h"
#include "monitoring/notifier.h"
#include "monitoring/observer.h"
#include "monitoring/http_connector.h"
#include "tasks/scheduler.h"
#include "tasks/sync_plan.h"
#include "tasks/send_score.h"
#include "shared/defmac.h"
#include "shared/utils.h"
#include "shared/logger/logger.h"
#include "shared/logger/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/qhashex.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/message.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/config/config.h"
#include "shared/qt/version/version_number.h"
#include "shared/thread/thread_pool.h"

#include <QNetworkProxy>

#ifdef MINGW
#include <windows.h>
#else
#include <signal.h>
#endif
#include <unistd.h>

using namespace std;
using namespace communication;
using namespace communication::transport;

/**
  Используется для уведомления основного потока о завершении работы программы.
*/
#ifdef MINGW
BOOL WINAPI stopProgramHandler(DWORD sig)
{
    if ((sig == CTRL_C_EVENT) || (sig == CTRL_BREAK_EVENT))
    {
        const char* sigName = (sig == CTRL_C_EVENT) ? "CTRL_C_EVENT" : "CTRL_BREAK_EVENT";
        log_verbose << "Signal " << sigName << " is received. Program will be stopped";
        Application::stop();
    }
    else
        log_verbose << "Signal " << sig << " is received";
    return TRUE;
}
#else
void stopProgramHandler(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
    {
        const char* sigName = (sig == SIGTERM) ? "SIGTERM" : "SIGINT";
        log_verbose << "Signal " << sigName << " is received. Program will be stopped";
        Application::stop();
    }
    else
        log_verbose << "Signal " << sig << " is received";
}
#endif // #ifdef MINGW

void initLoggerExt()
{
    QString logConf;
#ifdef MINGW
    config::base().getValue("logger.conf_win", logConf);
#else
    config::base().getValue("logger.conf", logConf);
#endif
    if (!logConf.isEmpty())
    {
        config::dirExpansion(logConf);
        if (QFile::exists(logConf))
            alog::loadSavers(logConf.toStdString());
        else
            log_error << "Logger config file not exists: " << logConf;
    }
    alog::printSaversInfo();
}

void stopLog()
{
    alog::logger().flush();
    alog::logger().waitingFlush();
    alog::logger().stop();
}

void stopProgram()
{
    monitoring::httpConnector().close();

    #define STOP_THREAD(THREAD_FUNC, NAME, TIMEOUT) \
        if (!THREAD_FUNC.stop(TIMEOUT * 1000)) { \
            log_info << "Thread '" NAME "': Timeout expired, thread will be terminated"; \
            THREAD_FUNC.terminate(); \
        }

    STOP_THREAD(task::sendScore(),      "SendScore",        5*60 /* 5 мин*/)
    STOP_THREAD(task::syncPlan(),       "SyncPlan",         5*60 /* 5 мин*/)
    STOP_THREAD(task::scheduler(),      "Scheduler",        3*60 /* 3 мин*/)
    STOP_THREAD(monitoring::mailSmtp(), "MonitorMailSmtp",  65   /*65 сек*/)
    STOP_THREAD(monitoring::notifier(), "MonitorNotifier",  30   /*30 сек*/)
    STOP_THREAD(monitoring::observer(), "MonitorObserver",  30   /*30 сек*/)
    STOP_THREAD(garbageCollector(),     "GarbageCollector", 30   /*30 сек*/)

    #undef STOP_THREAD

    webCon().reset();
    tcp::listener().close();

    fomsCon().disconnect();
    upoolCon()->disconnect();

    // Удаляем сэйвер для записи лог-сообщений в БД
    alog::logger().removeSaver("eventlog");

    // Перед отключением БД нужно сбросить все лог-сообщения
    alog::logger().flush();
    alog::logger().waitingFlush();

    dbpool().close();

    log_info << "'Ais Expert' service is stopped";
    stopLog();

    trd::threadPool().stop();
}

void helpInfo(/*const char * binary*/)
{
    alog::logger().clearSavers();
    alog::logger().addSaverStdOut(alog::Level::Info, true);

    log_info << "'Ais Expert' service"
             << " (version: " << productVersion().toString()
             << "; protocol version: "
             << BPROTOCOL_VERSION_LOW << "-" << BPROTOCOL_VERSION_HIGH
             << "; gitrev: " << GIT_REVISION << ")";
    log_info << "Usage: aisexpert";
    log_info << "  -n do not daemonize";
    log_info << "  -h this help";
    alog::logger().flush();
}

//std::vector<int> testNRVO(int value, size_t size, const std::vector<int> **localVec)
//{
//    std::vector<int> vec(size, value);
//    (void) vec;
//    *localVec = &vec;

//    return vec;
//}

int main(int argc, char *argv[])
{
//    fnc().first = 3;
//    fnc().second = 5;

//    QSetEx<QUuidEx> ss;

//    ss.insert(QUuid::createUuid());
//    ss.insert(QUuid::createUuid());
//    ss.insert(QUuid::createUuid());
//    ss.insert(QUuid::createUuid());
//    ss.insert(QUuid::createUuid());

//    QHashEx<quint32, QHashDummyValue> qq;
//    qq.insert(1, QHashDummyValue());
//    qq.insert(2, QHashDummyValue());
//    qq.insert(2, QHashDummyValue());
//    qq.insert(5, QHashDummyValue());

//    for (auto it = qq.constBegin(); it != qq.constEnd(); ++it)
//        qDebug() << it.key();

//    QHashEx<int, QHashDummyValue> qqe;
//    qqe.insert(7, QHashDummyValue());
//    qqe.insert(8, QHashDummyValue());
//    qqe.insert(8, QHashDummyValue());
//    qqe.insert(9, QHashDummyValue());
//    qqe.insert(-1, QHashDummyValue());

//    qDebug() << qqe.contains(8);

//    return 0;

//    QSetEx<quint32> st;
//    st.insert(12);
//    st.insert(13);
//    st.insert(13);
//    st.insert(13);
//    st.insert(15);
//    st.insert(15);

//    qDebug() << st.contains(15);
//    qDebug() << st.contains(16);

//    QMap<quint64, int> hashMap;
//    hashMap[1]++;
//    hashMap[1]++;
//    for (int i = 0; i < 100000000; ++i)
//    {
//        quint64 h = hash64(QUuid::createUuid());
//        hashMap[h]++;
//        if (i % 1000000 == 0)
//            qDebug() << i;
//    }

//    for (auto it = hashMap.constBegin(); it != hashMap.constEnd(); ++it)
//        if (it.value() > 1)
//        {
//            qDebug() << it.key() << ":" << it.value();
//        }

//    communication::SResult s {true, 10};
//    communication::SResult s2 {s};
//    communication::SResult s3 = std::move(s);

//    QDate dd = QDate(1900, 1, 1);


//    QDateTime dt = QDateTime::fromMSecsSinceEpoch(-100);
//    QDateTime dt2 = QDateTime::fromMSecsSinceEpoch(qint64(2000));

//    QDate dd2 = dt2.date();

//    bool b1 = dt.isValid();
//    bool b2 = dd2.isValid();

//    const std::vector<int> *localVec = nullptr;
//    std::vector<int> vec = testNRVO(0, 10, &localVec);

//    if (&vec == localVec)
//        std::cout << "NRVO was applied" << std::endl;
//    else
//        std::cout << "NRVO was not applied" << std::endl;

//    return 0;


    // Устанавливаем в качестве разделителя целой и дробной части символ '.',
    // если этого не сделать - функции преобразования строк в числа (std::atof)
    // буду неправильно работать.
    qputenv("LC_NUMERIC", "C");

    // Пул потоков запустим после кода демонизации
    trd::threadPool().stop();

    // Именуем web-листенер
    tcp::listener().setName("web");

    int ret = 0;
    try
    {
        alog::logger().start();

#ifdef NDEBUG
        alog::logger().addSaverStdOut(alog::Level::Info, true);
#else
        alog::logger().addSaverStdOut(alog::Level::Debug2);
#endif
#ifdef MINGW
        if (!SetConsoleCtrlHandler(stopProgramHandler, TRUE))
        {
            log_error << "Could not set control handler";
            alog::logger().flush();
            stopLog();
            return 1;
        }
#else
        signal(SIGTERM, &stopProgramHandler);
        signal(SIGINT,  &stopProgramHandler);
#endif

        QDir homeDir = QDir::home();
        if (!homeDir.exists())
        {
            log_error << "Home dir " << homeDir.path() << " not exists";
            stopLog();
            return 1;
        }

        int c;
        bool isDaemon = true;
        while ((c = getopt(argc, argv, "nh")) != EOF)
        {
            switch (c)
            {
                case 'h':
                    helpInfo();
                    stopLog();
                    exit(0);
                case 'n':
                    isDaemon = false;
                    break;
                case '?':
                    log_error << "Invalid option";
                    stopLog();
                    return 1;
            }
        }

        // Путь к основному конфиг-файлу
        QString configFile = config::dir() + "/aisexpert.conf";
        if (!QFile::exists(configFile))
        {
            log_error << "Config file " << configFile << " not exists";
            stopLog();
            return 1;
        }

        config::base().setReadOnly(true);
        config::base().setSaveDisabled(true);
        if (!config::base().readFile(configFile.toStdString()))
        {
            stopLog();
            return 1;
        }

        QString configFileS;
#ifdef MINGW
        config::base().getValue("state.file_win", configFileS);
#else
        config::base().getValue("state.file", configFileS);
#endif
        config::dirExpansion(configFileS);
        config::state().readFile(configFileS.toStdString());

        QString logFile;
#ifdef MINGW
        config::base().getValue("logger.file_win", logFile);
        config::dirExpansion(logFile);
#else
        config::base().getValue("logger.file", logFile);
#endif
        config::dirExpansion(logFile);
        QFileInfo logFileInfo {logFile};
        QString logFileDir = logFileInfo.absolutePath();
        if (!QDir(logFileDir).exists())
            if (!QDir().mkpath(logFileDir))
            {
                log_error << "Failed create log directory: " << logFileDir;
                stopLog();
                return 1;
            }

        // Создаем дефолтный сэйвер для логгера
        {
            std::string logLevelStr = "info";
            config::base().getValue("logger.level", logLevelStr);

            bool logContinue = true;
            config::base().getValue("logger.continue", logContinue);

            alog::Level logLevel = alog::levelFromString(logLevelStr);
            alog::SaverPtr saver {new alog::SaverFile("default",
                                                      logFile.toStdString(),
                                                      logLevel,
                                                      logContinue)};
            alog::logger().addSaver(saver);
        }

        // Создаем eventlog сэйвер для логгера
        {
            alog::SaverPtr saver {new alog::EventLogSaver("eventlog",
                                                          alog::Level::Verbose)};
            alog::logger().addSaver(saver);
        }

        log_info << "'Ais Expert' service is running"
                 << " (version " << productVersion().toString() << ")";
        alog::logger().flush();

#ifndef MINGW
        if (isDaemon)
        {
            stopLog();
            alog::logger().removeSaverStdOut();
            alog::logger().removeSaverStdErr();

            if (daemon(1, 0) != 0)
                return 0;

            alog::logger().start();
            log_verbose << "Demonization success";
        }
#endif
        alog::logger().removeSaverStdOut();
        alog::logger().removeSaverStdErr();

        // Создаем дополнительные сэйверы для логгера
        initLoggerExt();

        if (!communication::command::checkUnique())
        {
            stopProgram();
            return 1;
        }

        if (!communication::error::checkUnique())
        {
            stopProgram();
            return 1;
        }

        // Пул потоков нужно активировать после кода демонизации
        trd::threadPool().start();

        Application appl {argc, argv};

        // Устанавливаем текущую директорию. Эта конструкция работает только
        // когда создан экземпляр QCoreApplication.
        if (QDir::setCurrent(QCoreApplication::applicationDirPath()))
        {
            log_debug << "Set work directory: " << QCoreApplication::applicationDirPath();
        }
        else
        {
            log_error << "Failed set work directory";
            stopProgram();
            return 1;
        }

        // Подключение к БД
        if (!db::firebird::setIgnoreSIGTERM())
        {
            stopProgram();
            return 1;
        }

        if (!dbpool().init())
        {
            stopProgram();
            return 1;
        }

        // Инициализацию Application делаем после подключения к БД
        if (!appl.init())
        {
            stopProgram();
            return 1;
        }

        QNetworkProxy::setApplicationProxy(QNetworkProxy::NoProxy);

        // Инициализация communication::listener::tcp
        QHostAddress hostAddress = QHostAddress::Any;
        config::readHostAddress("listener.address", hostAddress);

        int port = 62062;
        config::base().getValue("listener.port", port);
        if (!tcp::listener().init({hostAddress, port}))
        {
            stopProgram();
            return 1;
        }
        tcp::listener().setCompressionLevel(0);
        tcp::listener().setCheckProtocolCompatibility(false);

        chk_connect_q(&tcp::listener(), SIGNAL(message(communication::Message::Ptr)),
                      &appl, SLOT(message(communication::Message::Ptr)))

        chk_connect_d(&tcp::listener(), SIGNAL(message(communication::Message::Ptr)),
                      &appl, SLOT(messageTest(communication::Message::Ptr)))

        chk_connect_q(&tcp::listener(), SIGNAL(socketConnected(communication::SocketDescriptor)),
                      &appl, SLOT(webSocketConnected(communication::SocketDescriptor)))

        chk_connect_q(&tcp::listener(), SIGNAL(socketDisconnected(communication::SocketDescriptor)),
                      &appl, SLOT(webSocketDisconnected(communication::SocketDescriptor)))

        // Инициализация подключения к ФОМС
        hostAddress = QHostAddress::Any;
        config::readHostAddress("foms.address", hostAddress);

        port = 62065;
        config::base().getValue("foms.port", port);
        if (!fomsCon().socket()->init({hostAddress, port}))
        {
            stopProgram();
            return 1;
        }
        fomsCon().socket()->setCompressionLevel(0);
        fomsCon().socket()->setCheckProtocolCompatibility(false);
        fomsCon().socket()->setMessageFormat(SerializationFormat::Json);

        chk_connect_q(fomsCon().socket().get(), SIGNAL(message(communication::Message::Ptr)),
                      &appl, SLOT(message(communication::Message::Ptr)))

        chk_connect_q(fomsCon().socket().get(), SIGNAL(connected(communication::SocketDescriptor)),
                      &appl, SLOT(fomsSocketConnected(communication::SocketDescriptor)))

        chk_connect_q(fomsCon().socket().get(), SIGNAL(disconnected(communication::SocketDescriptor)),
                      &appl, SLOT(fomsSocketDisconnected(communication::SocketDescriptor)))

        // Инициализация подключения к UsersPool
        hostAddress = QHostAddress::Any;
        config::readHostAddress("users_pool.address", hostAddress);

        port = 62063;
        config::base().getValue("users_pool.port", port);
        if (!upoolCon()->init({hostAddress, port}))
        {
            stopProgram();
            return 1;
        }
        upoolCon()->setCompressionLevel(0);
        upoolCon()->setCheckProtocolCompatibility(false);
        upoolCon()->setMessageFormat(SerializationFormat::Json);

        chk_connect_q(upoolCon().get(), SIGNAL(message(communication::Message::Ptr)),
                      &appl, SLOT(message(communication::Message::Ptr)))

        chk_connect_q(upoolCon().get(), SIGNAL(connected(communication::SocketDescriptor)),
                      &appl, SLOT(upoolSocketConnected(communication::SocketDescriptor)))

        chk_connect_q(upoolCon().get(), SIGNAL(disconnected(communication::SocketDescriptor)),
                      &appl, SLOT(upoolSocketDisconnected(communication::SocketDescriptor)))

        if (!xgb::init())
        {
            stopProgram();
            return 1;
        }

        if (!uright().init())
        {
            stopProgram();
            return 1;
        }

        if (!task::scheduler().init())
        {
            stopProgram();
            return 1;
        }
        task::scheduler().start();

        if (!task::syncPlan().init())
        {
            stopProgram();
            return 1;
        }
        task::syncPlan().start();

        if (!task::sendScore().init())
        {
            stopProgram();
            return 1;
        }
        task::sendScore().start();

        if (!monitoring::observer().init())
        {
            stopProgram();
            return 1;
        }
        monitoring::observer().start();

        chk_connect_d(&task::syncPlan(), SIGNAL(commandProcessed(bool)),
                      &monitoring::observer(), SLOT(fomsCommandProcessed(bool)))

        chk_connect_d(&task::sendScore(), SIGNAL(commandProcessed(bool)),
                      &monitoring::observer(), SLOT(fomsCommandProcessed(bool)))

        if (!monitoring::notifier().init())
        {
            stopProgram();
            return 1;
        }
        monitoring::notifier().start();

        if (!monitoring::mailSmtp().init())
        {
            stopProgram();
            return 1;
        }
        monitoring::mailSmtp().start();

        if (!monitoring::httpConnector().init())
        {
            stopProgram();
            return 1;
        }

        if (Application::isStopped())
        {
            stopProgram();
            return 0;
        }

        appl.initCheckConnect();

        ret = appl.exec();
        appl.deinit();
    }
    catch (std::exception& e)
    {
        log_error << "Failed initialization. Detail: " << e.what();
        ret = 1;
    }
    catch (...)
    {
        log_error << "Failed initialization. Unknown error";
        ret = 1;
    }

    stopProgram();
    return ret;
}
