#include "aisexpert_appl.h"
#include "event_log.h"
#include "functions.h"
#include "user_rights.h"
#include "functions.h"
#include "garbage_collector.h"
#include "tasks/scheduler.h"
#include "tasks/sync_plan.h"
#include "tasks/send_score.h"
#include "xgboost/options.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "database/settings.h"
#include "monitoring/http_connector.h"
#include "monitoring/mail_smtp.h"
#include "monitoring/observer.h"

#include "commands/authorization.h"
#include "commands/commands.h"
#include "commands/error.h"
#include "commands/event_log.h"
#include "commands/group.h"
#include "commands/model.h"
#include "commands/monitor.h"
#include "commands/notify.h"
#include "commands/nsi.h"
#include "commands/paging.h"
#include "commands/score.h"
#include "commands/send_score.h"
#include "commands/sync_data.h"
#include "commands/task.h"
#include "commands/upool.h"
#include "commands/user.h"
#include "commands/version.h"
#include "commands/xgboost.h"

#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/config/config.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/logger_operators.h"
#include "shared/qt/communication/transport/base.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/version/version_number.h"
#include "shared/thread/thread_pool.h"
#include "shared/thread/thread_utils.h"

#include <unistd.h>
#include <sys/stat.h>
#include <string>
#include <thread>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Application")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Application")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Application")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Application")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Application")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Application")

using namespace db::firebird;
using namespace sql;

void initLoggerExt();

volatile bool Application::_stop = false;
std::atomic_int Application::_exitCode = {0};

std::time_t loggerConfModifyTime()
{
    QString logConf;
    std::time_t confTime = 0;
#ifdef MINGW
    config::base().getValue("logger.conf_win", logConf);
#else
    config::base().getValue("logger.conf", logConf);
#endif
    if (!logConf.isEmpty())
    {
        config::dirExpansion(logConf);
        if (QFile::exists(logConf))
        {
            struct stat st;
            ::stat(logConf.toUtf8().constData(), &st);
            confTime = st.st_mtime;
        }
    }
    return confTime;
}

Application::Application(int& argc, char** argv)
    : QCoreApplication(argc, argv)
{
    _stopTimerId = startTimer(1000);
    //_databaseTimerId = startTimer(5*60*1000 /*5 мин*/);

    struct stat st;
    ::stat(config::base().filePath().c_str(), &st);
    _configBaseModify = st.st_mtime;
    _configBaseTimerId = startTimer(15*1000 /*15 сек*/);
    _garbageTimerId = startTimer(1*60*60*1000 /*1 час*/);

    _configLoggerModify = loggerConfModifyTime();

    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &Application::command_##COMMAND, this);

    FUNC_REGISTRATION(UPoolAuthorization)
    FUNC_REGISTRATION(FomsAuthorization)
    FUNC_REGISTRATION(WebAuthorization)
    FUNC_REGISTRATION(UPoolUserAuth)
    FUNC_REGISTRATION(UserActiveStatus)
    FUNC_REGISTRATION(UserActiveSet)
    FUNC_REGISTRATION(UserResetAuth)
    FUNC_REGISTRATION(UserLogout)
    FUNC_REGISTRATION(UPoolChangesNotify)
    FUNC_REGISTRATION(GroupCreate)
    FUNC_REGISTRATION(GroupEdit)
    FUNC_REGISTRATION(GroupDelete)
    FUNC_REGISTRATION(GroupInfo)
    FUNC_REGISTRATION(GroupList)
    FUNC_REGISTRATION(AssignGroup)
    FUNC_REGISTRATION(UserInfo)
    FUNC_REGISTRATION(UserList)

    FUNC_REGISTRATION(ModelEdit)
    FUNC_REGISTRATION(ModelDelete)
    FUNC_REGISTRATION(ModelInfo)
    FUNC_REGISTRATION(ModelList)

    FUNC_REGISTRATION(ScoreEdit)
    FUNC_REGISTRATION(ScoreDelete)
    FUNC_REGISTRATION(ScoreInfo)
    FUNC_REGISTRATION(ScoreList)

    FUNC_REGISTRATION(NeedSendScore)

    FUNC_REGISTRATION(NsiVidmpList)
    FUNC_REGISTRATION(NsiProfileList)
    FUNC_REGISTRATION(NsiLpuList)
    FUNC_REGISTRATION(NsiMkbList)

    FUNC_REGISTRATION(ModelXgbDelete)
    FUNC_REGISTRATION(ModelXgbInfo)
    FUNC_REGISTRATION(ModelXgbEdit)
    FUNC_REGISTRATION(ModelXgbOption)
    FUNC_REGISTRATION(EventLogList)
    FUNC_REGISTRATION(AisVersion)

    FUNC_REGISTRATION(Monitoring)
    FUNC_REGISTRATION(MonitorThresholdInfo)
    FUNC_REGISTRATION(MonitorThresholdEdit)

    FUNC_REGISTRATION(SendMailDebug)

    FUNC_REGISTRATION(NotifyAddressCreate)
    FUNC_REGISTRATION(NotifyAddressEdit)
    FUNC_REGISTRATION(NotifyAddressDelete)
    FUNC_REGISTRATION(NotifyAddressInfo)
    FUNC_REGISTRATION(NotifyAddressList)

    FUNC_REGISTRATION(NotifyTriggerCreate)
    FUNC_REGISTRATION(NotifyTriggerEdit)
    FUNC_REGISTRATION(NotifyTriggerDelete)
    FUNC_REGISTRATION(NotifyTriggerInfo)
    FUNC_REGISTRATION(NotifyTriggerList)

    FUNC_REGISTRATION(SyncProgress)

    #undef FUNC_REGISTRATION

    _upoolRoute.commands.insert(command::UPoolUserList);
    _upoolRoute.commands.insert(command::UPoolUserAuth);
    _upoolRoute.point1.name = "Users Pool";
    _upoolRoute.point2.name = "Web Server";

//    _inputRoute.commands.insert(command::GetSyncData);
//    _inputRoute.point1.name = "Input socket";
//    _inputRoute.point2.name = "Ais Server";

}

bool Application::init()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT DB_VERSION FROM VERSION"))
        return false;

    if (!q.first())
        return false;

    int dbVersion = -1;
    assignValue(dbVersion, q.record(), "DB_VERSION");

    if (dbVersion != VERSION_DATABASE)
    {
        log_error_m << "Incompatible database versions"
                    << ". Version required: " << VERSION_DATABASE
                    << ". Current version: "  << dbVersion
                    << ". Need update database structure to version "
                    << VERSION_DATABASE;
        return false;
    }

    return true;
}

void Application::deinit()
{
    _threadIds.lock([](std::vector<pid_t>& tids) {
        for (pid_t tid : tids)
            dbpool().abortOperation(tid);
    });

    while (!_threadIds.empty())
        usleep(100*1000);
}

void Application::initCheckConnect()
{
    QMetaObject::invokeMethod(this, "checkFomsConnect",  Qt::QueuedConnection);
    QMetaObject::invokeMethod(this, "checkUpoolConnect", Qt::QueuedConnection);
}

void Application::timerEvent(QTimerEvent* event)
{
    if (event->timerId() == _stopTimerId)
    {
        if (_stop)
        {
            killTimer(_stopTimerId);
            killTimer(_fomsConTimerId);
            killTimer(_upoolConTimerId);
            //killTimer(_databaseTimerId);
            killTimer(_webAuthorizTimerId);
            killTimer(_configBaseTimerId);
            killTimer(_garbageTimerId);
            exit(_exitCode);
            return;
        }
    }
    else if (event->timerId() == _fomsConTimerId)
    {
        checkFomsConnect();
    }
    else if (event->timerId() == _upoolConTimerId)
    {
        checkUpoolConnect();
    }
    else if (event->timerId() == _webAuthorizTimerId)
    {
        checkWebAuthorization();
    }
    else if (event->timerId() == _configBaseTimerId)
    {
        struct stat st;
        ::stat(config::base().filePath().c_str(), &st);
        if (_configBaseModify != st.st_mtime)
        {
            _configBaseModify = st.st_mtime;
            config::base().rereadFile();
            log_verbose_m << "Config file was reread: " << config::base().filePath();

            killTimer(_fomsConTimerId);  _fomsConTimerId  = -1;
            killTimer(_upoolConTimerId); _upoolConTimerId = -1;

            xgb::init();
            initCheckConnect();
            monitoring::httpConnector().reinit();
            monitoring::mailSmtp().awake();
        }

        std::time_t configLoggerModify = loggerConfModifyTime();
        if (_configLoggerModify != configLoggerModify)
        {
            _configLoggerModify = configLoggerModify;
            initLoggerExt();
        }
    }
    else if (event->timerId() == _garbageTimerId)
    {
        if (!garbageCollector().isRunning())
            garbageCollector().start();
    }
}

void Application::stop(int exitCode)
{
    _exitCode = exitCode;
    stop();
}

void Application::message(const Message::Ptr& message)
{
    if (message->processed())
        return;

    if (lst::FindResult fr = _funcInvoker.findCommand(message->command()))
    {
        if (!command::pool().commandIsMultiproc(message->command()))
            message->markAsProcessed();
        _funcInvoker.call(message, fr);
    }

    if (message->processed())
        return;

    // Эта команда обрабатывается в command_UPoolUserAuth() поэтому
    // форвардить ее не нужно
    if (message->command() == command::UPoolUserAuth)
        return;

    _upoolRoute.forwarding(message);
}

void Application::messageTest(const communication::Message::Ptr& message)
{
    if (message->command() == command::SpeedTest)
    {
        data::SpeedTest speedTest;
        READ_FROM_MESSAGE(message, speedTest);
        speedTest.uuid = QUuidEx::createUuid();

        Message::Ptr answer = message->cloneForAnswer();
        writeToJsonMessage(speedTest, answer);

        webCon().send(answer);
    }
}

void Application::webSocketConnected(SocketDescriptor socketDescriptor)
{
    if (!webCon().isEmpty())
    {
        data::CloseConnection closeConnection;
        closeConnection.description = "Web-server is already connected";

        Message::Ptr m = createJsonMessage(closeConnection);
        m->destinationSocketDescriptors().insert(socketDescriptor);
        tcp::listener().send(m);
        return;
    }

    webCon().init(socketDescriptor);
    _webAuthorizTimerId = startTimer(1000);

    _upoolRoute.point2.socket = webCon().socket();
}

void Application::webSocketDisconnected(SocketDescriptor socketDescriptor)
{
    if (webCon().socketDescriptor() == socketDescriptor)
    {
        if (_webAuthorizTimerId != -1)
        {
            killTimer(_webAuthorizTimerId);
            _webAuthorizTimerId = -1;
        }
        _upoolRoute.point2.socket.reset();
        webCon().reset();
        uright().setWebHashId(0);
    }
}

void Application::fomsSocketConnected(SocketDescriptor /*socketDescriptor*/)
{
    log_verbose_m << EventLog(u8"Соединение c сервером ФОМС установлено");

    QString password;
    config::base().rereadFile();
    config::base().getValue("foms.password", password);

    if (!password.isEmpty())
    {
        data::FomsAuthorization fomsAuthorization;
        fomsAuthorization.password = password;

        Message::Ptr m = createJsonMessage(fomsAuthorization);
        fomsCon().socket()->send(m);
    }
}

void Application::fomsSocketDisconnected(SocketDescriptor /*socketDescriptor*/)
{
    fomsCon().setAuthorized(false);
    log_error_m << EventLog(u8"Соединение c сервером ФОМС потеряно");
}

void Application::upoolSocketConnected(SocketDescriptor /*socketDescriptor*/)
{
    log_verbose_m << EventLog(u8"Соединение с сервером 'Users Pool' установлено");

    QString password;
    config::base().rereadFile();
    config::base().getValue("users_pool.password", password);

    if (!password.isEmpty())
    {
        data::UPoolAuthorization upoolAuthorization;
        upoolAuthorization.password = password;

        Message::Ptr m = createJsonMessage(upoolAuthorization);
        upoolCon()->send(m);
    }
    _upoolRoute.point1.socket = upoolCon();
}

void Application::upoolSocketDisconnected(SocketDescriptor /*socketDescriptor*/)
{
    _upoolRoute.point1.socket.reset();
    log_error_m << EventLog(u8"Соединение c сервером 'Users Pool' потеряно");
}

void Application::checkFomsConnect()
{
    if (_fomsConTimerId == -1)
    {
        int interval = 60; /*в секундах*/
        config::base().getValue("foms.reconnect_timeout", interval);
        _fomsConTimerId = startTimer(interval * 1000);
    }

    if (!fomsCon().isConnected())
    {
        log_info_m << EventLog(u8"Попытка установки соединения с сервером ФОМС");
        fomsCon().socket()->connect();
    }
}

void Application::checkUpoolConnect()
{
    if (_upoolConTimerId == -1)
    {
        int interval = 60; /*в секундах*/
        config::base().getValue("users_pool.reconnect_timeout", interval);
        _upoolConTimerId = startTimer(interval * 1000);
    }

    if (!upoolCon()->isConnected())
    {
        log_info_m << EventLog(u8"Попытка установки соединения с сервером 'Users Pool'");
        upoolCon()->connect();
    }
}

void Application::validateUsers()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT SID, LOGIN, NAME, IS_ADMIN, IS_VALID FROM USERS"))
        return;

    data::UPoolValidUsers upoolValidUsers;
    while (q.next())
    {
        QSqlRecord r = q.record();
        data::UPoolUserInfo user;

        assignValue(user.sid,     r, "SID"      );
        assignValue(user.login,   r, "LOGIN"    );
        assignValue(user.name,    r, "NAME"     );
        assignValue(user.isAdmin, r, "IS_ADMIN" );
        assignValue(user.isValid, r, "IS_VALID" );

        upoolValidUsers.items.append(user);
    }

    Message::Ptr m = createJsonMessage(upoolValidUsers);
    writeToJsonMessage(upoolValidUsers, m);
    upoolCon()->send(m);
}

void Application::command_UPoolAuthorization(const Message::Ptr& message)
{
    if (message->type() == Message::Type::Answer)
    {
        if (message->execStatus() == Message::ExecStatus::Success)
        {
            log_verbose_m << EventLog(u8"Соединение c сервером 'Users Pool' авторизовано успешно");

            // После успешной авторизации, проверяем список пользователей из БД
            validateUsers();
        }
        else
        {
            log_error_m << EventLog(u8"Соединение c сервером 'Users Pool' не прошло авторизацию");
            //QString msg = errorDescription(message);
        }
    }
}

void Application::command_FomsAuthorization(const Message::Ptr& message)
{
    if (message->type() == Message::Type::Answer)
    {
        if (message->execStatus() == Message::ExecStatus::Success)
        {
            fomsCon().setAuthorized(true);
            log_verbose_m << EventLog(u8"Соединение c сервером ФОМС авторизовано успешно");

            //data::SyncCommandList commandList;

            //Message::Ptr m = createJsonMessage(commandList);

            //fomsCon().socket()->send(m);
        }
        else
        {
            fomsCon().setAuthorized(false);
            log_error_m << EventLog(u8"Соединение c сервером ФОМС не прошло авторизацию");
        }
    }
}

void Application::command_WebAuthorization(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Command)
        return;

    data::WebAuthorization webAuthorization;
    READ_FROM_MESSAGE(message, webAuthorization)

    QString password;
    config::base().rereadFile();
    config::base().getValue("listener.password", password);

    if (password == webAuthorization.password)
    {
        quint64 hashId = hash64(QUuidEx::createUuid());

        // Даем права администратора для hashId
        uright().setWebHashId(hashId);

        webCon().setAuthorized(true);
        killTimer(_webAuthorizTimerId);
        _webAuthorizTimerId = -1;

        // Отправляем подтверждение об успешной операции: пустое сообщение.
        // В поле tag(0) отправляем hashId, с эти hashId wed-сервер должен
        // обращаться к функциям системы
        Message::Ptr answer = message->cloneForAnswer();
        answer->setTag(hashId);
        webCon().send(answer);

        log_verbose_m << EventLog(u8"Web-сервер авторизован успешно")
                      << u8". Удаленный хост: " << message->sourcePoint();

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        sql::exec(q, "SELECT HASH_ID, IS_ACTIVE, IS_VALID FROM USERS");

        while (q.next())
        {
            QSqlRecord r = q.record();

            quint64 userHashId = 0;
            assignValue(userHashId, r, "HASH_ID");

            bool isActive = false;
            assignValue(isActive, r, "IS_ACTIVE");

            bool isValid = false;
            assignValue(isValid, r, "IS_VALID");

            if (!isActive || !isValid)
            {
                Message::Ptr m = createJsonMessage(command::UserResetAuth);
                m->setTag(userHashId);
                webCon().send(m);
            }
        }
        return;
    }

    // Если не прошли авторизацию
    Message::Ptr answer = message->cloneForAnswer();
    writeToJsonMessage(error::failed_web_authoriz, answer);
    webCon().send(answer);

    uright().setWebHashId(0);

    data::CloseConnection closeConnection;
    closeConnection.description = error::failed_web_authoriz.description;

    Message::Ptr m = createJsonMessage(closeConnection);
    webCon().send(m);

    log_error_m << EventLog(u8"Web-сервер не прошел авторизацию")
                << u8". Удаленный хост: " << message->sourcePoint()
                << u8". Соединение будет закрыто";
}

void Application::checkWebAuthorization()
{
    if (webCon().isConnected()
        && !webCon().isAuthorized()
        && webCon().waitingAuthorizExpired())
    {
        killTimer(_webAuthorizTimerId);
        _webAuthorizTimerId = -1;

        data::CloseConnection closeConnection;
        closeConnection.description = "Authorization timeout expired";

        Message::Ptr m = createJsonMessage(closeConnection);
        webCon().send(m);

        log_error_m << EventLog(u8"Время ожидания авторизации web-сервера истекло")
                    << u8". Удаленный хост: " << webCon().socket()->peerPoint()
                    << u8". Соединение будет закрыто";
    }
}

void Application::command_UPoolChangesNotify(const Message::Ptr& message)
{
    data::UPoolChangesNotify upoolChangesNotify;
    READ_FROM_MESSAGE(message, upoolChangesNotify);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QUuidEx defaultGroupId;
    if (sql::exec(q, "SELECT ID FROM GROUPS WHERE IS_DEFAULT = 1"))
    {
        if (q.first())
            assignValue(defaultGroupId, q.record(), "ID");
    }

    // Новые пользователи
    if (!upoolChangesNotify.newUsers.isEmpty())
    {
        QString fields =
            "  ID        "
            ", HASH_ID   "
            ", LOGIN     "
            ", NAME      "
            ", SID       "
            ", IS_ACTIVE "
            ", IS_VALID  "
            ", IS_ADMIN  ";

        if (!defaultGroupId.isNull())
            fields += ", GROUP_ID";

        QString sql = sql::INSERT_INTO("USERS", fields);

        if (q.prepare(sql))
        {
            for (const data::UPoolUserInfo& user : upoolChangesNotify.newUsers)
            {
                /*
                  Проверка логина пользователя на уникальность.
                */
                QSqlQuery q2 {dbcon->createResult()};
                if (sql::exec(q2, "SELECT ID FROM USERS WHERE LOGIN = ?", user.login))
                {
                    if (q2.first())
                    {
                        QUuidEx userId;
                        assignValue(userId, q2.record(), "ID");

                        if (userId != user.sid)
                        {
                            // Удалить логин пользователя, если для этого логина
                            // сменился SID.
                            sql::exec(q2, "DELETE FROM USERS WHERE ID = ?", userId);
                        }
                    }
                }

                if (user.login.trimmed().isEmpty())
                {
                    // Логин не должен быть пустым, разобраться.
                    break_point
                    log_error_m << "User login is empty";
                }

                QUuidEx uuid = QUuid::createUuid();
                bindValue(q, ":ID",        uuid );
                bindValue(q, ":HASH_ID",   hash64(uuid));
                bindValue(q, ":LOGIN",     user.login);
                bindValue(q, ":NAME",      user.name);
                bindValue(q, ":SID",       user.sid);
                bindValue(q, ":IS_ACTIVE", true); // по умолчанию - включен.
                bindValue(q, ":IS_VALID",  user.isValid);
                bindValue(q, ":IS_ADMIN",  user.isAdmin);

                if (!defaultGroupId.isNull())
                    bindValue(q, ":GROUP_ID", defaultGroupId);

                q.exec();
            }
        }
    }

    // Измененные пользователи
    for (const data::UPoolUserInfo& user : upoolChangesNotify.changeUsers)
    {
        if (sql::exec(q,
            "UPDATE USERS SET IS_VALID = ?, NAME = ?, IS_ADMIN = ? WHERE SID = ?",
            user.isValid, user.name, user.isAdmin, user.sid ))
        {
            if (!user.isValid)
            {
                sql::exec(q, "SELECT HASH_ID FROM USERS WHERE SID = ?", user.sid);
                if (q.first())
                {
                    quint64 hashId;
                    assignValue(hashId, q.record(), "HASH_ID");

                    Message::Ptr m = createJsonMessage(command::UserResetAuth);
                    m->setTag(hashId);
                    webCon().send(m);
                }
            }
        }
    }

    // Удаленные пользователи
    for (const QString& sid : upoolChangesNotify.deleteUsers)
    {
        if (sql::exec(q, "UPDATE USERS SET IS_VALID = 0 WHERE SID = ?", sid))
        {
            sql::exec(q, "SELECT HASH_ID FROM USERS WHERE SID = ?", sid);
            if (q.first())
            {
                quint64 hashId;
                assignValue(hashId, q.record(), "HASH_ID");

                Message::Ptr m = createJsonMessage(command::UserResetAuth);
                m->setTag(hashId);
                webCon().send(m);
            }
        }
    }

    uright().updatePermissions();
}

void Application::command_UPoolUserAuth(const Message::Ptr& message)
{
    if (message->type() == Message::Type::Command)
    {
        _upoolRoute.forwarding(message);
        return;
    }

    //--- Message::Type::Answer ---
    if (message->execStatus() != Message::ExecStatus::Success)
    {
        const char* msg = u8"Произошла техническая ошибка при аутентификации "
                          u8"пользователя. Детали: %1";
        QString err = errorDescription(message);
        log_error_m << EventLog(msg, err)
                    << ". Command: " << CommandNameLog(message->command());

        _upoolRoute.forwarding(message);
        return;
    }

    data::UPoolUserAuth upoolUserAuth;
    READ_FROM_MESSAGE(message, upoolUserAuth);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT      "
        "   ID        "
        "  ,HASH_ID   "
        "  ,IS_ACTIVE "
        " FROM        "
        "   USERS     "
        " WHERE       "
        "   SID = ?   ", upoolUserAuth.sid))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        _upoolRoute.forwarding(answer);
        return;
    }

    bool isNewUser = false;
    bool isActive = false;

    if (q.first())
    {
        QSqlRecord r = q.record();
        assignValue(upoolUserAuth.id,     r, "ID       ");
        assignValue(upoolUserAuth.hashId, r, "HASH_ID  ");
        assignValue(isActive,             r, "IS_ACTIVE");
        answer->setTag(upoolUserAuth.hashId);
    }
    else
    {
        /*
           Флаг isNewUser будет выставлен в true только в том случае, если пользователь
           введёт правильную пару логин+пароль и сделает это до выполнения функции
           validateUsres. После выполнения этой функции в БД уже будет пристуствовать
           данный пользователь. Будет выполнена только проверка на корректность
           учётных данных.
        */
        isNewUser = true;
        upoolUserAuth.id = QUuid::createUuid();
        upoolUserAuth.hashId = hash64(upoolUserAuth.id);
        answer->setTag(upoolUserAuth.hashId);
    }

    if (upoolUserAuth.status == false)
    {
        const char* msg = u8"Ошибка аутентификации пользователя. Логин: '%1'";
        alog::Line logLine = log_error_m << EventLog(msg, upoolUserAuth.login);
        if (!isNewUser)
            logLine << EventUser(upoolUserAuth.id);

        writeToJsonMessage(error::failed_user_auth.asFailed(), answer);
        _upoolRoute.forwarding(answer);
        return;
    }

    if (isNewUser)
    {
        if (upoolUserAuth.login.trimmed().isEmpty())
        {
            // Логин не должен быть пустым, разобраться.
            break_point
            log_error_m << "User login is empty";
        }

        if (q.prepare(
            " INSERT INTO USERS                                                        "
            " (ID, HASH_ID, LOGIN, NAME, SID, IS_ACTIVE, IS_VALID, IS_ADMIN, GROUP_ID) "
            " VALUES                                                                   "
            " (:ID, :HASH_ID, :LOGIN, :NAME, :SID, :IS_ACTIVE, :IS_VALID, :IS_ADMIN,   "
            "   (SELECT FIRST 1 ID FROM GROUPS WHERE IS_DEFAULT = 1)                   "
            " )                                                                        " ))
        {
            bindValue(q, ":ID       ", upoolUserAuth.id);
            bindValue(q, ":HASH_ID  ", upoolUserAuth.hashId);
            bindValue(q, ":LOGIN    ", upoolUserAuth.login);
            bindValue(q, ":NAME     ", upoolUserAuth.name);
            bindValue(q, ":SID      ", upoolUserAuth.sid);
            bindValue(q, ":IS_ACTIVE", true);
            bindValue(q, ":IS_VALID ", true);
            bindValue(q, ":IS_ADMIN ", upoolUserAuth.isAdmin);

            if (!q.exec())
            {
                writeToJsonMessage(error::insert_sql_statement, answer);
                _upoolRoute.forwarding(answer);
                return;
            }
        }
    }
    else
    {
        // Пользователь с привилегиями администратора может пройти аутентификацию
        // даже когда его учетная запись неактивна
        if (!isActive && !upoolUserAuth.isAdmin)
        {
            const char* msg = u8"Аутентификации пользователя запрещена. Логин: '%1'";
            log_error_m << EventLog(msg, upoolUserAuth.login)
                        << EventUser(upoolUserAuth.id);

            writeToJsonMessage(error::user_is_inactive, answer);
            _upoolRoute.forwarding(answer);
            return;
        }

        QString sql = (upoolUserAuth.isAdmin)
            ? " UPDATE USERS SET IS_VALID = 1, IS_ADMIN = 1, IS_ACTIVE = 1 "
              " WHERE SID = ?                                              "
            : " UPDATE USERS SET IS_VALID = 1, IS_ADMIN = 0                "
              " WHERE SID = ?                                              ";

        if (!sql::exec(q, sql, upoolUserAuth.sid))
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            _upoolRoute.forwarding(answer);
            return;
        }
    }

    sql::exec(q, "UPDATE USERS SET LAST_LOGON = ? WHERE ID = ?",
                 QDateTime::currentDateTime(),
                 upoolUserAuth.id);

    const char* msg = u8"Выполнена аутентификации пользователя. Логин: '%1'";
    log_verbose_m << EventLog(msg, upoolUserAuth.login)
                  << EventUser(upoolUserAuth.id);

    writeToJsonMessage(upoolUserAuth, answer);
    _upoolRoute.forwarding(answer);
}

void Application::command_UserActiveStatus(const Message::Ptr& message)
{
    data::UserActiveStatus userActiveStatus;
    READ_FROM_MESSAGE(message, userActiveStatus);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT IS_ACTIVE FROM USERS WHERE ID = ?", userActiveStatus.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (q.first())
        assignValue(userActiveStatus.value, q.record(), "IS_ACTIVE");
    else
        userActiveStatus.value = false;

    writeToJsonMessage(userActiveStatus, answer);
    webCon().send(answer);
}

void Application::command_UserActiveSet(const Message::Ptr& message)
{
    data::UserActiveSet userActiveSet;
    READ_FROM_MESSAGE(message, userActiveSet);

    quint64 userHashId = message->tag();
    Message::Ptr answer = message->cloneForAnswer();

    if (!uright().isAdmin(userHashId))
    {
        writeToJsonMessage(error::admin_privileges, answer);
        webCon().send(answer);
        return;
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "UPDATE USERS SET IS_ACTIVE = ? WHERE ID = ?",
                      userActiveSet.value, userActiveSet.id))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    // Обновление прав доступа
    uright().updatePermissions(userHashId);

    writeToJsonMessage(userActiveSet, answer);
    webCon().send(answer);

    // Отправляем запрос на аннулирование сессий пользователя
    if (!userActiveSet.value)
    {
        Message::Ptr m = createJsonMessage(command::UserResetAuth);
        m->setTag(hash64(userActiveSet.id));
        webCon().send(m);
    }
}

void Application::command_UserResetAuth(const Message::Ptr& message)
{
    log_verbose_m << EventLog(u8"Сессия пользователя аннулирована")
                  << EventUser(message->tag());
}

void Application::command_UserLogout(const Message::Ptr& message)
{
    log_verbose_m << EventLog(u8"Сессия пользователя завершена")
                  << EventUser(message->tag());

    Message::Ptr answer = message->cloneForAnswer();
    webCon().send(answer);
}

void Application::command_Monitoring(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();
    data::Monitoring::Ptr monitoring = monitoring::observer().get();

    writeToJsonMessage(*monitoring, answer);
    webCon().send(answer);
}

void Application::command_MonitorThresholdInfo(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();
    data::MonitorThresholdInfo monitorThresholdInfo;

    monitorThresholdInfo.cpu.warn = 80;
    monitorThresholdInfo.cpu.crit = 95;
    db::settings::getValue("monitoring.threshold.cpu.warn", monitorThresholdInfo.cpu.warn);
    db::settings::getValue("monitoring.threshold.cpu.crit", monitorThresholdInfo.cpu.crit);

    monitorThresholdInfo.ram.warn = 70;
    monitorThresholdInfo.ram.crit = 90;
    db::settings::getValue("monitoring.threshold.ram.warn", monitorThresholdInfo.ram.warn);
    db::settings::getValue("monitoring.threshold.ram.crit", monitorThresholdInfo.ram.crit);

    monitorThresholdInfo.hdd.warn = 65;
    monitorThresholdInfo.hdd.crit = 85;
    db::settings::getValue("monitoring.threshold.hdd.warn", monitorThresholdInfo.hdd.warn);
    db::settings::getValue("monitoring.threshold.hdd.crit", monitorThresholdInfo.hdd.crit);

    monitorThresholdInfo.cputmp.warn = 50;
    monitorThresholdInfo.cputmp.crit = 75;
    db::settings::getValue("monitoring.threshold.cputmp.warn", monitorThresholdInfo.cputmp.warn);
    db::settings::getValue("monitoring.threshold.cputmp.crit", monitorThresholdInfo.cputmp.crit);

    writeToJsonMessage(monitorThresholdInfo, answer);
    webCon().send(answer);
}

void Application::command_MonitorThresholdEdit(const Message::Ptr& message)
{
    data::MonitorThresholdEdit monitorThresholdEdit;
    READ_FROM_MESSAGE(message, monitorThresholdEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    db::settings::setValue(q, "monitoring.threshold.cpu.warn", monitorThresholdEdit.cpu.warn);
    db::settings::setValue(q, "monitoring.threshold.cpu.crit", monitorThresholdEdit.cpu.crit);

    db::settings::setValue(q, "monitoring.threshold.ram.warn", monitorThresholdEdit.ram.warn);
    db::settings::setValue(q, "monitoring.threshold.ram.crit", monitorThresholdEdit.ram.crit);

    db::settings::setValue(q, "monitoring.threshold.hdd.warn", monitorThresholdEdit.hdd.warn);
    db::settings::setValue(q, "monitoring.threshold.hdd.crit", monitorThresholdEdit.hdd.crit);

    db::settings::setValue(q, "monitoring.threshold.cputmp.warn", monitorThresholdEdit.cputmp.warn);
    db::settings::setValue(q, "monitoring.threshold.cputmp.crit", monitorThresholdEdit.cputmp.crit);

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    // В случае успеха отправляем назад пустой ответ
    webCon().send(answer);

    // Рассылаем уведомление об изменении настроек
    data::MonitorThresholdInfo monitorThresholdInfo;
    static_cast<data::MonitorThreshold&>(monitorThresholdInfo) = monitorThresholdEdit;
    Message::Ptr m = createJsonMessage(monitorThresholdInfo, Message::Type::Event);
    webCon().send(m);
}

void Application::command_SendMailDebug(const Message::Ptr& message)
{
    data::SendMailDebug sendMailDebug;
    READ_FROM_MESSAGE(message, sendMailDebug);

    quint64 userHashId = message->tag();
    Message::Ptr answer = message->cloneForAnswer();

    if (!uright().isWebAppl(userHashId))
    {
        writeToJsonMessage(error::webapp_privileges, answer);
        webCon().send(answer);
        return;
    }

    typedef monitoring::MailSmtp::MailData MailData;
    MailData::Ptr mail {new MailData({"cpu", QUuidEx(),
                                      sendMailDebug.addresses,
                                      sendMailDebug.subject,
                                      sendMailDebug.body})};
    monitoring::mailSmtp().send(mail);
    monitoring::mailSmtp().awake();

    webCon().send(answer);
}

static bool notifyAddressModify(const data::NotifyAddress& address, Message::Ptr& answer)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        return false;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT ID FROM NOTIFY_ADDRESS WHERE TRIM(UPPER(EMAIL)) = ?",
        address.email.toUpper().trimmed() ))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        return false;
    }
    if (q.first())
    {
        QUuidEx id;
        assignValue(id, q.record(), "ID");
        if (address.id != id)
        {
            writeToJsonMessage(error::notify_address_not_unique, answer);
            return false;
        }
    }

    QString fields = "ID, NAME, EMAIL";
    QString sql = sql::UPDATE_OR_INSERT_INTO("NOTIFY_ADDRESS", fields, "ID");

    if (!q.prepare(sql))
    {
        writeToJsonMessage(error::insert_or_update_sql, answer);
        return false;
    }

    bindValue(q, ":ID    ", address.id    );
    bindValue(q, ":NAME  ", address.name  );
    bindValue(q, ":EMAIL ", address.email );

    if (!q.exec())
    {
        writeToJsonMessage(error::insert_or_update_sql, answer);
        return false;
    }

    if (!sql::exec(q,
        "DELETE FROM NOTIFY_ADDRESS_LINK WHERE ADDRESS_ID = ?", address.id))
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        return false;
    }

    if (address.triggers.count())
    {
        fields = "ADDRESS_ID, TRIGGER_ID";
        sql = sql::INSERT_INTO("NOTIFY_ADDRESS_LINK", fields);

        if (!q.prepare(sql))
        {
            writeToJsonMessage(error::insert_sql_statement, answer);
            return false;
        }

        for (const QUuidEx& triggerId : address.triggers)
        {
            bindValue(q, ":ADDRESS_ID", address.id );
            bindValue(q, ":TRIGGER_ID", triggerId );

            if (!q.exec())
            {
                writeToJsonMessage(error::insert_sql_statement, answer);
                return false;
            }
        }
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        return false;
    }
    return true;
}

void Application::command_NotifyAddressCreate(const Message::Ptr& message)
{
    data::NotifyAddressCreate notifyAddressCreate;
    READ_FROM_MESSAGE(message, notifyAddressCreate);

    Message::Ptr answer = message->cloneForAnswer();

    notifyAddressCreate.id = QUuidEx::createUuid();
    if (!notifyAddressModify(notifyAddressCreate, answer))
    {
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Создан адресат отправки уведомлений '%1'";
    log_verbose_m << EventLog(msg, notifyAddressCreate.name)
                  << EventUser(message->tag());

    writeToJsonMessage(notifyAddressCreate, answer);
    webCon().send(answer);
}

void Application::command_NotifyAddressEdit(const Message::Ptr& message)
{
    data::NotifyAddressEdit notifyAddressEdit;
    READ_FROM_MESSAGE(message, notifyAddressEdit);

    Message::Ptr answer = message->cloneForAnswer();

    if (!notifyAddressModify(notifyAddressEdit, answer))
    {
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Изменен адресат отправки уведомлений '%1'";
    log_verbose_m << EventLog(msg, notifyAddressEdit.name)
                  << EventUser(message->tag());

    writeToJsonMessage(notifyAddressEdit, answer);
    webCon().send(answer);
}

void Application::command_NotifyAddressDelete(const Message::Ptr& message)
{
    data::NotifyAddressDelete notifyAddressDetete;
    READ_FROM_MESSAGE(message, notifyAddressDetete);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT NAME FROM NOTIFY_ADDRESS WHERE ID = ?", notifyAddressDetete.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    QString addressName;
    assignValue(addressName, q.record(), "NAME");

    if (!sql::exec(q,
        "DELETE FROM NOTIFY_ADDRESS WHERE ID = ?", notifyAddressDetete.id))
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!sql::exec(q,
        "DELETE FROM NOTIFY_ADDRESS_LINK WHERE ADDRESS_ID = ?", notifyAddressDetete.id))
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Удален адресат отправки уведомлений '%1'";
    log_verbose_m << EventLog(msg, addressName)
                  << EventUser(message->tag());

    writeToJsonMessage(notifyAddressDetete, answer);
    webCon().send(answer);
}

static bool notifyAddressList(const QUuidEx& addressId, QVector<data::NotifyAddress>& addresses)
{
    addresses.clear();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q  {db::firebird::createResult(transact)};
    QSqlQuery q2 {db::firebird::createResult(transact)};

    if (!transact->begin())
        return false;

    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    QString sql =
        " SELECT           "
        "   ID             "
        "  ,NAME           "
        "  ,EMAIL          "
        " FROM             "
        "   NOTIFY_ADDRESS "
        " WHERE            "
        "   (1 = 1)        ";

    if (!addressId.isNull())
        sql += " AND ID = :ID ";

    if (!q.prepare(sql))
        return false;

    if (!addressId.isNull())
        bindValue(q, ":ID", addressId);

    if (!q.exec())
        return false;

    while (q.next())
    {
        QSqlRecord r = q.record();
        data::NotifyAddress address;

        assignValue(address.id    , r, "ID    ");
        assignValue(address.name  , r, "NAME  ");
        assignValue(address.email , r, "EMAIL ");

        sql::exec(q2, " SELECT TRIGGER_ID FROM NOTIFY_ADDRESS_LINK "
                      " WHERE ADDRESS_ID = ?", address.id);
        while (q2.next())
        {
            QUuidEx triggerId;
            assignValue(triggerId, q2.record(), "TRIGGER_ID");

            address.triggers.append(triggerId);
        }
        addresses.append(address);
    }

    transact->rollback();
    return true;
}

void Application::command_NotifyAddressInfo(const Message::Ptr& message)
{
    data::NotifyAddressInfo notifyAddressInfo;
    READ_FROM_MESSAGE(message, notifyAddressInfo);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::NotifyAddress> addresses;
    if (!::notifyAddressList(notifyAddressInfo.id, addresses))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (addresses.isEmpty())
    {
        writeToJsonMessage(error::notify_address_not_found, answer);
        webCon().send(answer);
        return;
    }
    data::NotifyAddressInfoA notifyAddressInfoA;
    //*(static_cast<data::Group*>(&groupInfoA)) = groups[0];
    static_cast<data::NotifyAddress&>(notifyAddressInfoA) = addresses[0];

    writeToJsonMessage(notifyAddressInfoA, answer);
    webCon().send(answer);
}

void Application::command_NotifyAddressList(const Message::Ptr& message)
{
    data::NotifyAddressList notifyAddressList;
    READ_FROM_MESSAGE(message, notifyAddressList);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::NotifyAddress> addresses;
    if (!::notifyAddressList(QUuidEx(), addresses))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    notifyAddressList.items = addresses;
    writeToJsonMessage(notifyAddressList, answer);
    webCon().send(answer);
}

static bool notifyTriggerModify(const data::NotifyTrigger& trigger, Message::Ptr& answer)
{
    static QSet<QString> triggerTypes;
    if (triggerTypes.isEmpty())
        triggerTypes << "cpu" << "ram" << "hdd" << "cputmp" << "foms" << "evlog";

    if (!triggerTypes.contains(trigger.type))
    {
        writeToJsonMessage(error::notify_unknown_trigger_type, answer);
        return false;
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QString fields =
        "  ID                 "
        ", NAME               "
        ", TRIGGER_TYPE       "
        ", THRESHOLD          "
        ", EMISSION_INTERVAL  "
        ", EMISSION_NEXT      "
        ", SUBJECT            "
        ", MESSAGE_TEMPLATE   ";

    QString sql = sql::UPDATE_OR_INSERT_INTO("NOTIFY_TRIGGER", fields, "ID");

    if (!q.prepare(sql))
    {
        writeToJsonMessage(error::insert_or_update_sql, answer);
        return false;
    }

    QDateTime emissionNext =
        QDateTime::currentDateTime().addSecs(trigger.emissionInterval * 60);

    bindValue(q, ":ID                ", trigger.id               );
    bindValue(q, ":NAME              ", trigger.name             );
    bindValue(q, ":TRIGGER_TYPE      ", trigger.type             );
    bindValue(q, ":THRESHOLD         ", trigger.threshold        );
    bindValue(q, ":EMISSION_INTERVAL ", trigger.emissionInterval );
    bindValue(q, ":EMISSION_NEXT     ", emissionNext             );
    bindValue(q, ":SUBJECT           ", trigger.subject          );
    bindValue(q, ":MESSAGE_TEMPLATE  ", trigger.messageTemplate  );

    if (!q.exec())
    {
        writeToJsonMessage(error::insert_or_update_sql, answer);
        return false;
    }
    return true;
}

void Application::command_NotifyTriggerCreate(const Message::Ptr& message)
{
    data::NotifyTriggerCreate notifyTriggerCreate;
    READ_FROM_MESSAGE(message, notifyTriggerCreate);

    Message::Ptr answer = message->cloneForAnswer();

    if (notifyTriggerCreate.type == "foms")
    {
        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            "SELECT COUNT(*) FROM NOTIFY_TRIGGER WHERE TRIGGER_TYPE = 'foms'"))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        q.first();
        if (q.value(0).toInt() != 0)
        {
            writeToJsonMessage(error::notify_foms_already_exists, answer);
            webCon().send(answer);
            return;
        }
    }
    if (notifyTriggerCreate.type == "evlog")
    {
        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            "SELECT COUNT(*) FROM NOTIFY_TRIGGER WHERE TRIGGER_TYPE = 'evlog'"))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        q.first();
        if (q.value(0).toInt() != 0)
        {
            writeToJsonMessage(error::notify_evlog_already_exists, answer);
            webCon().send(answer);
            return;
        }
    }

    notifyTriggerCreate.id = QUuidEx::createUuid();
    if (!notifyTriggerModify(notifyTriggerCreate, answer))
    {
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Создан триггер уведомления '%1'", notifyTriggerCreate.name)
                  << EventUser(message->tag());

    writeToJsonMessage(notifyTriggerCreate, answer);
    webCon().send(answer);
}

void Application::command_NotifyTriggerEdit(const Message::Ptr& message)
{
    data::NotifyTriggerEdit notifyTriggerEdit;
    READ_FROM_MESSAGE(message, notifyTriggerEdit);

    Message::Ptr answer = message->cloneForAnswer();

    if (!notifyTriggerModify(notifyTriggerEdit, answer))
    {
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Изменен триггер уведомления '%1'", notifyTriggerEdit.name)
                  << EventUser(message->tag());

    writeToJsonMessage(notifyTriggerEdit, answer);
    webCon().send(answer);
}

void Application::command_NotifyTriggerDelete(const Message::Ptr& message)
{
    data::NotifyTriggerDelete notifyTriggerDelete;
    READ_FROM_MESSAGE(message, notifyTriggerDelete);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT NAME FROM NOTIFY_TRIGGER WHERE ID = ?", notifyTriggerDelete.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    QString triggerName;
    assignValue(triggerName, q.record(), "NAME");

    if (!sql::exec(q,
        "DELETE FROM NOTIFY_TRIGGER WHERE ID = ?", notifyTriggerDelete.id))
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!sql::exec(q,
        "DELETE FROM NOTIFY_ADDRESS_LINK WHERE TRIGGER_ID = ?", notifyTriggerDelete.id))
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Удален триггер уведомления '%1'", triggerName)
                  << EventUser(message->tag());

    writeToJsonMessage(notifyTriggerDelete, answer);
    webCon().send(answer);
}

static bool notifyTriggerList(const QUuidEx& triggerId, QVector<data::NotifyTrigger>& triggers)
{
    triggers.clear();

    QString sql =
        " SELECT               "
        "   ID                 "
        "  ,NAME               "
        "  ,TRIGGER_TYPE       "
        "  ,THRESHOLD          "
        "  ,EMISSION_INTERVAL  "
        "  ,SUBJECT            "
        "  ,MESSAGE_TEMPLATE   "
        " FROM                 "
        "   NOTIFY_TRIGGER     "
        " WHERE                "
        "   (1 = 1)            ";

    if (!triggerId.isNull())
        sql += " AND ID = :TRIGGER_ID ";

    sql += " ORDER BY TRIGGER_TYPE, NAME ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
        return false;

    if (!triggerId.isNull())
        bindValue(q, ":TRIGGER_ID", triggerId);

    if (!q.exec())
        return false;

    while (q.next())
    {
        QSqlRecord r = q.record();
        data::NotifyTrigger trigger;

        assignValue(trigger.id              , r, "ID                ");
        assignValue(trigger.name            , r, "NAME              ");
        assignValue(trigger.type            , r, "TRIGGER_TYPE      ");
        assignValue(trigger.threshold       , r, "THRESHOLD         ");
        assignValue(trigger.emissionInterval, r, "EMISSION_INTERVAL ");
        assignValue(trigger.subject         , r, "SUBJECT           ");
        assignValue(trigger.messageTemplate , r, "MESSAGE_TEMPLATE  ");

        triggers.append(trigger);
    }
    return true;
}

void Application::command_NotifyTriggerInfo(const Message::Ptr& message)
{
    data::NotifyTriggerInfo notifyTriggerInfo;
    READ_FROM_MESSAGE(message, notifyTriggerInfo);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::NotifyTrigger> triggers;
    if (!::notifyTriggerList(notifyTriggerInfo.id, triggers))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (triggers.isEmpty())
    {
        writeToJsonMessage(error::notify_trigger_not_found, answer);
        webCon().send(answer);
        return;
    }
    data::NotifyTriggerInfoA notifyTriggerInfoA;
    //*(static_cast<data::Group*>(&groupInfoA)) = groups[0];
    static_cast<data::NotifyTrigger&>(notifyTriggerInfoA) = triggers[0];

    writeToJsonMessage(notifyTriggerInfoA, answer);
    webCon().send(answer);
}

void Application::command_NotifyTriggerList(const Message::Ptr& message)
{
    data::NotifyTriggerList notifyTriggerList;
    READ_FROM_MESSAGE(message, notifyTriggerList);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::NotifyTrigger> triggers;
    if (!::notifyTriggerList(QUuidEx(), triggers))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    notifyTriggerList.items = triggers;
    writeToJsonMessage(notifyTriggerList, answer);
    webCon().send(answer);
}

template<typename T>
static bool contentEdit(const QString& tableName, T& content, bool isAdmin)
{
    QString sql =
        " UPDATE %1 SET                "
        "   NAME        = :NAME        "
        "  ,DESCRIPTION = :DESCRIPTION "
        "  ,IS_PUBLIC   = :IS_PUBLIC   "
        " WHERE                        "
        "   ID = :ID                   ";

    if (!isAdmin)
        sql += " AND USER_ID = :USER_ID ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql.arg(tableName)))
        return false;

    bindValue(q, ":ID"          , content.id          );
    bindValue(q, ":NAME"        , content.name        );
    bindValue(q, ":DESCRIPTION" , content.description );
    bindValue(q, ":IS_PUBLIC"   , content.isPublic    );

    if (!isAdmin)
        bindValue(q, ":USER_ID", content.userId);

    return q.exec();
}

static void contentSqlSpecial(QString&, const data::ModelFilter&)
{}

static void contentBindSpecial(QSqlQuery&, const data::ModelFilter&)
{}

static void contentListSpecial(data::Model&, const QSqlRecord&)
{}

static void contentSqlSpecial(QString& sql, const data::ScoreFilter& filter)
{
    if (!filter.modelId.isNull())
        sql += " AND MODEL_ID = :MODEL_ID ";
}

static void contentBindSpecial(QSqlQuery& q, const data::ScoreFilter& filter)
{
    if (!filter.modelId.isNull())
        bindValue(q, ":MODEL_ID", filter.modelId);
}

static void contentListSpecial(data::Score& content, const QSqlRecord& r)
{
    assignValue(content.modelId   , r, "MODEL_ID    ");
    assignValue(content.sendStatus, r, "SEND_STATUS ");
    assignValue(content.taskName  , r, "TASK_NAME   ");
    assignValue(content.modelName , r, "MODEL_NAME  ");
}

template<typename T, typename Filter>
static bool contentList(const QString& tableName,
                        data::PagingInfo& paging,
                        const Filter& filter,
                        QVector<T>& list)
{
    list.clear();

    QString sql =
        " SELECT                       "
        "   %1                         "
        " FROM                         "
        "   %2 C                       "
        " LEFT JOIN                    "
        "   TASK T ON T.ID = C.TASK_ID "
        " %3                           "
        " WHERE (1 = 1)                ";

    //-- tableName == "MODEL" --
    /* %1 */
    QString fieds =
        "  C.ID                 "
        ", C.TASK_ID            "
        ", C.USER_ID            "
        ", C.NAME               "
        ", C.DESCRIPTION        "
        ", C.CREATE_DATE        "
        ", C.PERIOD_BEGIN       "
        ", C.PERIOD_END         "
      //", C.PERIOD_BEGIN_REAL  "
      //", C.PERIOD_END_REAL    "
        ", C.IS_PUBLIC          "
        ", C.IS_ARCHIVE         "
        ", C.OUTDATED_ROWS      "
        ", T.IS_PERIODIC        ";

    QString join;
    if (tableName == "SCORE")
    {
        /* %1 */
        fieds +=
        ", C.MODEL_ID           "
        ", C.SEND_STATUS        "
        ", T.NAME AS TASK_NAME  "
        ", M.NAME AS MODEL_NAME ";

        /* %3 */
        join =
        " LEFT JOIN                      "
        "   MODEL M ON M.ID = C.MODEL_ID ";
    }
    //            %1         %2             %3
    sql = sql.arg(fieds).arg(tableName).arg(join);

    if (!filter.id.isNull())
        sql += " AND C.ID = :ID ";

    if (!filter.taskId.isNull())
        sql += " AND C.TASK_ID = :TASK_ID ";

    if (!filter.userId.isNull())
    {
        if (filter.show == data::filter::Show::Self)
            sql += " AND C.USER_ID = :USER_ID ";

        else if (filter.show == data::filter::Show::Other)
            sql += " AND (C.USER_ID != :USER_ID AND C.IS_PUBLIC = 1) ";

        else if (filter.show == data::filter::Show::All)
            sql += " AND (C.USER_ID = :USER_ID OR C.IS_PUBLIC = 1) ";
    }

    if (filter.archive == data::filter::Archive::No
        || filter.archive == data::filter::Archive::Yes)
    {
        sql += " AND C.IS_ARCHIVE = %1 ";
        sql = sql.arg((filter.archive == data::filter::Archive::Yes) ? "1" : "0");
    }

    if (!filter.name.isEmpty())
        sql += " AND C.NAME CONTAINING :NAME ";

    if (filter.createDate.begin.isValid())
        sql += " AND C.CREATE_DATE >= :CREATE_DATE_BEGIN ";

    if (filter.createDate.end.isValid())
        sql += " AND C.CREATE_DATE <= :CREATE_DATE_END ";

    if (filter.dataPeriod.begin.isValid() && filter.dataPeriod.end.isValid())
        // a.start <= b.end AND a.end >= b.start
        sql += " AND C.PERIOD_BEGIN <= :PERIOD_END AND C.PERIOD_END >= :PERIOD_BEGIN ";

    sql += " ORDER BY             "
           "   T.IS_PERIODIC DESC "
           "  ,C.CREATE_DATE DESC ";

    contentSqlSpecial(sql, filter);
    pagingPrepare(paging, sql);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
        return false;

    if (!filter.id.isNull())
        bindValue(q, ":ID", filter.id);

    if (!filter.taskId.isNull())
        bindValue(q, ":TASK_ID", filter.taskId);

    if (!filter.userId.isNull())
        bindValue(q, ":USER_ID", filter.userId);

    if (!filter.name.isEmpty())
        bindValue(q, ":NAME", filter.name);

    if (filter.createDate.begin.isValid())
        bindValue(q, ":CREATE_DATE_BEGIN", filter.createDate.begin);

    if (filter.createDate.end.isValid())
        bindValue(q, ":CREATE_DATE_END", filter.createDate.end);

    if (filter.dataPeriod.begin.isValid() && filter.dataPeriod.end.isValid())
    {
        bindValue(q, ":PERIOD_BEGIN", filter.dataPeriod.begin);
        bindValue(q, ":PERIOD_END", filter.dataPeriod.end);
    }

    contentBindSpecial(q, filter);

    if (!q.exec())
        return false;

    while (q.next())
    {
        T content;
        QSqlRecord r = q.record();

        assignValue(content.userId, r, "USER_ID");

        if (filter.show == data::filter::Show::Other
            || filter.show == data::filter::Show::All)
        {
            if (filter.userId != content.userId
                && !filter.otherUsers.isEmpty()
                && !filter.otherUsers.contains(content.userId))
                continue;
        }

        assignValue(content.id               , r, "ID                ");
        assignValue(content.taskId           , r, "TASK_ID           ");
        assignValue(content.name             , r, "NAME              ");
        assignValue(content.description      , r, "DESCRIPTION       ");
        assignValue(content.createDate       , r, "CREATE_DATE       ");
        assignValue(content.period.begin     , r, "PERIOD_BEGIN      ");
        assignValue(content.period.end       , r, "PERIOD_END        ");
      //assignValue(content.periodReal.begin , r, "PERIOD_BEGIN_REAL ");
      //assignValue(content.periodReal.end   , r, "PERIOD_END_REAL   ");
        assignValue(content.isPublic         , r, "IS_PUBLIC         ");
        assignValue(content.isArchive        , r, "IS_ARCHIVE        ");
        assignValue(content.isPeriodic       , r, "IS_PERIODIC       ");

        contentListSpecial(content, r);
        list.append(content);
    }

    paging.total = q.size();
    return true;
}

void Application::command_ModelEdit(const Message::Ptr& message)
{
    data::ModelEdit modelEdit;
    READ_FROM_MESSAGE(message, modelEdit)

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != modelEdit.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    if (!contentEdit("MODEL", modelEdit, isAdmin))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);

        log_error_m << EventLog(u8"Ошибка редактирования модели '%1'", modelEdit.name)
                    << EventContent(modelEdit.id)
                    << EventUser(message->tag());
        return;
    }
    log_verbose_m << EventLog(u8"Модель '%1' изменена", modelEdit.name)
                  << EventContent(modelEdit.id)
                  << EventUser(message->tag());

    writeToJsonMessage(modelEdit, answer);
    webCon().send(answer);
}

void Application::command_ModelDelete(const Message::Ptr& message)
{
    data::ModelDelete modelDelete;
    READ_FROM_MESSAGE(message, modelDelete)

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != modelDelete.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!modelDelete.force)
    {
        // Если модель указана в качестве родительской для score задачи, то такую
        // модель удалить нельзя.
        // На данный момент выключения задач 'score_calc' при удалении не происходит,
        // так как данная проверка не позволит удалить модель, которая является
        // родительской для 'score_calc' задачи.
        if (!sql::exec(q,
            "SELECT ID FROM TASK WHERE TASK_TYPE = ? AND MODEL_ID = ?",
            getTaskString(TaskType::ScoreCalc), modelDelete.id))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        // Если запрос вернул данные, значит подчиненные задачи присуствуют
        if (q.first())
        {
            writeToJsonMessage(error::model_has_child, answer);
            webCon().send(answer);
            return;
        }
    }

    QString modelName;
    if (!sql::exec(q, "SELECT NAME FROM MODEL WHERE ID = ?", modelDelete.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    assignValue(modelName, q.record(), "NAME");

    bool sqlRes = false;
    if (isAdmin)
    {
        sqlRes = sql::exec(q,
            "UPDATE MODEL SET IS_ARCHIVE = 1 WHERE ID = ?", modelDelete.id);
    }
    else
    {
        sqlRes = sql::exec(q,
            "UPDATE MODEL SET IS_ARCHIVE = 1 WHERE ID = ? AND USER_ID = ?",
            modelDelete.id, modelDelete.userId);
    }
    if (!sqlRes)
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Модель '%1' удалена", modelName)
                  << EventContent(modelDelete.id)
                  << EventUser(message->tag());

    writeToJsonMessage(modelDelete, answer);
    webCon().send(answer);
}

void Application::validateScores(const QUuidEx& modelId, QList<QUuidEx>& scores)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT ID FROM TASK WHERE MODEL_ID = ?", modelId))
        return;

    while (q.next())
    {
        QSqlRecord r = q.record();

        QUuidEx scoreId;
        assignValue(scoreId, q.record(), "ID");
        scores.append(scoreId);
   }
}

void Application::command_ModelInfo(const Message::Ptr& message)
{
    data::ModelInfo modelInfo;
    READ_FROM_MESSAGE(message, modelInfo)

    Message::Ptr answer = message->cloneForAnswer();

    data::PagingInfo paging;
    data::ModelFilter filter;
    filter.id = modelInfo.id;
    filter.archive = data::filter::Archive::All;

    QVector<data::Model> list;
    if (!contentList("MODEL", paging, filter, list))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (list.isEmpty())
    {
        writeToJsonMessage(error::model_not_found, answer);
        webCon().send(answer);
        return;
    }

    data::ModelInfoA modelInfoA;
    static_cast<data::Model&>(modelInfoA) = list[0];

    writeToJsonMessage(modelInfoA, answer);
    webCon().send(answer);
}

void Application::command_ModelList(const Message::Ptr& message)
{
    data::ModelList modelList;
    READ_FROM_MESSAGE(message, modelList)

    Message::Ptr answer = message->cloneForAnswer();

    if (!contentList("MODEL", modelList.paging, modelList.filter, modelList.items))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(modelList, answer);
    webCon().send(answer);
}

void Application::command_ScoreEdit(const Message::Ptr& message)
{
    data::ScoreEdit scoreEdit;
    READ_FROM_MESSAGE(message, scoreEdit);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != scoreEdit.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    if (!contentEdit("SCORE", scoreEdit, isAdmin))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);

        log_error_m << EventLog(u8"Ошибка редактирования применения '%1'", scoreEdit.name)
                    << EventContent(scoreEdit.id)
                    << EventUser(message->tag());
        return;
    }

    log_verbose_m << EventLog(u8"Применение '%1' изменено", scoreEdit.name)
                  << EventContent(scoreEdit.id)
                  << EventUser(message->tag());

    writeToJsonMessage(scoreEdit, answer);
    webCon().send(answer);
}

void Application::command_ScoreDelete(const Message::Ptr& message)
{
    data::ScoreDelete scoreDelete;
    READ_FROM_MESSAGE(message, scoreDelete);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != scoreDelete.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!scoreDelete.force)
    {
        // Проверка на наличие связи удалямого скора с созданными отчетами.
        // Если в Базе Данных присутствует отчет, для которого удаляемый скор
        // является родительским, то необходимо отказать в удалении
        if (!sql::exec(q,
            "SELECT ID FROM REPORT WHERE SCORE_ID = ? ", scoreDelete.id))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        // Если запрос вернул данные, значит отчеты созданные по этому скору
        // присуствуют в базе данных
        if (q.first())
        {
            writeToJsonMessage(error::score_has_child, answer);
            webCon().send(answer);
            return;
        }
    }

    QString scoreName;
    if (!sql::exec(q, "SELECT NAME FROM SCORE WHERE ID = ?", scoreDelete.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    assignValue(scoreName, q.record(), "NAME");

    bool sqlRes = false;
    if (isAdmin)
    {
        sqlRes = sql::exec(q,
            "UPDATE SCORE SET IS_ARCHIVE = 1 WHERE ID = ?", scoreDelete.id);
    }
    else
    {
        sqlRes = sql::exec(q,
            "UPDATE SCORE SET IS_ARCHIVE = 1 WHERE ID = ? AND USER_ID = ?",
            scoreDelete.id, scoreDelete.userId);
    }
    if (!sqlRes)
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Применение '%1' удалено", scoreName)
                  << EventContent(scoreDelete.id)
                  << EventUser(message->tag());

    writeToJsonMessage(scoreDelete, answer);
    webCon().send(answer);
}

void Application::command_ScoreInfo(const Message::Ptr& message)
{
    data::ScoreInfo scoreInfo;
    READ_FROM_MESSAGE(message, scoreInfo);

    Message::Ptr answer = message->cloneForAnswer();

    data::PagingInfo paging;
    data::ScoreFilter filter;
    filter.id = scoreInfo.id;
    filter.archive = data::filter::Archive::All;

    QVector<data::Score> list;
    if (!contentList("SCORE", paging, filter, list))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (list.isEmpty())
    {
        writeToJsonMessage(error::score_not_found, answer);
        webCon().send(answer);
        return;
    }

    data::ScoreInfoA scoreInfoA;
    static_cast<data::Score&>(scoreInfoA) = list[0];

    writeToJsonMessage(scoreInfoA, answer);
    webCon().send(answer);
}

void Application::command_ScoreList(const Message::Ptr& message)
{
    data::ScoreList scoreList;
    READ_FROM_MESSAGE(message, scoreList);

    Message::Ptr answer = message->cloneForAnswer();

    if (!contentList("SCORE", scoreList.paging, scoreList.filter, scoreList.items))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(scoreList, answer);
    webCon().send(answer);
}

void Application::command_NeedSendScore(const Message::Ptr& message)
{
    data::NeedSendScore needSendScore;
    READ_FROM_MESSAGE(message, needSendScore);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != needSendScore.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT COUNT(*) FROM SCORE WHERE ID = ?", needSendScore.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    if (q.value(0).toInt() == 0)
    {
        writeToJsonMessage(error::score_not_found, answer);
        webCon().send(answer);
        return;
    }

    bool sqlRes = false;
    if (isAdmin)
    {
        sqlRes = sql::exec(q,
            "UPDATE SCORE SET SEND_STATUS = ? WHERE ID = ?",
            SendScoreStatus::NeedSend, needSendScore.id);
    }
    else
    {
        sqlRes = sql::exec(q,
            "UPDATE SCORE SET SEND_STATUS = ? WHERE ID = ? AND USER_ID = ?",
            SendScoreStatus::NeedSend, needSendScore.id, needSendScore.userId);
    }
    if (!sqlRes)
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(needSendScore, answer);
    webCon().send(answer);

    task::sendScore().awake();
}

static bool groupModify(const data::Group& group, Message::Ptr& answer)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        return false;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (group.isDefault)
        if (!sql::exec(q, "UPDATE GROUPS SET IS_DEFAULT = 0"))
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            return false;
        }

    QString fields =
        "  ID          "
        ", NAME        "
        ", LEARN_MODEL "
        ", APPLY_MODEL "
        ", REPORT      "
        ", REPORT_FED  "
        ", MONITORING  "
        ", IS_DEFAULT  ";

    QString sql = sql::UPDATE_OR_INSERT_INTO("GROUPS", fields, "ID");

    if (!q.prepare(sql))
    {
        writeToJsonMessage(error::insert_or_update_sql, answer);
        return false;
    }

    bindValue(q, ":ID          ", group.id            );
    bindValue(q, ":NAME        ", group.name          );
    bindValue(q, ":LEARN_MODEL ", group.canLearnModel );
    bindValue(q, ":APPLY_MODEL ", group.canApplyModel );
    bindValue(q, ":REPORT      ", group.canRWReport   );
    bindValue(q, ":REPORT_FED  ", group.canReportFed  );
    bindValue(q, ":MONITORING  ", group.canMonitoring );
    bindValue(q, ":IS_DEFAULT  ", group.isDefault     );

    if (!q.exec())
    {
        writeToJsonMessage(error::insert_or_update_sql, answer);
        return false;
    }
    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        return false;
    }

    return true;
}

void Application::command_GroupCreate(const Message::Ptr& message)
{
    data::GroupCreate groupCreate;
    READ_FROM_MESSAGE(message, groupCreate);

    Message::Ptr answer = message->cloneForAnswer();

    groupCreate.id = QUuid::createUuid();
    if (!groupModify(groupCreate, answer))
    {
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Создана группа '%1'", groupCreate.name)
                  << EventUser(message->tag());

    writeToJsonMessage(groupCreate, answer);
    webCon().send(answer);
}

void Application::command_GroupEdit(const Message::Ptr& message)
{
    data::GroupEdit groupEdit;
    READ_FROM_MESSAGE(message, groupEdit);

    Message::Ptr answer = message->cloneForAnswer();

    if (!groupModify(groupEdit, answer))
    {
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Изменена группа '%1'", groupEdit.name)
                  << EventUser(message->tag());

    writeToJsonMessage(groupEdit, answer);
    webCon().send(answer);
}

void Application::command_GroupDelete(const Message::Ptr& message)
{
    data::GroupDelete deleteGroup;
    READ_FROM_MESSAGE(message, deleteGroup);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT COUNT(*) FROM USERS WHERE GROUP_ID = ?", deleteGroup.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    int countUsers = q.record().value(0).toInt();
    if (countUsers > 0)
    {
        data::GroupDeleteFail err;
        err.assign(error::delete_group_with_users.asFailed());
        err.id = deleteGroup.id;

        writeToJsonMessage(err, answer);
        webCon().send(answer);
        return;
    }

    if (!sql::exec(q,
        "SELECT COUNT(*) FROM GROUPS WHERE ID = ? AND IS_DEFAULT = 1", deleteGroup.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    bool isDefault = (q.record().value(0).toInt() > 0);
    if (isDefault)
    {
        writeToJsonMessage(error::delete_default_group, answer);
        webCon().send(answer);
        return;
    }

    if (!sql::exec(q, "SELECT NAME FROM GROUPS WHERE ID = ?", deleteGroup.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    q.first();
    QString groupName;
    assignValue(groupName, q.record(), "NAME");

    if (!sql::exec(q, "DELETE FROM GROUPS WHERE id = ?", deleteGroup.id))
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    log_verbose_m << EventLog(u8"Удалена группа '%1'", groupName)
                  << EventUser(message->tag());

    writeToJsonMessage(deleteGroup, answer);
    webCon().send(answer);
}

static bool groupList(const QUuidEx& groupId, QVector<data::Group>& groups)
{
    groups.clear();
    QString sql =
        " SELECT "
        "   G.ID           "
        "  ,G.NAME         "
        "  ,G.LEARN_MODEL  "
        "  ,G.APPLY_MODEL  "
        "  ,G.REPORT       "
        "  ,G.REPORT_FED   "
        "  ,G.MONITORING   "
        "  ,G.IS_DEFAULT   "
        "  ,(SELECT COUNT(*) FROM USERS U WHERE U.GROUP_ID = G.ID) AS USER_COUNT "
        " FROM             "
        "   GROUPS G       "
        " WHERE            "
        "   (1 = 1)        ";

    if (!groupId.isNull())
        sql += " AND G.ID = :GROUP_ID ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
        return false;

    if (!groupId.isNull())
        bindValue(q, ":GROUP_ID", groupId);

    if (!q.exec())
        return false;

    while (q.next())
    {
        QSqlRecord r = q.record();
        data::Group group;
        assignValue(group.id,            r, "ID          ");
        assignValue(group.name,          r, "NAME        ");
        assignValue(group.canLearnModel, r, "LEARN_MODEL ");
        assignValue(group.canApplyModel, r, "APPLY_MODEL ");
        assignValue(group.canRWReport,   r, "REPORT      ");
        assignValue(group.canReportFed,  r, "REPORT_FED  ");
        assignValue(group.canMonitoring, r, "MONITORING  ");
        assignValue(group.isDefault,     r, "IS_DEFAULT  ");
        assignValue(group.userCount,     r, "USER_COUNT  ");

        groups.append(group);
    }
    return true;
}

void Application::command_GroupInfo(const Message::Ptr& message)
{
    data::GroupInfo groupInfo;
    READ_FROM_MESSAGE(message, groupInfo);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::Group> groups;
    if (!::groupList(groupInfo.id, groups))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (groups.isEmpty())
    {
        writeToJsonMessage(error::group_not_found, answer);
        webCon().send(answer);
        return;
    }
    data::GroupInfoA groupInfoA;
    static_cast<data::Group&>(groupInfoA) = groups[0];

    writeToJsonMessage(groupInfoA, answer);
    webCon().send(answer);
}

void Application::command_GroupList(const Message::Ptr& message)
{
    data::GroupList groupList;
    READ_FROM_MESSAGE(message, groupList);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::Group> groups;
    if (!::groupList(QUuidEx(), groups))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    groupList.items = groups;
    writeToJsonMessage(groupList, answer);
    webCon().send(answer);
}

static bool userList(const QUuidEx& userId, QVector<data::User>& users)
{
    users.clear();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QString sql =
        " SELECT        "
        "   ID          "
        "  ,HASH_ID     "
        "  ,LOGIN       "
        "  ,NAME        "
        "  ,IS_ACTIVE   "
        "  ,IS_VALID    "
        "  ,IS_ADMIN    "
        "  ,GROUP_ID    "
        " FROM          "
        "   USERS       "
        " WHERE (1 = 1) ";

    if (!userId.isNull())
        sql += " AND ID = :ID ";

    if (!q.prepare(sql))
        return false;

    if (!userId.isNull())
        bindValue(q, ":ID", userId);

    if (!q.exec())
        return false;

    while (q.next())
    {
        data::User user;
        QSqlRecord r = q.record();

        assignValue(user.id,        r, "ID        ");
        assignValue(user.hashId,    r, "HASH_ID   ");
        assignValue(user.login,     r, "LOGIN     ");
        assignValue(user.name,      r, "NAME      ");
        assignValue(user.isActive,  r, "IS_ACTIVE ");
        assignValue(user.isValid,   r, "IS_VALID  ");
        assignValue(user.isAdmin,   r, "IS_ADMIN  ");

        // Для администраторов системы не назначаем группу, т.к. это не имеет
        // смысла, администратор  обладает  полными  правами, которые  нельзя
        // ограничить при помощи групп.
        // На стороне веб-интерфейса эта ситуация обрабатывается: поле выбора
        // группы будет недоступно для изменения если groupId  равен нулю
        if (!user.isAdmin)
            assignValue(user.groupId, r, "GROUP_ID");

        users.append(user);
    }
    return true;
}

void Application::command_UserInfo(const Message::Ptr& message)
{
    data::UserInfo userInfo;
    READ_FROM_MESSAGE(message, userInfo);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::User> users;
    if (!userList(userInfo.id, users))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (users.isEmpty())
    {
        writeToJsonMessage(error::user_not_found, answer);
        webCon().send(answer);
        return;
    }
    data::UserInfoA userInfoA;
    *(static_cast<data::User*>(&userInfoA)) = users[0];

    writeToJsonMessage(userInfoA, answer);
    webCon().send(answer);
}

void Application::command_UserList(const Message::Ptr& message)
{
    //auto userHashId = message->tag();
    data::UserList userList;
    READ_FROM_MESSAGE(message, userList);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::User> users;
    if (!::userList(QUuidEx(), users))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    userList.items = users;
    writeToJsonMessage(userList, answer);
    webCon().send(answer);
}

void Application::command_AssignGroup(const Message::Ptr& message)
{
    data::AssignGroup assignGroup;
    READ_FROM_MESSAGE(message, assignGroup);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare("UPDATE USERS SET GROUP_ID = :GROUP_ID WHERE ID = :ID"))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    bindValue(q, ":ID",       assignGroup.userId);
    bindValue(q, ":GROUP_ID", assignGroup.groupId);

    if (!q.exec())
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(assignGroup, answer);
    webCon().send(answer);
}

void Application::command_NsiVidmpList(const Message::Ptr& message)
{
    auto tfunc = [this](Message::Ptr message)
    {
        log_debug_m << "Begin func for NsiVidmp command";

        trd::ThreadIdLock threadIdLock(&_threadIds); (void) threadIdLock;

        data::NsiVidmpList nsiVidmpList;
        READ_FROM_MESSAGE(message, nsiVidmpList);

        Message::Ptr answer = message->cloneForAnswer();

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (nsiVidmpList.scoreId.isNull())
        {
            if (!sql::exec(q,
                " SELECT                          "
                "   ID      AS NSI_VIDMP__ID      "
                "  ,VMPNAME AS NSI_VIDMP__VMPNAME "
                " FROM                            "
                "   NSI_VIDMP                     "
                " ORDER BY                        "
                "   NSI_VIDMP.ID ASC              "))
            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }
        }
        else
        {
            if (!sql::exec(q,
                " SELECT                                    "
                "   T.VID_MP          AS NSI_VIDMP__ID      "
                "  ,NSI_VIDMP.VMPNAME AS NSI_VIDMP__VMPNAME "
                " FROM                                      "
                " (                                         "
                "   SELECT DISTINCT                         "
                "     SC.VID_MP                             "
                "   FROM                                    "
                "     SCORE_DATA SC                         "
                "   WHERE                                   "
                "     SC.SCORE_ID = ?                       "
                " ) T                                       "
                " LEFT JOIN                                 "
                "   NSI_VIDMP ON T.VID_MP = NSI_VIDMP.ID    "
                " WHERE                                     "
                "   T.VID_MP IS NOT NULL                    "
                " ORDER BY                                  "
                "   T.VID_MP ASC                            ",
                nsiVidmpList.scoreId))
            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }
        }
        log_debug_m << "Exec SQL for NsiVidmp command";

        nsiVidmpList.items.clear();
        while (q.next())
        {
            data::NsiVidmp nsiVidmp;
            QSqlRecord r = q.record();

            assignValue(nsiVidmp.id,   r, "NSI_VIDMP__ID");
            assignValue(nsiVidmp.name, r, "NSI_VIDMP__VMPNAME");

            if (!nsiVidmp.name.isEmpty())
                nsiVidmp.name[0] = nsiVidmp.name[0].toUpper();

            nsiVidmpList.items.append(nsiVidmp);
        }
        log_debug_m << "Fetch data for NsiVidmp command";

        if (dbcon->operationIsAborted())
            return;

        writeToJsonMessage(nsiVidmpList, answer);
        webCon().send(answer);

        log_debug_m << "End func for NsiVidmp command";
    };

    std::thread t {tfunc, message};
    t.detach();
}

void Application::command_NsiProfileList(const Message::Ptr& message)
{
    auto tfunc = [this](Message::Ptr message)
    {
        log_debug_m << "Begin func for NsiProfile command";

        trd::ThreadIdLock threadIdLock(&_threadIds); (void) threadIdLock;

        data::NsiProfileList nsiProfileList;
        READ_FROM_MESSAGE(message, nsiProfileList);

        Message::Ptr answer = message->cloneForAnswer();

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (nsiProfileList.scoreId.isNull())
        {
            if (!sql::exec(q,
                " SELECT                          "
                "   ID     AS NSI_PROFILE__ID     "
                "  ,PRNAME AS NSI_PROFILE__PRNAME "
                " FROM                            "
                "   NSI_PROFILE                   "
                " ORDER BY                        "
                "   NSI_PROFILE.ID ASC            "))

            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }
        }
        else
        {
            if (!sql::exec(q,
               " SELECT                                      "
               "   T.PROFIL           AS NSI_PROFILE__ID     "
               "  ,NSI_PROFILE.prname AS NSI_PROFILE__PRNAME "
               " FROM                                        "
               " (                                           "
               "   SELECT DISTINCT                           "
               "     SC.PROFIL                               "
               "   FROM                                      "
               "     SCORE_DATA SC                           "
               "   WHERE                                     "
               "     SC.SCORE_ID = ?                         "
               " ) T                                         "
               " LEFT JOIN                                   "
               "   NSI_PROFILE ON T.PROFIL = NSI_PROFILE.ID  "
               " ORDER BY                                    "
               "   T.PROFIL ASC                              ",
               nsiProfileList.scoreId))
            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }
        }
        log_debug_m << "Exec SQL for NsiProfile command";

        nsiProfileList.items.clear();
        while (q.next())
        {
            data::NsiProfile nsiProfile;
            QSqlRecord r = q.record();

            assignValue(nsiProfile.id,   r, "NSI_PROFILE__ID");
            assignValue(nsiProfile.name, r, "NSI_PROFILE__PRNAME");

            if (!nsiProfile.name.isEmpty())
                nsiProfile.name[0] = nsiProfile.name[0].toUpper();

            nsiProfileList.items.append(nsiProfile);
        }
        log_debug_m << "Fetch data for NsiProfile command";

        if (dbcon->operationIsAborted())
            return;

        writeToJsonMessage(nsiProfileList, answer);
        webCon().send(answer);

        log_debug_m << "End func for NsiProfile command";
    };

    std::thread t {tfunc, message};
    t.detach();
}

void Application::command_NsiLpuList(const Message::Ptr& message)
{
    auto tfunc = [this](Message::Ptr message)
    {
        log_debug_m << "Begin func for NsiLpu command";

        trd::ThreadIdLock threadIdLock(&_threadIds); (void) threadIdLock;

        data::NsiLpuList nsiLpuList;
        READ_FROM_MESSAGE(message, nsiLpuList);

        Message::Ptr answer = message->cloneForAnswer();

        QString sql;
        const data::NsiLpuFilter& filter = nsiLpuList.filter;

        if (filter.scoreId.isNull())
        {
            sql =
            " SELECT                                       "
            "   NSI_LPU.CODE        AS NSI_LPU__CODE       "
            "  ,NSI_LPU.FULL_NAME   AS NSI_LPU__FULL_NAME  "
            "  ,NSI_LPU.SHORT_NAME  AS NSI_LPU__SHORT_NAME "
            " FROM                                         "
            "    NSI_LPU                                   "
            " WHERE (1 = 1)                                ";
        }
        else
        {
            sql =
            " SELECT                                       "
            "   T.CODE_LPU          AS NSI_LPU__CODE       "
            "  ,NSI_LPU.FULL_NAME   AS NSI_LPU__FULL_NAME  "
            "  ,NSI_LPU.SHORT_NAME  AS NSI_LPU__SHORT_NAME "
            " FROM                                         "
            " (                                            "
            "   SELECT DISTINCT                            "
            "     SC.CODE_LPU                              "
            "   FROM                                       "
            "     SCORE_DATA SC                            "
            "   WHERE                                      "
            "     SC.SCORE_ID = :SCORE_ID                  "
            " ) T                                          "
            " LEFT JOIN                                    "
            "   NSI_LPU ON T.CODE_LPU = NSI_LPU.CODE       "
            " WHERE                                        "
            "   T.CODE_LPU IS NOT NULL                     ";
        }

    //    if (!filter.fullName.isEmpty())
    //        sql += " AND FULL_NAME CONTAINING :FULL_NAME ";

    //    if (!filter.shortName.isEmpty())
    //        sql += " AND SHORT_NAME CONTAINING :SHORT_NAME ";

        if (!filter.lpuSearch.isEmpty())
        {
            QString s = filter.lpuSearch.toUpper();
            sql += " AND (UPPER(NSI_LPU.CODE) STARTING WITH '" + s + "'    "
                   "      OR  UPPER(NSI_LPU.FULL_NAME) LIKE '%" + s + "%') ";
        }

        //pagingPrepare(nsiLpuList.paging, sql);

        sql += " ORDER BY T.CODE_LPU ASC ";

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!q.prepare(sql))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        if (!filter.scoreId.isNull())
            bindValue(q, ":SCORE_ID", filter.scoreId);

    //    if (!filter.fullName.isEmpty())
    //        bindValue(q, ":FULL_NAME", filter.fullName);

    //    if (!filter.shortName.isEmpty())
    //        bindValue(q, ":SHORT_NAME", filter.shortName);

        if (!q.exec())
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        log_debug_m << "Exec SQL for NsiLpu command";

        nsiLpuList.items.clear();
        while (q.next())
        {
            data::NsiLpu nsiLpu;
            QSqlRecord r = q.record();

            assignValue(nsiLpu.code,      r, "NSI_LPU__CODE"       );
            assignValue(nsiLpu.fullName,  r, "NSI_LPU__FULL_NAME"  );
            assignValue(nsiLpu.shortName, r, "NSI_LPU__SHORT_NAME" );

            nsiLpuList.items.append(nsiLpu);
        }
        log_debug_m << "Fetch data for NsiLpu command";

        if (dbcon->operationIsAborted())
            return;

        //nsiLpuList.paging.total = q.size();

        writeToJsonMessage(nsiLpuList, answer);
        webCon().send(answer);

        log_debug_m << "End func for NsiLpu command";
    };

    std::thread t {tfunc, message};
    t.detach();
}

void Application::command_NsiMkbList(const Message::Ptr& message)
{
    auto tfunc = [this](Message::Ptr message)
    {
        log_debug_m << "Begin func for NsiMkb command";

        trd::ThreadIdLock threadIdLock(&_threadIds); (void) threadIdLock;

        data::NsiMkbList nsiMkbList;
        READ_FROM_MESSAGE(message, nsiMkbList);

        Message::Ptr answer = message->cloneForAnswer();

        QString sql;
        const data::NsiMkbFilter& filter = nsiMkbList.filter;

        if (filter.scoreId.isNull())
        {
            sql =
            " SELECT                                   "
            "   NSI_MKB.MKB_CODE  AS NSI_MKB__MKB_CODE "
            "  ,NSI_MKB.MKB_NAME  AS NSI_MKB__MKB_NAME "
            " FROM                                     "
            "   NSI_MKB                                ";
        }
        else
        {
            sql =
            " SELECT                                            "
            "   NSI_MKB.MKB_CODE  AS NSI_MKB__MKB_CODE          "
            "  ,NSI_MKB.MKB_NAME  AS NSI_MKB__MKB_NAME          "
            " FROM                                              "
            " (                                                 "
            "   SELECT DISTINCT                                 "
            "     SC.MKB1                                       "
            "   FROM                                            "
            "     SCORE_DATA SC                                 "
            "   WHERE                                           "
            "     SC.SCORE_ID = :SCORE_ID                       "
            "     AND                                           "
            "     SC.MKB1 IS NOT NULL                           "
            " ) T                                               "
            " LEFT JOIN                                         "
            "  NSI_MKB ON NSI_MKB.MKB_CODE STARTING WITH T.MKB1 "
            " WHERE                                             "
            "   NSI_MKB.MKB_CODE IS NOT NULL                    ";
        }

//        if (filter.parentId == 0)
//            sql += " AND NSI_MKB.ID_PARENT IS NULL ";
//        else if (filter.parentId > 0)
//            sql += " AND NSI_MKB.ID_PARENT = :ID_PARENT ";

        if (filter.onlyMkbCode)
            sql += " AND NSI_MKB.MKB_CODE IS NOT NULL ";

    //    if (!filter.mkbCode.isEmpty())
    //    {
    //        sql += " AND UPPER(MKB_CODE) STARTING WITH :MKB_CODE ";
    //        //sql += filter.mkbCode.toUpper() + "% ";
    //    }

    //    if (!filter.mkbName.isEmpty())
    //    {
    //        sql += " AND UPPER(MKB_NAME) LIKE ";
    //        sql += "'%" + filter.mkbName.toUpper() + "%' ";
    //    }

        if (!filter.mkbSearch.isEmpty())
        {
            QString s = filter.mkbSearch.toUpper();
            sql += " AND (UPPER(NSI_MKB.MKB_CODE) STARTING WITH '" + s + "'"
                   "      OR  UPPER(NSI_MKB.MKB_NAME) LIKE '%" + s + "%') ";

        }

        sql += " ORDER BY T.MKB1 ASC ";

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!q.prepare(sql))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        if (!filter.scoreId.isNull())
            bindValue(q, ":SCORE_ID", filter.scoreId);

//        if (filter.parentId > 0)
//            bindValue(q, ":ID_PARENT", filter.parentId);

    //    if (!filter.mkbCode.isEmpty())
    //        bindValue(q, ":MKB_CODE", filter.mkbCode.toUpper());

        if (!q.exec())
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        log_debug_m << "Exec SQL for NsiMkb command";

        nsiMkbList.items.clear();
        while (q.next())
        {
            data::NsiMkb nsiMkb;
            QSqlRecord r = q.record();

            assignValue(nsiMkb.mkbCode,  r, "NSI_MKB__MKB_CODE " );
            assignValue(nsiMkb.mkbName,  r, "NSI_MKB__MKB_NAME " );

            nsiMkbList.items.append(nsiMkb);
        }
        log_debug_m << "Fetch data for NsiMkb command";

        if (dbcon->operationIsAborted())
            return;

        writeToJsonMessage(nsiMkbList, answer);
        webCon().send(answer);

        log_debug_m << "End func for NsiMkb command";
    };

    std::thread t {tfunc, message};
    t.detach();
}

void Application::command_ModelXgbDelete(const Message::Ptr& message)
{
    data::ModelXgbDelete modelXgbDelete;
    READ_FROM_MESSAGE(message, modelXgbDelete);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    for (const data::XgbParam& param : modelXgbDelete.items)
    {
        if (!sql::exec(q, "DELETE FROM MODEL_XGB WHERE ID = ? AND NAME = ?",
                          modelXgbDelete.id, param.name))
        {
            writeToJsonMessage(error::delete_sql_statement, answer);
            webCon().send(answer);
            return;
        }
    }

    data::ModelXgbDeleteA modelXgbDeleteA;
    modelXgbDeleteA.id = modelXgbDelete.id;
    writeToJsonMessage(modelXgbDeleteA, answer);
    webCon().send(answer);
}

void Application::command_ModelXgbInfo(const Message::Ptr& message)
{
    data::ModelXgbInfo modelXgbInfo;
    READ_FROM_MESSAGE(message, modelXgbInfo);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::XgbParam> options;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT NAME, CONTENT, ENABLED FROM MODEL_XGB WHERE ID = ?",
                      modelXgbInfo.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    while (q.next())
    {
        QSqlRecord r = q.record();
        QString name, value;
        bool enabled;
        assignValue(name,    r, "NAME"  );
        assignValue(value,   r, "CONTENT" );
        assignValue(enabled, r, "ENABLED" );

        options.append(data::XgbParam{name, value, enabled});
    }

    data::ModelXgbInfoA modelXgbInfoA;
    modelXgbInfoA.id = modelXgbInfo.id;
    modelXgbInfoA.items = options;
    writeToJsonMessage(modelXgbInfoA, answer);
    webCon().send(answer);
}

void Application::command_ModelXgbEdit(const Message::Ptr& message)
{
    data::ModelXgbEdit modelXgbEdit;
    READ_FROM_MESSAGE(message, modelXgbEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    for (const data::XgbParam& param : modelXgbEdit.items)
    {
        QString name = param.name;
        QString value = param.value;
        bool enabled = param.enabled;

        if (!sql::exec(q, " UPDATE OR INSERT INTO MODEL_XGB "
                          " (ID, NAME, CONTENT, ENABLED)    "
                          " VALUES                          "
                          " (?, ?, ?, ?)                    "
                          " MATCHING (ID, NAME)             ",
                          modelXgbEdit.id, name, value, enabled))
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            webCon().send(answer);
            return;
        }
    }

    data::ModelXgbEditA modelXgbEditA;
    modelXgbEditA.id = modelXgbEdit.id;
    writeToJsonMessage(modelXgbEditA, answer);
    webCon().send(answer);
}

void Application::command_ModelXgbOption(const Message::Ptr& message)
{
    data::ModelXgbOption modelXgbOption;
    READ_FROM_MESSAGE(message, modelXgbOption);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " UPDATE MODEL_XGB      "
        "   SET ENABLED = ?     "
        " WHERE                 "
        "   ID = ? AND NAME = ? ",
        modelXgbOption.enable,
        modelXgbOption.id,
        modelXgbOption.name))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    data::ModelXgbOptionA modelXgbOptionA;
    modelXgbOptionA.id = modelXgbOption.id;
    writeToJsonMessage(modelXgbOptionA, answer);
    webCon().send(answer);
}

void Application::command_SyncProgress(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    data::SyncInfo::List syncList = task::SyncPlan::syncList();
    if (!syncList.empty())
        syncList[0].syncStatus = true;

    data::SyncProgress syncProgressA;
    syncProgressA.items.swap(syncList);

    writeToJsonMessage(syncProgressA, answer);
    webCon().send(answer);
}

void Application::command_EventLogList(const Message::Ptr& message)
{
    auto tfunc = [this](Message::Ptr message)
    {
        log_debug_m << "Begin func for EventLog command";

        trd::ThreadIdLock threadIdLock(&_threadIds); (void) threadIdLock;

        data::EventLogList eventLogList;
        READ_FROM_MESSAGE(message, eventLogList);

        Message::Ptr answer = message->cloneForAnswer();

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        QString sql =
            " SELECT                        "
            "   E.CREATE_DATE               "
            "  ,E.USER_ID                   "
            "  ,E.TASK_ID                   "
            "  ,E.LOG_LEVEL                 "
            "  ,E.DESCRIPTION               "
            "  ,E.LOCATION                  "
            "  ,U.NAME AS USER_NAME         "
            "  ,T.NAME AS TASK_NAME         "
            " FROM                          "
            "   EVENT_LOG E                 "
            " LEFT JOIN                     "
            "   USERS U ON E.USER_ID = U.ID "
            " LEFT JOIN                     "
            "   TASK T ON E.TASK_ID = T.ID  "
            " WHERE (1 = 1)                 ";

        const data::EventFilter& filter = eventLogList.filter;

        if (!filter.taskId.isNull())
            sql += " AND E.TASK_ID = :TASK_ID ";

        if (!filter.userId.isNull()) // Если userId в фильтре не пуст
        {
            sql += " AND (E.USER_ID = :USER_ID ";
            if (filter.selectAll)
                sql += " OR E.USER_ID IS NULL  ";
            sql += " ) ";
        }
        else // Если userId в фильтре пуст
        {
            if (!filter.selectAll)
                sql += " AND E.USER_ID IS NOT NULL ";
            else
                {/* Иначе отображаем весь список */}
        }

//        if (!filter.selectAll)
//        {
//            if (!filter.userId.isNull())
//                sql += " AND E.USER_ID = :USER_ID ";

//            if (!filter.taskId.isNull())
//                sql += " AND E.TASK_ID = :TASK_ID ";
//        }

        if (filter.createDate.begin.isValid())
            sql += " AND E.CREATE_DATE >= :CREATE_DATE_BEGIN ";

        if (filter.createDate.end.isValid())
            sql += " AND E.CREATE_DATE <= :CREATE_DATE_END ";

        pagingPrepare(eventLogList.paging, sql);

        sql += (filter.descendSort)
               ? " ORDER BY E.CREATE_DATE DESC "
               : " ORDER BY E.CREATE_DATE ASC ";

        if (!q.prepare(sql))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        if (!filter.taskId.isNull())
            bindValue(q, ":TASK_ID", filter.taskId);

        if (!filter.userId.isNull())
            bindValue(q, ":USER_ID", filter.userId);

//        if (!filter.selectAll)
//        {
//            if (!filter.userId.isNull())
//                bindValue(q, ":USER_ID", filter.userId);

//            if (!filter.taskId.isNull())
//                bindValue(q, ":TASK_ID", filter.taskId);
//        }

        if (filter.createDate.begin.isValid())
            bindValue(q, ":CREATE_DATE_BEGIN", filter.createDate.begin);

        if (filter.createDate.end.isValid())
            bindValue(q, ":CREATE_DATE_END", filter.createDate.end);

        if (!q.exec())
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        log_debug_m << "Exec SQL for EventLog command";

        eventLogList.items.clear();
        while (q.next())
        {
            data::EventLog eventLog;
            QSqlRecord r = q.record();

            assignValue(eventLog.createDate  , r, "CREATE_DATE" );
            assignValue(eventLog.userId      , r, "USER_ID    " );
            assignValue(eventLog.taskId      , r, "TASK_ID    " );
            assignValue(eventLog.userName    , r, "USER_NAME  " );
            assignValue(eventLog.taskName    , r, "TASK_NAME  " );
            assignValue(eventLog.level       , r, "LOG_LEVEL  " );
            assignValue(eventLog.description , r, "DESCRIPTION" );
            assignValue(eventLog.location    , r, "LOCATION   " );

            eventLogList.items.append(eventLog);
        }
        log_debug_m << "Fetch data for EventLog command";

        if (dbcon->operationIsAborted())
            return;

        eventLogList.paging.total = q.size();
        log_debug_m << "Calc records count for EventLog command";

        writeToJsonMessage(eventLogList, answer);
        webCon().send(answer);

        log_debug_m << "End func for EventLog command";
    };

    std::thread t {tfunc, message};
    t.detach();
}

void Application::command_AisVersion(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    data::AisVersion aisVersion;
    aisVersion.major  = productVersion().ver.major;
    aisVersion.minor  = productVersion().ver.minor;
    aisVersion.patch  = productVersion().ver.patch;
    aisVersion.strver = productVersion().toString();
    aisVersion.gitrev = GIT_REVISION;

    writeToJsonMessage(aisVersion, answer);
    webCon().send(answer);
}

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
