#include "event_log.h"
#include "user_rights.h"
#include "commands/event_log.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "qt/logger/logger_operators.h"

QUuidEx EventUser::id() const
{
    if (!_id.isNull())
        return _id;

    if (_hashId == 0)
        return QUuidEx();

    _id = uright().userId(_hashId);
    return _id;
}

QMap<QUuidEx, QString> EventTask::_names;
QMutex EventTask::_namesLock;

EventTask::EventTask(const QUuidEx& id, qint32 retCode, qint16 isLife)
    : _id(id), _retCode(retCode), _isLife(isLife)
{
    QMutexLocker locker(&_namesLock); (void) locker;
    _name = _names[_id];
    if (_name.isEmpty())
    {
        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (sql::exec(q, "SELECT NAME FROM TASK WHERE ID = ?", _id) && q.first())
        {
            sql::assignValue(_name, q.record(), "NAME");
            _names[_id] = _name;
        }
    }
}

namespace alog {

using namespace communication;

Line& operator<< (Line& line, const EventLog& el)
{
    if (line.toLogger())
    {
        line << el.descript();
        line.impl->module = "EventLog";
    }
    return line;
}

Line& operator<< (Line& line, const EventUser& eu)
{
    if (line.toLogger())
    {
        Something* s = EventLogSaver::something(line);
        if (EventLogSaver::Data* d = dynamic_cast<EventLogSaver::Data*>(s))
            d->userId = eu.id();
    }
    return line;
}

Line& operator<< (Line& line, const EventTask& et)
{
    if (line.toLogger())
    {
        Something* s = EventLogSaver::something(line);
        if (EventLogSaver::Data* d = dynamic_cast<EventLogSaver::Data*>(s))
        {
            d->taskId = et.id();
            d->taskName = et.name();
            d->taskRetCode = et.retCode();
            d->taskLife = et.isLife();
        }
    }
    return line;
}

Line& operator<< (Line& line, const EventContent& ec)
{
    if (line.toLogger())
    {
        Something* s = EventLogSaver::something(line);
        if (EventLogSaver::Data* d = dynamic_cast<EventLogSaver::Data*>(s))
            d->contentId = ec.id();
    }
    return line;
}

EventLogSaver::EventLogSaver(const string& name, Level level)
    : Saver(name, level)
{}

void EventLogSaver::flushImpl(const MessageList& messages)
{
    using namespace sql;

    for (Message* message : messages)
    {
        if (message->module == "EventLog")
        {
            QUuidEx userId;
            QUuidEx taskId;
            qint32  taskRetCode = {-1};
            qint16  taskLife  = {-1};
            if (EventLogSaver::Data* d =
                    dynamic_cast<EventLogSaver::Data*>(message->something.get()))
            {
                userId = d->userId;
                taskId = d->taskId;
                taskRetCode = d->taskRetCode;
                taskLife = d->taskLife;
            }
            int level = static_cast<int>(message->level);
            QString descript = QString::fromUtf8(message->str.c_str());
            QString location = QString("%1:%2").arg(message->file.c_str())
                                               .arg(message->line);

            qint64 time_msec = message->timeVal.tv_sec * 1000;
            qint64 msec = qint64(double(message->timeVal.tv_usec) / 1000);
            time_msec += msec;

            db::firebird::Driver::Ptr dbcon = dbpool().connect();
            if (!dbcon->isOpen())
            {
                log_error << "Failed save to EventLog. Database not connected";
                break;
            }

            QSqlQuery q {dbcon->createResult()};
            QDateTime createDate = QDateTime::fromMSecsSinceEpoch(time_msec);

            QString fields =
                "CREATE_DATE, USER_ID, TASK_ID, LOG_LEVEL, DESCRIPTION, LOCATION";

            if (taskRetCode >= 0)
                fields += ", TASK_RET_CODE";

            if (taskLife >= 0)
                fields += ", TASK_LIFE";

            QString sql = sql::INSERT_INTO("EVENT_LOG", fields);

            if (q.prepare(sql))
            {
                bindValue(q, ":CREATE_DATE ", createDate );
                bindValue(q, ":USER_ID     ", userId     );
                bindValue(q, ":TASK_ID     ", taskId     );
                bindValue(q, ":LOG_LEVEL   ", level      );
                bindValue(q, ":DESCRIPTION ", descript   );
                bindValue(q, ":LOCATION    ", location   );

                if (taskRetCode >= 0)
                    bindValue(q, ":TASK_RET_CODE", taskRetCode);

                if (taskLife >= 0)
                    bindValue(q, ":TASK_LIFE", taskLife);

                q.exec();
            }
        }
    }
}

bool EventLogSaver::Data::canModifyMessage() const
{
    return true;
}

string EventLogSaver::Data::modifyMessage(const string& str) const
{
    string s = str;
    if (!taskId.isNull())
    {
        QString sId = taskId.toString(); sId.chop(1);
        s += ". TaskId: ";
        s += (sId.toUtf8().constData() + 1);
    }
    if (!taskName.isEmpty())
    {
        s += ". TaskName: '";
        s += taskName.toUtf8().constData();
        s += "'";
    }
    if (!contentId.isNull())
    {
        s += ". ContentId: " + contentId.toString().toStdString();
    }
    if (!userId.isNull())
    {
        QString userName = uright().userName(userId);
        s += QString(". User: '%1'").arg(userName).toUtf8().constData();
    }
    return s;
}

Something* EventLogSaver::something(Line& line)
{
    if (line.impl->something.empty())
        line.impl->something = Something::Ptr(new EventLogSaver::Data);

    return line.impl->something.get();
}

} // namespace alog
