#pragma once

#include "shared/logger/logger.h"
#include "shared/qt/quuidex.h"
#include <QtCore>

class EventLog
{
public:
    template<typename... Args>
    EventLog(QString descript, const Args&... args)
    {
        buildFunc(descript, args...);
        _descript = descript;
    }

    template<typename... Args>
    EventLog(const char* descript, const Args&... args)
    {
        QString descr = QString::fromUtf8(descript);
        buildFunc(descr, args...);
        _descript = descr;
    }

    QString  descript() const {return _descript;}

private:
    template<typename T, typename... Args>
    void buildFunc(QString& descript, const T t, const Args&... args)
    {
        static_assert(std::is_integral<T>::value || std::is_floating_point<T>::value,
                      "The T must be integral type");

        descript = descript.arg(t);
        buildFunc(descript, args...);
    }
    template<typename... Args>
    void buildFunc(QString& descript, const char* t, const Args&... args)
    {
        descript = descript.arg(QString::fromUtf8(t));
        buildFunc(descript, args...);
    }
    template<typename... Args>
    void buildFunc(QString& descript, const QString& s, const Args&... args)
    {
        descript = descript.arg(s);
        buildFunc(descript, args...);
    }
    template<typename... Args>
    void buildFunc(QString& descript, const QUuidEx& u, const Args&... args)
    {
        QString s = u.toString();
        s.remove(0, 1);
        s.chop(1);
        descript = descript.arg(s);
        buildFunc(descript, args...);
    }
    template<typename... Args>
    void buildFunc(QString& descript, const QDateTime& dt, const Args&... args)
    {
        descript = descript.arg(dt.toString("dd.MM.yyyy hh:mm:ss.zzz"));
        buildFunc(descript, args...);
    }
    template<typename... Args>
    void buildFunc(QString& descript, const QDate& d, const Args&... args)
    {
        descript = descript.arg(d.toString("dd.MM.yyyy"));
        buildFunc(descript, args...);
    }
    void buildFunc(QString&) {}

    QString  _descript;
};

/**
  Вспомогательная структура, используется для записи в журнал событий
  идентификатора пользователя
*/
struct EventUser
{
    EventUser(const QUuidEx& id) : _id(id) {}
    EventUser(quint64 hashId) : _hashId(hashId) {}
    QUuidEx id() const;

private:
    mutable QUuidEx _id;
    quint64 _hashId = {0};
};

/**
  Вспомогательная структура, используется для записи в журнал событий
  идентификатора задачи
*/
struct EventTask
{
    EventTask(const QUuidEx& id, qint32 retCode = -1, qint16 isLife = -1);
    QUuidEx id() const {return _id;}
    QString name() const {return _name;}
    qint32  retCode() const {return _retCode;}
    qint16  isLife() const {return _isLife;}

private:
    QUuidEx _id;
    QString _name;
    qint32  _retCode = {-1};
    qint16  _isLife  = {-1};

    static QMap<QUuidEx /*Id*/, QString /*Name*/> _names;
    static QMutex _namesLock;
};

/**
  Вспомогательная структура, используется для записи в журнал событий
  идентификатора контента (модель, применение, отчет)
*/
struct EventContent
{
    EventContent(const QUuidEx& id) : _id(id) {}
    QUuidEx id() const {return _id;}

private:
    mutable QUuidEx _id;
};

namespace alog {

using namespace std;

Line& operator<< (Line&, const EventLog&);
Line& operator<< (Line&, const EventUser&);
Line& operator<< (Line&, const EventTask&);
Line& operator<< (Line&, const EventContent&);

struct EventLogSaver : public Saver
{
    struct Data : Something
    {
        QUuidEx userId;
        QUuidEx contentId;
        QUuidEx taskId;
        QString taskName;
        qint32  taskRetCode = {-1};
        qint16  taskLife  = {-1};

        bool canModifyMessage() const override;
        string modifyMessage(const string&) const override;
    };

    static Something* something(Line&);

    EventLogSaver(const string& name, Level level = Error);
    void flushImpl(const MessageList&) override;
};

} // namespace alog

