#include "functions.h"
#include "database/connect.h"
#include "database/sql_func.h"

#include "shared/safe_singleton.h"
#include "shared/spin_locker.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Functions")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Functions")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Functions")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Functions")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Functions")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Functions")

namespace communication {
namespace transport {

void WebCon::init(SocketDescriptor socketDescript)
{
    SpinLocker locker(_lock); (void) locker;

    _socketDescriptor = socketDescript;
    _socket = tcp::listener().socketByDescriptor(_socketDescriptor)
                             .dynamic_cast_to<tcp::Socket::Ptr>();
    _isAuthorized = false;
    _authorizTimer.reset();
}

bool WebCon::isEmpty() const
{
    SpinLocker locker(_lock); (void) locker;
    return _socket.empty();
}

bool WebCon::isConnected() const
{
    SpinLocker locker(_lock); (void) locker;
    return (_socket && _socket->isConnected());
}

tcp::Socket::Ptr WebCon::socket() const
{
    SpinLocker locker(_lock); (void) locker;
    return _socket;
}

void WebCon::send(const Message::Ptr& message) const
{
    SpinLocker locker(_lock); (void) locker;
    if (_socket)
        if (message->type() == Message::Type::Event
            || _socket->socketDescriptor() == message->socketDescriptor())
        {
            _socket->send(message);
        }
}
void WebCon::reset()
{
    SpinLocker locker(_lock); (void) locker;
    if (_socket)
        _socket->stop();
    _socket.reset();
    _socketDescriptor = {-1};
    _isAuthorized = false;
}

SocketDescriptor WebCon::socketDescriptor() const
{
    SpinLocker locker(_lock); (void) locker;
    return _socketDescriptor;
}

bool WebCon::isAuthorized() const
{
    SpinLocker locker(_lock); (void) locker;
    return _isAuthorized;
}

void WebCon::setAuthorized(bool val)
{
    SpinLocker locker(_lock); (void) locker;
    _isAuthorized = val;
}

bool WebCon::waitingAuthorizExpired() const
{
    SpinLocker locker(_lock); (void) locker;
    return (_authorizTimer.elapsed() > 5*1000 /*5 сек*/);
}

WebCon& webCon()
{
    return ::safe_singleton<WebCon>();
}

bool FomsCon::isConnected() const
{
    SpinLocker locker(_lock); (void) locker;
    return (_socket && _socket->isConnected());
}

tcp::Socket::Ptr FomsCon::socket() const
{
    SpinLocker locker(_lock); (void) locker;
    return _socket;
}

bool FomsCon::send(const Message::Ptr& message) const
{
    SpinLocker locker(_lock); (void) locker;
    if (_socket && _isAuthorized)
        return _socket->send(message);

    return false;
}

void FomsCon::disconnect()
{
    SpinLocker locker(_lock); (void) locker;
    _socket->stop();
    _isAuthorized = false;
}

bool FomsCon::isAuthorized() const
{
    SpinLocker locker(_lock); (void) locker;
    return _isAuthorized;
}

void FomsCon::setAuthorized(bool val)
{
    SpinLocker locker(_lock); (void) locker;
    _isAuthorized = val;
}

FomsCon& fomsCon()
{
    return ::safe_singleton<FomsCon>();
}

tcp::Socket::Ptr& upoolCon()
{
    return ::safe_singleton_ptr<tcp::Socket, 1>();
}

bool RouteCommands::forwarding(const Message::Ptr& message)
{
    if (!commands.contains(message->command()))
        return false;

    quint64 curTime = std::time(nullptr);
    auto removeExpiredTime= [curTime](Point& point)
    {
        for (int i = 0; i < point.transferredCommands.count(); ++i)
            if (point.transferredCommands[i].second < curTime)
                point.transferredCommands.remove(i--);
    };
    removeExpiredTime(point1);
    removeExpiredTime(point2);

    auto func = [&message, curTime](Point& p1, Point& p2) -> int
    {
        if (p1.socket)
        {
            if (p1.socket->socketDescriptor() == message->socketDescriptor())
            {
                if (!p2.socket)
                {
                    log_error_m << "Unable forwarding command " << CommandNameLog(message->command())
                                << " from socket '" << p1.name << "'"
                                << " to socket '" << p2.name << "'"
                                << "; Socket '" << p2.name << "' is not available";

                    data::Error error;
                    error.commandId = message->command();
                    error.messageId = message->id();
                    error.description = "Unable forwarding message to socket '" + p2.name + "'"
                                        ". Socket is not available";

                    Message::Ptr m = createJsonMessage(error);
                    p1.socket->send(m);
                    return 0;
                }

                if (message->type() == Message::Type::Command)
                {
                    quint64 time = (message->maxTimeLife() != 0)
                                   ? message->maxTimeLife() : (curTime + 10);
                    p1.transferredCommands.append(qMakePair(message->id(), time));
                    p2.socket->send(message);
                    return 1;
                }
                else if (message->type() == Message::Type::Answer)
                {
                    bool found = false;
                    for (int i = 0; i < p2.transferredCommands.count(); ++i)
                        if (p2.transferredCommands[i].first == message->id())
                        {
                            found = true;
                            p2.transferredCommands.remove(i);
                            break;
                        }

                    if (found)
                    {
                        p2.socket->send(message);
                        return 1;
                    }

                    log_error_m << "Unable forwarding command " << CommandNameLog(message->command())
                                << " from socket '" << p1.name << "'"
                                << " to socket '" << p2.name << "'"
                                << ". Timeout has expired. Message id: " << message->id();

                    data::Error error;
                    error.commandId = message->command();
                    error.messageId = message->id();
                    error.description = "Unable forwarding message to socket '" + p2.name + "'"
                                        ". Timeout for this message has expired";

                    Message::Ptr m = createJsonMessage(error);
                    p1.socket->send(m);
                    return 0;
                }
                else // Message::Type::Event
                {
                    break_point
                    p2.socket->send(message);
                    return 1;
                }
            }
        }
        return -1;
    };

    int res = func(point1, point2);
    if (res != -1)
        return bool(res);

    res = func(point2, point1);
    if (res != -1)
        return bool(res);

    log_error_m << "Failed forwarding message " << CommandNameLog(message->command());
    return false;
}

} // namespace transport
} // namespace communication

void pagingPrepare(const communication::data::PagingInfo &pi, QString& sql)
{
    if (pi.limit > 0)
    {
        sql = QString("SELECT FIRST %1 SKIP %2 ").arg(pi.limit).arg(pi.offset)
              + sql.mid(sql.indexOf("SELECT", Qt::CaseInsensitive) + 6);
    }
}

void assignTaskT(communication::TaskType& taskType, const QSqlRecord& r)
{
    QString taskTypeStr;
    sql::assignValue(taskTypeStr, r, "TASK_TYPE");
    taskType = communication::getTaskType(taskTypeStr);
}

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
