/*****************************************************************************
  Модуль с функциями общего назначения

*****************************************************************************/

#pragma once

#include "commands/paging.h"
#include "commands/task.h"

#include "shared/defmac.h"
#include "shared/simple_timer.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/qhashex.h"
#include "shared/qt/communication/message.h"
#include "shared/qt/communication/transport/tcp.h"

#include <QtCore>
#include <QtSql>

#ifndef QTEXT
#define QTEXT(MSG) QString::fromUtf8(u8##MSG)
#endif

#define READ_FROM_MESSAGE(MESSAGE, DATA) { \
    SResult res = readFromMessage(MESSAGE, DATA); \
    if (!res) { \
        communication::data::MessageError err {error::json_parse_error}; \
        err.description = res.description(); \
        communication::Message::Ptr answer = MESSAGE->cloneForAnswer(); \
        writeToJsonMessage(err, answer); \
        webCon().send(answer); \
        return; \
    }}

#define Q_FIRST_CHECK(ERRMSG) \
    if (!q.first()) { \
        writeToJsonMessage(ERRMSG, answer); \
        webCon().send(answer); \
        return; \
    }

namespace communication {
namespace transport {

/**
  Используется для обработки подключений от web-севера
*/
class WebCon
{
public:
    void init(SocketDescriptor);

    bool isEmpty() const;
    bool isConnected() const;

    void send(const Message::Ptr& message) const;
    void reset();

    tcp::Socket::Ptr socket() const;
    SocketDescriptor socketDescriptor() const;

    bool isAuthorized() const;
    void setAuthorized(bool);

    bool waitingAuthorizExpired() const;

private:
    tcp::Socket::Ptr _socket;
    SocketDescriptor _socketDescriptor = {-1};

    bool _isAuthorized = {false};
    simple_timer _authorizTimer;

    mutable std::atomic_flag  _lock = ATOMIC_FLAG_INIT;
};

WebCon& webCon();

/**
  Используется для обработки подключений к ФОМС-серверу
*/
class FomsCon
{
public:
    bool isConnected() const;

    bool isAuthorized() const;
    void setAuthorized(bool);

    tcp::Socket::Ptr socket() const;
    bool send(const Message::Ptr& message) const;
    void disconnect();

private:
    FomsCon() = default;
    DISABLE_DEFAULT_COPY(FomsCon)

    bool _isAuthorized = {false};
    tcp::Socket::Ptr _socket {new tcp::Socket};

    mutable std::atomic_flag  _lock = ATOMIC_FLAG_INIT;
    template<typename T, int> friend T& ::safe_singleton();
};

FomsCon& fomsCon();

// Сокет для подключения к пулу пользователей (ActiveDirectory)
tcp::Socket::Ptr& upoolCon();

/**
  Используется для перенаправления команд между сокетами
*/
struct RouteCommands
{
    QSetEx<QUuidEx> commands;

    struct Point
    {
        QString name;
        tcp::Socket::Ptr socket;

        // Идентификаторы переданных команд и время ожидания ответа
        QVector<QPair<QUuidEx, quint64/*Time*/>> transferredCommands;
    };
    Point point1;
    Point point2;

    bool forwarding(const Message::Ptr&);
};

} // namespace transport
} // namespace communication

/**
  Автосчетчик для инкрементальных типов
*/
//template<typename T>
//struct AutoCounter
//{
//    T& cnt;
//    AutoCounter(T& counter) : cnt(counter) {++cnt;}
//    ~AutoCounter() {--cnt;}
//};


// Функция предподготовки SQL-запроса для постраничного выбора данных
void pagingPrepare(const communication::data::PagingInfo&, QString& sql);

//
void assignTaskT(communication::TaskType&, const QSqlRecord&);
