#include "garbage_collector.h"
#include "commands/task.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "shared/logger/logger.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "GarbageCollector")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "GarbageCollector")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "GarbageCollector")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "GarbageCollector")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "GarbageCollector")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "GarbageCollector")

#define QUERY_ERROR(TEXT) u8"Ошибка в запросе '" TEXT "'"

using namespace sql;
using namespace communication;

GarbageCollector& garbageCollector()
{
    return ::safe_singleton<GarbageCollector>();
}

bool GarbageCollector::collect()
{
    /*
      Важно: коннект к БД создаем только один раз. Это нужно для того чтобы
      в случае прерывания выполнения программы можно было  корректно  выйти
      из потока. В этом случае текущий dbcon будет  помечен  как  "aborted",
      и никакие  запросы  с  ним  более  не  выполнятся.  Если  же  вызвать
      dbpool().connect()  повторно,  то  будет  создано  новое  подключение,
      и поток может начать выполнение длительного запроса.  Далее  основной
      поток программы может не дождется остановки этого потока,  и завершит
      его принудительно. В результате этих  действий  программа  завершится
      крашем
    */
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    /*
      Удаление всех пользователей с флагом IS_VALID = 0,
      и датой последнего входа более 3 месяцев.
    */
    if (!sql::exec(q,
        " SELECT ID FROM USERS "
        " WHERE IS_VALID = 0 AND DATEDIFF (month, LAST_LOGON, current_date) > 3 "))
    {
        log_error_m << QUERY_ERROR("Выбор пользователей");
        return false;
    }

    while (q.next())
    {
        QUuidEx userId;
        assignValue(userId, q.record(), "ID");

        // Удалить логи, связанные с пользователем
        if (!sql::exec(q, "DELETE FROM EVENT_LOG WHERE USER_ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Очистка логов");
            return false;
        }

        // Удалить данные моделей, созданных пользователем
        if (!sql::exec(q,
            " DELETE FROM MODEL_DATA                               "
            " WHERE ID IN (SELECT ID FROM MODEL WHERE USER_ID = ?) ", userId))
        {
            log_error_m << QUERY_ERROR("Удаление данных модели");
            return false;
        }

        // Удалить настройки моделей, созданных пользователем
        if (!sql::exec(q,
            " DELETE FROM MODEL_XGB                                "
            " WHERE ID IN (SELECT ID FROM MODEL WHERE USER_ID = ?) ", userId))
        {
            log_error_m << QUERY_ERROR("Удаление настроек модели");
            return false;
        }

        // Удалить модели, созданных пользователем
        if (!sql::exec(q, "DELETE FROM MODEL WHERE USER_ID = ?",
                       userId))
        {
            log_error_m << QUERY_ERROR("Удаление модели");
            return false;
        }

        // Удалить данные отчета, созданных пользователем
        if (!sql::exec(q,
            " DELETE FROM REPORT_DATA                               "
            " WHERE ID IN (SELECT ID FROM REPORT WHERE USER_ID = ?) ", userId))
        {
            log_error_m << QUERY_ERROR("Удаление данных отчета");
            return false;
        }

        // Удалить отчеты, созданные пользователем
        if (!sql::exec(q, "DELETE FROM REPORT WHERE USER_ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Удаление отчетов");
            return false;
        }

        // Удалить федеральные отчеты, созданных пользователем
        if (!sql::exec(q, "DELETE FROM REPORT_FED WHERE USER_ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Удаление федеральных отчетов");
            return false;
        }

        // Удалить данные оценок, созданных пользователем
        if (!sql::exec(q,
            " DELETE FROM SCORE_DATA                                     "
            " WHERE SCORE_ID IN (SELECT ID FROM SCORE WHERE USER_ID = ?) ", userId))
        {
            log_error_m << QUERY_ERROR("Удаление даннных для оценок");
            return false;
        }

        // Удалить оценки, созданных пользователем
        if (!sql::exec(q, "DELETE FROM SCORE WHERE USER_ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Удалить оценки");
            return false;
        }

        // Удалить запросы на синхронизацию, созданные задачами пользователя
        if (!sql::exec(q, "DELETE FROM SYNC_PLANNING WHERE USER_ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Удаление оценкок пользователя");
            return false;
        }

        // Удалить задачи пользователя
        if (!sql::exec(q, "DELETE FROM TASK WHERE USER_ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Удаление задач пользователя");
            return false;
        }

        // Удалить пользователя
        if (!sql::exec(q, "DELETE FROM USERS WHERE ID = ?", userId))
        {
            log_error_m << QUERY_ERROR("Удаление пользователя");
            return false;
        }
    }
    // Очистка записей логов старше 3-ёх месяцев.
    if (!sql::exec(q,
        " DELETE FROM EVENT_LOG "
        " WHERE DATEDIFF (month, CREATE_DATE, current_date) > 3 "))
    {
        log_error_m << QUERY_ERROR("Удаление лог-таблицы");
        return false;
    }

    QVector<QUuidEx> removeList;

    // Выборка моделей, для которых не существует пользователей.
    if (!sql::exec(q,
         " SELECT                                      "
         "   MODEL.ID                                  "
         " FROM                                        "
         "   MODEL                                     "
         " LEFT JOIN USERS ON MODEL.USER_ID = USERS.ID "
         " WHERE                                       "
         "   USERS.ID IS NULL                          "
         "   AND                                       "
         "   MODEL.USER_ID <> ?                        ",
         QUuidEx(0)))
    {
        log_error_m << QUERY_ERROR("Выборка моделей не существующих пользователей");
        return false;
    }

    // Сохранение выбранных моделей
    while (q.next())
    {
        QUuidEx id;
        assignValue(id, q.record(), "ID");
        removeList.append(id);
    }

    // Выборка моделей, для которых отсутствуют задачи применения,
    // и которые были удалены пользователем
    if (!sql::exec(q,
         " SELECT                                     "
         "   MODEL.ID                                 "
         " FROM                                       "
         "   MODEL                                    "
         " LEFT JOIN TASK ON TASK.MODEL_ID = MODEL.ID "
         " WHERE                                      "
         "   TASK.ID IS NULL                          "
         "   AND                                      "
         "   IS_ARCHIVE = 1                           "))
    {
        log_error_m << QUERY_ERROR("Выборка удаленных моделей");
        return false;
    }

    // Сохранение выбранных моделей
    while (q.next())
    {
        QUuidEx id;
        assignValue(id, q.record(), "ID");
        removeList.append(id);
    }

    // Удаление всех связанных с моделью данных
    for (QUuidEx id : removeList)
    {
        QSqlQuery q2 {dbcon->createResult()};

        // Очистка бинарных данных моделей.
        if (!sql::exec(q2, "DELETE FROM MODEL_DATA WHERE MODEL_DATA.ID = ?", id))
        {
            log_error_m << QUERY_ERROR("Удаление бинарных данных модели");
            return false;
        }

        // Очистка настроек моделей.
        if (!sql::exec(q2, "DELETE FROM MODEL_XGB WHERE MODEL_XGB.ID = ?", id))
        {
            log_error_m << QUERY_ERROR("Удаление настроек модели");
            return false;
        }

        if (!sql::exec(q2,
             " DELETE FROM                         "
             " MODEL_DICTIONARY                    "
             " WHERE MODEL_DICTIONARY.MODEL_ID = ? ", id))
        {
            log_error_m << QUERY_ERROR("Удаление словаря модели");
            return false;
        }

        if (!sql::exec(q2,
             " DELETE FROM                                   "
             " MODEL_DICTIONARY_MOST_FREQ                    "
             " WHERE MODEL_DICTIONARY_MOST_FREQ.MODEL_ID = ? ", id))
        {
            log_error_m << QUERY_ERROR("Удаление частотного словаря модели");
            return false;
        }

        if (!sql::exec(q2, " DELETE FROM MODEL WHERE ID = ?", id))
        {
            log_error_m << QUERY_ERROR("Удаление модели");
            return false;
        }
    }

    removeList.clear();

    // Выборка оценок не существующих пользовтелей
    if (!sql::exec(q,
         " SELECT                                        "
         "   SCORE.ID                                    "
         " FROM                                          "
         "   SCORE                                       "
         " LEFT JOIN USERS ON SCORE.USER_ID = USERS.ID   "
         " WHERE                                         "
         "   SCORE.USER_ID IS NULL                       "
         "   AND                                         "
         "   SCORE.USER_ID <> ?                          ", QUuidEx(0)))
    {
        log_error_m << QUERY_ERROR("Выборка оценок с удалёнными пользователями");
        return false;
    }

    // Сохранение выбранных оценок
    while (q.next())
    {
        QUuidEx id;
        assignValue(id, q.record(), "ID");
        removeList.append(id);
    }

    // Выборка оценок, для которых отсутствуют отчеты и которые были удалены пользователем
    if (!sql::exec(q,
         " SELECT                                         "
         "   SCORE.ID                                     "
         " FROM                                           "
         "   SCORE                                        "
         " LEFT JOIN REPORT ON REPORT.SCORE_ID = SCORE.ID "
         " WHERE                                          "
         "   REPORT.ID IS NULL                            "
         "   AND                                          "
         "   SCORE.IS_ARCHIVE = 1                         "))
    {
        log_error_m << QUERY_ERROR("Выборка удаленных оценок");
        return false;
    }

    // Сохранение выбранных оценок
    while (q.next())
    {
        QUuidEx id;
        assignValue(id, q.record(), "ID");
        removeList.append(id);
    }

    // Удаление данных связанными с оценками
    for (QUuidEx id : removeList)
    {
        QSqlQuery q2 {dbcon->createResult()};

        if (!sql::exec(q2,
            "DELETE FROM SCORE_DATA WHERE SCORE_DATA.SCORE_ID = ?", id))
        {
            log_error_m << QUERY_ERROR("Удаление данных оценок");
            return false;
        }

        // Удаление оценок, для которых отсутсвуют отчеты и которые были удален пользователем
        if (!sql::exec(q2, " DELETE FROM SCORE WHERE ID = ?", id))
        {
            log_error_m << QUERY_ERROR("Удаление оценки");
            return false;
        }
    }

    // Очистка отчетов удаленных пользователей.
    if (!sql::exec(q,
         " DELETE FROM REPORT WHERE ID IN                 "
         " (                                              "
         "   SELECT REPORT.ID                             "
         "     FROM                                       "
         "   REPORT                                       "
         "   LEFT JOIN USERS ON REPORT.USER_ID = USERS.ID "
         "   WHERE                                        "
         "     REPORT.USER_ID IS NULL                     "
         "     AND REPORT.USER_ID <> ?                    "
         " )                                              ", QUuidEx(0)))
    {
        log_error_m << QUERY_ERROR("Удаление отчетов");
        return false;
    }

    // Очистка федеральных отчетов удаленных пользователей.
    if (!sql::exec(q,
         " DELETE FROM REPORT_FED WHERE ID IN                 "
         " (                                                  "
         "   SELECT REPORT_FED.ID                             "
         "     FROM                                           "
         "   REPORT_FED                                       "
         "   LEFT JOIN USERS ON REPORT_FED.USER_ID = USERS.ID "
         "   WHERE                                            "
         "     REPORT_FED.USER_ID IS NULL                     "
         "     AND REPORT_FED.USER_ID <> ?                    "
         " )                                                  ", QUuidEx(0)))
    {
        log_error_m << QUERY_ERROR("Удаление федеральных отчетов");
        return false;
    }

    // Очистка данных отчетов удаленных пользователей.
    if (!sql::exec(q,
         " SELECT                                 "
         "   T.REPORT_ID                          "
         " FROM                                   "
         " (                                      "
         "   SELECT                               "
         "     REPORT_DATA.REPORT_ID              "
         "   FROM                                 "
         "     REPORT_DATA                        "
         "   GROUP BY                             "
         "     REPORT_DATA.REPORT_ID              "
         " ) T                                    "
         " LEFT JOIN                              "
         " (                                      "
         "   SELECT                               "
         "     ID, TASK_ID                        "
         "   FROM                                 "
         "     REPORT                             "
         "   UNION                                "
         "   SELECT                               "
         "     ID, TASK_ID                        "
         "   FROM                                 "
         "     REPORT_FED                         "
         " ) R ON R.ID = T.REPORT_ID              "
         " LEFT JOIN TASK ON R.TASK_ID  = TASK.ID "
         " WHERE                                  "
         "   R.ID IS NULL                         "
         "   AND                                  "
         "   (                                    "
         "     TASK.EXEC_STATUS <> ?              "
         "     OR                                 "
         "     TASK.EXEC_STATUS IS NULL           "
         "   )                                    ", TaskExecStatus::Running))
    {
        log_error_m << QUERY_ERROR("Удаление данных отчетов");
        return false;
    }

    while (q.next())
    {
        QUuidEx id;
        assignValue(id, q.record(), "ID");

        QSqlQuery q2 {dbcon->createResult()};

        if (!sql::exec(q2,
            "DELETE FROM REPORT_DATA WHERE REPORT_DATA.ID = ?", id))
        {
            log_error_m << QUERY_ERROR("Удаление словарей");
            return false;
        }
    }

    return true;
}

void GarbageCollector::run()
{
    log_info_m << u8"Started";
    log_info_m << u8"Процесс сборки мусора запущен";

    _threadId = trd::gettid();

    if (!collect())
        log_info_m << u8"Процесс сборки мусора прерван";
    else
        log_info_m << u8"Процесс сборки мусора завершен";

    log_info_m << u8"Stopped";
}

void GarbageCollector::threadStopEstablished()
{
    dbpool().abortOperation(_threadId);
}
