#include "http_connector.h"
#include "observer.h"

#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/logger_operators.h"
#include "shared/qt/communication/transport/tcp.h"

#include <unistd.h>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "HttpConnector")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "HttpConnector")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "HttpConnector")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "HttpConnector")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "HttpConnector")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "HttpConnector")

namespace monitoring {

HttpConnector& httpConnector()
{
    return ::safe_singleton<HttpConnector>();
}

bool HttpConnector::init()
{
    QHostAddress hostAddress = QHostAddress::Any;
    config::readHostAddress("monitoring.external.address", hostAddress);

    int port = 8088;
    config::base().getValue("monitoring.external.port", port);

    int attempt = 0;
    while (!QTcpServer::listen(hostAddress, port))
    {
       if (++attempt > 10)
           break;
       usleep(200*1000);
    }
    if (attempt > 10)
        log_error_m << "Start listener of connection to " << HostPoint(hostAddress, port)
                    << " is failed. Detail: " << errorString();
    else
       log_verbose_m << "Start listener of connection with params: "
                     << serverAddress() << ":" << serverPort();

    return (attempt <= 10);
}

void HttpConnector::reinit()
{
    QTcpServer::close();
    init();
}

void HttpConnector::incomingConnection(communication::SocketDescriptor handle)
{
    QTcpSocket* socket = new QTcpSocket(this);
    socket->setSocketDescriptor(handle);

    chk_connect_d(socket, SIGNAL(readyRead()), this, SLOT(onReadyRead()));
    chk_connect_d(socket, SIGNAL(disconnected()), this, SLOT(onDisconnected()));
}

void HttpConnector::onReadyRead()
{
    QTcpSocket* socket = qobject_cast<QTcpSocket*>(sender());
    QByteArray ba = socket->readAll();
    log_debug_m << ba;

    ba = "HTTP/1.0 200 Ok\n"
         "Content-Type: application/json; charset=\"utf-8\"\n\n";

    data::Monitoring::Ptr monitoring = monitoring::observer().get();
    ba += monitoring->toJson() + "\n";

    log_debug_m << "Sent to http monitor: " << ba;

    socket->write(ba);
    socket->disconnectFromHost();
}

void HttpConnector::onDisconnected()
{
    QTcpSocket* socket = qobject_cast<QTcpSocket*>(sender());
    socket->close();
    socket->deleteLater();
}

} // namespace monitoring

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
