#include "mail_smtp.h"
#include "event_log.h"

//#include "shared/utils.h"
#include "shared/simple_ptr.h"
#include "shared/simple_timer.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"

#include <QHostInfo>
#include <QSslSocket>
#include <QStringList>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "MailSmtp")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "MailSmtp")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "MailSmtp")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "MailSmtp")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "MailSmtp")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "MailSmtp")

namespace monitoring {

MailSmtp& mailSmtp()
{
    return ::safe_singleton<MailSmtp>();
}

bool MailSmtp::init()
{
    return true;
}

void MailSmtp::awake()
{
    QMutexLocker locker(&_threadLock); (void) locker;
    _threadCond.wakeAll();
}

void MailSmtp::threadStopEstablished()
{
    awake();
}

void MailSmtp::send(const MailSmtp::MailData::Ptr& mail)
{
    QMutexLocker locker(&_threadLock); (void) locker;
    if (mail->triggerType != "evlog")
        for (MailData::Ptr m : _mailList)
            if (m->triggerId == mail->triggerId)
                return;

    _mailList.append(mail);

    log_debug2_m << "Mail was append to engine send"
                 << ". Subject: '" << mail->subject << "'"
                 << ". Addresses: " << mail->addresses.join(",");
}

static bool waitForResponse(QAbstractSocket* socket, int& responseCode)
{
    responseCode = 0;
    int breakFlag = 0;
    QString responseText;
    QString responseLine;

    while (true)
    {
        if (!socket->waitForReadyRead(30*1000 /*30 сек*/))
        {
            log_error_m << "Timeout expired for response from server";
            return false;
        }

        while (socket->canReadLine())
        {
            responseLine = socket->readLine();
            responseText += responseLine;

            // Extract the respose code from the server's responce (first 3 digits)
            responseCode = responseLine.left(3).toInt();

            if (responseCode == 400)
            {
                log_error_m << "An error has occurred on the client side";
                breakFlag = 2;
                break;
            }
            if (responseCode == 500)
            {
                log_error_m << "An error occurred on the server side";
                breakFlag = 2;
                break;
            }
            if (responseLine[3] == ' ')
            {
                breakFlag = 1;
                break;
            }
        }
        if (breakFlag != 0)
            break;
    }

    log_debug2_m << "Server response code: " <<  responseCode;
    log_debug2_m << "Server response: " << responseText;
    return (breakFlag == 1);
}

static bool sendMessage(QAbstractSocket* socket, const QString &text)
{
    QByteArray message = text.toUtf8();
    log_debug2_m << "Send message: " << message;

    socket->write(message + "\r\n");
    if (!socket->waitForBytesWritten(30*1000 /*30 сек*/))
    {
        log_error_m << "Timeout expired to write data to server";
        return false;
    }
    return true;
}

void MailSmtp::run()
{
    log_info_m << "Started";

    while (true)
    {
        CHECK_QTHREADEX_STOP

        { //Block for QMutexLocker
            QMutexLocker locker(&_threadLock); (void) locker;
            if (_mailList.isEmpty())
            {
                _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
                continue;
            }
        }

        bool failConfRead = false;

        QString hostName;
        if (!config::base().getValue("monitoring.mail_smtp.address", hostName))
            failConfRead |= true;

        int port = 0;
        if (!config::base().getValue("monitoring.mail_smtp.port", port))
            failConfRead |= true;

        QString userName;
        if (!config::base().getValue("monitoring.mail_smtp.user", userName))
            failConfRead |= true;

        QString userPassw;
        if (!config::base().getValue("monitoring.mail_smtp.password", userPassw))
            failConfRead |= true;

        if (failConfRead)
        {
            QMutexLocker locker(&_threadLock); (void) locker;
            _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            continue;
        }

        container_ptr<MailData> mail;
        { //Block for QMutexLocker
            QMutexLocker locker(&_threadLock); (void) locker;
            if (!_mailList.isEmpty())
            {
                mail = _mailList.first();
                _mailList.removeAt(0);
            }
        }
        if (mail.empty())
        {
            QMutexLocker locker(&_threadLock); (void) locker;
            _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            continue;
        }
        log_verbose_m << "Try send mail message '" << mail->subject << "'";

        simple_ptr<QAbstractSocket> socket;
        if (port == 465)
        {
            socket = simple_ptr<QAbstractSocket>(new QSslSocket(0));
            chk_connect_d(socket.get(), SIGNAL(disconnected()), this, SLOT(socketDisconnected()))

            log_verbose_m << "Try connect to host (with encryption) " << hostName << ":" << port;
            static_cast<QSslSocket*>(socket.get())->connectToHostEncrypted(hostName, port);
        }
        else
        {
            socket = simple_ptr<QAbstractSocket>(new QTcpSocket(0));
            chk_connect_d(socket.get(), SIGNAL(disconnected()), this, SLOT(socketDisconnected()))

            log_verbose_m << "Try connect to host (without encryption) " << hostName << ":" << port;
            socket->connectToHost(hostName, port);
        }

        if (!socket->waitForConnected(10*1000 /*10 сек*/))
        {
            log_error_m << "Failed connect to host " << hostName << ":" << port
                        << ". Error code: " << int(socket->error())
                        << ". Detail: " << socket->errorString();

            QMutexLocker locker(&_threadLock); (void) locker;
            _mailList.prepend(mail);
            _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            continue;
        }
        log_verbose_m << "Connect to host " << hostName << ":" << port;

        int responseCode; (void) responseCode;

        #define WAIT_FOR_RESPONSE() \
            if (!waitForResponse(socket, responseCode)) { \
                log_verbose_m << "Close socket connection"; \
                socket->close(); \
                { QMutexLocker locker(&_threadLock); (void) locker; \
                  _mailList.prepend(mail); } \
                sleep(2*60 /*2 мин*/); \
                continue; \
            }

        #define SEND_MESSAGE(MESSAGE) \
            if (!sendMessage(socket, MESSAGE)) { \
                log_verbose_m << "Close socket connection"; \
                socket->close(); \
                { QMutexLocker locker(&_threadLock); (void) locker; \
                  _mailList.prepend(mail); } \
                sleep(2*60 /*2 мин*/); \
                continue; \
            }

        #define CHECK_RESPONSE_CODE(CODE) \
            if (responseCode != CODE) { \
                log_error_m << "Incorrect response code. Expected code " << CODE \
                            << ", received code " << responseCode; \
                log_verbose_m << "Close socket connection"; \
                socket->close(); \
                { QMutexLocker locker(&_threadLock); (void) locker; \
                  _mailList.prepend(mail); } \
                sleep(2*60 /*2 мин*/); \
                continue; \
            }

        // Ждем первое сообщение от почтового сервера
        WAIT_FOR_RESPONSE()

        // If the response code is not 220 (Service ready)
        // means that is something wrong with the server
        CHECK_RESPONSE_CODE(220)

        // Send a EHLO/HELO message to the server
        // The client's first command must be EHLO/HELO
        SEND_MESSAGE("EHLO localhost")
        WAIT_FOR_RESPONSE()

        // The response code needs to be 250
        CHECK_RESPONSE_CODE(250)

        // Sending command: AUTH LOGIN
        SEND_MESSAGE("AUTH LOGIN")
        WAIT_FOR_RESPONSE()

        // Wait for 334 response code
        CHECK_RESPONSE_CODE(334)

        // Send the username in base64
        SEND_MESSAGE(userName.toUtf8().toBase64());
        WAIT_FOR_RESPONSE()

        // Wait for 334
        CHECK_RESPONSE_CODE(334)

        // Send the password in base64
        SEND_MESSAGE(userPassw.toUtf8().toBase64())
        WAIT_FOR_RESPONSE()

        // If the response is not 235 then the authentication was faild
        CHECK_RESPONSE_CODE(235)

        log_verbose_m << "Success authentication. User: " << userName;

        // Send the MAIL command with the sender
        SEND_MESSAGE("MAIL FROM:<" + userName + ">")
        WAIT_FOR_RESPONSE()
        CHECK_RESPONSE_CODE(250)

        // Send RCPT command for each recipient
        for (const QString& address : mail->addresses)
        {
            SEND_MESSAGE("RCPT TO:<" + address + ">")
            WAIT_FOR_RESPONSE()

            if (responseCode != 250)
                break;
        }
        CHECK_RESPONSE_CODE(250)

        // Send DATA command
        SEND_MESSAGE("DATA")
        WAIT_FOR_RESPONSE()
        CHECK_RESPONSE_CODE(354)

        QStringList mailAddresses;
        for (const QString& address : mail->addresses)
            mailAddresses.append("<" + address + ">");

        QString data;
        data += "To: " + mailAddresses.join(",") + "\n";
        data += "From: " + userName + "\n";
        data += "Subject: " + mail->subject + "\n";

        //Let's intitiate multipart MIME with cutting boundary "frontier"
        data += "MIME-Version: 1.0\n";
        data += "Content-Type: multipart/mixed; boundary=frontier\n\n";

        data += "--frontier\n";
        //message.append( "Content-Type: text/html\n\n" );  //Uncomment this for HTML formating, coment the line below
        //message.append( "Content-Type: text/plain\n\n" );
        data += "Content-Type: text/plain;charset=utf-8\n\n";
        data += mail->body;
        data += "\n\n";

        data += "--frontier--\n";

        data.replace("\n", "\r\n");
        data.replace("\r\n.\r\n", "\r\n..\r\n");

        SEND_MESSAGE(data)

        // Send \r\n.\r\n to end the mail data
        SEND_MESSAGE(".")
        WAIT_FOR_RESPONSE()
        CHECK_RESPONSE_CODE(250)

        SEND_MESSAGE("QUIT")
        WAIT_FOR_RESPONSE()
        CHECK_RESPONSE_CODE(221)

        if (socket->state() != QAbstractSocket::UnconnectedState)
        {
            socket->disconnectFromHost();
            if (socket->state() != QAbstractSocket::UnconnectedState)
                socket->waitForDisconnected(2*1000 /*2 сек*/);
        }
        socket->close();

        const char* msg = u8"Почтовое уведомление '%1' отправлено следующим адресатам: %2";
        log_verbose_m << EventLog(msg, mail->subject, mailAddresses.join(","));

        // Пауза между отправлениями
        sleep(10);

    } // while (true)

    log_info_m << "Stopped";
}

void MailSmtp::socketDisconnected()
{
    log_verbose_m << "Disconnected from host";
}

} // namespace monitoring

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m

