#include "notifier.h"
#include "observer.h"
#include "mail_smtp.h"
#include "event_log.h"
#include "functions.h"
#include "commands/event_log.h"
#include "database/connect.h"
#include "database/sql_func.h"

#include "shared/utils.h"
#include "shared/simple_timer.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Notifier")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Notifier")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Notifier")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Notifier")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Notifier")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Notifier")

namespace monitoring {

using namespace db::firebird;
using namespace sql;

Notifier& notifier()
{
    return ::safe_singleton<Notifier>();
}

bool Notifier::init()
{
    return true;
}

void Notifier::run()
{
    log_info_m << "Started";

    _threadId = trd::gettid();

    // Значения параметров мониторинга за последнюю минуту
    QList<int> cpu;
    QList<int> ram;
    QList<int> hdd;
    QList<int> cputmp;
    QList<int> foms;

    simple_timer timer;
    simple_timer timer2;

    while (true)
    {
        CHECK_QTHREADEX_STOP

        data::Monitoring::Ptr monitor = monitoring::observer().get();

        if (monitor->hardware.cpu.total != 0)
        {
            // Уже в процентах
            cpu.append(monitor->hardware.cpu.system);
        }
        if (monitor->hardware.ram.total != 0)
        {
            // Переводим в проценты
            ram.append(double(monitor->hardware.ram.system) / monitor->hardware.ram.total * 100);
        }
        if (monitor->hardware.hdd.total != 0)
        {
            // Переводим в проценты
            hdd.append(double(monitor->hardware.hdd.system) / monitor->hardware.hdd.total * 100);
        }

        // Температура задается в градусах
        cputmp.append(monitor->hardware.cpuMaxCoreTemp);

        // Задается как логический признак ДА/НЕТ
        foms.append(monitor->software.fomsIsActive ? 1 : 0);

        if (cpu.count() < 3)
        {
            sleep(5);
            continue;
        }

        // Удаляем показатели старше 1 минуты
        while (cpu.count()    > 12) cpu.removeAt(0);
        while (ram.count()    > 12) ram.removeAt(0);
        while (hdd.count()    > 12) hdd.removeAt(0);
        while (cputmp.count() > 12) cputmp.removeAt(0);
        while (foms.count()   > 12) foms.removeAt(0);

        if (timer.elapsed() > 15*1000 /*15 сек*/)
        {
            qint16 cpuAvg = utl::average(cpu);
            qint16 ramAvg = utl::average(ram);
            qint16 hddAvg = utl::average(hdd);
            qint16 cputmpAvg = utl::average(cputmp);

            checkTrigger("cpu", cpuAvg);
            checkTrigger("ram", ramAvg);
            checkTrigger("hdd", hddAvg);
            checkTrigger("cputmp", cputmpAvg);

            for (int i : foms)
                if (i == 0)
                {
                    checkTrigger("foms", 1);
                    break;
                }

            timer.reset();
        }

        if (timer2.elapsed() > 60*1000 /*1 мин*/)
        {
            checkTrigger("evlog", 1);
            timer2.reset();
        }
        sleep(5);

    } // while (true)

    log_info_m << "Stopped";
}

void Notifier::threadStopEstablished()
{
    dbpool().abortOperation(_threadId);
}

void Notifier::checkTrigger(const QString& triggerType, qint16 threshold)
{
    if (threadStop())
        return;

    QString sql =
        " SELECT                          "
        "   ID                            "
        "  ,NAME                          "
        "  ,TRIGGER_TYPE                  "
        "  ,THRESHOLD                     "
        "  ,EMISSION_INTERVAL             "
        "  ,EMISSION_NEXT                 "
        "  ,SUBJECT                       "
        "  ,MESSAGE_TEMPLATE              "
        " FROM                            "
        "   NOTIFY_TRIGGER                "
        " WHERE                           "
        "   TRIGGER_TYPE = :TRIGGER_TYPE  "
        "   AND                           ";

    if ((triggerType != "foms") && (triggerType != "evlog"))
        sql += " THRESHOLD <= :THRESHOLD  "
               " ORDER BY                 "
               "   THRESHOLD DESC         ";
    else
        sql += " THRESHOLD > 0 ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
    {
        sleep(15);
        return;
    }
    bindValue(q, ":TRIGGER_TYPE", triggerType);

    if ((triggerType != "foms") && (triggerType != "evlog"))
        bindValue(q, ":THRESHOLD", threshold);

    if (!q.exec())
    {
        sleep(15);
        return;
    }
    if (q.first())
    {
        QUuidEx   triggerId;
        QString   triggerName;
        qint16    emissionInterval = 30;
        QDateTime emissionNext = QDateTime(QDate(1900, 1, 1));
        QString   subject;
        QString   messageTemplate;

        QSqlRecord r = q.record();
        assignValue(triggerId        , r, "ID                ");
        assignValue(triggerName      , r, "NAME              ");
        assignValue(emissionInterval , r, "EMISSION_INTERVAL ");
        assignValue(emissionNext     , r, "EMISSION_NEXT     ");
        assignValue(subject          , r, "SUBJECT           ");
        assignValue(messageTemplate  , r, "MESSAGE_TEMPLATE  ");

        if (emissionInterval < 1)
        {
            emissionInterval = 1;
            log_warn_m << "Trigger emission interval cannot be less than 1 minute"
                       << ". Emission interval set to 1 minute. Trigger name: "
                       << triggerName;
        }

        if (QDateTime::currentDateTime() > emissionNext)
        {
            if (triggerType == "evlog")
            {
                QString messages = getEvlogMessages();
                if (messages.isEmpty())
                {
                    updateEmissionTime(emissionInterval, triggerId);
                    return;
                }
                messageTemplate += "\n" + messages;
            }

            log_debug2_m << "Trigger corresponds operation threshold"
                         << ". Trigger: '" << triggerName << "'"
                         << ", id: " << triggerId
                         << ", type: " << triggerType
                         << ", threshold: " << threshold;

            QStringList addresses = readAddresses(triggerId);
            if (addresses.isEmpty())
            {
                const char* msg = u8"У триггера '%1' нет сопоставленных "
                                  u8"почтовых адресов, он не будет отправлен";
                log_warn_m << EventLog(msg, triggerName);

                updateEmissionTime(emissionInterval, triggerId);
                return;
            }

            // Отправляем сообщение
            typedef monitoring::MailSmtp::MailData MailData;
            MailData::Ptr mail {new MailData({triggerType, triggerId, addresses,
                                              subject, messageTemplate})};
            mailSmtp().send(mail);
            mailSmtp().awake();

            updateEmissionTime(emissionInterval, triggerId);
        }
    }
}

QStringList Notifier::readAddresses(const QUuidEx& triggerId)
{
    if (threadStop())
        return {};

    QStringList sl;
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                                    "
        "   A.NAME                                  "
        "  ,A.EMAIL                                 "
        " FROM                                      "
        "   NOTIFY_ADDRESS_LINK L                   "
        " LEFT JOIN                                 "
        "   NOTIFY_ADDRESS A ON L.ADDRESS_ID = A.ID "
        " WHERE                                     "
        "   L.TRIGGER_ID = ?                        ", triggerId))
    {
        sleep(15);
        return {};
    }
    while (q.next())
    {
        QString addr;
        assignValue(addr, q.record(), "EMAIL");
        sl.append(addr);
    }
    return sl;
}

void Notifier::updateEmissionTime(qint16 emissionInterval, const QUuidEx& triggerId)
{
    if (threadStop())
        return;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QDateTime emissionTime = QDateTime::currentDateTime().addSecs(emissionInterval * 60);

    if (!sql::exec(q,
         "UPDATE NOTIFY_TRIGGER SET EMISSION_NEXT = ? WHERE ID = ?",
         emissionTime, triggerId))
    {
        sleep(15);
    }
}

QString Notifier::getEvlogMessages()
{
    if (threadStop())
        return {};

    qint64 evlogTimeMarker = -1;
    config::state().getValue("monitoring.evlog_time_marker", evlogTimeMarker);

    QDateTime createDateMarker {QDate::currentDate()};
    if (evlogTimeMarker > 0)
        createDateMarker = QDateTime::fromMSecsSinceEpoch(evlogTimeMarker);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    if (!dbcon->isOpen())
        return {};

    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                          "
        "   E.CREATE_DATE                 "
        "  ,E.USER_ID                     "
        "  ,E.TASK_ID                     "
        "  ,E.LOG_LEVEL                   "
        "  ,E.DESCRIPTION                 "
      //"  ,E.LOCATION                    "
        "  ,U.NAME AS USER_NAME           "
        "  ,T.NAME AS TASK_NAME           "
        " FROM                            "
        "   EVENT_LOG E                   "
        " LEFT JOIN                       "
        "   USERS U ON E.USER_ID = U.ID   "
        " LEFT JOIN                       "
        "   TASK T ON E.TASK_ID = T.ID    "
        " WHERE                           "
        "   E.CREATE_DATE > :CREATE_DATE  "
        "   AND                           "
        "   E.LOG_LEVEL <= :LOG_LEVEL     "
        " ORDER BY                        "
        "   E.CREATE_DATE DESC            ",
        createDateMarker, alog::Level::Error ))
    {
        sleep(15);
        return {};
    }

    bool first = true;
    QString messages;

    while (q.next())
    {
        data::EventLog eventLog;
        QSqlRecord r = q.record();

        assignValue(eventLog.createDate  , r, "CREATE_DATE" );
        assignValue(eventLog.userId      , r, "USER_ID    " );
        assignValue(eventLog.taskId      , r, "TASK_ID    " );
        assignValue(eventLog.userName    , r, "USER_NAME  " );
        assignValue(eventLog.taskName    , r, "TASK_NAME  " );
        assignValue(eventLog.level       , r, "LOG_LEVEL  " );
        assignValue(eventLog.description , r, "DESCRIPTION" );
        //assignValue(eventLog.location  , r, "LOCATION"    );

        messages += "\n";
        messages += eventLog.createDate.toString("dd.MM.yyyy hh:mm:ss.zzz");
        messages += " " + eventLog.description;

        if (!eventLog.taskId.isNull())
        {
            messages += "\n";
            messages += QTEXT("Задача: ") + eventLog.taskName;
            messages += " " + eventLog.taskId.toString();
        }
        if (!eventLog.userId.isNull())
        {
            messages += "\n";
            messages += QTEXT("Пользователь: ") + eventLog.userName;
            messages += " " + eventLog.userId.toString();
        }
        if (!eventLog.taskId.isNull() || !eventLog.userId.isNull())
            messages += "\n";

        if (first)
        {
            config::state().setValue("monitoring.evlog_time_marker",
                                     eventLog.createDate.toMSecsSinceEpoch());
            config::state().save();
            first = false;
        }
    }

    return messages;
}

} // namespace monitoring

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
