#include "system_info.h"

// Linux Core headers
#include <sys/stat.h>
#include <sys/statfs.h>
#include <sys/sysinfo.h>

#include <pwd.h>
#include <unistd.h>


namespace monitoring {

static const qint32 sizeKB = 1024;
const QString SystemInfo::_aisHome = "/var/opt/aisexpert";

SystemInfo::SystemInfo()
{
    _cpuStatusNow = readCpuStats();
}

void SystemInfo::update(data::MonitorHardware& monitoring)
{
    readProcInfo(monitoring);
    readCpuInfo(monitoring);
    readMemoryInfo(monitoring);
    readHddInfo(monitoring);

    monitoring.hdd.process = sizeFolder(_aisHome) / sizeKB / sizeKB; // размер в мегабайтах
}

void SystemInfo::readProcInfo(data::MonitorHardware& monitoring)
{
    monitoring.PID = getpid();
    QFile data {"/proc/" + QString::number(monitoring.PID) + "/status"};
    data.open(QFile::ReadOnly);
    QTextStream stream(&data);
    QString s = stream.readAll();

    static QRegExp rx1 {R"(Name:\s+([A-z]+))"};
    static QRegExp rx2 {R"(VmRSS:\s+(\d+))"};
    static QRegExp rx3 {R"(Threads:\s+(\d+))"};

    auto regVal = [](const QString& s,  const QRegExp& rx) -> QString
    {
        int pos = rx.indexIn(s);
        return (pos != -1) ? rx.cap(1) : QString{};
    };

    monitoring.processName = regVal(s, rx1);
    monitoring.ram.process = regVal(s, rx2).toUInt() / sizeKB;
    monitoring.nThreads    = regVal(s, rx3).toUInt();
}

void SystemInfo::readCpuInfo(data::MonitorHardware& monitoring)
{
    monitoring.cpu.total      = 100; // По просьбе веберов возвращаем 100
    monitoring.cpu.system     = cpuLoad();
    monitoring.cpu.process    = quint32(procLoad());
    monitoring.cpuMaxCoreTemp = cpuMaxTemp();
}

void SystemInfo::readMemoryInfo(data::MonitorHardware& monitoring)
{
    QFile data("/proc/meminfo");
    data.open(QFile::ReadOnly);
    QTextStream stream(&data);
    QString str = stream.readAll();

    static QRegExp rx1 {R"(MemTotal:\s+(\d+))"};
    static QRegExp rx2 {R"(MemFree:\s+(\d+))"};
    static QRegExp rx3 {R"(Buffers:\s+(\d+))"};
    static QRegExp rx4 {R"(Cached:\s+(\d+))"};

    auto regVal = [](const QString& s,  const QRegExp& rx) -> quint32
    {
        int pos = rx.indexIn(s);
        return ((pos != -1) ? rx.cap(1) : QString{}).toUInt();
    };

    monitoring.ram.total  = regVal(str, rx1);
    quint32 freeMemory    = regVal(str, rx2);

    quint32 buffersMemory = regVal(str, rx3);
    quint32 cachedMemory  = regVal(str, rx4);

    monitoring.ram.system = (monitoring.ram.total - freeMemory - buffersMemory - cachedMemory) / sizeKB;
    monitoring.ram.total  /= sizeKB;
}

void SystemInfo::readHddInfo(data::MonitorHardware& monitoring)
{
    // Для информации о жестком диске необходима информация о любом из файлов на нем
    static const char* dirPath {"/var/opt/aisexpert/log/aisexpert.log"};
    struct stat   stst;
    struct statfs stfs;

    if (::stat(dirPath, &stst) == -1)
        return;

    if (::statfs(dirPath, &stfs) == -1)
        return;

    quint32 freeSpace     = stfs.f_bavail  * stst.st_blksize / sizeKB / sizeKB;
    monitoring.hdd.total  = stfs.f_blocks  * stst.st_blksize / sizeKB / sizeKB;
    monitoring.hdd.system = monitoring.hdd.total - freeSpace;
}

quint32 SystemInfo::cpuLoad()
{
    _cpuStatusPrev = _cpuStatusNow;
    _pstatPrev     = _pstatNow;
    _cpuStatusNow  = readCpuStats();

    int size1 = _cpuStatusPrev.size();
    int size2 = _cpuStatusNow.size();

    if (!size1 || !size2 || size1 != size2)
        return 0;

    _cpuTotalTime = 1;

    for (int i = 0; i < size1; ++i)
        _cpuTotalTime += (_cpuStatusNow[i] - _cpuStatusPrev[i]);

    quint32 load = 100 - ((_cpuStatusNow[size2 - 1] - _cpuStatusPrev[size1 - 1]) * 100 / _cpuTotalTime);
    return load;
}

double SystemInfo::procLoad()
{
    double load = 100.0 * (((_pstatNow.utime_ticks + _pstatNow.cutime_ticks)
                           - (_pstatPrev.utime_ticks + _pstatPrev.cutime_ticks))
       / double(_cpuTotalTime));

    return load;
}

double SystemInfo::cpuMaxTemp()
{
    static bool first {true};
    static QStringList tempFiles;

    if (first)
    {
        QDir dir {"/sys/class/hwmon"};
        QFileInfoList files = dir.entryInfoList(QDir::Dirs | QDir::Files | QDir::NoDotAndDotDot);

        QDir coretempDir;
        bool readCoretemp = false;
        for (const QFileInfo& fi : files)
            if (fi.isSymLink())
            {
                QString path = fi.symLinkTarget();
                if (path.contains("/coretemp")
                    && QDir(path).exists())
                {
                    readCoretemp = true;
                    coretempDir.setPath(path);
                    break;
                }
            }

        if (readCoretemp)
        {
            QStringList filters; filters << "temp*_input";
            tempFiles = coretempDir.entryList(filters, QDir::Files);
            if (tempFiles.isEmpty())
            {
                coretempDir.cdUp();
                coretempDir.cdUp();
                tempFiles = coretempDir.entryList(filters, QDir::Files);
            }
        }
        for (QString& fileName : tempFiles)
            fileName = coretempDir.filePath(fileName);

        first = false;
    }

    double maxCoreTemp = 0;
    for (const QString& filePath : tempFiles)
    {
        QFile file {filePath};
        if (file.open(QIODevice::ReadOnly | QIODevice::Text))
        {
            QTextStream stream {&file};
            double temp;
            stream >> temp;
            maxCoreTemp = qMax(maxCoreTemp, temp);
            file.close();
        }
    }

    return maxCoreTemp / 1000;
}

QVector<float> SystemInfo::readCpuStats()
{
    // Открываем данные о процессорном времени в целом:
    QVector<float> ret;
    QFile dataCPU {"/proc/stat"};
    if (!dataCPU.open(QFile::ReadOnly | QFile::Text))
        return ret;

    // Открываем данные о процессорном времени AisExpert:
    qint32 PID = getpid();
    QString path = "/proc/" + QString::number(PID) + "/stat";
    FILE* dataProc = fopen(path.toStdString().c_str(), "r");
    if (dataProc == 0)
        return ret;

    QTextStream cpuStatFile(&dataCPU);

    // Процессорное время в целом:
    QString dummy;
    cpuStatFile >> dummy; (void) dummy;
    for (int i = 0; i < 4; ++i)
    {
        int val;
        cpuStatFile >> val;
        ret.push_back(val);
    }

    // Процессорное время AisExpert:
    fscanf(dataProc, "%*d %*s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u "
                     "%lu %lu %ld %ld",
                     &_pstatNow.utime_ticks, &_pstatNow.stime_ticks,
                     &_pstatNow.cutime_ticks, &_pstatNow.cstime_ticks);

    dataCPU.close();
    fclose(dataProc);
    return ret;
}

quint64 SystemInfo::sizeFolder(const QString& path)
{
    QDir currentFolder {path};
    quint64 totalsize = 0;

    QFileInfoList folderItems = currentFolder.entryInfoList(QDir::Dirs
                                                            | QDir::Files
                                                            | QDir::NoSymLinks
                                                            | QDir::NoDotAndDotDot,
                                                            QDir::SortFlag::Name);
    for (const QFileInfo& i : folderItems)
    {
        const QString& iname = i.fileName();
        if (i.isDir())
            totalsize += sizeFolder(path + "/" + iname);
        else
            totalsize += i.size();
    }
    return totalsize;
}

void SystemInfo::readHostInfo()
{
    _hostInfo.hostName = hostName();
    _hostInfo.userName = userName();
    _hostInfo.uptime   = uptime();
}

QString SystemInfo::hostName()
{
    QFile data("/proc/sys/kernel/hostname");
    data.open(QFile::ReadOnly | QFile::Text);
    QTextStream stream(&data);
    QString str = stream.readLine();
    data.close();
    return str;
}

QString SystemInfo::userName()
{
    uid_t uid = geteuid();
    passwd* pw = getpwuid(uid);
    return QString::fromLatin1(pw->pw_name);
}

long SystemInfo::uptime()
{
    struct sysinfo o;
    sysinfo(&o);
    return o.uptime;
}

} // namespace monitoring
