#include "utils.h"
#include "base_task.h"
#include "sync_plan.h"
#include "task_messages.h"
#include "functions.h"
#include "database/connect.h"
#include "database/sql_func.h"

#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"

#include <unistd.h>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "BaseTask")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "BaseTask")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "BaseTask")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "BaseTask")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "BaseTask")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "BaseTask")

namespace task {

using namespace sql;

BaseTask::BaseTask(TaskType type, const QUuidEx& taskId, const QUuidEx& userId)
    : _type(type),
      _id(taskId),
      _hashId(hash64(_id)),
      _userId(userId)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (sql::exec(q, "SELECT SKIP_SYNC, IS_PERIODIC FROM TASK WHERE ID = ?", taskId))
    {
        if (q.first())
        {
            assignValue(_skipSync,   q.record(), "SKIP_SYNC  ");
            assignValue(_isPeriodic, q.record(), "IS_PERIODIC");
        }
        else
            _baseInit = false;
    }
    else
        _baseInit = false;
}

const QUuidEx& BaseTask::contentId() const
{
    static QUuidEx uuid;
    return uuid;
}

const data::TaskProgress& BaseTask::progress() const
{
    QMutexLocker locker(&_progressLock); (void) locker;

    if (_progress.taskType == TaskType::Undefined)
    {
        _progress.taskType = type();
        _progress.taskId = id();
        _progress.userId = userId();
        _progress.taskExecStatus = TaskExecStatus::Running;
        //_progress.contentId = contentId();
        _progress.isPeriodic = _isPeriodic;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        sql::exec(q, " SELECT        "
                     "   NAME        "
                     "  ,DESCRIPTION "
                     "  ,CREATE_DATE "
                     " FROM          "
                     "   TASK        "
                     " WHERE         "
                     "   ID = ?      ", id());

        if (q.first())
        {
            QSqlRecord r = q.record();
            assignValue(_progress.taskName       , r, "NAME        " );
            assignValue(_progress.taskDescript   , r, "DESCRIPTION " );
            assignValue(_progress.taskCreateDate , r, "CREATE_DATE " );
        }
    }

    _progress.current = _progressCurrent;
    _progress.total = _progressTotal;
    return _progress;
}

RetInfo BaseTask::initPeriod()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                "
        "   PERIOD_BEGIN        "
        "  ,PERIOD_END          "
        "  ,REL_PERIOD_DURATION "
        " FROM                  "
        "   TASK                "
        " WHERE                 "
        "   ID = ?              ", id() ))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        return {RetInfo::Error::Sql};
    }

    QSqlRecord r = q.record();

    assignValue(_period.begin,      r, "PERIOD_BEGIN        ");
    assignValue(_period.end,        r, "PERIOD_END          ");
    assignValue(_relPeriodDuration, r, "REL_PERIOD_DURATION ");

    // Если задача периодическая, интервал необходимо вычислить исходя
    // из текущего месяца.
    if (_isPeriodic)
    {
//        QDate now = QDate::currentDate();
//        _period.begin = QDateTime(QDate(now.year(), now.month(), 1));
//        _period.end   = QDateTime(QDate(now.year(), now.month(), now.daysInMonth()));
        QDateTime now = QDateTime::currentDateTime();
        _period.begin = QDateTime(firstDay(now));
        _period.end   = QDateTime(lastDay(now));

        _period.begin = _period.begin.addMonths(-1 * _relPeriodDuration);
    }

    return {RetInfo::Success};
}

void BaseTaskThread::threadStopEstablished()
{
    dbpool().abortOperation(_threadId);
}

void BaseTaskThread::interrupt()
{
    QThreadEx::stop(0);
    _interrupted = true;
}

bool BaseTaskThread::taskIsRunning()
{
    return QThreadEx::isRunning();
}

bool BaseTaskThread::taskIsFinished()
{
    return QThreadEx::isFinished();
}

void BaseTaskThread::taskStart()
{
    QThreadEx::start();
}

bool BaseTaskThread::taskStop(unsigned long timeout)
{
    return QThreadEx::stop(timeout);
}

void BaseTaskThread::taskTerminate()
{
    QThreadEx::terminate();
}

RetInfo syncCheck(const BaseTask* task)
{
    if (task->skipSync())
    {
        log_verbose_m << TASK_EVENT_LOG2(u8"Синхронизация пропущена (SKIP_SYNC = TRUE)");
        return {RetInfo::Success};
    }

    int isComplete = -1;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    // Проверка статуса "Синхронизировано" для текущей задачи
    if (!sql::exec(q,
        "SELECT IS_COMPLETE FROM SYNC_PLANNING WHERE TASK_ID = ?", task->id()))
    {
        log_error_m << TASK_EVENT_LOG2(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (q.first())
        assignValue(isComplete, q.record(), "IS_COMPLETE");

    if (isComplete == 0)
        return {RetInfo::Error::NeedSync};

    // Если запись о задаче существует и флаг установлен в состояние "Синхронизировано",
    // то задача может продолжить выполнение, так как диапазон данных синхронизирован с ФОМС
    if (isComplete == 1)
        return {RetInfo::Success};

    QDate beginAlign = firstDay(task->period().begin);
    QDate endAlign = lastDay(task->period().end);

    QString fields =
        "  TASK_ID        "
        ", USER_ID        "
        ", CREATE_DATE    "
        ", PERIOD_BEGIN   "
        ", PERIOD_END     "
        ", PERIOD_CURRENT "
        ", IS_COMPLETE    "
        ", IS_PERIODIC    ";

    QString sql = sql::UPDATE_OR_INSERT_INTO("SYNC_PLANNING", fields, "TASK_ID");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG2(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    if (task->type() == TaskType::LearnModel)
        beginAlign = beginAlign.addMonths(-1);

    bindValue(q, ":TASK_ID        ", task->id()                   );
    bindValue(q, ":USER_ID        ", task->userId()               );
    bindValue(q, ":CREATE_DATE    ", QDateTime::currentDateTime() );
    bindValue(q, ":PERIOD_BEGIN   ", beginAlign                   );
    bindValue(q, ":PERIOD_END     ", endAlign                     );
    bindValue(q, ":PERIOD_CURRENT ", QVariant(QVariant::Date)     );
    bindValue(q, ":IS_COMPLETE    ", 0                            );
    bindValue(q, ":IS_PERIODIC    ", task->isPeriodic()           );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG2(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    return {RetInfo::Error::NeedSync};
}

RetInfo dataIsPresent(const BaseTaskThread* task, const char* logMessage)
{
    log_info_m << TASK_EVENT_LOG2(DATA_QUALITY_START);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT IIF(EXISTS(        "
        "   SELECT * FROM SYNC_DATA "
        "   WHERE                   "
        "     DATE_OT_PER >= ?         "
        "     AND                   "
        "     DATE_OT_PER <= ?         "
        "   ), 1, 0)                "
        "   AS DATA_EXISTS          "
        " FROM                      "
        "   RDB$DATABASE            ",
        task->period().begin, task->period().end ))
    {
        log_error_m << TASK_EVENT_LOG2(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    q.first();
    bool dataExists = false;
    assignValue(dataExists, q.record(), "DATA_EXISTS");

    if (!dataExists)
    {
        log_error_m << TASK_EVENT_LOG2(TASK_ERR_NO_DATA);
        return {RetInfo::Error::NoData};
    }

    QMap<QDate, int> meeMonth;
    QMap<QDate, int> ekmpMonth;
    int meeRatio = 0;
    int ekmpRatio = 0;
    std::atomic_int sqlExecError = {0};

    trd::ThreadIdList threadIds;

    auto funcMee = [&threadIds, task, &meeMonth, &sqlExecError]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            " SELECT                                           "
            "   SRC.EMONTH,                                    "
            "   SRC.EYEAR,                                     "
            "   SUM(ROWCOUNT) AS ROWSUM                        "
            " FROM                                             "
            " (                                                "
            "   SELECT                                         "
            "     EXTRACT(MONTH FROM DATE_OT_PER) AS EMONTH,      "
            "     EXTRACT(YEAR FROM DATE_OT_PER) AS EYEAR,        "
            "     COUNT(*) as ROWCOUNT                         "
            "   FROM                                           "
            "     SYNC_DATA                                    "
            "   WHERE                                          "
            "     DATE_OT_PER >= ? AND DATE_OT_PER <= ? AND MEE >= 0 "
            "   GROUP BY                                       "
            "     DATE_OT_PER                                     "
            " ) SRC                                            "
            " GROUP BY                                         "
            "    EMONTH, EYEAR                                 ",
            task->period().begin, task->period().end))
        {
            ++sqlExecError;
        }
        int month = 0;
        int year = 0;
        int rowSum = 0;
        while (q.next())
        {
            assignValue(month,  q.record(), "EMONTH" );
            assignValue(year,   q.record(), "EYEAR"  );
            assignValue(rowSum, q.record(), "ROWSUM" );

            if (rowSum < 150*1000)
                meeMonth.insert(QDate(year, month, 1), rowSum);
        }
    };

    auto funcEkmp = [&threadIds, task, &ekmpMonth, &sqlExecError]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            " SELECT                                            "
            "   SRC.EMONTH,                                     "
            "   SRC.EYEAR,                                      "
            "   SUM(ROWCOUNT) AS ROWSUM                         "
            " FROM                                              "
            " (                                                 "
            "   SELECT                                          "
            "     EXTRACT(MONTH FROM DATE_OT_PER) AS EMONTH,       "
            "     EXTRACT(YEAR FROM DATE_OT_PER) AS EYEAR,         "
            "     COUNT(*) as ROWCOUNT                          "
            "   FROM                                            "
            "     SYNC_DATA                                     "
            "   WHERE                                           "
            "     DATE_OT_PER >= ? AND DATE_OT_PER <= ? AND EKMP >= 0 "
            "   GROUP BY                                        "
            "     DATE_OT_PER                                      "
            " ) SRC                                             "
            " GROUP BY                                          "
            "   EMONTH, EYEAR                                   ",
            task->period().begin, task->period().end))
        {
            ++sqlExecError;
        }
        int month = 0;
        int year = 0;
        int rowSum = 0;
        while (q.next())
        {
            assignValue(month,  q.record(), "EMONTH" );
            assignValue(year,   q.record(), "EYEAR"  );
            assignValue(rowSum, q.record(), "ROWSUM" );

            if (rowSum < 150*1000)
                ekmpMonth.insert(QDate(year, month, 1), rowSum);
        }
    };

    auto funcMeeRatio = [&threadIds, task, &sqlExecError, &meeRatio]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            " SELECT                 "
            " (                      "
            "   SELECT               "
            "     COUNT(*)           "
            "   FROM                 "
            "     SYNC_DATA          "
            "   WHERE                "
            "     DATE_OT_PER >= ?      "
            "     AND DATE_OT_PER <= ?  "
            "     AND MEE = 0        "
            " ) AS NEGATIVE,         "
            " (                      "
            "   SELECT               "
            "     COUNT(*)           "
            "   FROM                 "
            "     SYNC_DATA          "
            "   WHERE                "
            "     DATE_OT_PER >= ?      "
            "     AND DATE_OT_PER <= ?  "
            "     AND MEE > 0        "
            " ) AS POSITIVE          "
            " FROM                   "
            "   RDB$DATABASE         ",
            task->period().begin,
            task->period().end,
            task->period().begin,
            task->period().end))
        {
            ++sqlExecError;
        }
        int negativeRec = 0;
        int positiveRec = 0;

        if (q.first())
        {
            QSqlRecord r = q.record();
            assignValue(negativeRec, r, "NEGATIVE");
            assignValue(positiveRec, r, "POSITIVE");
        }

        if (negativeRec > 0)
            meeRatio = ((double)positiveRec / negativeRec) * 100;
        else
            meeRatio = 100;
    };

    auto funcEkmpRatio = [&threadIds, task, &sqlExecError, &ekmpRatio]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            " SELECT                 "
            " (                      "
            "   SELECT               "
            "     COUNT(*)           "
            "   FROM                 "
            "     SYNC_DATA          "
            "   WHERE                "
            "     DATE_OT_PER >= ?      "
            "     AND DATE_OT_PER <= ?  "
            "     AND EKMP = 0       "
            " ) AS NEGATIVE,         "
            " (                      "
            "   SELECT               "
            "     COUNT(*)           "
            "   FROM                 "
            "     SYNC_DATA          "
            "   WHERE                "
            "     DATE_OT_PER >= ?      "
            "     AND DATE_OT_PER <= ?  "
            "     AND EKMP > 0       "
            " ) AS POSITIVE          "
            " FROM                   "
            "   RDB$DATABASE         ",
            task->period().begin,
            task->period().end,
            task->period().begin,
            task->period().end))
        {
            ++sqlExecError;
        }
        int negativeRec = 0;
        int positiveRec = 0;

        if (q.first())
        {
            QSqlRecord r = q.record();
            assignValue(negativeRec   , r, "NEGATIVE");
            assignValue(positiveRec, r, "POSITIVE");
        }

        if (negativeRec > 0)
            ekmpRatio = ((double)positiveRec / negativeRec) * 100;
        else
            ekmpRatio = 100;
    };


    std::thread t1 {funcMee};
    t1.detach();

    std::thread t2 {funcEkmp};
    t2.detach();

    std::thread t3 {funcMeeRatio};
    t3.detach();

    std::thread t4 {funcEkmpRatio};
    t4.detach();

    // Вместо sleep(1);
    while (threadIds.empty())
        usleep(1);

    while (!threadIds.empty())
    {
        usleep(100*1000);
        if (task->threadStop())
        {
            threadIds.lock([](std::vector<pid_t>& tids) {
                for (pid_t tid : tids)
                    dbpool().abortOperation(tid);
            });
            break;
        }
    }
    while (!threadIds.empty())
        usleep(100*1000);

    if (task->threadStop())
        return {RetInfo::Error::Interrupt};

    if (sqlExecError > 0)
    {
        log_error_m << TASK_EVENT_LOG2(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QList<QDate> meeMonthKeys = meeMonth.keys();
    qSort(meeMonthKeys.begin(), meeMonthKeys.end(),
          [](const QDate& d1, const QDate& d2) {return d2 < d1;});

    for (QDate& item : meeMonthKeys)
        log_warn_m << TASK_EVENT_LOG2(logMessage, u8"МЭЭ",
                                      item.toString("MM.yyyy"), meeMonth.value(item));

    QList<QDate> ekmpMonthKeys = ekmpMonth.keys();
    qSort(ekmpMonthKeys.begin(), ekmpMonthKeys.end(),
          [](const QDate& d1, const QDate& d2) {return d2 < d1;});

    for (QDate& item : ekmpMonthKeys)
        log_warn_m << TASK_EVENT_LOG2(logMessage, u8"ЭКМП",
                                      item.toString("MM.yyyy"), ekmpMonth.value(item));

    const int RATIO_NORMAL = 20;

    if (meeRatio < (RATIO_NORMAL - 5) || meeRatio > (RATIO_NORMAL + 5))
        log_warn_m << TASK_EVENT_LOG2(TASK_ERR_BAD_RATIO, u8"МЭЭ", meeRatio, 100 - meeRatio);

    if (ekmpRatio < (RATIO_NORMAL - 5) || ekmpRatio > (RATIO_NORMAL + 5))
        log_warn_m << TASK_EVENT_LOG2(TASK_ERR_BAD_RATIO, u8"ЭКМП", ekmpRatio, 100 - ekmpRatio);

    log_info_m << TASK_EVENT_LOG2(DATA_QUALITY_STOP);

    return {RetInfo::Success};
}

void initLearnPeriod(TimeRange& tr, const quint16 relPeriodDuration)
{
    /*
        Расчет периода для периодической задачи.
        Правый край - текущий месяц минус два.
        Левый край - правый край минус глубина.
    */

    QDateTime now = QDateTime::currentDateTime();
    QDateTime begin = QDateTime(firstDay(now));
    QDateTime end   = QDateTime(firstDay(now));

    /*
      Пропуск текущего и предыдущего месяца.
      Для текущего ещё нет данных.
      Предыдущий будем скорить.
    */
    begin = end.addMonths(-2);

    // Сдвиг начала периода на глубину обучения.
    begin = begin.addMonths(-1 * relPeriodDuration);
    end   = end.addMonths(-2);

    // Корректировка дат на первый и последние числа.
    tr.begin = QDateTime(firstDay(begin));
    tr.end   = QDateTime(lastDay(end));

}

void initScorePeriod(TimeRange& tr)
{
    QDateTime now = QDateTime::currentDateTime();
    now = QDateTime(QDate(now.date().year(), now.date().month(), 1));
    now = now.addMonths(-1);

    tr.begin = QDateTime(firstDay(now));
    tr.end   = QDateTime(lastDay(now));
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
