#pragma once

#include "ret_info.h"
#include "commands/error.h"
#include "commands/task.h"
#include "commands/time_range.h"

#include "shared/list.h"
#include "shared/defmac.h"
#include "shared/clife_base.h"
#include "shared/clife_ptr.h"
#include "shared/thread/thread_utils.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/thread/qthreadex.h"

#include <QtCore>
#include <atomic>

namespace task {

using namespace communication;
using namespace communication::transport;

enum class ScoreType
{
    MEE  = 0,
    EKMP = 1
};

class BaseTask : public clife_base
{
public:
    // Тип задачи
    TaskType type() const {return _type;}

    // Идентификатор исполняемой задачи
    QUuidEx id() const {return _id;}

    // Хэш-код идентификатор исполняемой задачи
    quint64 hashId() const {return _hashId;}

    // Идентификатор пользователя
    QUuidEx userId() const {return _userId;}

    // Идентификатор создаваемого контента (модели, оценки, отчета),
    // соотносится с TaskType.
    virtual const QUuidEx& contentId() const;

    // Код выполнения задачи
    RetInfo retInfo() const {return _retInfo;}
    qint32  retCode() const {return _retInfo.code();}

    // Признак пропуска синхронизации
    bool skipSync() const {return _skipSync;}

    // Признак периодичности задачи
    bool isPeriodic() const {return _isPeriodic;}

    // Левая и правая границы диапазона, с которой работает задача
    const TimeRange& period() const {return _period;}

    // Возвращает информацию о прогрессе выполнения задачи
    const data::TaskProgress& progress() const;

    // Прерывает выполнение задачи. Действует аналогично функции stop(), но при
    // этом не ждет завершения работы потока
    virtual void interrupt() = 0;

    // Возвращает TRUE если задача была прервана посредством функции interrupt()
    bool interrupted() const {return _interrupted;}

    // Возвращает TRUE если поток задачи выполняется
    virtual bool taskIsRunning() = 0;

    // Возвращает TRUE если поток задачи завершен
    virtual bool taskIsFinished() = 0;

    virtual void taskStart() = 0;
    virtual bool taskStop(unsigned long timeout) = 0;
    virtual void taskTerminate() = 0;

public:
    struct Find
    {
        int operator() (const QUuidEx* taskId, const BaseTask* item2, void*) const
            {return QUuidEx::compare(*taskId, item2->id());}

        int operator() (const quint32* hashId, const BaseTask* item2, void*) const
            {return LIST_COMPARE_ITEM(*hashId, item2->hashId());}

        int operator() (const TaskType* taskType, const BaseTask* item2, void*) const
            {return LIST_COMPARE_ITEM(*taskType, item2->type());}
    };
    struct Allocator
    {
        void destroy(BaseTask* x) {if (x) x->release();}
    };
    typedef clife_ptr<BaseTask> Ptr;
    typedef lst::List<BaseTask, Find, Allocator> List;

protected:
    BaseTask(TaskType, const QUuidEx& taskId, const QUuidEx& userId);

    /**
      @brief Инициализация задачи
      @return RetInfo - с информацие об ошибке
    */
    virtual RetInfo initPeriod();

    virtual void sanitize() {}

protected:
    bool _baseInit = {true};
    std::atomic_bool _interrupted = {false};

    RetInfo _retInfo = {RetInfo::Error::Undef};
    TimeRange _period;

    // Глубина периода для периодических задач
    short _relPeriodDuration = {0};

    std::atomic_int _progressCurrent = {-1};
    std::atomic_int _progressTotal   = {-1};

private:
    DISABLE_DEFAULT_FUNC(BaseTask)

    const TaskType _type;
    const QUuidEx  _id;
    const quint64  _hashId;
    const QUuidEx  _userId;

    bool _skipSync = {false};
    bool _isPeriodic = {false};

    mutable data::TaskProgress _progress;
    mutable QMutex _progressLock;
};

class BaseTaskThread : public QThreadEx, public BaseTask
{
protected:
    using BaseTask::BaseTask;

    void threadStopEstablished() override;
    void interrupt() override;

    bool taskIsRunning() override;
    bool taskIsFinished() override;

    void taskStart() override;
    bool taskStop(unsigned long timeout) override;
    void taskTerminate() override;

protected:
    // Идентификатор потока
    pid_t _threadId = {0};

private:
    DISABLE_DEFAULT_FUNC(BaseTaskThread)
};

/**
  @brief Проверка актуальности данных
  @details Метод произведёт проверку на наличие устаревших данных. Если
           все данные, которые подлежат выборке свежие, то задача приступит
           к обучению модели. Если данные нуждаются в синхронизации, то
           задача будет отложена до того момента, пока синхронизация не
           будет выполнена.
  @return RetInfo - с информацие об ошибке
*/
RetInfo syncCheck(const BaseTask*);

/**
  @brief Проверка на наличие данных ЭКМП/МЭЭ
*/
RetInfo dataIsPresent(const BaseTaskThread*, const char* logMessage);

void initLearnPeriod(TimeRange&, const quint16);

void initScorePeriod(TimeRange&);

} // namespace task
