#include "create_report.h"
#include "task_messages.h"
#include "functions.h"

#include "database/connect.h"
#include "database/sql_func.h"
#include "commands/error.h"

namespace task {

using namespace db::firebird;
using namespace sql;

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "CreateReport")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "CreateReport")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "CreateReport")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "CreateReport")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "CreateReport")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "CreateReport")

CreateReport::CreateReport(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTaskThread(TaskType::CreateReport, taskId, userId)
{}

void CreateReport::run()
{
    #define TASK_NAME u8"Отчет"

    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        "UPDATE REPORT SET EXEC_STATUS = ? WHERE TASK_ID = ?",
        data::Report::ExecStatus::Creating, id()))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    if (!sql::exec(q,
        " SELECT             "
        "   ID               "
        "  ,TASK_ID          "
        "  ,USER_ID          "
        "  ,SCORE_ID         "
        "  ,NAME             "
        "  ,REPORT_TYPE      "
        "  ,CREATE_DATE      "
        "  ,PERIOD_BEGIN     "
        "  ,PERIOD_END       "
        "  ,CODE_LPU         "
        "  ,CODE_MKB         "
        "  ,VIDMP            "
        "  ,MED_PROFILE      "
        "  ,THRESHOLD        "
        "  ,RECORD_COUNT     "
        "  ,RANGING_TYPE     "
        "  ,PRESENT_TYPE     "
        "  ,SORT_TYPE        "
        "  ,EXEC_STATUS      "
        " FROM               "
        "   REPORT           "
        " WHERE              "
        "   TASK_ID = ?      ", id()))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }
    if (!q.first())
    {
        _retInfo = {RetInfo::Error::General};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    { //Block for QSqlRecord
        QSqlRecord r = q.record();

        assignValue(_report.id,            r, "ID           " );
        assignValue(_report.taskId,        r, "TASK_ID      " );
        assignValue(_report.userId,        r, "USER_ID      " );
        assignValue(_report.scoreId,       r, "SCORE_ID     " );
        assignValue(_report.type,          r, "REPORT_TYPE  " );
        assignValue(_report.name,          r, "NAME         " );
        assignValue(_report.createDate,    r, "CREATE_DATE  " );
        assignValue(_report.period.begin,  r, "PERIOD_BEGIN " );
        assignValue(_report.period.end,    r, "PERIOD_END   " );
        assignValue(_report.lpu,           r, "CODE_LPU     " );
        assignValue(_report.mkb,           r, "CODE_MKB     " );
        assignValue(_report.vidmp,         r, "VIDMP        " );
        assignValue(_report.medProfile,    r, "MED_PROFILE  " );
        assignValue(_report.threshold,     r, "THRESHOLD    " );
        assignValue(_report.recordCount,   r, "RECORD_COUNT " );
        assignValue(_report.ranging,       r, "RANGING_TYPE " );
        assignValue(_report.present,       r, "PRESENT_TYPE " );
        assignValue(_report.sort,          r, "SORT_TYPE    " );
        assignValue(_report.execStatus,    r, "EXEC_STATUS  " );
    }

    _progressCurrent = 0;
    _progressTotal = 100;

    // Первое событие отправляем после того как получим _report.id,
    // иначе contentId окажется не заполненным
    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    // Суб. sql-запрос
    QString sqlSub =
        " SELECT                            "
      //"   SYNC_DATA.CODE_LPU AS CODE_LPU  " // Код ЛПУ возвращается в любом случае
        "   COUNT( %1 )        AS CNT       " // Количество записей, которые были
        "                                   " // сгруппированы в зависимости от типа отчета
        "  ,SUM( %2 )          AS PREDICT   " // Сумма скоров
        "  ,%3                 AS CODE      " // В зависимости от типа отчета поле может
        "                                   " // содержать значения полей CODE_LPU, MKB1,
        " FROM                              " // VID_MP, PROFIL
        "   SCORE_DATA                      "
        " LEFT JOIN                         "
        "   SYNC_DATA ON SYNC_DATA.GKEY = SCORE_DATA.GKEY "
        " WHERE                             "
        "   SCORE_DATA.SCORE_ID = :SCORE_ID "  // отбираем по id score
        "   %4                              "  // Фильтры
        " GROUP BY                          "
        "   %5                              ";

    QString field;

    /* %1 */
    switch (_report.type)
    {
        case ReportType::Lpu:        field = "SYNC_DATA.CODE_LPU"; break;
        case ReportType::Mkb:        field = "SYNC_DATA.MKB1";     break;
        case ReportType::Vidmp:      field = "SYNC_DATA.VID_MP";   break;
        case ReportType::MedProfile: field = "SYNC_DATA.PROFIL";   break;
        default:                     field = "0"; break;
    }
    sqlSub = sqlSub.arg(field);

    /* %2 */
    switch (_report.ranging)
    {
        case ReportRanging::MeeScores:
            field = "SCORE_DATA.MEE";
            break;

        case ReportRanging::MeeRisk:
            field = "SCORE_DATA.MEE * SYNC_DATA.SUM_RUB";
            break;

        case ReportRanging::EkmpScores:
            field = "SCORE_DATA.EKMP";
            break;

        case ReportRanging::EkmpRisk:
            field = "SCORE_DATA.EKMP * SYNC_DATA.SUM_RUB";
            break;

        default:
            field = "0";
            break;
    }
    sqlSub = sqlSub.arg(field);

    /* %3 */
    switch (_report.type)
    {
        case ReportType::Lpu:        field = "SYNC_DATA.CODE_LPU"; break;
        case ReportType::Mkb:        field = "SYNC_DATA.MKB1";     break;
        case ReportType::Vidmp:      field = "SYNC_DATA.VID_MP";   break;
        case ReportType::MedProfile: field = "SYNC_DATA.PROFIL";   break;
        default:                     field = "0"; break;
    }
    sqlSub = sqlSub.arg(field);

    /* %4 фильтры */
    QString filters;
    switch (_report.ranging)
    {
        case ReportRanging::MeeScores:
        case ReportRanging::MeeRisk:
            filters = " AND SCORE_DATA.MEE %1 :SCORE_LIMIT ";
            break;

        case ReportRanging::EkmpScores:
        case ReportRanging::EkmpRisk:
            filters = " AND SCORE_DATA.EKMP %1 :SCORE_LIMIT ";
            break;

        default:
            break;
    }
    filters = filters.arg((_report.sort == ReportSort::Ascending) ? "<=" : ">=");

    auto normalizeEnumeration = [](QString s)
    {
        return s.remove(QChar(' ')).remove(QChar('\'')).replace(",", "','");
    };

    if (!_report.mkb.trimmed().isEmpty())
    {
        filters += " AND SYNC_DATA.MKB1 IN ('%1') ";
        filters = filters.arg(normalizeEnumeration(_report.mkb));
    }
    if (!_report.lpu.trimmed().isEmpty())
    {
        filters += " AND SYNC_DATA.CODE_LPU IN ('%1') ";
        filters = filters.arg(normalizeEnumeration(_report.lpu));
    }
    if (!_report.vidmp.trimmed().isEmpty())
    {
        filters += " AND SYNC_DATA.VID_MP IN ('%1') ";
        filters = filters.arg(normalizeEnumeration(_report.vidmp));
    }
    if (!_report.medProfile.trimmed().isEmpty())
    {
        filters += " AND SYNC_DATA.PROFIL IN ('%1') ";
        filters = filters.arg(normalizeEnumeration(_report.medProfile));
    }
    sqlSub = sqlSub.arg(filters);

    /* %5 */
    switch (_report.type)
    {
        case ReportType::Lpu:        field = "SYNC_DATA.CODE_LPU"; break;
        case ReportType::Mkb:        field = "SYNC_DATA.MKB1";     break;
        case ReportType::Vidmp:      field = "SYNC_DATA.VID_MP";   break;
        case ReportType::MedProfile: field = "SYNC_DATA.PROFIL";   break;
        default:                     field = ""; break;
    }
    sqlSub = sqlSub.arg(field);

    // Основной sql-запрос
    QString sql =
        " SELECT FIRST %1    "
        "   RESULT.CNT       "
        "  ,RESULT.PREDICT   " // Сумма (Скор) или Сумма (Скор * Цена)
        "  ,RESULT.CODE      "
        "  ,%2 AS NAME       "
        " FROM (%3) RESULT   ";

    /* %1 */
    sql = sql.arg(_report.recordCount);

    /* %2 */
    switch (_report.type)
    {
        case ReportType::Lpu:        field = "NSI_LPU.SHORT_NAME"; break;
        case ReportType::Mkb:        field = "NSI_MKB.MKB_NAME";   break;
        case ReportType::Vidmp:      field = "NSI_VIDMP.VMPNAME";  break;
        case ReportType::MedProfile: field = "NSI_PROFILE.PRNAME"; break;
        default:                     field = "''"; break;
    }
    sql = sql.arg(field);

    /* %3 */
    sql = sql.arg(sqlSub);

    if (_report.type == ReportType::Lpu)
        sql += " LEFT JOIN NSI_LPU ON NSI_LPU.CODE = RESULT.CODE";

    if (_report.type == ReportType::Mkb)
        sql += " LEFT JOIN NSI_MKB ON NSI_MKB.MKB_CODE = RESULT.CODE ";

    else if (_report.type == ReportType::Vidmp)
        sql += " LEFT JOIN NSI_VIDMP ON NSI_VIDMP.ID = RESULT.CODE ";

    else if (_report.type == ReportType::MedProfile)
        sql += " LEFT JOIN NSI_PROFILE ON NSI_PROFILE.ID = RESULT.CODE ";

    sql += " ORDER BY RESULT.PREDICT %1 ";
    sql = sql.arg((_report.sort == ReportSort::Ascending) ? "ASC" : "DESC");
    //---

    Transaction::Ptr transact = dbcon->createTransact();
    q = QSqlQuery(db::firebird::createResult(transact));

    if (!transact->begin())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!q.prepare(sql))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    bindValue(q, ":SCORE_ID   ", _report.scoreId);
    bindValue(q, ":SCORE_LIMIT", _report.threshold);

    if (!q.exec())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    QString dataFields =
        "  ID           "
        ", REPORT_ID    "
        ", RESULT_NAME  "
        ", RESULT_CODE  "
        ", RESULT_VALUE "
        ", RESULT_COUNT ";
      //", CODE_LPU     "
      //", NAME_LPU     ";

    QString dataSql = sql::INSERT_INTO("REPORT_DATA", dataFields);

    QSqlQuery qdata {dbcon->createResult(transact)};
    if (!qdata.prepare(dataSql))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    while (q.next())
    {
        QSqlRecord r = q.record();

        bindValue(qdata, ":ID           ", QUuidEx::createUuid());
        bindValue(qdata, ":REPORT_ID    ", _report.id );
        bindValue(qdata, ":RESULT_COUNT ", r.value("CNT").toString());
        bindValue(qdata, ":RESULT_VALUE ", r.value("PREDICT").toDouble());
      //bindValue(qdata, ":CODE_LPU     ", r.value("CODE_LPU").toString());
      //bindValue(qdata, ":NAME_LPU     ", r.value("NAME_LPU").toString());
        bindValue(qdata, ":RESULT_CODE  ", r.value("CODE").toString());
        bindValue(qdata, ":RESULT_NAME  ", r.value("NAME").toString());

//        if (_report.type == ReportType::Lpu)
//        {
//            bindValue(qdata, ":RESULT_CODE", r.value("CODE_LPU").toString());
//            bindValue(qdata, ":RESULT_NAME", r.value("NAME_LPU").toString());
//        }
//        else
//        {
//            bindValue(qdata, ":RESULT_CODE", r.value("CODE").toString());
//            bindValue(qdata, ":RESULT_NAME", r.value("NAME").toString());
//        }

        if (!qdata.exec())
        {
            _retInfo = {RetInfo::Error::Sql};
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            TASK_CLEAN_AND_RETURN(TASK_NAME);
        }
    } // while (q.next())

    if (!transact->commit())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    q = QSqlQuery(dbcon->createResult());

    if (!sql::exec(q,
        "UPDATE REPORT SET EXEC_STATUS = ? WHERE TASK_ID = ?",
        data::Report::ExecStatus::Success, id()))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    _progressCurrent = 100;
    m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    data::TaskContentCreate taskContentCreate;
    taskContentCreate.taskType = {TaskType::CreateReport};
    taskContentCreate.taskId = id();
    taskContentCreate.userId = userId();
    taskContentCreate.contentId = contentId();
    taskContentCreate.isPeriodic = isPeriodic();

    m = createJsonMessage(taskContentCreate, Message::Type::Event);
    webCon().send(m);

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME);

    #undef TASK_NAME
}

void CreateReport::sanitize()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    sql::exec(q, "UPDATE REPORT SET EXEC_STATUS = ? WHERE TASK_ID = ?",
                 data::Report::ExecStatus::Failed, id());

    sql::exec(q, "DELETE FROM REPORT_DATA WHERE REPORT_ID = ?", _report.id);
}

const QUuidEx& CreateReport::contentId() const
{
    return _report.id;
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
