#include "create_report_fed.h"
#include "task_messages.h"
#include "functions.h"
#include "utils.h"

#include "commands/error.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "database/settings.h"
#include "shared/simple_timer.h"
#include "shared/qt/config/config.h"

#include <QProcess>
#include <QDomDocument>

namespace task {

using namespace db::firebird;
using namespace sql;

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "CreateRepFed")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "CreateRepFed")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "CreateRepFed")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "CreateRepFed")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "CreateRepFed")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "CreateRepFed")

CreateReportFed::CreateReportFed(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTaskThread(TaskType::CreateReportFed, taskId, userId)
{}

void CreateReportFed::run()
{
    #define TASK_NAME u8"Отчет (федеральный)"

    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();

    { //Block for QSqlQuery
        QSqlQuery q {dbcon->createResult()};
        if (!sql::exec(q,
            "UPDATE REPORT_FED SET EXEC_STATUS = ? WHERE TASK_ID = ?",
            data::ReportFed::ExecStatus::Creating, id()))
        {
            _retInfo = {RetInfo::Error::Sql};
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            TASK_CLEAN_AND_RETURN(TASK_NAME);
        }
    }

    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {dbcon->createResult(transact)};

    if (!transact->begin())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        " SELECT                          "
        "   R.ID                          "
        "  ,R.TASK_ID                     "
        "  ,R.USER_ID                     "
        "  ,R.SCORE_ID                    "
        "  ,R.NAME                        "
        "  ,R.CREATE_DATE                 "
        "  ,R.PERIOD_BEGIN                "
        "  ,R.PERIOD_END                  "
        "  ,R.IS_PERIODIC                 "
        "  ,R.IS_CORRECTION               "
        "  ,R.ORDER_NUM                   "
        "  ,R.EXEC_STATUS                 "
        "  ,S.NAME AS SCORE_NAME          "
        " FROM                            "
        "   REPORT_FED R                  "
        " LEFT JOIN                       "
        "   SCORE S ON S.ID = R.SCORE_ID  "
        " WHERE                           "
        "   R.TASK_ID = ?                 ", id()))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }
    if (!q.first())
    {
        _retInfo = {RetInfo::Error::General};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    { //Block for QSqlRecord
        QSqlRecord r = q.record();

        assignValue(_report.id           , r, "ID           " );
        assignValue(_report.taskId       , r, "TASK_ID      " );
        assignValue(_report.userId       , r, "USER_ID      " );
        assignValue(_report.scoreId      , r, "SCORE_ID     " );
        assignValue(_report.name         , r, "NAME         " );
        assignValue(_report.createDate   , r, "CREATE_DATE  " );
        assignValue(_report.period.begin , r, "PERIOD_BEGIN " );
        assignValue(_report.period.end   , r, "PERIOD_END   " );
        assignValue(_report.isPeriodic   , r, "IS_PERIODIC  " );
        assignValue(_report.isCorrection , r, "IS_CORRECTION" );
        assignValue(_report.orderNum     , r, "ORDER_NUM    " );
        assignValue(_report.execStatus   , r, "EXEC_STATUS  " );
        assignValue(_report.scoreName    , r, "SCORE_NAME   " );
    }

    log_debug_m << "Report name: " << _report.name
                << ". Id: " << _report.id;
    log_debug_m << "Report score: '" << _report.scoreName << "'"
                << ". Id: " << _report.scoreId;
    log_debug_m << "Report periodic: "  << _report.isPeriodic
                << ", correction: " << _report.isCorrection;

    _progressCurrent = 0;
    _progressTotal = 100;

    // Первое событие отправляем после того как получим _report.id,
    // иначе contentId окажется не заполненным
    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    QDateTime now {QDateTime::currentDateTime()};

    //--- Формирование xml-файла ---
    QDomDocument xml;
    QDomText xtext;

    auto setDateRange = [&](QDomElement& xdate, QDate begin, QDate end)
    {
        QDomElement xbegin = xml.createElement("BEGIN");
        xdate.appendChild(xbegin);
        xtext = xml.createTextNode(begin.toString("yyyy-MM-dd"));
        xbegin.appendChild(xtext);

        QDomElement xend = xml.createElement("END");
        xdate.appendChild(xend);
        xtext = xml.createTextNode(end.toString("yyyy-MM-dd"));
        xend.appendChild(xtext);
    };

    auto setScoreRangeAis = [&](QDomElement& xscores, double begin, double end, int count)
    {
        QDomElement xscoreRange = xml.createElement("SCORE_RANGE");
        xscores.appendChild(xscoreRange);

        QDomElement xbegin = xml.createElement("BEGIN");
        xscoreRange.appendChild(xbegin);
        xtext = xml.createTextNode(QString::number(int(begin * 100)));
        xbegin.appendChild(xtext);

        QDomElement xend = xml.createElement("END");
        xscoreRange.appendChild(xend);
        xtext = xml.createTextNode(QString::number(int(end * 100)));
        xend.appendChild(xtext);

        QDomElement xcount = xml.createElement("COUNT");
        xscoreRange.appendChild(xcount);
        xtext = xml.createTextNode(QString::number(count));
        xcount.appendChild(xtext);
    };

    auto setScoreTotalCountAis = [&](QDomElement& elem, int totalCount)
    {
        QDomElement xtotalCount = xml.createElement("TOTAL_COUNT");
        elem.appendChild(xtotalCount);

        xtext = xml.createTextNode(QString::number(totalCount));
        xtotalCount.appendChild(xtext);
    };

    auto setScoreRangeExp =
         [&](QDomElement& xscores, double begin, double end, int count, int count2)
    {
        QDomElement xscoreRange = xml.createElement("SCORE_RANGE");
        xscores.appendChild(xscoreRange);

        QDomElement xbegin = xml.createElement("BEGIN");
        xscoreRange.appendChild(xbegin);
        xtext = xml.createTextNode(QString::number(int(begin * 100)));
        xbegin.appendChild(xtext);

        QDomElement xend = xml.createElement("END");
        xscoreRange.appendChild(xend);
        xtext = xml.createTextNode(QString::number(int(end * 100)));
        xend.appendChild(xtext);

        QDomElement xcount = xml.createElement("CHECKED");
        xscoreRange.appendChild(xcount);
        xtext = xml.createTextNode(QString::number(count));
        xcount.appendChild(xtext);

        QDomElement xcount2 = xml.createElement("DEFECT");
        xscoreRange.appendChild(xcount2);
        xtext = xml.createTextNode(QString::number(count2));
        xcount2.appendChild(xtext);
    };

    QDomElement xroot = xml.createElement("RP_EXP");
    xml.appendChild(xroot);

    QDomElement zglv = xml.createElement("ZGLV");
    xroot.appendChild(zglv);

    QDomElement xvers = xml.createElement("VERSION");
    zglv.appendChild(xvers);
    xtext = xml.createTextNode("1.0");
    xvers.appendChild(xtext);

    QDomElement xdata = xml.createElement("DATA");
    zglv.appendChild(xdata);
    xtext = xml.createTextNode(now.date().toString("yyyy-MM-dd"));
    xdata.appendChild(xtext);

    QDomElement xfileName = xml.createElement("FILENAME");
    zglv.appendChild(xfileName);
    xtext = xml.createTextNode(_report.name);
    xfileName.appendChild(xtext);

    // Временной период
    QDate periodBegin = firstDay(_report.period.begin);
    QDate periodEnd = firstDay(_report.period.end).addMonths(1).addDays(-1);

    log_debug_m << "Period range: " << periodBegin << " / " << periodEnd;

    // Для тестирования lastDay(_report.period.end), потом удалить
    QDate periodEnd2 = lastDay(_report.period.end);
    log_debug_m << "Period range (test): " << periodBegin << " / " << periodEnd2;

    //<!-- Секция с данными от АИС Эксперт -->
    QDomElement xais = xml.createElement("AIS_EXP");
    xroot.appendChild(xais);

    //<!-- Диапазон дат выборки (помесячно) -->
    QDomElement xdateRange = xml.createElement("DATE_RANGE");
    xais.appendChild(xdateRange);

    setDateRange(xdateRange, periodBegin, periodEnd);

    //<!-- Список отчетных месяцев -->
     QDomElement xaisMonths = xml.createElement("MONTHS");
     xais.appendChild(xaisMonths);

    //<!-- Секция с данными размеченными экспертами -->
    QDomElement xexp = xml.createElement("EXPERTS");
    xroot.appendChild(xexp);

    //<!-- Диапазон дат выборки (помесячно) -->
    QDomNode xdateRange2 = xdateRange.cloneNode(); // DATE_RANGE
    xexp.appendChild(xdateRange2);

    //<!-- Список отчетных месяцев -->
    QDomElement xexpMonths = xml.createElement("MONTHS");
    xexp.appendChild(xexpMonths);
    //---

    QString sql, sql2;
    int count, defects, totalCount;

    QString sql0 =
        " SELECT                                                      "
        "   COUNT(%1)                                                 "
        " FROM                                                        "
        "   SCORE_DATA SC                                             "
        " LEFT JOIN                                                   "
        "   SYNC_DATA_FED SY ON SY.GKEY = SC.GKEY                     "
        " WHERE                                                       "
        "   SC.SCORE_ID = :SCORE_ID                                   "
        "   AND                                                       "
        "   (SY.DATE_OT_PER >= :CURRENT AND SY.DATE_OT_PER < :CURRENT_NEXT) "
        "   AND                                                       "
        "   (%2 > :LOW_BORDER AND %2 <= :HIGH_BORDER)                 ";

    #define SQL_PREPARE(Q, SQL) \
        if (!Q.prepare(SQL)) { \
            _retInfo = {RetInfo::Error::Sql}; \
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE); \
            TASK_CLEAN_AND_RETURN(TASK_NAME); \
        } \
        if (threadStop()) \
            continue;

    #define SQL_EXEC(Q) \
        if (!Q.exec()) { \
            _retInfo = {RetInfo::Error::Sql}; \
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC); \
            TASK_CLEAN_AND_RETURN(TASK_NAME); \
        } \
        if (threadStop()) \
            continue;

    #define GET_COUNT(COUNT) \
        q.first(); \
        COUNT = q.value(0).toInt();

    QDate current = periodBegin;
    QDate currentNext = periodBegin.addMonths(1);

    // В лоб считаем количество месяцев
    int monthCount = 0;
    while (current <= periodEnd)
    {
        ++monthCount;
        current = current.addMonths(1);
    }
    log_debug_m << "Month count: " << monthCount;

    auto bindValues = [&](QSqlQuery& q_, double lo, double hi)
    {
        bindValue(q_, ":SCORE_ID    ", _report.scoreId);
        bindValue(q_, ":CURRENT     ", current        );
        bindValue(q_, ":CURRENT_NEXT", currentNext    );
        bindValue(q_, ":LOW_BORDER  ", lo             );
        bindValue(q_, ":HIGH_BORDER ", hi             );
    };

    double persentStep = 4.762 / monthCount; // шаг в процентах
    #define PERSENT_STEP { \
        _progressCurrent += persentStep; \
        Message::Ptr m = createJsonMessage(progress(), Message::Type::Event); \
        webCon().send(m); \
    }

    PERSENT_STEP // 1
    current = periodBegin;

    while (current <= periodEnd)
    {
        if (threadStop())
        {
            _retInfo = {RetInfo::Error::Interrupt};
            //log_error_m << TASK_EVENT_LOG(TASK_MSG_INT, TASK_NAME);
            //TASK_CLEAN_AND_RETURN(TASK_NAME);
            TASK_RETURN_FOR_INT(TASK_NAME);
        }

        QString qAdd =
               " INSERT INTO SYNC_DATA_FED (GKEY, DATE_OUT, MEE, EKMP) "
               " SELECT                                                "
               "   GKEY,                                               "
               "   DATE_OT_PER,                                           "
               "   MEE,                                                "
               "   EKMP                                                "
               " FROM                                                  "
               "   SYNC_DATA SY                                        "
               " WHERE                                                 "
               "   SY.DATE_OT_PER >= :CURRENT                             "
               "   AND                                                 "
               "   SY.DATE_OT_PER < :CURRENT_NEXT                         ";
        SQL_PREPARE(q, qAdd);
        bindValue(q, ":CURRENT     ", current        );
        bindValue(q, ":CURRENT_NEXT", currentNext    );
        SQL_EXEC(q);

        PERSENT_STEP // 2

        log_debug_m << "Date range: " << current << " / " << currentNext;

        //--- МЭЭ. АИС Эксперт ---
        QDomElement xaisMon = xml.createElement("MONTH");
        xaisMonths.appendChild(xaisMon);

        QDomElement xaisDate = xml.createElement("DATE");
        xaisMon.appendChild(xaisDate);

        setDateRange(xaisDate, current, currentNext.addDays(-1));

        QDomElement xaisMee = xml.createElement("MEE");
        xaisMon.appendChild(xaisMee);

        QDomElement xaisScores = xml.createElement("SCORES");
        xaisMee.appendChild(xaisScores);

        sql = sql0.arg("SC.MEE").arg("SC.MEE");
        totalCount = 0;

        SQL_PREPARE(q, sql)
        bindValues(q, 0.0, 0.5);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 3

        log_debug_m << "MEE AIS count (0-50): " << count;
        totalCount += count;

        // Запись в xml
        setScoreRangeAis(xaisScores, 0.0, 0.5, count);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.5, 0.75);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 4

        log_debug_m << "MEE AIS count (50-75): " << count;
        totalCount += count;

        // Запись в xml
        setScoreRangeAis(xaisScores, 0.5, 0.75, count);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.75, 1.0);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 5

        log_debug_m << "MEE AIS count (75-100): " << count;
        totalCount += count;

        // Запись в xml
        setScoreRangeAis(xaisScores, 0.75, 1.0, count);
        setScoreTotalCountAis(xaisMee, totalCount);


        //--- ЭКМП. АИС Эксперт ---
        QDomElement xaisEkmp = xml.createElement("EKMP");
        xaisMon.appendChild(xaisEkmp);

        xaisScores = xml.createElement("SCORES");
        xaisEkmp.appendChild(xaisScores);

        sql = sql0.arg("SC.EKMP").arg("SC.EKMP");
        totalCount = 0;

        SQL_PREPARE(q, sql)
        bindValues(q, 0.0, 0.5);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 6

        log_debug_m << "EKMP AIS count (0-50): " << count;
        totalCount += count;

        // Запись в xml
        setScoreRangeAis(xaisScores, 0.0, 0.5, count);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.5, 0.75);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 7

        log_debug_m << "EKMP AIS count (50-75): " << count;
        totalCount += count;

        // Запись в xml
        setScoreRangeAis(xaisScores, 0.5, 0.75, count);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.75, 1.0);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 8

        log_debug_m << "EKMP AIS count (75-100): " << count;
        totalCount += count;

        // Запись в xml
        setScoreRangeAis(xaisScores, 0.75, 1.0, count);
        setScoreTotalCountAis(xaisEkmp, totalCount);


        //--- MEE. Эксперты ---
        QDomElement xexpMon = xml.createElement("MONTH");
        xexpMonths.appendChild(xexpMon);

        QDomElement xexpDate = xml.createElement("DATE");
        xexpMon.appendChild(xexpDate);

        setDateRange(xexpDate, current, currentNext.addDays(-1));

        QDomElement xexpMee = xml.createElement("MEE");
        xexpMon.appendChild(xexpMee);

        QDomElement xexpScores = xml.createElement("SCORES");
        xexpMee.appendChild(xexpScores);

        sql = sql0.arg("SY.MEE").arg("SC.MEE");
        sql += " AND SY.MEE >= 0 "; // Для вычисления количества проверок

        SQL_PREPARE(q, sql)
        bindValues(q, 0.0, 0.5);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 9

        log_debug_m << "MEE EXP count (0-50): " << count;

        sql2 = sql0.arg("SY.MEE").arg("SC.MEE");
        sql2 += " AND SY.MEE > 0 "; // Для вычисления количества дефектов

        SQL_PREPARE(q, sql2)
        bindValues(q, 0.0, 0.5);
        SQL_EXEC(q)
        GET_COUNT(defects)

        PERSENT_STEP // 10

        log_debug_m << "MEE EXP defects (0-50): " << defects;

        // Запись в xml
        setScoreRangeExp(xexpScores, 0.0, 0.5, count, defects);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.5, 0.75);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 11

        log_debug_m << "MEE EXP count (50-75): " << count;

        SQL_PREPARE(q, sql2)
        bindValues(q, 0.5, 0.75);
        SQL_EXEC(q)
        GET_COUNT(defects)

        PERSENT_STEP // 12

        log_debug_m << "MEE EXP defects (50-75): " << defects;

        // Запись в xml
        setScoreRangeExp(xexpScores, 0.5, 0.75, count, defects);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.75, 1.0);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 13

        log_debug_m << "MEE EXP count (75-100): " << count;

        SQL_PREPARE(q, sql2)
        bindValues(q, 0.75, 1.0);
        SQL_EXEC(q)
        GET_COUNT(defects)

        PERSENT_STEP // 14

        log_debug_m << "MEE EXP defects (75-100): " << defects;

        // Запись в xml
        setScoreRangeExp(xexpScores, 0.75, 1.0, count, defects);

        //--- ЭКМП. Эксперты ---
        QDomElement xexpEkmp = xml.createElement("EKMP");
        xexpMon.appendChild(xexpEkmp);

        xexpScores = xml.createElement("SCORES");
        xexpEkmp.appendChild(xexpScores);

        sql = sql0.arg("SY.EKMP").arg("SC.EKMP");
        sql += " AND SY.EKMP >= 0 "; // Для вычисления количества проверок

        SQL_PREPARE(q, sql)
        bindValues(q, 0.0, 0.5);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 15

        log_debug_m << "EKMP EXP count (0-50): " << count;

        sql2 = sql0.arg("SY.EKMP").arg("SC.EKMP");
        sql2 += " AND SY.EKMP > 0 "; // Для вычисления количества дефектов

        SQL_PREPARE(q, sql2)
        bindValues(q, 0.0, 0.5);
        SQL_EXEC(q)
        GET_COUNT(defects)

        PERSENT_STEP // 16

        log_debug_m << "EKMP EXP defects (0-50): " << defects;

        // Запись в xml
        setScoreRangeExp(xexpScores, 0.0, 0.5, count, defects);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.5, 0.75);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 17

        log_debug_m << "EKMP EXP count (50-75): " << count;

        SQL_PREPARE(q, sql2)
        bindValues(q, 0.5, 0.75);
        SQL_EXEC(q)
        GET_COUNT(defects)

        PERSENT_STEP // 18

        log_debug_m << "EKMP EXP deects (50-75): " << defects;

        // Запись в xml
        setScoreRangeExp(xexpScores, 0.5, 0.75, count, defects);

        SQL_PREPARE(q, sql)
        bindValues(q, 0.75, 1.0);
        SQL_EXEC(q)
        GET_COUNT(count)

        PERSENT_STEP // 19

        log_debug_m << "EKMP EXP count (75-100): " << count;

        SQL_PREPARE(q, sql2)
        bindValues(q, 0.75, 1.0);
        SQL_EXEC(q)
        GET_COUNT(defects)

        PERSENT_STEP // 20

        log_debug_m << "EKMP EXP defects (75-100): " << defects;

        // Запись в xml
        setScoreRangeExp(xexpScores, 0.75, 1.0, count, defects);

        QString qDel = "DELETE FROM SYNC_DATA_FED";
        SQL_PREPARE(q, qDel);
        SQL_EXEC(q);

        PERSENT_STEP // 21

        current = current.addMonths(1);
        currentNext = currentNext.addMonths(1);

    } // while (current <= periodEnd)

    QString xmlStr = R"(<?xml version="1.0" encoding="windows-1251"?>)";
    xmlStr += "\n";
    xmlStr += xml.toString(2);

    //QFile file {"/tmp/ais_reportfed.xml"};
    QString reportFileName = "/tmp/" + _report.name;
    QFile file {reportFileName};
    if (!file.open(QIODevice::WriteOnly))
    {
        _retInfo = {RetInfo::Error::General};
        const char* msg = u8"Ошибка открытия временного файла %1 в режиме записи";
        log_error_m << TASK_EVENT_LOG(msg, reportFileName);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }
    file.write(xmlStr.toUtf8());
    file.close();

    if (_report.isPeriodic)
    {
        QString storeDir = "/var/opt/aisexpert/reportfed";
        db::settings::getValue("foms.report_fed.store_dir", storeDir);

        if (QDir(storeDir).exists())
        {
            QProcess proc;
            QStringList arguments;
            arguments << "-j";
            arguments << storeDir + "/" + _report.name + ".oms";
            arguments << reportFileName;

            proc.start("/usr/bin/zip", arguments);
            if (!proc.waitForStarted(5000))
            {
                proc.kill();
                _retInfo = {RetInfo::Error::General};
                log_error_m << TASK_EVENT_LOG("Ошибка запуска архиватора ZIP");
                TASK_CLEAN_AND_RETURN(TASK_NAME);
            }
            else
            {
                // Ждем окончания работы архиватора
                if (!proc.waitForFinished(30*1000 /*30 сек*/))
                {
                    proc.kill();
                    _retInfo = {RetInfo::Error::General};
                    const char* msg = u8"Превышено время работы архиватора ZIP";
                    log_error_m << TASK_EVENT_LOG(msg);
                    TASK_CLEAN_AND_RETURN(TASK_NAME);
                }
            }
            int exitCode = proc.exitCode();
            if (exitCode != 0)
            {
                _retInfo = {RetInfo::Error::General};
                const char* msg = u8"Архиватор ZIP завершил работу с кодом ошибки: %1";
                log_error_m << TASK_EVENT_LOG(msg, exitCode);
                TASK_CLEAN_AND_RETURN(TASK_NAME);
            }

            const char* msg = u8"Отчет %1.oms сохранен в директорию %2";
            log_info_m << TASK_EVENT_LOG(msg, _report.name, storeDir);
        }
        else
        {
            _retInfo = {RetInfo::Error::General};
            const char* msg = u8"Не обнаружена директория для сохранения отчетов: %1";
            log_error_m << TASK_EVENT_LOG(msg, storeDir);
            TASK_CLEAN_AND_RETURN(TASK_NAME);
        }
    }

    if (!sql::exec(q,
        "UPDATE REPORT_FED SET XML_DATA = ? WHERE TASK_ID = ?",
        xmlStr.toUtf8(), id()))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    if (!sql::exec(q,
        "UPDATE REPORT_FED SET EXEC_STATUS = ? WHERE TASK_ID = ?",
        data::ReportFed::ExecStatus::Success, id()))
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    if (!transact->commit())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    _progressCurrent = 100;
    m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    data::TaskContentCreate taskContentCreate;
    taskContentCreate.taskType = {TaskType::CreateReportFed};
    taskContentCreate.taskId = id();
    taskContentCreate.userId = userId();
    taskContentCreate.contentId = contentId();
    taskContentCreate.isPeriodic = isPeriodic();

    m = createJsonMessage(taskContentCreate, Message::Type::Event);
    webCon().send(m);

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME);

    #undef TASK_NAME
}

void CreateReportFed::threadStopEstablished()
{
    dbpool().abortOperation(_threadId);
}

void CreateReportFed::sanitize()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    sql::exec(q, "UPDATE REPORT_FED SET EXEC_STATUS = ? WHERE TASK_ID = ?",
                 data::ReportFed::ExecStatus::Failed, id());
}

const QUuidEx& CreateReportFed::contentId() const
{
    return _report.id;
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
