#include "learn_model.h"
#include "task_messages.h"
#include "functions.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "commands/sync_data.h"
#include "commands/error.h"
#include "commands/model.h"
#include "shared/qt/config/config.h"
#include "shared/defmac.h"
#include "xgboost/options.h"
#include "xgboost/autofree.h"
#include "3rdparty/xgboost/include/xgboost/learner.h"
#include "3rdparty/xgboost/include/xgboost/c_api.h"
#include "scheduler.h"
#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"

#include <unistd.h>
#include <chrono>
#include <string>
#include <QtSql>
#include <QDateTime>
#include <QDate>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "LearnModel")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "LearnModel")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "LearnModel")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "LearnModel")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "LearnModel")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "LearnModel")

namespace task {

using namespace db::firebird;
using namespace sql;

LearnModel::LearnModel(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTaskThread(TaskType::LearnModel, taskId, userId)
{}

LearnModel::~LearnModel()
{
    log_debug_m << "Call ~LearnModel() TaskId: " << id();
}

void LearnModel::run()
{
    #define TASK_NAME u8"Обучение"

    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    _interrupted = false;

    log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME);

    if (!_baseInit)
    {
        _retInfo = {RetInfo::Error::Init};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_INIT);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    _progressCurrent = 0;
    _progressTotal = 100;

    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    _diffLimit = 0.01;
    config::base().getValue("learn_model.diff_limit", _diffLimit);

    _iterCount = 10;
    config::base().getValue("learn_model.iter_count", _iterCount);

    _thresholdStep = 0.01;
    config::base().getValue("learn_model.threshold_step", _thresholdStep);

    _dumpFlush = false;
    config::base().getValue("dump.flush", _dumpFlush);

    if (_dumpFlush)
    {
        log_debug_m << "Dump flush enabled";

        _dumpDir.clear();
        config::base().getValue("dump.dir", _dumpDir);

        if (!QDir(_dumpDir).exists())
            log_error_m << "Dump directory not exists: " << _dumpDir;
    }
    else
        log_debug_m << "Dump flush disabled";

    try
    {
        _retInfo = initPeriod();
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_INIT)

        _retInfo = syncCheck(this);
        TASK_CHECK_NEEDSYNC_AND_ERROR(TASK_NAME, TASK_ERR_SYNC)

        _retInfo = dataIsPresent(this, TASK_ERR_NODATA_ENOUGH);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_DATA_CHECK)

        _retInfo = makeDictionary();
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_LEARN_MEE)

        _retInfo = makeLearn(ScoreType::MEE);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_LEARN_MEE)

        _retInfo = makeLearn(ScoreType::EKMP);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_LEARN_EKMP)

        _retInfo = insertModel();
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_MODEL_SAVE)

        fmeraShiftPeriod();

        _retInfo = dataIsPresent(this, TASK_ERR_NODATA_FMERA);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_DATA_FMERA)

        _retInfo = FMeasure(ScoreType::MEE);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_CALC_FMERA, u8"МЭЭ")

        _retInfo = FMeasure(ScoreType::EKMP);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_CALC_FMERA, u8"ЭКМП")
    }
    catch (const std::exception& e)
    {
        _retInfo = {RetInfo::Error::General};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_UNHANDLED, TASK_NAME, e.what());
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    data::TaskContentCreate taskContentCreate;
    taskContentCreate.taskType = {TaskType::LearnModel};
    taskContentCreate.taskId = id();
    taskContentCreate.userId = userId();
    taskContentCreate.contentId = contentId();
    taskContentCreate.isPeriodic = isPeriodic();

    m = createJsonMessage(taskContentCreate, Message::Type::Event);
    webCon().send(m);

    if (isPeriodic())
    {
        _retInfo = startChild();
        if (!_retInfo/*.isSuccess()*/)
            log_warn_m << TASK_EVENT_LOG(u8"Не удалось запустить зависимые задачи");
    }

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME);

    #undef TASK_NAME
}

const QUuidEx& LearnModel::contentId() const
{
    return _modelId;
}

RetInfo LearnModel::initPeriod()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(" SELECT                "
                   "   PERIOD_BEGIN        "
                   "  ,PERIOD_END          "
                   "  ,REL_PERIOD_DURATION "
                   " FROM                  "
                   "   TASK                "
                   " WHERE                 "
                   "   ID = :ID            "))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }
    bindValue(q, ":ID" , id());

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        return {RetInfo::Error::Sql};
    }

    QSqlRecord r = q.record();

    assignValue(_period.begin,      r, "PERIOD_BEGIN        ");
    assignValue(_period.end,        r, "PERIOD_END          ");
    assignValue(_relPeriodDuration, r, "REL_PERIOD_DURATION ");

    // Если задача периодическая, интервал необходимо вычислить исходя
    // из текущего месяца.
    if (isPeriodic())
    {
        initLearnPeriod(_period, _relPeriodDuration);
    }

    return {RetInfo::Success};
}

RetInfo LearnModel::insertModel()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    QString query = " SELECT                 "
                    "   ID                  ,"
                    "   USER_ID             ,"
                    "   PARENT_ID           ,"
                    "   TASK_TYPE           ,"
                    "   NAME                ,"
                    "   DESCRIPTION         ,"
                    "   RUN_DATETIME        ,"
                    "   REGULARITY_MONTH    ,"
                    "   REGULARITY_DAY      ,"
                    "   REGULARITY_HOUR     ,"
                    "   ATTEMPT_LIMIT       ,"
                    "   ATTEMPT_COUNTER     ,"
                    "   ATTEMPT_INTERVAL    ,"
                    "   IS_ENABLED          ,"
                    "   IS_PUBLIC           ,"
                    "   PERIOD_BEGIN        ,"
                    "   PERIOD_END          ,"
                    "   REL_PERIOD_DURATION ,"
                    "   MODEL_ID             "
                    " FROM TASK              "
                    "   WHERE ID = :ID       ";

    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":ID" , id());

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        return {RetInfo::Error::Sql};
    }

    QSqlRecord r = q.record();

    QUuidEx taskId, userId;
    assignValue(taskId, r, "ID");
    assignValue(userId, r, "USER_ID");

    QString name, decr;
    assignValue(name,   r, "NAME");
    assignValue(decr,   r, "DESCRIPTION");

    QDateTime create = QDateTime::currentDateTime();

    short isPublic;
    assignValue(isPublic, r, "IS_PUBLIC");

    QString fields =
        " ID,                "
        " TASK_ID,           "
        " USER_ID,           "
        " NAME,              "
        " DESCRIPTION,       "
        " CREATE_DATE,       "
        " PERIOD_BEGIN,      "
        " PERIOD_END,        "
      //" PERIOD_BEGIN_REAL, "
      //" PERIOD_END_REAL,   "
        " IS_PUBLIC,         "
        " OUTDATED_ROWS      ";

    query = sql::INSERT_INTO("MODEL", fields);
    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":ID               " , contentId()       );
    bindValue(q, ":TASK_ID          " , id()              );
    bindValue(q, ":USER_ID          " , userId            );
    bindValue(q, ":NAME             " , name              );
    bindValue(q, ":DESCRIPTION      " , decr              );
    bindValue(q, ":CREATE_DATE      " , create            );
    bindValue(q, ":PERIOD_BEGIN     " , _period.begin     );
    bindValue(q, ":PERIOD_END       " , _period.end       );
  //bindValue(q, ":PERIOD_BEGIN_REAL" , _periodReal.begin );
  //bindValue(q, ":PERIOD_END_REAL  " , _periodReal.end   );
    bindValue(q, ":IS_PUBLIC        " , isPublic          );
    bindValue(q, ":OUTDATED_ROWS    " , 0                 );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!sql::exec(q, "DELETE FROM SYNC_PLANNING WHERE TASK_ID = ?", id()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        return {RetInfo::Error::Sql};
    }

    return {RetInfo::Success};
}

RetInfo LearnModel::fillArrays(Ret2DArray& data, Ret2DArray& labels,
                               ScoreType type, bool forFMera, bool showProgress)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QString sql =
        " SELECT COUNT(*) FROM SYNC_DATA "
        " WHERE                          "
        "   DATE_OT_PER >= :PERIOD_BEGIN    "
        "   AND                          "
        "   DATE_OT_PER <= :PERIOD_END      "
        "   AND                          "
        "   %1 >= 0                      ";

    sql = sql.arg((type == ScoreType::MEE) ? "MEE" : "EKMP");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":PERIOD_BEGIN", _period.begin );
    bindValue(q, ":PERIOD_END",   _period.end   );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    q.first();
    int recCount = q.value(0).toInt();
    int step, limit;

    if (forFMera)
    {
        if (recCount == 0)
        {
            log_error_m << TASK_EVENT_LOG(TASK_MSG_NODATA);
            return {RetInfo::Error::NoData};
        }

        if (alog::logger().level() >= alog::Level::Debug)
        {
            log_debug_m << "Filling array for Score: " << contentId()
                        << ", baseModel: " << _modelId
                        << ", begin: "     << _period.begin.date()
                        << ", end: "       << _period.end.date()
                        << ", recCount: "  << recCount;
        }

        step = recCount / 4;
        step = (step == 0) ? 1 : step;

        _progressCurrent = (type == ScoreType::MEE) ? 90 : 95;
        limit            = (type == ScoreType::MEE) ? 94 : 99;
    }
    else
    {
        if (recCount == 0)
        {
            const char* score = (type == ScoreType::MEE) ? u8"МЭЭ" : u8"ЭКМП";
            log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_CALC_DATA, score);
            return {RetInfo::Error::NoData};
        }

        if (alog::logger().level() >= alog::Level::Debug)
        {
            log_debug_m << "Filling array for Model: " << _modelId
                        << ", TaskId: "   << id()
                        << ", begin: "    << _period.begin.date()
                        << ", end: "      << _period.end.date()
                        << ", recCount: " << recCount;
        }

        step = recCount / 9;
        step = (step == 0) ? 1 : step;

        _progressCurrent = (type == ScoreType::MEE) ? 1  : 46;
        limit            = (type == ScoreType::MEE) ? 10 : 55;
    } // if (forFMera)

    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    if (showProgress)
        webCon().send(m);

    QString colNames =
        " MSK_OT_ORD, VID_MP_ORD, USL_OK_ORD, PROFIL_ORD, MKB1_ORD, MKB2_ORD, CODE_USL_ORD,         "
        " CODE_MD_ORD, KOL_USL, KOL_FACT, ISH_MOV_ORD, RES_GOSP_ORD, TARIF_B, TARIF_S, VID_TR_ORD,  "
        " EXTR_ORD, SPEC_MD_ORD, DOMC_TYPE_ORD, CODE_LPU_ORD, VID_SF_ORD, PERSCODE_ORD, VID_KOEFF,  "
        " USL_TMP, SEX, FOR_POM, P_CEL_ORD, DN_ORD, MKB0_ORD, DS_ONK, MKB1SS_ORD, MKB1S_ORD,        "
        " DATE_NPR_OF_WEEK, DATE_NPR_MONTH, DATE_NPR_DIF, DATE_IN_OF_WEEK, DATE_IN_MONTH,           "
        " DATE_OUT_OF_WEEK, DATE_OUT_MONTH, DATE_DIFF, OLD_VAR, KOL_USL_per_Day, KOL_FACT_per_Day,  "
        " DATE_NPR_DIF_RATE,KOL_USLKOL_FACT, TARIF_BTARIF_D, TARIF_BSUM_RUB, KOL_USLKOL_FACT_p,     "
        " TARIF_BSUM_RUB_p, KOL_USL_rate, KOL_FACT_rate, DIF_OT, AGGR_LPU_KOL_DEF,  AGGR_LPU_COUNT, "
        " AGGR_LPU_RATE, AGGR_MSK_OT_KOL_DEF, AGGR_MSK_OT_COUNT, AGGR_MSK_OT_RATE, OKATO_INS_ORD, CODE_MSK_ORD, SUM_PER_DAY ";

    sql = " SELECT                                            ";
    for (const QString& column : colNames.split(","))
    {
        sql += column.trimmed() + ",";
    }
    sql += "   KOL_DEF                                        "
           " FROM                                             "
           "   %1(:MODEL_ID, :PERIOD_BEGIN, :PERIOD_END, %2); ";

    sql = sql.arg("GET_LEARN_ROWS");
    sql = sql.arg((type == ScoreType::MEE) ? "0" : "1");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":MODEL_ID     ", contentId()   );
    bindValue(q, ":PERIOD_BEGIN ", _period.begin );
    bindValue(q, ":PERIOD_END   ", _period.end   );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    const int colCount = colNames.split(",").count();

    QList<QVector<float>> records;
    QVector<float> reclbl;

    while (q.next())
    {
        CHECK_QTHREADEX_STOP

        int iCol = 0;
        QSqlRecord r = q.record();

        QVector<float> rec;
        rec.resize(colCount);

        for (const QString& column : colNames.split(","))
        {
            rec[iCol++] = toFloat(r, column.trimmed());
        }

        records.append(rec);
        reclbl.append(toFloat( r, "KOL_DEF" ));

#ifdef TEST_MODE
        // Логгирование для проверки данных с БД
        if (alog::logger().level() >= alog::Level::Debug)
        {
            alog::Line logLine = log_debug_m << "Values: ";
            for (int i = 0; i < iCol - 1; ++i)
                logLine << rec[i] << "; ";
        }
        // Логгирование для проверки данных с БД
        if (alog::logger().level() >= alog::Level::Debug)
        {
            alog::Line logLine = log_debug_m << "DefValues: ";
            logLine << recDef[0] << "; ";
        }
#endif

        if (records.count() % step == 0)
        {
            _progressCurrent =
                (_progressCurrent < limit) ? (_progressCurrent + 1) : limit;
            m = createJsonMessage(progress(), Message::Type::Event);
            if (showProgress)
                webCon().send(m);
        }
    } // while (q.next())

    if (threadStop())
        return {RetInfo::Error::Interrupt};

    if (records.isEmpty())
    {
        if (forFMera)
        {
            log_error_m << TASK_EVENT_LOG(TASK_MSG_NODATA);
        }
        else
        {
            const char* score = (type == ScoreType::MEE) ? u8"МЭЭ" : u8"ЭКМП";
            log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_CALC_DATA, score);
        }
        return {RetInfo::Error::NoData};
    }

    data.alloc(records.count(), colCount);

    for (int i = 0; i < records.count(); ++i)
    {
        float* r1 = (float*)records[i].constData();
        float* r2 = data.ptr() + i * data.columns();
        for (int j = 0; j < colCount; ++j)
            *r2++ = *r1++;
    }

    labels.alloc(records.count(), 1);

    for (int i = 0; i < reclbl.count(); ++i)
    {
        if (reclbl.at(i) != 0 && reclbl.at(i) != 1)
        {
             log_debug2_m << "Label error. Not binary input for model: " << contentId()
                          << " because it more not 0 or 1, value: " << reclbl.at(i);
        }
        labels.ptr()[i] = reclbl.at(i);
    }

    // Сохранение дампа бинарных данных в файл, имя которого имеет формат
    // [Идентификатор Модели]_[Тип Модели].dump
    if (_dumpFlush)
    {
        QString modelId = _modelId.toString();
        modelId.remove(0, 1);
        modelId.chop(1);

        const char* modelType = (type == ScoreType::MEE) ? "mee" : "ekmp";
        QString dataType = (forFMera == true) ? "apply" : "train";
        QString dumpPath = _dumpDir + "/" + modelId + "_" + modelType + "_" + dataType + ".dump";

        QFile dumpFile {dumpPath};
        if (dumpFile.open(QIODevice::WriteOnly))
        {
            QTextStream dumpStream {&dumpFile};
            for (int i = 0; i < records[0].count(); ++i)
            {
                // TODO Володя, объясни работу этого кода
                dumpStream << colNames.split(",").toVector()[i].trimmed() << ";";
            }
            dumpStream << "KOL_DEF" << endl;

            for (int i = 0; i< records.count(); ++i)
            {
                const QVector<float>& line = records[i];
                for (float item : line)
                    dumpStream << item << ";";

                dumpStream << reclbl[i] << endl;
            }
            dumpFile.close();
        }
        else
            log_error_m << "Failed open dump file: " << dumpPath;
    }

    return {RetInfo::Success};
}

RetInfo LearnModel::makeDictionary()
{
    log_info_m << TASK_EVENT_LOG(DICTIONARY_START);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql, RetInfo::Critical::Yes};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    std::atomic_int sqlExecError = {0};
    trd::ThreadIdList threadIds;

    auto funcMakeDictionary = [this, &threadIds, &transact, &sqlExecError]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult(transact)};

        log_debug_m << "Создание словаря для модели: " << contentId() << " выполняется.";

        if (!sql::exec(q, "EXECUTE PROCEDURE MAKE_DICTIONARY(?, ?, ?)",
            contentId(),
            _period.begin,
            _period.end))
        {
            ++sqlExecError;
            log_debug_m << "В процессе создания словаря для модели: " << contentId() << " возникла ошибка.";
        }

        log_debug_m << "Создание словаря для модели: " << contentId() << " завершено.";
    };

    auto funcMakeDictionaryMostFreq = [this, &threadIds, &transact, &sqlExecError]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult(transact)};

        log_debug_m << "Создание частотного словаря для модели: " << contentId() << " выполняется.";

        if (!sql::exec(q, "EXECUTE PROCEDURE MAKE_DICTIONARY_MOST_FREQ(?, ?, ?)",
            contentId(),
            _period.begin,
            _period.end))
        {
            ++sqlExecError;
            log_debug_m << "В процессе создания частотного словаря для модели: " << contentId() << " возникла ошибка.";
        }

        log_debug_m << "Создание частотного словаря для модели: " << contentId() << " завершено.";
    };

    auto funcMakeAggrMee = [this, &threadIds, &transact, &sqlExecError]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult(transact)};

        log_debug_m << "Создание агрегатов (МЭЭ) для модели: " << contentId() << " выполняется.";

        if (!sql::exec(q, "EXECUTE PROCEDURE MAKE_AGGR(?, ?, ?, 0)",
            contentId(),
            _period.begin,
            _period.end))
        {
            ++sqlExecError;
            log_debug_m << "В процессе создания агрегатов (МЭЭ) для модели: " << contentId() << " возникла ошибка.";
        }

        log_debug_m << "Создание агрегатов (МЭЭ) для модели: " << contentId() << " завершено.";
    };

    auto funcMakeAggrEkmp = [this, &threadIds, &transact, &sqlExecError]()
    {
        trd::ThreadIdLock threadIdLock(&threadIds); (void) threadIdLock;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult(transact)};

        log_debug_m << "Создание агрегатов (ЭКМП) для модели: " << contentId() << " выполняется.";

        if (!sql::exec(q, "EXECUTE PROCEDURE MAKE_AGGR(?, ?, ?, 1)",
            contentId(),
            _period.begin,
            _period.end))
        {
            ++sqlExecError;
            log_debug_m << "В процессе создания агрегатов (ЭКМП) для модели: " << contentId() << " возникла ошибка.";
        }

        log_debug_m << "Создание агрегатов (ЭКМП) для модели: " << contentId() << " завершено.";
    };

    std::thread t1 {funcMakeDictionary};
    t1.detach();

    std::thread t2 {funcMakeDictionaryMostFreq};
    t2.detach();

    std::thread t3 {funcMakeAggrMee};
    t3.detach();

    std::thread t4 {funcMakeAggrEkmp};
    t4.detach();

    // Вместо sleep(1);
    while (threadIds.empty())
        usleep(1);

    while (!threadIds.empty())
    {
        usleep(100*1000);
        if (threadStop())
        {
            threadIds.lock([](std::vector<pid_t>& tids) {
                for (pid_t tid : tids)
                    dbpool().abortOperation(tid);
            });
            break;
        }
    }
    while (!threadIds.empty())
        usleep(100*1000);

    if (threadStop())
        return {RetInfo::Error::Interrupt};

    if (sqlExecError > 0)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        return {RetInfo::Error::Sql, RetInfo::Critical::Yes};
    }

    log_info_m << TASK_EVENT_LOG(DICTIONARY_STOP);

    return {RetInfo::Success};
}

#define ITERATION_PRECISION_BREAK

RetInfo LearnModel::makeLearn(ScoreType type)
{
    int res;
    RetInfo retInfo = {RetInfo::Error::Undef};

    Ret2DArray inData;
    Ret2DArray lblData;

    if (!(retInfo = fillArrays(inData, lblData, type, false, true)))
        return retInfo;

#ifdef ITERATION_PRECISION_BREAK
    // Массивы данных для Ф-Меры
    Ret2DArray inDataF;
    Ret2DArray lblDataF;
    // Сохранение текущих границ периода
    QDateTime begin = _period.begin;
    QDateTime end = _period.end;

    _period.begin = _period.begin.addMonths(-1);
    _period.end = QDateTime(lastDay(_period.begin));

    // Заполнение массива данных на предыдущем месяце
    if (!(retInfo = fillArrays(inDataF, lblDataF, type, true, false)))
        return retInfo;

    log_verbose_m << "F-mera iteration check period: " << _period.begin << " - " << _period.end
                  << ". TaskId: " << id()
                  << ". Task period: " << _period.begin << " - " << _period.end;

    // Восстановление границ периода
    _period.begin = begin;
    _period.end = end;

    DMatrixHandle h_trainF[1];

    res = XGDMatrixCreateFromMat(inDataF.ptr(), inDataF.rows(), inDataF.columns(), NAN, &h_trainF[0]);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL2, "XGDMatrixCreateFromMat()", res);
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }

    XGBOOST_MATRIX_AUTO_FREE2(h_trainF[0])
#endif

    _progressCurrent = (type == ScoreType::MEE) ? 10 : 55;
    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    DMatrixHandle h_train[1];

    res = XGDMatrixCreateFromMat(inData.ptr(), inData.rows(), inData.columns(), NAN, &h_train[0]);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL2, "XGDMatrixCreateFromMat()", res);
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }
    XGBOOST_MATRIX_AUTO_FREE(h_train[0])

    res = XGDMatrixSetFloatInfo(h_train[0], "label", lblData.ptr(), lblData.rows());
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL2, "XGDMatrixSetFloatInfo()", res);
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }

    BoosterHandle h_booster;
    res = XGBoosterCreate(h_train, 1, &h_booster);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL2, "XGBoosterCreate()", res);
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }
    XGBOOST_HANDLE_AUTO_FREE(h_booster)

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql, RetInfo::Critical::Yes};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    QMap<QString, QString> options;
    if (!saveXgbOptions(q, options, type))
    {
        log_error_m << TASK_EVENT_LOG(u8"Ошибка сохранения настроек XGBoost");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::No};
    }

    if (!(retInfo = xgb::applyOptions(h_booster, options)))
        return retInfo;

    int iterationLimit = 500;
    iterationLimit = (type == ScoreType::MEE) ? 500 : 200;

    int step = int(iterationLimit / 35.);
    step = (step == 0) ? 1 : step;

    int limit = (type == ScoreType::MEE) ? 45 : 90;

    QString modelId = _modelId.toString();
    modelId.remove(0, 1);
    modelId.chop(1);

    const char* modelType = (type == ScoreType::MEE) ? "mee" : "ekmp";
    QString dumpPath  = _dumpDir + "/" + modelId + "_" + modelType + "_scores.dump";

    QFile dumpFile {dumpPath};
    if (_dumpFlush)
        if (!dumpFile.open(QIODevice::WriteOnly))
        {
            log_error_m << "Failed open dump file: " << dumpPath;
        }

#ifdef ITERATION_PRECISION_BREAK
    // Площадь предудущей фигуры для каждой итерации
    double squareOnePrev = 0;
    // Площадь предудущей фигуры для нескольких итераций
    double squareManyPrev = 0;

#endif
    for (int i = 0; i <= iterationLimit; i++)
    {
        XGBoosterUpdateOneIter(h_booster, i, h_train[0]);
        CHECK_QTHREADEX_STOP

#ifdef ITERATION_PRECISION_BREAK
        // Разница между соседними итерациями
        double diffOne = 0;
        // Разница между несколькими итерациями
        double diffMany = 0;

        double f1MeasureMax = 0;
        double f1ThresholdMax = 0;

        bst_ulong out_len;
        const float* f;
        res = XGBoosterPredict(h_booster, h_trainF[0], 0, 0, &out_len, &f);
        if (res)
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterPredict()");
            return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
        }

        double TP = 0;
        double TN = 0;
        double FP = 0;
        double FN = 0;
        double threshold = 0;

        // Обнуление значения текущей площади
        double squareCurrent = 0;

        if (_dumpFlush
            && dumpFile.isOpen()
            && i == iterationLimit)
        {
            QTextStream dumpStream {&dumpFile};
            dumpStream << "predict;expert" << endl;
        }

        for (threshold = _thresholdStep; threshold <= 1.0; threshold += _thresholdStep)
        {
            CHECK_QTHREADEX_STOP

            TP = 0;
            TN = 0;
            FP = 0;
            FN = 0;

            bool predict = false;
            bool fact = false;

            for (int j = 0; j < int(out_len); ++j)
            {
                CHECK_QTHREADEX_STOP

                // Оценка АИС
                float predictScore = f[j];

                if (predictScore <= threshold)
                    predict = false;
                else
                    predict = true;

                // Оценка эксперта
                float expertScore = lblDataF.ptr()[j];
                if (expertScore > 0)
                    fact = true;
                else
                    fact = false;

                if (_dumpFlush
                    && dumpFile.isOpen()
                    && threshold == _thresholdStep
                    && i == iterationLimit)
                {
                    QTextStream dumpStream {&dumpFile};
                    dumpStream << QString::number(predictScore, 'f', 5) << ";"
                               << QString::number(expertScore,  'f', 5) << endl;
                }

                if (predict && fact)
                    ++TP;
                else if (predict && !fact)
                    ++FP;
                else if (!predict && fact)
                    ++FN;
                else if (!predict && !fact)
                    ++TN;
            }

            if (threadStop())
                return {RetInfo::Error::Interrupt};

            double precision = 0;
            if ((TP + FP) > 0)
                precision = TP / ( TP + FP );

            double recall = 0;
            if ((TP + FN) > 0)
                recall = TP / ( TP + FN );

            double f1Measure = 0;

            if ((precision + recall) < 0 || (precision + recall) > 0)
                f1Measure = 2 * ( (precision * recall) / (precision + recall) );

            // Если найдено максимальное значение Ф-Меры, то необходимо
            // сохранить значение Ф-Меры и значение порога, на котором
            // данное максимальное значение найдено.
            if (f1Measure > f1MeasureMax)
            {
                f1MeasureMax = f1Measure;
                f1ThresholdMax = threshold;
            }

            // Общая площадь фигуры
            squareCurrent += _thresholdStep * f1Measure;
        }

        modelType = (type == ScoreType::MEE) ? "МЭЭ" : "ЭКМП";
        log_debug_m << u8"F-мера (" << modelType << "): " << f1MeasureMax
                    << u8", итерация: " << i
                    << u8", порог: " << f1ThresholdMax;

//        if (type == ScoreType::MEE)
//            log_debug_m << u8"F-мера (МЭЭ): "  << f1MeasureMax
//                        << u8", итерация: "    << i
//                        << u8", порог: "       << f1ThresholdMax;
//        else
//            log_debug_m << u8"F-мера (ЭКМП): " << f1MeasureMax
//                        << u8", итерация: "    << i
//                        << u8", порог: "       << f1ThresholdMax;

        // Для первой итерации значение предыдущего заполняется первым вычисленным значением
        // и расчёт разницы не прозводится
        if (i == 0)
        {
            squareOnePrev = squareCurrent;
            squareManyPrev = squareCurrent;
            continue;
        }
        // Разница текущей и предыдущей площадей
        diffOne = squareCurrent - squareOnePrev;

        // Разница между текущей площадью и предыдущей с разницей в _iterCount итераций
        if (i % _iterCount == 0)
        {
            diffMany = squareCurrent - squareManyPrev;
        }

        log_debug_m << u8"Разность (между соседними итерациями): " << QString::number(diffOne, 'f', 5)
                    << u8", итерация: " << i;

        squareOnePrev = squareCurrent;

//        if (diffOne < 0.001)
//        {
//            log_debug_m << u8"Обучение остановлено, так как прирост Ф-Меры составил: " << diffOne
//                        << ". При пороге прироста : " << 0.001;
//            break;
//        }

        (void) diffOne;
        (void) diffMany;
//        if (i % _iterCount == 0)
//        {
//            log_debug_m << u8"Разность (между " << _iterCount << " итераций): " << QString::number(diffMany, 'f', 5)
//                        << u8", итерация: " << i;

//            squareManyPrev = squareCurrent;

//            if (diffMany < _diffLimit)
//            {
//                log_debug_m << u8"Обучение остановлено, так как прирост Ф-Меры составил: " << diffMany
//                            << ". При пороге прироста : " << _diffLimit;

//                break;
//            }
//        }
#endif
        if ((i + 1) % step == 0)
        {
            _progressCurrent = (_progressCurrent < limit) ? ++_progressCurrent : limit;
            m = createJsonMessage(progress(), Message::Type::Event);
            webCon().send(m);
        }
    }
    dumpFile.close();

    if (threadStop())
        return {RetInfo::Error::Interrupt};

    uint64_t out_len_model = 0;
    const char* out_dptr = nullptr;

    XGBoosterGetModelRaw(h_booster, &out_len_model, &out_dptr);

    log_debug_m << u8"Размер сохранямой модели: " << out_len_model << u8" байт"
                << " (" << contentId() << ")";

    QByteArray rawData {out_dptr, int(out_len_model)};

    QString query = " UPDATE OR INSERT INTO MODEL_DATA "
                    " ( ID, KIND, RAW_DATA)            "
                    " VALUES                           "
                    " ( :ID, :KIND, :RAW_DATA )        "
                    " MATCHING (ID, KIND)              ";

    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":ID       ", contentId()                      );
    bindValue(q, ":KIND     ", (type == ScoreType::MEE) ? 0 : 1 );
    bindValue(q, ":RAW_DATA ", rawData                          );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        return {RetInfo::Error::Sql, RetInfo::Critical::Yes};
    }

    _progressCurrent = (type == ScoreType::MEE) ? 45 : 90;
    m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

#ifdef ITERATION_CHECK
    _period.begin = begin;
    _period.end = end;
#endif

    return {RetInfo::Success};
}

RetInfo LearnModel::startChild()
{
    QString query = " SELECT                      "
                    "   ID                        "
                    " FROM                        "
                    "   TASK                      "
                    " WHERE                       "
                    "   TASK_TYPE = :TASK_TYPE    "
                    "   AND PARENT_ID = :MODEL_ID "
                    "   AND USER_ID = :USER_ID    "
                    "   AND IS_PERIODIC = 1       "
                    "   AND RUN_CHILD_TASK = 1    ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":TASK_TYPE", getTaskString(TaskType::ScoreCalc));
    bindValue(q, ":MODEL_ID ", id());
    bindValue(q, ":USER_ID  ", userId());

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QUuidEx taskId;
    // Все связанные задачи будут запущены немедленно.
    while(q.next())
    {
        assignValue(taskId, q.record(), "ID"  );

        data::TaskStartNow taskStartNow;
        taskStartNow.taskId = taskId;
        taskStartNow.userId = userId();

        log_info_m << TASK_EVENT_LOG(u8"Запуск зависимой задачи 'Применение': %1"
                                     u8"; Родительская задача: %2", taskId, id());

        Message::Ptr m = createJsonMessage(taskStartNow);
        scheduler().message(m);
    }

    return {RetInfo::Success};
}

bool LearnModel::loadXgbOptions(QMap<QString, QString>& options, ScoreType type)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QString sql = "SELECT NAME, CONTENT FROM MODEL_XGB WHERE ID = ? AND KIND = %1";
    sql = sql.arg((type == ScoreType::MEE) ? "0" : "1");

    if (!sql::exec(q, sql, _modelId))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return false;
    }

    while (q.next())
    {
        QSqlRecord r = q.record();
        QString name, value;
        assignValue(name,  r, "NAME"  );
        assignValue(value, r, "CONTENT" );
        options[name] = value;
    }

    return true;
}

bool LearnModel::saveXgbOptions(QSqlQuery& q, QMap<QString, QString>& options, ScoreType type)
{
    QString sql = " SELECT                  "
                  "   NAME, CONTENT         "
                  " FROM                    "
                  "   MODEL_XGB             "
                  " WHERE                   "
                  "   ID = ? AND ENABLED = 1 AND KIND = %1";
    sql = sql.arg((type == ScoreType::MEE) ? "0" : "1");

    // Получить настройки профиля
    if (!sql::exec(q, sql, QUuidEx{}))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return false;
    }

    while (q.next())
    {
        QSqlRecord r = q.record();
        QString name, value;
        assignValue(name,    r, "NAME"  );
        assignValue(value,   r, "CONTENT" );
        options[name] = value;
    }
    // Сохранить настройки профиля
    for (const QString& key : options.keys())
    {
        QString value = options.value(key);
        if (!sql::exec(q, " UPDATE OR INSERT INTO MODEL_XGB "
                          " (ID, KIND, NAME, CONTENT, ENABLED)    "
                          " VALUES                          "
                          " (?, ?, ?, ?, ?)                    "
                          " MATCHING (ID, KIND, NAME)             ",
                          contentId(), (type == ScoreType::MEE) ? 0 : 1, key, value, true))
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            return false;
        }
    }
    return true;
}

void LearnModel::sanitize()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "DELETE FROM MODEL_DATA WHERE ID = ?", contentId()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
    }

    if (!sql::exec(q, "DELETE FROM MODEL_XGB WHERE ID = ?", contentId()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
    }

    if (!sql::exec(q, "DELETE FROM MODEL_DICTIONARY WHERE MODEL_ID = ?", contentId()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
    }

    if (!sql::exec(q, "DELETE FROM MODEL_DICTIONARY_MOST_FREQ WHERE MODEL_ID = ?", contentId()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
    }
}

void LearnModel::fmeraShiftPeriod()
{
    QDateTime start = _period.begin.addMonths(-1);
    QDateTime left  = QDateTime(firstDay(start));
    QDateTime right = QDateTime(lastDay(start));

    log_verbose_m << "F-mera period: " << left << " - " << right
                  << ". TaskId: " << id()
                  << ". Task period: " << _period.begin << " - " << _period.end;

    _period.begin = left;
    _period.end = right;
}

RetInfo LearnModel::FMeasure(ScoreType type)
{
    RetInfo retInfo = {RetInfo::Error::Undef};
    int res;

    Ret2DArray inData;
    Ret2DArray lblExpert;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!q.prepare(" SELECT RAW_DATA FROM MODEL_DATA "
                   " WHERE ID = :ID AND KIND = :KIND "))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":ID"   , _modelId);
    bindValue(q, ":KIND" , (type == ScoreType::MEE) ? 0 : 1);

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_NOMODEL);
        return {RetInfo::Error::NoModel, RetInfo::Critical::Yes};
    }

    QByteArray ba;
    assignValue(ba, q.record(), "RAW_DATA");

    if (!(retInfo = fillArrays(inData, lblExpert, type, true, true)))
        return retInfo;

    if (inData.rows() == 0)
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_NODATA);
        return {RetInfo::Error::NoData, RetInfo::Critical::No};
    }

    _progressCurrent = (type == ScoreType::MEE) ? 94 : 99;
    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    // create the booster and load some parameters
    BoosterHandle h_booster;

    res = XGBoosterCreate(0, 0, &h_booster);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterCreate()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }
    XGBOOST_HANDLE_AUTO_FREE(h_booster)

    QMap<QString, QString> options;
    if (!loadXgbOptions(options, type))
        return {RetInfo::Error::Xgboost, RetInfo::Critical::No};

    if (!(retInfo = xgb::applyOptions(h_booster, options)))
        return retInfo;

    res = XGBoosterLoadModelFromBuffer(h_booster, ba.data(), ba.size());
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterLoadModelFromBuffer()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }

    DMatrixHandle inMatrix;
    res = XGDMatrixCreateFromMat(inData.ptr(), inData.rows(), inData.columns(), NAN, &inMatrix);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGDMatrixCreateFromMat()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }
    XGBOOST_MATRIX_AUTO_FREE(inMatrix)

    inData.free();

    bst_ulong out_len;
    const float* f;
    res = XGBoosterPredict(h_booster, inMatrix, 0, 0, &out_len, &f);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterPredict()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }

    double TP = 0;
    double TN = 0;
    double FP = 0;
    double FN = 0;
    double threshold = 0;

    double f1_measure_max = 0;
    double thresholdForMax = 0;

    for (threshold = 0.01; threshold <= 1.0; threshold += 0.01)
    {
        CHECK_QTHREADEX_STOP

        TP = 0;
        TN = 0;
        FP = 0;
        FN = 0;

        bool predict = false;
        bool fact = false;

        for (int i = 0; i < int(out_len); ++i)
        {
            CHECK_QTHREADEX_STOP

            // Оценка АИС
            float predictScore = f[i];

            if (predictScore <= threshold)
                predict = false;
            else
                predict = true;

            // Оценка эксперта
            float expertScore = lblExpert.ptr()[i];
            if (expertScore > 0)
                fact = true;
            else
                fact = false;

            if (predict && fact)
                ++TP;
            else if (predict && !fact)
                ++FP;
            else if (!predict && fact)
                ++FN;
            else if (!predict && !fact)
                ++TN;

        }
        if (threadStop())
            return {RetInfo::Error::Interrupt};

        double precision = 0;
        if ((TP + FP) < 0 || (TP + FP) > 0)
            precision = TP / ( TP + FP );

        double recall = 0;
        if ((TP + FN) < 0 || (TP + FN) > 0)
            recall = TP / ( TP + FN );

        double f1_measure = 0;

        if (precision + recall < 0 || precision + recall > 0)
            f1_measure = 2 * ( (precision * recall) / (precision + recall) );

        if (type == ScoreType::MEE)
            log_debug2_m << u8"F-мера (МЭЭ): " << f1_measure << u8", порог " << threshold;
        else
            log_debug2_m << u8"F-мера (ЭКМП): " << f1_measure << u8", порог " << threshold;

        if (f1_measure > f1_measure_max)
        {
            f1_measure_max = f1_measure;
            thresholdForMax = threshold;
        }

    }

    lblExpert.free();

    if (threadStop())
        return {RetInfo::Error::Interrupt};

    if (type == ScoreType::MEE)
        log_info_m << TASK_EVENT_LOG(TASK_MSG_F1_MEASURE, u8"МЭЭ", QString::number(f1_measure_max, 'f', 3), QString::number(thresholdForMax, 'f', 2));
    else
        log_info_m << TASK_EVENT_LOG(TASK_MSG_F1_MEASURE, u8"ЭКМП", QString::number(f1_measure_max, 'f', 3), QString::number(thresholdForMax, 'f', 2));

    if (!sql::exec(q, " UPDATE OR INSERT INTO  "
                      "   MODEL_DATA           "
                      " (ID, KIND, F1_MEASURE) "
                      " VALUES                 "
                      " (?, ?, ?)              "
                      " MATCHING (ID, KIND)    ",
                      contentId(),
                      (type == ScoreType::MEE) ? 0 : 1,
                      f1_measure_max))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        return {RetInfo::Error::Sql};
    }

    _progressCurrent = (type == ScoreType::MEE) ? 95 : 100;
    m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    return {RetInfo::Success};
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
