#pragma once

#include "base_task.h"
#include "commands/error.h"
#include "commands/time_range.h"
#include "database/firebird_driver.h"
#include "tasks/utils.h"

namespace task {

using namespace communication;

/**
  Задача по вычислению модели
*/
class LearnModel : public BaseTaskThread
{
public:
    LearnModel(const QUuidEx& taskId, const QUuidEx& userId);
    ~LearnModel();

private:
    Q_OBJECT
    DISABLE_DEFAULT_FUNC(LearnModel)

    void run() override;
    const QUuidEx& contentId() const override;

    /**
      @brief Инициализация задачи
      @return RetInfo - с информацие об ошибке
    */
    RetInfo initPeriod() override;

    /**
      @brief Создание модели в БД
      @return RetInfo - с информацие об ошибке
    */
    RetInfo insertModel();

    /**
      @brief Заполнить массив data данными и массив меток arrLbl
      @param arrVals      - строки данных
      @param arrLbl       - метки для обучения
      @param type         - вид оценки МЭЭ или ЭКМП
      @param forFMera     - вид заполнения данных обычное или для Ф-Меры
      @param showProgress - включение/отключение индикации заполнения массива
      @return RetInfo   - с информацие об ошибке
    */
    RetInfo fillArrays(Ret2DArray& data, Ret2DArray& labels, ScoreType type,
                       bool forFMera, bool showProgress);

    RetInfo makeDictionary();

    /**
      @brief Метод создает модель для заданного диапазона данных. Тип модели
             задаётся переменной "type"
      @return RetInfo - с информацие об ошибке
    */
    RetInfo makeLearn(ScoreType type);

    /**
      @brief Поиск задачи применения и её запуск, для которой данная задача обучения
             является связанной.
      @return RetInfo - с информацие об ошибке
    */
    RetInfo startChild();

    /**
      @brief Загрузка сохраненных настроек для модели
      @param options - список, куда будут помещены настройки.
      @return true - в случае успешной загрузки, false - иначе.
    */

    bool loadXgbOptions(QMap<QString, QString>& options, ScoreType type);

    /**
      @brief Сохранение копии(клона) настроек базового профиля
      @return true - если нет ошибок, false - иначе.
    */
    bool saveXgbOptions(QSqlQuery& q, QMap<QString, QString>& options, ScoreType type);

    /**
      @brief Действия, которые производит метод в случае досрочного выхода при
             возникновении ошибки.
    */
    void sanitize() override;

    /**
      @brief Изменение периода _period модели для вычисления F-меры.
    */
    void fmeraShiftPeriod();

    /**
      @brief Вычисление F-меры для периода _period задачи и типа type.
    */
    RetInfo FMeasure(ScoreType type);

    // Идентификатор созданной модели
    const QUuidEx _modelId = {QUuidEx::createUuid()};

    // Фактическая левая и правая  граница диапазона, с которой работает задача обучения
    //TimeRange _periodReal;

    // Самая старшая дата и время модификации записи содержащяяся в кеше
    // для диапазона данных _begin и _end.
    QDateTime _timeMark;

    // Глубина обучения
    qint16 _relPeriodDuration;

    // Элемент вектора номер N указывает на строку данных в массиве, который
    // был заполнен методом fillArrays.
    //QVector<QUuidEx> _rowIndexer;

    double  _diffLimit;
    int     _iterCount;
    double  _thresholdStep;
    bool    _dumpFlush = {false};
    QString _dumpDir;

    //Ret2DArray _colData;
    //Ret2DArray _lblData;
};

} // namespace task
