#include "report_fed.h"
#include "scheduler.h"
#include "task_messages.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "commands/error.h"
#include "commands/report_fed.h"

#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "ReportFed")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "ReportFed")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "ReportFed")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "ReportFed")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "ReportFed")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "ReportFed")

namespace task {

using namespace db::firebird;
using namespace sql;

ReportFed::ReportFed(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTaskThread(TaskType::SyncData, taskId, userId)
{}

void ReportFed::run()
{
    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    _interrupted = false;

    log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME_REPORTFED);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    // Берем первую периодическую задачу по вычислению скоров
    if (!sql::exec(q,
        "SELECT FIRST 1 ID FROM TASK WHERE TASK_TYPE = ? AND IS_PERIODIC = 1",
        getTaskString(TaskType::ScoreCalc)))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME_REPORTFED);
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        TASK_CLEAN_AND_RETURN(TASK_NAME_REPORTFED);
    }

    QUuidEx taskId;
    assignValue(taskId, q.record(), "ID");

    // Берем последний скор созданный периодической задачей
    if (!sql::exec(q,
        " SELECT FIRST 1     "
        "   ID               "
        " FROM               "
        "   SCORE            "
        " WHERE              "
        "   TASK_ID = ?      "
        " ORDER BY           "
        "   CREATE_DATE DESC ", taskId))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        TASK_CLEAN_AND_RETURN(TASK_NAME_REPORTFED);
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_REG_NOSCORE, taskId);
        TASK_CLEAN_AND_RETURN(TASK_NAME_REPORTFED);
    }

    data::TaskReportFedCreate taskReportFedCreate;

    assignValue(taskReportFedCreate.scoreId, q.record(), "ID");
    taskReportFedCreate.isPeriodic = true;
    taskReportFedCreate.isCorrection = false;
    taskReportFedCreate.userId = userId();
    taskReportFedCreate.name = "Task Report Fed";

    Message::Ptr m = createJsonMessage(taskReportFedCreate);
    scheduler().message(m);

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME_REPORTFED);
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
