#pragma once
#include "container_ptr.h"
#include <QtCore>

namespace task
{

class RetInfo
{
public:
    // Псевдо-структура, используется для возврата из функции положительного
    // результата в виде:
    //   return {RetInfo::Success};
    // Такая запись выглядит предпочтительнее чем:
    //   return {RetInfo::Error::None};
    struct SuccessType {};
    static const SuccessType Success;

    // Коды ошибок
    enum class Error
    {
        Undef       =  -1,
      //Ok          =   0,
        None        =   0, // Не ошибка (успех)

        // Коды от 1 до 199 считаем ошибками (условно)
        General     =   1,
        Sql         =   2,
        NoData      =   3,
        Sync        =   4,
        Xgboost     =   5,
        JsonParse   =   6,
        NoModel     =   7,
        NoDatabase  =   8,
        RecordCount =   9,
        Init        =  10,
        Foms        =  11,
        DBError     =  12,

        // Коды от 200 не считаем ошибками, но это и не успех
        Interrupt   =  200,
        NeedSync    =  201,
        Defered     =  202

    };

    enum class Critical
    {
        No  = 0,  // Некритическая ошибка, при возниковении которой возможен
                  // повторный запуск задачи
        Yes = 1   // Критическая ошибка, повтороный запуск не имеет смысла
    };


    bool correspond(Error err) const {return (_d->error == err);}

    bool isSuccess()  const {return (_d->error == Error::None);}
    //bool isError()    const {return (_d->error != Error::Ok);}
    bool isCritical() const {return (_d->critical == Critical::Yes);}
    //bool isInterupt() const {return (_d->error == Error::Interrupt);}

    qint32 code() const {return static_cast<qint32>(_d->error);}

    explicit operator bool () const {return isSuccess();}
    bool operator! () const {return !isSuccess();}

    RetInfo(const RetInfo&) = default;
    RetInfo& operator = (const RetInfo&) = default;

    RetInfo(RetInfo&&);
    RetInfo& operator= (RetInfo&&);

    bool operator== (const RetInfo&) const;

    RetInfo(SuccessType);

    /**
      @brief cоздание структуры с кодом ошибки code.
      @param code - код ошибки
      @details - в случае отсутствия ошибки (код = 0), степень фатальности
                 isFatal будет установлена Pass. Даже в том случае если степень
                 фатальности не задана.
    */
    RetInfo(Error);

    /**
      @brief cоздание структуры с кодом ошибки code и степенью критичности.
      @param code - код ошибки
      @param isFatal - степень критичности
    */
    RetInfo(Error, Critical);

private:
    RetInfo() = delete;

    struct Data
    {
        typedef container_ptr<Data> Ptr;

        Error error = {Error::None};
        Critical critical = {Critical::Yes};
    };
    Data::Ptr _d = {Data::Ptr::create_join_ptr()};
};

} // namespace task
