﻿#include "scheduler.h"
#include "functions.h"
#include "learn_model.h"
#include "task_messages.h"
#include "score_calc.h"
#include "sync_data.h"
#include "sync_nsi.h"
#include "sync_plan.h"
#include "event_log.h"
#include "create_report.h"
#include "create_report_fed.h"
#include "report_fed.h"
#include "user_rights.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "database/settings.h"

#include "commands/commands.h"
#include "commands/error.h"
#include "commands/report.h"
#include "commands/score.h"
#include "commands/sync_data.h"

#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/logger_operators.h"
#include "shared/qt/communication/transport/tcp.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Scheduler")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Scheduler")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Scheduler")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Scheduler")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Scheduler")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Scheduler")

namespace task {

using namespace db::firebird;
using namespace sql;

Scheduler& scheduler()
{
    return ::safe_singleton<Scheduler>();
}

Scheduler::Scheduler()
{
    chk_connect_d(&tcp::listener(), SIGNAL(message(communication::Message::Ptr)),
                  this, SLOT(message(communication::Message::Ptr)))

    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &Scheduler::command_##COMMAND, this);

    FUNC_REGISTRATION(TaskSyncDataNow)
    FUNC_REGISTRATION(TaskSyncDataStop)
    FUNC_REGISTRATION(TaskSyncDataInfo)
    FUNC_REGISTRATION(TaskSyncDataEdit)

    //FUNC_REGISTRATION(CreateSyncPlanning)
    //FUNC_REGISTRATION(CreateSyncPlanningDebug)

    FUNC_REGISTRATION(TaskStartNow)
    FUNC_REGISTRATION(TaskProgress)
    FUNC_REGISTRATION(TaskInterrupt)

    FUNC_REGISTRATION(TaskModelCreate)
    FUNC_REGISTRATION(TaskModelEdit)
    FUNC_REGISTRATION(TaskModelDelete)
    FUNC_REGISTRATION(TaskModelInfo)
    FUNC_REGISTRATION(TaskModelList)

    FUNC_REGISTRATION(TaskScoreCreate)
    FUNC_REGISTRATION(TaskScoreEdit)
    FUNC_REGISTRATION(TaskScoreDelete)
    FUNC_REGISTRATION(TaskScoreInfo)
    FUNC_REGISTRATION(TaskScoreList)
            
    FUNC_REGISTRATION(TaskReportCreate)
    FUNC_REGISTRATION(TaskReportFedCreate)

    FUNC_REGISTRATION(ReportEdit)
    FUNC_REGISTRATION(ReportDelete)
    FUNC_REGISTRATION(ReportInfo)
    FUNC_REGISTRATION(ReportData)
    FUNC_REGISTRATION(ReportList)

    FUNC_REGISTRATION(ReportFedEdit)
    FUNC_REGISTRATION(ReportFedDelete)
    FUNC_REGISTRATION(ReportFedInfo)
    FUNC_REGISTRATION(ReportFedList)
    FUNC_REGISTRATION(ReportFedName)

    FUNC_REGISTRATION(TaskReportFedNow)
    FUNC_REGISTRATION(TaskReportFedInfo)
    FUNC_REGISTRATION(TaskReportFedEdit)

    FUNC_REGISTRATION(TaskSyncNsiNow)
    FUNC_REGISTRATION(TaskSyncNsiInfo)
    FUNC_REGISTRATION(TaskSyncNsiEdit)

    #undef FUNC_REGISTRATION

    //_errorSenderWeb = [](const Message::Ptr& err) {webCon().send(err);};
}

bool Scheduler::init()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT               "
        "   ID,                "
        "   USER_ID            "
        " FROM TASK            "
        " WHERE                "
        "   EXEC_STATUS = ?    "
        "   OR EXEC_STATUS = ? "
        "   OR EXEC_STATUS = ? ",
        TaskExecStatus::Running,
        TaskExecStatus::Attempt,
        TaskExecStatus::WaitSync))
    {
        return false;
    }

    while (q.next())
    {
        QSqlRecord r = q.record();
        TaskInfo t;
        assignValue(t.id     , r, "ID      ");
        assignValue(t.userId , r, "USER_ID ");
        log_info_m << EventLog(u8"Задача завершена из-за перезапуска приложения")
                   << EventTask(t.id)
                   << EventUser(t.userId);
    }

    // Задачи синхронизации не сбрасываем, что бы не нарушить порядок
    // их выполнения после перезапуска программы
    if (!sql::exec(q, " UPDATE TASK SET      "
                      "   EXEC_STATUS = ?    "
                      " WHERE                "
                      "   EXEC_STATUS = ?    "
                      "   OR EXEC_STATUS = ? ",
                      //"   OR EXEC_STATUS = ? ",
                      TaskExecStatus::NotRun,
                      TaskExecStatus::Running,
                      TaskExecStatus::Attempt))
                      //TaskExecStatus::WaitSync))
    {
        return false;
    }
    if (!initSingleTasks())
        return false;

    // Удаление плана на синхронизацию, так как в случае перезапуска приложения
    // запуск синхронизации должен быть инициирован перезапущенной задачей
    if (!sql::exec(q, "DELETE FROM SYNC_PLANNING"))
    {
        return false;
    }

    return true;
}

void Scheduler::awake()
{
    _passTimer = true;

    QMutexLocker locker(&_threadLock); (void) locker;
    _threadCond.wakeAll();
}

void Scheduler::message(const communication::Message::Ptr& message)
{
    if (message->processed())
        return;

    if (_funcInvoker.containsCommand(message->command()))
    {
        if (!command::pool().commandIsMultiproc(message->command()))
            message->markAsProcessed();

        message->add_ref();
        QMutexLocker locker(&_threadLock); (void) locker;
        _messages.add(message.get());
        _threadCond.wakeAll();
    }
}

void Scheduler::loadTasks()
{
    QDateTime now {QDateTime::currentDateTime()};

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, " UPDATE TASK SET NEXT_DATETIME = RUN_DATETIME "
                      " WHERE NEXT_DATETIME IS NULL                  "))
    {
        return;
    }

    // Удаление задач, не привязанных к пользовтелю
    if (!sql::exec(q,
        " DELETE FROM TASK WHERE ID IN                                                 "
        " (                                                                            "
        "   SELECT                                                                     "
        "     TASK.ID                                                                  "
        "   FROM TASK LEFT JOIN USERS ON TASK.USER_ID = USERS.ID                       "
        "   WHERE                                                                      "
        "     TASK.USER_ID IS NOT NULL                                                 "
        "     AND TASK.USER_ID <> char_to_uuid('00000000-0000-0000-0000-000000000000') "
        "     AND USERS.ID IS NULL                                                     "
        " )                                                                            "))
    {
        return;
    }

    // Исправляем deadlock запуска задачи со статусом "ожидается синхронизация"
    if (!sql::exec(q,
        " UPDATE TASK SET EXEC_STATUS = ? WHERE ID IN (     "
        "   SELECT T.ID FROM TASK T                         "
        "   LEFT JOIN SYNC_PLANNING SP ON T.ID = SP.TASK_ID "
        "   WHERE T.EXEC_STATUS = ? AND SP.TASK_ID IS NULL  "
        " )                                                 ",
        TaskExecStatus::NotRun,
        TaskExecStatus::WaitSync ))
    {
        return;
    }

    if (!q.prepare(
        " SELECT                                               "
        "   T.ID                                               "
        "  ,T.TASK_TYPE                                        "
        "  ,T.USER_ID                                          "
        "  ,T.FORCED                                           "
        "  ,T.ATTEMPT_INTERVAL                                 "
        " FROM                                                 "
        "   TASK T                                             "
        " LEFT JOIN                                            "
        "   SYNC_PLANNING SP ON SP.TASK_ID = T.ID              "
        " WHERE                                                "
        " (                                                    "
        "   (                                                  " // Разовая задача по времени старта
        "     T.IS_PERIODIC = 0                                "
        "     AND T.NEXT_DATETIME <= :NEXT_DATETIME1           "
        "     AND T.IS_COMPLETE = 0                            "
        "     AND                                              "
        "     (                                                "
        "       T.EXEC_STATUS = :EXEC_STATUS_NOT_RUN1          "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_SUCCESS1       "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_FAILED1        "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_INTERRUPTED1   "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_DEFFERED1      "
        "     )                                                "
        "   )                                                  "
        "   OR                                                 "
        "   (                                                  " // Периодическая задача по времени старта
        "     T.IS_PERIODIC = 1                                "
        "     AND T.NEXT_DATETIME <= :NEXT_DATETIME2           "
        "     AND T.IS_COMPLETE = 0                            "
        "     AND                                              "
        "     (                                                "
        "       T.EXEC_STATUS = :EXEC_STATUS_NOT_RUN2          "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_SUCCESS2       "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_FAILED2        "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_INTERRUPTED2   "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_DEFFERED2      "
        "     )                                                "
        "   )                                                  "
        "   OR                                                 "
        "   (                                                  " // Периодическая задача принудительно
        "     T.IS_PERIODIC = 1                                "
        "     AND FORCED = 1                                   "
        "     AND T.IS_COMPLETE = 0                            "
        "     AND                                              "
        "     (                                                "
        "       T.EXEC_STATUS = :EXEC_STATUS_NOT_RUN3          "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_SUCCESS3       "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_FAILED3        "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_INTERRUPTED3   "
        "       OR T.EXEC_STATUS = :EXEC_STATUS_DEFFERED3      "
        "     )                                                "
        "   )                                                  "
        "   OR                                                 "
        "   (                                                  " // Задача синхронизации
        "     T.EXEC_STATUS = :EXEC_STATUS_WAITSYNC4           "
        "     AND                                              "
        "     SP.IS_COMPLETE = 1                               "
        "   )                                                  "
        "   OR                                                 "
        "   (                                                  "
        "     T.EXEC_STATUS = :EXEC_STATUS_ATTEMPT5            "
        "     AND                                              "
        "     DATEADD(MINUTE, T.ATTEMPT_INTERVAL, T.RUN_DATETIME) <= :NOW_DATETIME5 "
        "   )                                                  "
        " )                                                    "
        " AND                                                  "
        "   T.IS_ENABLED = 1                                   " // Любая задача для запуска должна быть включена
        ))
    {
        return;
    }

    bindValue(q, ":NEXT_DATETIME1           ", now                         );
    bindValue(q, ":EXEC_STATUS_NOT_RUN1     ", TaskExecStatus::NotRun      );
    bindValue(q, ":EXEC_STATUS_SUCCESS1     ", TaskExecStatus::Success     );
    bindValue(q, ":EXEC_STATUS_FAILED1      ", TaskExecStatus::Failed      );
    bindValue(q, ":EXEC_STATUS_INTERRUPTED1 ", TaskExecStatus::Interrupted );
    bindValue(q, ":EXEC_STATUS_DEFFERED1    ", TaskExecStatus::Deffered    );

    bindValue(q, ":NEXT_DATETIME2           ", now                         );
    bindValue(q, ":EXEC_STATUS_NOT_RUN2     ", TaskExecStatus::NotRun      );
    bindValue(q, ":EXEC_STATUS_SUCCESS2     ", TaskExecStatus::Success     );
    bindValue(q, ":EXEC_STATUS_FAILED2      ", TaskExecStatus::Failed      );
    bindValue(q, ":EXEC_STATUS_INTERRUPTED2 ", TaskExecStatus::Interrupted );
    bindValue(q, ":EXEC_STATUS_DEFFERED2    ", TaskExecStatus::Deffered    );

    bindValue(q, ":EXEC_STATUS_NOT_RUN3     ", TaskExecStatus::NotRun      );
    bindValue(q, ":EXEC_STATUS_SUCCESS3     ", TaskExecStatus::Success     );
    bindValue(q, ":EXEC_STATUS_FAILED3      ", TaskExecStatus::Failed      );
    bindValue(q, ":EXEC_STATUS_INTERRUPTED3 ", TaskExecStatus::Interrupted );
    bindValue(q, ":EXEC_STATUS_DEFFERED3    ", TaskExecStatus::Deffered    );

    bindValue(q, ":EXEC_STATUS_WAITSYNC4    ", TaskExecStatus::WaitSync    );

    bindValue(q, ":EXEC_STATUS_ATTEMPT5     ", TaskExecStatus::Attempt     );
    bindValue(q, ":NOW_DATETIME5            ", QDateTime::currentDateTime());

    if (!q.exec())
        return;

    QVector<TaskInfo> tasks;
    while (q.next())
    {
        TaskInfo t;
        QSqlRecord r = q.record();

        assignValue(t.id               , r, "ID               ");
        assignValue(t.userId           , r, "USER_ID          ");
        assignValue(t.forced           , r, "FORCED           ");
        assignValue(t.attempt_interval , r, "ATTEMPT_INTERVAL ");
        assignTaskT(t.taskType         , r                     );

        tasks.append(t);
    }

    for (const TaskInfo& task : tasks)
    {
        if (lst::FindResult fr = _tasks.findRef(task.id))
            if (_tasks.item(fr.index())->taskIsRunning())
                continue;

        if (_tasks.count() > 20)
        {
            sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ? WHERE ID = ?",
                         TaskExecStatus::Deffered, task.id);
            continue;
        }

        BaseTask::Ptr newTask;
        if (task.taskType == TaskType::SyncData)
        {
            newTask = BaseTask::Ptr(new SyncData(task.id, QUuidEx()/*userId*/));
        }
        else if (task.taskType == TaskType::LearnModel)
        {
            newTask = BaseTask::Ptr(new LearnModel(task.id, task.userId));
        }
        else if (task.taskType == TaskType::ScoreCalc)
        {
            newTask = BaseTask::Ptr(new ScoreCalc(task.id, task.userId));
        }
        else if (task.taskType == TaskType::CreateReport)
        {
            newTask = BaseTask::Ptr(new CreateReport(task.id, task.userId));
        }
        else if (task.taskType == TaskType::CreateReportFed)
        {
            newTask = BaseTask::Ptr(new CreateReportFed(task.id, task.userId));
        }
        else if (task.taskType == TaskType::ReportFed)
        {
            newTask = BaseTask::Ptr(new ReportFed(task.id,  QUuidEx()/*userId*/));
        }
        else if (task.taskType == TaskType::SyncNsi)
        {
            newTask = BaseTask::Ptr(new SyncNsi(task.id,  QUuidEx()/*userId*/));
        }
        if (newTask)
        {
            _tasks.add(newTask.detach());
            _passTimer = true;
        }
    }
}

void Scheduler::run()
{
    log_info_m << "Started";

    QTime timer;
    Message::List messages;

    timer.start();
    _passTimer = true;

    while (true)
    {
        CHECK_QTHREADEX_STOP

        { //Block for QMutexLocker
            QMutexLocker locker(&_threadLock); (void) locker;
            messages.swap(_messages);
        }
        while (!messages.empty())
        {
            if (threadStop())
                break;

            Message::Ptr m {messages.release(0), false};

            log_debug2_m << "Before func invoker"
                         << ". Message id: " << m->id()
                         << ". Command: " << CommandNameLog(m->command());

            _funcInvoker.call(m);

            log_debug2_m << "After func invoker"
                         << ". Message id: " << m->id()
                         << ". Command: " << CommandNameLog(m->command());
        }
        CHECK_QTHREADEX_STOP

        { //Block for QMutexLocker
            QMutexLocker locker(&_threadLock); (void) locker;
            if (!_messages.empty())
                continue;

            if (!_passTimer)
            {
                _threadCond.wait(&_threadLock, 1000);
                if (!_messages.empty())
                    continue;
            }
        }

        CHECK_QTHREADEX_STOP

        for (int i = 0; i < _tasks.count(); ++i)
        {
            BaseTask* task = _tasks.item(i);
            if (task->taskIsRunning())
                continue;

            db::firebird::Driver::Ptr dbcon = dbpool().connect();
            QSqlQuery q {dbcon->createResult()};

            if (task->taskIsFinished())
            {
                TaskInfo taskInfo;
                if (!fillTaskInfo(task->id(), taskInfo))
                {
                    _tasks.remove(i--);
                    continue;
                }

                TaskExecStatus execStatus = TaskExecStatus::Success;

                if (task->interrupted())
                    execStatus = TaskExecStatus::Interrupted;

                // Завершение задачи для последующей синхронизации
                if (task->retInfo() == task::RetInfo::Error::NeedSync)
                {
                    execStatus = TaskExecStatus::WaitSync;
                    sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ? WHERE ID = ?",
                                 execStatus, taskInfo.id);

                    data::TaskProgress taskProgress;
                    taskProgress.taskType       = taskInfo.taskType;
                    taskProgress.taskId         = taskInfo.id;
                    taskProgress.taskName       = taskInfo.name;
                    taskProgress.taskDescript   = taskInfo.description;
                    taskProgress.taskCreateDate = taskInfo.created;
                    taskProgress.taskExecStatus = execStatus;
                    taskProgress.userId         = taskInfo.userId;
                    taskProgress.isPeriodic     = taskInfo.isPeriodic();

                    data::SyncInfo::List syncList = SyncPlan::syncList();
                    if (lst::FindResult fr = syncList.findRef(taskInfo.id, {lst::BruteForce::Yes}))
                    {
                        taskProgress.waitPosition = fr.index();
                        taskProgress.current = syncList[fr.index()].current;
                        taskProgress.total   = syncList[fr.index()].total;
                    }

                    Message::Ptr m = createJsonMessage(taskProgress, Message::Type::Event);
                    webCon().send(m);

                    if (!task->isPeriodic() && syncPlan().isPeriodic())
                    {
                        /*
                          Если добавленная задача разовая
                          И
                          Текущая задача периодическая
                          ТО
                          Произвести перечитывание плана синхронизации
                        */
                        syncPlan().interrupt();
                    }
                    else if (task->isPeriodic() && syncPlan().isSystem())
                    {
                        /*
                          Если добавленная задача периодическая (пользователь)
                          И
                          Текущая задача Системная Синхронзиация
                          ТО
                          Произвести перечитывание плана синхронизации
                        */
                        syncPlan().interrupt();
                    }
                    else
                        syncPlan().awake();
                }
                else if ((task->retInfo().code() <= 0) || task->interrupted())
                {
                    /*
                      Статус нужно устанавливать  для любой  задачи.  Независимо от типа
                      запуска необходимо установить статус - 'success' или 'interrupted'.
                      Эти статусы могут быть запущены  планировщиком  для  периодической
                      задачи.
                    */
                    sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ? WHERE ID = ?",
                                 execStatus, taskInfo.id);

                    /*
                      Завершение задачи успешное или прервано пользователем. Эти два
                      статуса завершения объединены в одну группу так как по сути
                      обладают одинаковым конечным действием. А именно - завершение задачи,
                      которое не подразумевает попыток повторения запусков.
                    */
                    if (!taskInfo.isPeriodic())
                    {
                        /*
                          Разовая задача - все модификаторы периодичности равны нулю.
                          Для прекращения работы разовой задачи достаточно установить
                          флаг IS_COMPLETE = 1;
                        */
                        sql::exec(q, "UPDATE TASK SET IS_COMPLETE = 1 WHERE ID = ?",
                                     taskInfo.id);
                    }
                    else // taskInfo.isPeriodic() == TRUE
                    {
                        /*
                          Периодическая задача - один или несколько модификаторов
                          периодичности имеют значение отличное от нуля.
                        */
                        if (taskInfo.isPeriodic() && taskInfo.forced)
                        {
                            /*
                              Обработка завершения периодической задачи, которая была
                              запущена принудительно. Флаг FORCED такой задачи обязательно
                              должен быть сброшен. Если данный флаг не сброисть, то задача
                              будет запускаться бесконечное число раз, до тех пор пока не будет
                              прервана вручную.
                            */
                            sql::exec(q, "UPDATE TASK SET FORCED = 0 WHERE ID = ?",
                                         taskInfo.id);
                        }
                        /*
                          Расчет следующего времени запуска. Выполняется для принудтельного
                          и планового запуска.

                          В случае планового запуска время сдвигается на необходимый отрезок времени
                          вперёд.

                          В случае принудительного запуска время не будет сдвинуто в том случае если
                          время завершения задачи случилось до времени следующего запуска.
                          Время будет сдвинуть на необходиммый отрезок вперед, если время завершения
                          задачи случилось после времени следующего запуска.
                        */
                        QDateTime nextRun;
                        calcNextRun(taskInfo.id, nextRun);

                        sql::exec(q, "UPDATE TASK SET NEXT_DATETIME = ? WHERE ID = ?",
                                     nextRun, taskInfo.id);
                    } // if (!taskInfo.isPeriodic())
                }
                else
                {
                    // Завершение задачи с ошибкой.
                    if (taskInfo.forced)
                    {
                        /*
                          Принудительный запуск задачи не предполагает нескольких
                          попыток запуска. Принудительная задача завершившаяся неудачей
                          не запускается до тех пор пока её не запустят заново.
                          Запуск возможен принудительно пользователем либо планировщиком
                          при наступлении времени запуска.
                        */
                        execStatus = TaskExecStatus::Failed;

                        notifyFailTask(taskInfo.id);
                    }
                    else
                    {
                        // Количество попыток = 0 или количество попыток исчерпано приводит
                        // к завершению задачи.
                        if (taskInfo.attempt_limit == 0
                            || (taskInfo.attempt_counter + 1) >= taskInfo.attempt_limit)
                        {
                            execStatus = TaskExecStatus::Failed;

                            // Обнуление счётчика неудачных попыток для сфейленной задачи.
                            // Для того что бы при следующем запуске начать отсчёт с нуля.
                            sql::exec(q, "UPDATE TASK SET ATTEMPT_COUNTER = 0 WHERE ID = ?",
                                         taskInfo.id);

                            notifyFailTask(taskInfo.id);

                            QDateTime nextRun;
                            calcNextRun(taskInfo.id, nextRun);

                            sql::exec(q, "UPDATE TASK SET NEXT_DATETIME = ? WHERE ID = ?",
                                         nextRun, taskInfo.id);

                            log_warn_m << u8"Количество попыток для задачи: "
                                       << taskInfo.id
                                       << u8" исчерпано";
                        }
                        else
                        {
                            // Увеличиваем число попыток на запуск
                            // Время старта не изменяем, время на перезапуск входит в общее время
                            // работы задачи.
                            execStatus = TaskExecStatus::Attempt;

                            log_warn_m << u8"Неудачная попытка для задачи: "
                                       << taskInfo.id
                                       << u8" Попытка № "
                                       << taskInfo.attempt_counter + 1;

                            sql::exec(q, "UPDATE TASK SET ATTEMPT_COUNTER = ? WHERE ID = ?",
                                         taskInfo.attempt_counter + 1, taskInfo.id);

                            sql::exec(q, "UPDATE TASK SET RUN_DATETIME = ? WHERE ID = ?",
                                         QDateTime::currentDateTime(), taskInfo.id);
                        }
                    }

                    // Удаление записи о синхронизации для проваленной задачи.
                    sql::exec(q, "DELETE FROM SYNC_PLANNING WHERE TASK_ID = ?", taskInfo.id);

                    sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ?, FORCED = 0 WHERE ID = ?",
                                 execStatus, taskInfo.id);
                }


                data::TaskProgress taskProgress;
                taskProgress.taskType       = taskInfo.taskType;
                taskProgress.taskId         = taskInfo.id;
                taskProgress.taskName       = taskInfo.name;
                taskProgress.taskDescript   = taskInfo.description;
                taskProgress.taskCreateDate = taskInfo.created;
                taskProgress.taskExecStatus = execStatus;
                taskProgress.userId         = taskInfo.userId;
                taskProgress.isPeriodic     = taskInfo.isPeriodic();

                data::SyncInfo::List syncList = SyncPlan::syncList();
                if (lst::FindResult fr = syncList.findRef(taskInfo.id, {lst::BruteForce::Yes}))
                {
                    taskProgress.waitPosition = fr.index();
                    taskProgress.current = syncList[fr.index()].current;
                    taskProgress.total   = syncList[fr.index()].total;
                }

                Message::Ptr m = createJsonMessage(taskProgress, Message::Type::Event);
                webCon().send(m);

                _tasks.remove(i--);
                continue;

            } // if (task->isFinished())

            // [16.06.2019 Карелин]
            // Здесь в запросе нельзя  устанавливать FORCED = 0,  т.к.  при  завершении  задачи
            // будет невозможно понять запущена она с FORCED или нет, соответственно вся логика
            // описанная выше в коде этой функции будет работать некорректно.
            // sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ?, FORCED = 0 WHERE ID = ?",

            // Запуск новой задачи
            sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ? WHERE ID = ?",
                         TaskExecStatus::Running, task->id());
            task->taskStart();

            // [27.03.2019 Карелин]
            // Во избежание двойного эмитирования события о старте задачи в этой
            // точке событие не отправляем.
            // Событие о старте задачи должно быть эмитировано из самой задачи.
            // Message::Ptr m = createJsonMessage(task->progress(), Message::Type::Event);
            // webCon().send(m);
        }

        if (!_passTimer && (timer.elapsed() < 60*1000 /*1 мин*/))
            continue;

        timer.restart();
        _passTimer = false;

        loadTasks();

    } // while (true)

    log_info_m << "Try stop tasks...";

    for (int i = 0; i != _tasks.size(); ++i)
    {
        BaseTask* task = _tasks.item(i);
        if (task->taskIsRunning())
        {
            log_info_m << "Stop task id : " << task->id();
            // Увеличенный таймаут, для того что бы задача успела завершить
            // долгие операции (например, rollback в БД).
            if (!task->taskStop(30*1000))
            {
                log_info_m << "Terminate task id : " << task->id();
                task->taskTerminate();
            }
        }
    }

    log_info_m << "Stopped";
}

void Scheduler::notifyFailTask(const QUuidEx& idTask)
{
    (void)idTask;
    // TODO: Уведомление о том, что все попытки были исчерпаны
}

void Scheduler::threadStopEstablished()
{
    QMutexLocker locker(&_threadLock); (void) locker;
    _threadCond.wakeAll();
}

bool Scheduler::initSingleTasks()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    auto removeRepetitions = [&](TaskType taskType)
    {
        // Удаляем возможные повторения
        return sql::exec(q,
            " DELETE FROM TASK                                              "
            " WHERE                                                         "
            "   TASK_TYPE = ?                                               "
            "   AND                                                         "
            "   ID NOT IN (SELECT FIRST 1 ID FROM TASK WHERE TASK_TYPE = ?) ",
            getTaskString(taskType),
            getTaskString(taskType));
    };

    auto taskExists = [&](TaskType taskType) -> bool
    {
        if (!sql::exec(q, "SELECT COUNT(*) FROM TASK WHERE TASK_TYPE = ?",
                          getTaskString(taskType)))
        {
            return false;
        }
        q.first();
        return (q.value(0).toInt() != 0);
    };

    auto periodicTaskExists = [&](TaskType taskType) -> bool
    {
        if (!sql::exec(q, "SELECT COUNT(*) FROM TASK WHERE TASK_TYPE = ? AND IS_PERIODIC = 1",
                          getTaskString(taskType)))
        {
            return false;
        }
        q.first();
        return (q.value(0).toInt() != 0);
    };

    auto defaultGroupExists = [&]() -> bool
    {
        if (!sql::exec(q, "SELECT COUNT(*) FROM GROUPS WHERE IS_DEFAULT = 1"))
        {
            return false;
        }
        q.first();
        return (q.value(0).toInt() != 0);
    };

    //--- Задача синхронизации ---
    removeRepetitions(TaskType::SyncData);
    if (!taskExists(TaskType::SyncData))
    {
        QString fields =
            "  ID             "
            ", TASK_TYPE      "
            ", NAME           "
            ", RUN_DATETIME   "
            ", REGULARITY_DAY "
            ", REGULARITY_HOUR";

        QString sql = sql::INSERT_INTO("TASK", fields);

        if (!q.prepare(sql))
            return false;

        bindValue(q, ":ID              ", QUuidEx::createUuid());
        bindValue(q, ":TASK_TYPE       ", getTaskString(TaskType::SyncData));
        bindValue(q, ":NAME            ", QString::fromUtf8(TASK_NAME_SYNCDATA));
        bindValue(q, ":RUN_DATETIME    ", QDateTime::currentDateTime());
        bindValue(q, ":REGULARITY_DAY  ", 1);
        bindValue(q, ":REGULARITY_HOUR ", 1);

        if (!q.exec())
            return false;
    }

    //--- Задача автоматической генерации отчетов для ФФОМС ---
    removeRepetitions(TaskType::ReportFed);
    if (!taskExists(TaskType::ReportFed))
    {
        QString fields =
            "  ID               "
            ", TASK_TYPE        "
            ", NAME             "
            ", RUN_DATETIME     "
            ", REGULARITY_MONTH ";

        QString sql = sql::INSERT_INTO("TASK", fields);

        if (!q.prepare(sql))
            return false;

        bindValue(q, ":ID               ", QUuidEx::createUuid());
        bindValue(q, ":TASK_TYPE        ", getTaskString(TaskType::ReportFed));
        bindValue(q, ":NAME             ", QString::fromUtf8(TASK_NAME_REPORTFED));
        bindValue(q, ":RUN_DATETIME     ", QDateTime::currentDateTime());
        bindValue(q, ":REGULARITY_MONTH ", 1);

        if (!q.exec())
            return false;
    }

    //--- Задача синхронизации справочников НСИ ---
    removeRepetitions(TaskType::SyncNsi);
    if (!taskExists(TaskType::SyncNsi))
    {
        QString fields =
            "  ID               "
            ", TASK_TYPE        "
            ", NAME             "
            ", RUN_DATETIME     "
            ", REGULARITY_MONTH ";

        QString sql = sql::INSERT_INTO("TASK", fields);

        if (!q.prepare(sql))
            return false;

        bindValue(q, ":ID               ", QUuidEx::createUuid());
        bindValue(q, ":TASK_TYPE        ", getTaskString(TaskType::SyncNsi));
        bindValue(q, ":NAME             ", QString::fromUtf8(TASK_NAME_SYNCNSI));
        bindValue(q, ":RUN_DATETIME     ", QDateTime::currentDateTime());
        bindValue(q, ":REGULARITY_MONTH ", 1);

        if (!q.exec())
            return false;
    }

    //--- Регулярная задача обучения ---
    if (!periodicTaskExists(TaskType::LearnModel))
    {
        QString fields =
            "  ID               "
            ", TASK_TYPE        "
            ", NAME             "
            ", RUN_DATETIME     "
            ", REGULARITY_MONTH ";

        QString sql = sql::INSERT_INTO("TASK", fields);

        if (!q.prepare(sql))
            return false;

        bindValue(q, ":ID               ", QUuidEx::createUuid());
        bindValue(q, ":TASK_TYPE        ", getTaskString(TaskType::LearnModel));
        bindValue(q, ":NAME             ", QString::fromUtf8(TASK_NAME_LEARN));
        bindValue(q, ":RUN_DATETIME     ", QDateTime::currentDateTime());
        bindValue(q, ":REGULARITY_MONTH ", 1);

        if (!q.exec())
            return false;
    }

    //--- Регулярная задача применения ---
    if (!periodicTaskExists(TaskType::ScoreCalc))
    {
        QString fields =
            "  ID               "
            ", TASK_TYPE        "
            ", NAME             "
            ", RUN_DATETIME     "
            ", REGULARITY_MONTH ";

        QString sql = sql::INSERT_INTO("TASK", fields);

        if (!q.prepare(sql))
            return false;

        bindValue(q, ":ID               ", QUuidEx::createUuid());
        bindValue(q, ":TASK_TYPE        ", getTaskString(TaskType::ScoreCalc));
        bindValue(q, ":NAME             ", QString::fromUtf8(TASK_NAME_APPLY));
        bindValue(q, ":RUN_DATETIME     ", QDateTime::currentDateTime());
        bindValue(q, ":REGULARITY_MONTH ", 1);

        if (!q.exec())
            return false;
    }

    //--- Группа по умолчанию для пользователей ---
    if (!defaultGroupExists())
    {
        QString fields =
            "  ID          "
            ", NAME        "
            ", LEARN_MODEL "
            ", APPLY_MODEL "
            ", REPORT      "
            ", REPORT_FED  "
            ", MONITORING  "
            ", IS_DEFAULT  ";

        QString sql = sql::UPDATE_OR_INSERT_INTO("GROUPS", fields, "ID");

        if (!q.prepare(sql))
            return false;

        bindValue(q, ":ID          ", QUuidEx::createUuid() );
        bindValue(q, ":NAME        ", "default"             );
        bindValue(q, ":LEARN_MODEL ", 2                     );
        bindValue(q, ":APPLY_MODEL ", 2                     );
        bindValue(q, ":REPORT      ", 2                     );
        bindValue(q, ":REPORT_FED  ", 0                     );
        bindValue(q, ":MONITORING  ", 0                     );
        bindValue(q, ":IS_DEFAULT  ", 1                     );

        if (!q.exec())
            return false;
    }

    return true;
}

void Scheduler::command_TaskSyncDataNow(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        "SELECT ID FROM TASK WHERE TASK_TYPE = ?", getTaskString(TaskType::SyncData)))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::task_not_found.asFailed())

    data::TaskStartNow taskStartNow;
    assignValue(taskStartNow.taskId, q.record(), "ID");

    Message::Ptr startTask = createJsonMessage(taskStartNow);
    startTask->setTag(message->tag());

    this->message(startTask);

    // Отправляем подтверждение об успешной операции: пустое сообщение
    webCon().send(answer);
}

void Scheduler::command_TaskSyncDataStop(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        "SELECT ID FROM TASK WHERE TASK_TYPE = ?", getTaskString(TaskType::SyncData)))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::task_not_found.asFailed())

    data::TaskInterrupt taskInterrupt;
    assignValue(taskInterrupt.taskId, q.record(), "ID");

    Message::Ptr stopTask = createJsonMessage(taskInterrupt);
    stopTask->setTag(message->tag());

    this->message(stopTask);

    // Отправляем подтверждение об успешной операции: пустое сообщение
    webCon().send(answer);
}

void Scheduler::command_TaskSyncDataInfo(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                "
        "   ID                  "
        "  ,RUN_DATETIME        "
        "  ,NEXT_DATETIME       "
        "  ,ATTEMPT_LIMIT       "
        "  ,ATTEMPT_INTERVAL    "
        "  ,EXEC_STATUS         "
        "  ,IS_ENABLED          "
        "  ,REL_PERIOD_DURATION "
        " FROM                  "
        "   TASK                "
        " WHERE                 "
        "   TASK_TYPE = ?       ", getTaskString(TaskType::SyncData)))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::syncdata_task_not_found)

    QSqlRecord r = q.record();
    data::TaskSyncDataInfoA taskSyncDataInfoA;

    assignValue(taskSyncDataInfoA.taskId           , r, "ID                 ");
    assignValue(taskSyncDataInfoA.runDateTime      , r, "RUN_DATETIME       ");
    assignValue(taskSyncDataInfoA.nextDateTime     , r, "NEXT_DATETIME      ");
    assignValue(taskSyncDataInfoA.attemptLimit     , r, "ATTEMPT_LIMIT      ");
    assignValue(taskSyncDataInfoA.attemptInterval  , r, "ATTEMPT_INTERVAL   ");
    assignValue(taskSyncDataInfoA.taskExecStatus   , r, "EXEC_STATUS        ");
    assignValue(taskSyncDataInfoA.isEnabled        , r, "IS_ENABLED         ");
    assignValue(taskSyncDataInfoA.relPeriodDuration, r, "REL_PERIOD_DURATION");

    int totalDays = 0;
    int currentDays = 0;

    sql::exec(q,
        " SELECT                                                     "
        "   PERIOD_BEGIN,                                            "
        "   PERIOD_END,                                              "
        "   COALESCE(PERIOD_CURRENT, PERIOD_BEGIN) as PERIOD_CURRENT "
        " FROM                                                       "
        "   SYNC_PLANNING SP                                         "
        " WHERE                                                      "
        "   SP.TASK_ID = ?                                           ",
        taskSyncDataInfoA.taskId);

    q.first();

    QDate begin, end, current;

    assignValue(begin  , q.record(), "PERIOD_BEGIN  ");
    assignValue(end    , q.record(), "PERIOD_END    ");
    assignValue(current, q.record(), "PERIOD_CURRENT");

    totalDays = begin.daysTo(end);
    currentDays = begin.daysTo(current);

    int percent = ((float)currentDays / totalDays) * 100;

    taskSyncDataInfoA.current = percent;
    taskSyncDataInfoA.total = 100;

    writeToJsonMessage(taskSyncDataInfoA, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskSyncDataEdit(const Message::Ptr& message)
{
    data::TaskSyncDataEdit taskSyncDataEdit;
    READ_FROM_MESSAGE(message, taskSyncDataEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(
        " UPDATE TASK SET                              "
        "   REGULARITY_MONTH    = :REGULARITY_MONTH    "
        "  ,REGULARITY_DAY      = :REGULARITY_DAY      "
        "  ,REGULARITY_HOUR     = :REGULARITY_HOUR     "
        "  ,RUN_DATETIME        = :RUN_DATETIME        "
        "  ,NEXT_DATETIME       = :NEXT_DATETIME       "
        "  ,ATTEMPT_LIMIT       = :ATTEMPT_LIMIT       "
        "  ,ATTEMPT_INTERVAL    = :ATTEMPT_INTERVAL    "
        "  ,IS_ENABLED          = :IS_ENABLED          "
        "  ,REL_PERIOD_DURATION = :REL_PERIOD_DURATION "
        " WHERE                                        "
        "   TASK_TYPE           = :TASK_TYPE           "))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        return;
    }

    bindValue(q, ":REGULARITY_MONTH    ", 0                                 );
    bindValue(q, ":REGULARITY_DAY      ", 1                                 );
    bindValue(q, ":REGULARITY_HOUR     ", 0                                 );
    bindValue(q, ":RUN_DATETIME        ", taskSyncDataEdit.nextDateTime     );
    bindValue(q, ":NEXT_DATETIME       ", taskSyncDataEdit.nextDateTime     );
    bindValue(q, ":ATTEMPT_LIMIT       ", taskSyncDataEdit.attemptLimit     );
    bindValue(q, ":ATTEMPT_INTERVAL    ", taskSyncDataEdit.attemptInterval  );
    bindValue(q, ":IS_ENABLED          ", taskSyncDataEdit.isEnabled        );
    bindValue(q, ":REL_PERIOD_DURATION ", taskSyncDataEdit.relPeriodDuration);
    bindValue(q, ":TASK_TYPE           ", getTaskString(TaskType::SyncData) );

    if (!q.exec())
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        return;
    }

    // Если все нормально, то посылаем пустой ответ
    webCon().send(answer);
}

void Scheduler::command_TaskStartNow(const Message::Ptr& message)
{
    data::TaskStartNow taskStartNow;
    READ_FROM_MESSAGE(message, taskStartNow);

    QUuidEx taskId = taskStartNow.taskId;
    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    // Запуск задач с UUID равным 0 невозможен, так как '0' - некорректное значение.
    if (taskId == QUuidEx(0))
    {
        writeToJsonMessage(error::task_identification_error.asFailed(), answer);
        webCon().send(answer);
        return;
    }

    if (lst::FindResult fr = _tasks.findRef(taskId))
        if (_tasks.item(fr.index())->taskIsRunning())
        {
            writeToJsonMessage(error::task_already_running.asFailed(), answer);
            webCon().send(answer);
            return;
        }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        "SELECT NAME, IS_PERIODIC FROM TASK WHERE ID = ?", taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::task_not_found.asFailed())

    //QString taskName;
    //assignValue(taskName, q.record(), "NAME");

    bool isPeriodic = false;
    assignValue(isPeriodic, q.record(), "IS_PERIODIC");

    if (isPeriodic && !isAdmin)
    {
        writeToJsonMessage(error::admin_privileges, answer);
        webCon().send(answer);
        return;
    }

    if (isPeriodic)
    {
        /*
          Запуск периодической задачи происходит путём установка флага FORCED = 1
          Время следующего запуска не изменяется. Время запуска изменится по завершению
          запущенной задачи.
        */
        QString sql = " UPDATE TASK SET       "
                      "   FORCED = 1          "
                      "  ,ATTEMPT_COUNTER = 0 "
                      " WHERE ID = :ID        ";

        if (!isAdmin)
            sql += " AND USER_ID = :USER_ID ";

        if (!q.prepare(sql))
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        bindValue(q, ":ID", taskId);

        if (!isAdmin)
            bindValue(q, ":USER_ID", taskStartNow.userId);

        if (!q.exec())
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            webCon().send(answer);
            return;
        }
    }
    else
    {
        /*
          Запуск разовой задачи производится путём установки флага IS_COMPLETE = 0,
          так же необходимо обнулить счетчик попыток (ATTEMPT_COUNTER), и выставить
          статус задачи EXEC_STATUS в TaskExecStatus::NotRun.
          Время следующего запуска меняется на текущее лишь для того, что бы понимать
          когда разовая задача была запущена в последний раз.
        */
        QDateTime now = QDateTime::currentDateTime().addSecs(-60);

        QString sql = " UPDATE TASK SET                  "
                      "   RUN_DATETIME  = :RUN_DATETIME  "
                      "  ,NEXT_DATETIME = :NEXT_DATETIME "
                      "  ,EXEC_STATUS   = :EXEC_STATUS   "
                      "  ,ATTEMPT_COUNTER = 0            "
                      "  ,IS_COMPLETE = 0                "
                      " WHERE ID = :ID                   ";

        if (!q.prepare(sql))
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        bindValue(q, ":RUN_DATETIME ", now);
        bindValue(q, ":NEXT_DATETIME", now);
        bindValue(q, ":EXEC_STATUS", TaskExecStatus::NotRun);
        bindValue(q, ":ID", taskId);

        if (!q.exec())
        {
            writeToJsonMessage(error::update_sql_statement, answer);
            webCon().send(answer);
            return;
        }
    }

    const char* msg = u8"Получена команда на запуск задачи";
    log_info_m << EventLog(msg)
               << EventTask(taskId)
               << EventUser(userHashId); // Пользователь выполнивший действие

    _passTimer = true;
    writeToJsonMessage(taskStartNow, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskProgress(const Message::Ptr& message)
{
    data::TaskProgressC taskProgressC;
    READ_FROM_MESSAGE(message, taskProgressC);

    Message::Ptr answer = message->cloneForAnswer();
    if (lst::FindResult fr = _tasks.findRef(taskProgressC.id))
    {
        writeToJsonMessage(_tasks.item(fr.index())->progress(), answer);
        webCon().send(answer);
        return;
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT         "
        "   TASK_TYPE    "
        "  ,NAME         "
        "  ,DESCRIPTION  "
        "  ,CREATE_DATE  "
        "  ,EXEC_STATUS  "
        "  ,USER_ID      "
        "  ,IS_PERIODIC  "
        " FROM           "
        "   TASK         "
        " WHERE          "
        "   ID = ?       ", taskProgressC.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::task_not_found.asFailed())

    data::TaskProgress taskProgress;
    taskProgress.taskId = taskProgressC.id;

    QSqlRecord r = q.record();
    assignTaskT(taskProgress.taskType       , r                );
    assignValue(taskProgress.taskName       , r, "NAME        ");
    assignValue(taskProgress.taskDescript   , r, "DESCRIPTION ");
    assignValue(taskProgress.taskCreateDate , r, "CREATE_DATE ");
    assignValue(taskProgress.taskExecStatus , r, "EXEC_STATUS ");
    assignValue(taskProgress.userId         , r, "USER_ID     ");
    assignValue(taskProgress.isPeriodic     , r, "IS_PERIODIC ");

    // Если статус "Ожидание Синхронизации", необходимо дополнительно
    // добавить номер в очереди.
    if (taskProgress.taskExecStatus == TaskExecStatus::WaitSync)
    {
        data::SyncInfo::List syncList = SyncPlan::syncList();
        if (lst::FindResult fr = syncList.findRef(taskProgressC.id, {lst::BruteForce::Yes}))
        {
            taskProgress.waitPosition = fr.index();
            taskProgress.current = syncList[fr.index()].current;
            taskProgress.total   = syncList[fr.index()].total;
        }
    }

    writeToJsonMessage(taskProgress, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskInterrupt(const Message::Ptr& message)
{
    data::TaskInterrupt taskInterrupt;
    READ_FROM_MESSAGE(message, taskInterrupt);

    QUuidEx taskId = taskInterrupt.taskId;
    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT           "
        "   NAME,          "
        "   EXEC_STATUS,   "
        "   NEXT_DATETIME, "
        "   IS_PERIODIC    "
        " FROM             "
        "   TASK           "
        " WHERE            "
        "   ID = ?         ", taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::task_not_found.asFailed())

    //QString taskName;
    TaskExecStatus execStatus = TaskExecStatus::NotRun;
    bool isPeriodic = false;
    QDateTime nextDatetime;

    //assignValue(taskName,     q.record(), "NAME         ");
    assignValue(execStatus,   q.record(), "EXEC_STATUS  ");
    assignValue(nextDatetime, q.record(), "NEXT_DATETIME");
    assignValue(isPeriodic,   q.record(), "IS_PERIODIC  ");

    if (isPeriodic && !isAdmin)
    {
        writeToJsonMessage(error::admin_privileges, answer);
        webCon().send(answer);
        return;
    }

    if (!sql::exec(q, "DELETE FROM SYNC_PLANNING WHERE TASK_ID = ?", taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    // Прерывать синхронизацию только в том случае, если в текущий момент
    // синхронизируется именно прерываемая задача.
    if (syncPlan().taskId() == taskId)
        syncPlan().interrupt();

    if (!sql::exec(q, "UPDATE TASK SET EXEC_STATUS = ? WHERE ID = ?",
                      TaskExecStatus::Interrupted, taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (execStatus == TaskExecStatus::WaitSync)
    {
        /*
          Блок: Прерывание задачи в режиме синхронизации.
          Данная операция не явлется полноценным прерыванием задачи, так как сам поток
          для задачи не запущен. Режим синхронизации это статус задачи 'WaitSync'.
          Поэтому для того, что бы прервать подобную задачу нужно поменять статус
          с 'WaitSync' на 'Interrupted'. Но смены статуса недостаточно, так как состояние
          'Interrupted' является запускаемым. Поэтому для полноценной остановки необходимо:
          А) Для периодической задачи:
             1) Сброисть флаг 'FORCED'. Это позволит избежать повторного запуска.
                Даже если этот флаг не был установлен, сброс не привет к ошибке.
             2) Вычислить время следующего запуска и установить его для задачи.
                Это делается для того, что бы задача со статусом 'Interrupted' ждала
                времени своего запуска.
          Б) Для разовой задачи:
             1) Установить флаг IS_COMPLETE, который блокирует запуск разовой задачи.
        */
        if (isPeriodic)
        {
            if (!sql::exec(q, "UPDATE TASK SET FORCED = 0 WHERE ID = ?", taskId))
            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }

            QDateTime nextRun;
            if (!calcNextRun(taskId, nextRun))
            {
                writeToJsonMessage(error::task_calc_next_run, answer);
                webCon().send(answer);
                return;
            }

            if (!sql::exec(q, "UPDATE TASK SET NEXT_DATETIME = ? WHERE ID = ?",
                               nextRun, taskId))
            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }
        }
        else
        {
            if (!sql::exec(q, "UPDATE TASK SET IS_COMPLETE = 1 WHERE ID = ?", taskId))
            {
                writeToJsonMessage(error::select_sql_statement, answer);
                webCon().send(answer);
                return;
            }
        }

        const char* msg = u8"Получена команда на прерывание задачи. Синхронизация отменена";
        log_info_m << EventLog(msg)
                   << EventTask(taskId)
                   << EventUser(userHashId); // Пользователь выполнивший действие

        writeToJsonMessage(taskInterrupt, answer);
        webCon().send(answer);

        if (!sql::exec(q,
            " SELECT        "
            "   ID          "
            "  ,USER_ID     "
            "  ,TASK_TYPE   "
            "  ,NAME        "
            "  ,DESCRIPTION "
            "  ,CREATE_DATE "
            "  ,IS_PERIODIC "
            "  ,EXEC_STATUS "
            " FROM          "
            "   TASK        "
            " WHERE         "
            "   ID = ?      ", taskId))
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        Q_FIRST_CHECK(error::task_not_found.asFailed())

        QSqlRecord r = q.record();
        data::TaskProgress taskProgress;

        assignValue(taskProgress.taskId           , r, "ID          ");
        assignValue(taskProgress.userId           , r, "USER_ID     ");
        assignTaskT(taskProgress.taskType         , r                );
        assignValue(taskProgress.taskName         , r, "NAME        ");
        assignValue(taskProgress.taskDescript     , r, "DESCRIPTION ");
        assignValue(taskProgress.taskCreateDate   , r, "CREATE_DATE ");
        assignValue(taskProgress.isPeriodic       , r, "IS_PERIODIC ");
        assignValue(taskProgress.taskExecStatus   , r, "EXEC_STATUS ");

        Message::Ptr m = createJsonMessage(taskProgress, Message::Type::Event);
        webCon().send(m);
        return;
    }

    lst::FindResult fr = _tasks.findRef(taskId);

    if (fr.failed())
    {
        //Message::Ptr m = createJsonMessage(taskProgress, Message::Type::Event);

        writeToJsonMessage(error::task_not_running, answer);
        webCon().send(answer);
        return;
    }

    if (!_tasks.item(fr.index())->taskIsRunning())
    {
        writeToJsonMessage(error::task_not_running, answer);
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Получена команда на прерывание задачи";
    log_info_m << EventLog(msg)
               << EventTask(taskId)
               << EventUser(userHashId); // Пользователь выполнивший действие

    _tasks.item(fr.index())->interrupt();

    writeToJsonMessage(taskInterrupt, answer);
    webCon().send(answer);
}

static bool mainTaskSqlQuery(QSqlQuery& q, bool createTask, TaskType taskType,
                             data::Task& task)
{
    // Общие поля, задаются при создании и редактировании задачи
    QString fields =
        "  ID                  "
        ", USER_ID             "
        ", PARENT_ID           "
        ", NAME                "
        ", DESCRIPTION         "
        ", REGULARITY_MONTH    "
        ", REGULARITY_DAY      "
        ", REGULARITY_HOUR     "
        ", ATTEMPT_LIMIT       "
        ", ATTEMPT_INTERVAL    "
        ", IS_ENABLED          "
        ", IS_PUBLIC           "
        ", PERIOD_BEGIN        "
        ", PERIOD_END          "
        ", REL_PERIOD_DURATION "
        ", MODEL_ID            "
        ", RUN_CHILD_TASK      "
        ", SKIP_SYNC           "
        ;

    // Поля задаются только при создании задачи
    if (createTask)
    {
        fields +=
        ", TASK_TYPE           "
        ", CREATE_DATE         ";

        // Задаем время создания задачи явно в коде, что бы потом вернуть
        // его web-разработчикам
        task.createDate = QDateTime::currentDateTime();

        if (task.runDateTime.isNull())
            task.runDateTime = task.createDate;
    }
    if (!task.nextDateTime.isNull())
    {
        task.runDateTime = task.nextDateTime;
        fields += ", NEXT_DATETIME ";
    }
    if (!task.runDateTime.isNull())
    {
        fields += ", RUN_DATETIME ";
    }

    // Поля задаются в БД при создании задачи или планировщиком
    //  " ATTEMPT_COUNTER     ,"
    //  " EXEC_STATUS         ,"
    //  " IS_COMPLETE         ,"
    //  " FORCED              ,"

    QString sql = sql::UPDATE_OR_INSERT_INTO("TASK", fields, "ID");

    if (!q.prepare(sql))
        return false;

    bindValue(q, ":ID                   ", task.id                );
    bindValue(q, ":USER_ID              ", task.userId            );
    bindValue(q, ":PARENT_ID            ", task.parentId          );
    bindValue(q, ":NAME                 ", task.name              );
    bindValue(q, ":DESCRIPTION          ", task.description       );
    bindValue(q, ":REGULARITY_MONTH     ", task.regularityMonth   );
    bindValue(q, ":REGULARITY_DAY       ", task.regularityDay     );
    bindValue(q, ":REGULARITY_HOUR      ", task.regularityHour    );
    bindValue(q, ":ATTEMPT_LIMIT        ", task.attemptLimit      );
    bindValue(q, ":ATTEMPT_COUNTER      ", task.attemptCounter    );
    bindValue(q, ":ATTEMPT_INTERVAL     ", task.attemptInterval   );
    bindValue(q, ":IS_ENABLED           ", task.isEnabled         );
    bindValue(q, ":IS_PUBLIC            ", task.isPublic          );
    bindValue(q, ":PERIOD_BEGIN         ", task.period.begin      );
    bindValue(q, ":PERIOD_END           ", task.period.end        );
    bindValue(q, ":REL_PERIOD_DURATION  ", task.relPeriodDuration );
    bindValue(q, ":MODEL_ID             ", task.modelId           );
    bindValue(q, ":RUN_CHILD_TASK       ", task.runChildTask      );
    bindValue(q, ":SKIP_SYNC            ", task.skipSync          );

    if (createTask)
    {
        bindValue(q, ":TASK_TYPE  ", getTaskString(taskType));
        bindValue(q, ":CREATE_DATE", task.createDate        );
    }

    if (!task.nextDateTime.isNull())
    {
        bindValue(q, ":NEXT_DATETIME", task.nextDateTime);
    }
    if (!task.runDateTime.isNull())
    {
        bindValue(q, ":RUN_DATETIME", task.runDateTime);
    }

    if (task.isPeriodic())
    {
        if (taskType == TaskType::LearnModel)
        {
            initLearnPeriod(task.period, task.relPeriodDuration);
        }
        else if (taskType == TaskType::ScoreCalc)
        {
            initScorePeriod(task.period);
        }
    }

    return q.exec();
}

static int taskDelete(QSqlQuery& q, TaskType taskType, const QUuidEx& taskId,
                      const QUuidEx& userId, bool isAdmin)
{
//    // Параметр userHashId предотвращает удаление задач не принадлежащих
//    // пользователю
//    if (!execSql(q,
//        " SELECT t.ID FROM TASK t"
//        " LEFT JOIN USERS u on t.USER_ID = u.ID "
//        " WHERE t.ID = ?  AND t.TASK_TYPE = ? AND u.HASH_ID = ?",
//        taskId, taskType, userHashId))
//    {
//        return 0;
//    }
//    if (!q.first())
//        return -1;

    QString sql =
        " DELETE FROM              "
        "   TASK                   "
        " WHERE                    "
        "   ID = :ID               "
        "   AND                    "
        "   TASK_TYPE = :TASK_TYPE ";

    if (!isAdmin)
        sql += " AND USER_ID = :USER_ID ";

    if (!q.prepare(sql))
        return 0;

    bindValue(q, ":ID       ", taskId );
    bindValue(q, ":TASK_TYPE", getTaskString(taskType) );

    if (!isAdmin)
        bindValue(q, ":USER_ID", userId );

    if (!q.exec())
        return 0;

    return 1;
}

int periodicTaskCount(QSqlQuery& q, TaskType taskType)
{
    QString taskTypeStr = getTaskString(taskType);

    if (!sql::exec(q,
        " SELECT                     "
        "   COUNT(*)                 "
        " FROM                       "
        "   TASK                     "
        " WHERE                      "
        "   TASK_TYPE = ?            "
        "   AND                      "
        "   (REGULARITY_MONTH > 0    "
        "    OR REGULARITY_DAY > 0   "
        "    OR REGULARITY_HOUR > 0) ", taskTypeStr))
    {
        return -1;
    }
    q.first();
    return q.value(0).toInt();
}

void Scheduler::command_TaskModelCreate(const Message::Ptr& message)
{
    data::TaskModelCreate taskModelCreate;
    READ_FROM_MESSAGE(message, taskModelCreate);

    Message::Ptr answer = message->cloneForAnswer();

    // Проверить на корректность имени и описания сразу, что бы
    // при неправильных входных данных не делать ничего лишнего.
    if (taskModelCreate.name.isEmpty()
        || taskModelCreate.description.isEmpty())
    {
        writeToJsonMessage(error::task_inncorrect_name, answer);
        webCon().send(answer);
        return;
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (taskModelCreate.isPeriodic())
    {
        int tastCount = periodicTaskCount(q, TaskType::LearnModel);
        if (tastCount < 0)
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        if (tastCount > 0)
        {
            writeToJsonMessage(error::learnmodel_periodic_exists, answer);
            webCon().send(answer);
            return;
        }
    }

    taskModelCreate.id = QUuidEx::createUuid();
    taskModelCreate.attemptLimit = 5;
    taskModelCreate.attemptCounter = 0;
    taskModelCreate.attemptInterval = 15;

    /* Время createDate устанавливается внутри функции mainTaskSqlQuery()
       taskModelCreate.createDate = QDateTime::currentDateTime(); */

    // Запрос на вставку/обновление данных в основную таблицу
    if (!mainTaskSqlQuery(q, true, TaskType::LearnModel, taskModelCreate))
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    _passTimer = true;
    writeToJsonMessage(taskModelCreate, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskModelEdit(const Message::Ptr& message)
{
    data::TaskModelEdit taskModelEdit;
    READ_FROM_MESSAGE(message, taskModelEdit);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != taskModelEdit.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT NAME, EXEC_STATUS, IS_PERIODIC FROM TASK WHERE ID = ?",
        taskModelEdit.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (!q.first())
    {
        data::MessageError msgerr = error::learnmodel_task_not_found;
        log_error_m << "Edit model task. Id: " << taskModelEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    //QString taskName;
    TaskExecStatus execStatus = TaskExecStatus::NotRun;
    bool isPeriodic = false;

    //assignValue(taskName,   q.record(), "NAME");
    assignValue(execStatus, q.record(), "EXEC_STATUS");
    assignValue(isPeriodic, q.record(), "IS_PERIODIC");

    if (execStatus == TaskExecStatus::Running
        || execStatus == TaskExecStatus::WaitSync
        || _tasks.findRef(taskModelEdit.id))
    {
        data::MessageError msgerr = error::edit_running_task;
        log_error_m << "Edit model task. Id: " << taskModelEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Запрет редактирование регулярной задачи для пользователей (не Администраторов)
    if (isPeriodic && !isAdmin)
    {
        data::MessageError msgerr = error::edit_task_denied;
        log_error_m << "Edit model task. Id: " << taskModelEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Проверить, нет ли попытки поменять тип задачи Разовая/Регулярная
    if (taskModelEdit.isPeriodic() != isPeriodic)
    {
        data::MessageError msgerr = error::change_run_mode_task;
        log_error_m << "Edit model task. Id: " << taskModelEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Запрос на вставку/обновление данных в основную таблицу
    if (!mainTaskSqlQuery(q, false, TaskType::LearnModel, taskModelEdit))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Параметры задачи 'Обучение' были изменены";
    log_verbose_m << EventLog(msg)
                  << EventTask(taskModelEdit.id)
                  << EventUser(message->tag());

    writeToJsonMessage(taskModelEdit, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskModelDelete(const Message::Ptr& message)
{
    data::TaskModelDelete taskModelDelete;
    READ_FROM_MESSAGE(message, taskModelDelete);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != taskModelDelete.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    #define LOG_ERROR_DELETE \
        log_error_m << "Delete model task. Id: " << taskModelDelete.taskId \
                    << ". " << msgerr.description;

    if (!sql::exec(q,
        "SELECT NAME, EXEC_STATUS FROM TASK WHERE ID = ?", taskModelDelete.taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (!q.first())
    {
        data::MessageError msgerr = error::learnmodel_task_not_found;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    QString taskName;
    TaskExecStatus execStatus = TaskExecStatus::NotRun;

    assignValue(taskName,   q.record(), "NAME");
    assignValue(execStatus, q.record(), "EXEC_STATUS");

    if (execStatus == TaskExecStatus::Running
        || execStatus == TaskExecStatus::WaitSync
        || _tasks.findRef(taskModelDelete.taskId))
    {
        data::MessageError msgerr = error::delete_running_task;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Если задача обучения указана в качестве родительской для score задачи,
    // то такую задачу обучения удалить нельзя.
    if (!sql::exec(q,
        " SELECT           "
        "   ID             "
        " FROM             "
        "   TASK           "
        " WHERE            "
        "   TASK_TYPE = ?  "
        "   AND            "
        "   PARENT_ID = ?  ",
        getTaskString(TaskType::ScoreCalc),
        taskModelDelete.taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    // Если запрос вернул данные, значит подчиненные задачи присуствуют
    if (q.first())
    {
        data::MessageError msgerr = error::learnmodel_task_has_child;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Если задача обучения указана в качестве родительской для созданных
    // оценок, то такую модель удалить нельзя.
    if (!sql::exec(q,
        " SELECT         "
        "   ID           "
        " FROM           "
        "   SCORE        "
        " WHERE          "
        "   MODEL_ID = ? ",
        taskModelDelete.taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    // Если запрос вернул данные, значит подчиненные задачи присуствуют
    if (q.first())
    {
        data::MessageError msgerr = error::learnmodel_task_has_child;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    int res = taskDelete(q, TaskType::LearnModel, taskModelDelete.taskId,
                                                  taskModelDelete.userId,
                                                  isAdmin);
    if (res < 0)
    {
        data::MessageError msgerr = error::learnmodel_task_not_found;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }
    else if (res == 0)
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Задача обучения '%1' удалена";
    log_verbose_m << EventLog(msg, taskName)
                  << EventTask(taskModelDelete.taskId)
                  << EventUser(message->tag());

    writeToJsonMessage(taskModelDelete, answer);
    webCon().send(answer);

    #undef LOG_ERROR_DELETE
}

bool Scheduler::taskList(TaskType taskType, const data::TaskFilter& filter,
                         QVector<data::Task>& list)
{
    QString sql =
        " SELECT                              "
        "   T.ID                              "
        "  ,T.USER_ID                         "
        "  ,T.PARENT_ID                       "
        "  ,T.TASK_TYPE                       "
        "  ,T.NAME                            "
        "  ,T.DESCRIPTION                     "
        "  ,T.CREATE_DATE                     "
        "  ,T.RUN_DATETIME                    "
        "  ,T.NEXT_DATETIME                   "
        "  ,T.REGULARITY_MONTH                "
        "  ,T.REGULARITY_DAY                  "
        "  ,T.REGULARITY_HOUR                 "
        "  ,T.ATTEMPT_LIMIT                   "
        "  ,T.ATTEMPT_COUNTER                 "
        "  ,T.ATTEMPT_INTERVAL                "
        "  ,T.EXEC_STATUS                     "
        "  ,T.IS_ENABLED                      "
        "  ,T.IS_PUBLIC                       "
        "  ,T.PERIOD_BEGIN                    "
        "  ,T.PERIOD_END                      "
        "  ,T.REL_PERIOD_DURATION             "
        "  ,T.MODEL_ID                        "
        "  ,T.RUN_CHILD_TASK                  "
        "  ,T.SKIP_SYNC                       "
        "  ,T.IS_PERIODIC                     "
        "  ,PT.NAME AS PARENT_NAME            "
        "  ,M.NAME  AS MODEL_NAME             "
        " FROM                                "
        "   TASK T                            "
        " LEFT JOIN                           "
        "   TASK PT ON T.PARENT_ID = PT.ID    "
        " LEFT JOIN                           "
        "   MODEL M ON T.MODEL_ID = M.ID      "
        " WHERE                               "
        "   T.TASK_TYPE = :TASK_TYPE          ";

    if (!filter.id.isNull())
        sql += " AND T.ID = :ID ";

    if (filter.mode == TaskMode::All)
    {
        sql +=
            " AND                   "
            " (                     "
            "   T.IS_PERIODIC = 1   "
            "   OR                  "
            "   (                   "
            "     T.IS_PERIODIC = 0 ";
            if (filter.userId.isNull())
            {
                return false;
            }
            else
            {
                if (filter.showOtherUsers && uright().isAdmin(filter.userId))
                {
                    // показать все задачи без фильтра по юзеру
                }
                else
                {
                    // показать все задачи c фильтром по юзеру
                    sql += " AND T.USER_ID = :USER_ID ";
                }
            }

        sql +=
            "   )  "
            " )    ";
    }
    else
    {
        if (!filter.userId.isNull())
            sql += " AND T.USER_ID = :USER_ID ";
    }

    if (filter.mode == TaskMode::OneTime)
        sql += " AND T.IS_PERIODIC = 0 ";

    if (filter.mode == TaskMode::Periodic)
        sql += " AND T.IS_PERIODIC = 1 ";

    if (!filter.modelId.isNull())
        sql += " AND T.MODEL_ID = :MODEL_ID ";

    if (!filter.parentId.isNull())
        sql += " AND T.PARENT_ID = :PARENT_ID ";

    sql += " ORDER BY             "
           "   T.IS_PERIODIC DESC "
           "  ,T.CREATE_DATE DESC ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
        return false;

    bindValue(q, ":TASK_TYPE", getTaskString(taskType));

    if (!filter.id.isNull())
        bindValue(q, ":ID", filter.id);

    if (filter.mode == TaskMode::All)
    {
        if (!filter.userId.isNull())
        {
            if (filter.showOtherUsers && uright().isAdmin(filter.userId))
            {
                // показать все задачи без фильтра по юзеру
            }
            else
            {
                // показать все задачи c фильтром по юзеру
                bindValue(q, ":USER_ID", filter.userId);
            }
        }
    }
    else
    {
        if (!filter.userId.isNull())
            bindValue(q, ":USER_ID", filter.userId);
    }

    if (!filter.modelId.isNull())
        bindValue(q, ":MODEL_ID", filter.modelId);

    if (!filter.parentId.isNull())
        bindValue(q, ":PARENT_ID", filter.parentId);

    if (!q.exec())
        return false;

    data::SyncInfo::List syncList = SyncPlan::syncList();

    while (q.next())
    {
        data::Task task;
        QSqlRecord r = q.record();

        bool isPeriodic = false;
        assignValue(isPeriodic              , r, "IS_PERIODIC");

        assignValue(task.id                 , r, "ID                  ");
        assignValue(task.userId             , r, "USER_ID             ");
        assignValue(task.parentId           , r, "PARENT_ID           ");
        assignValue(task.name               , r, "NAME                ");
        assignValue(task.description        , r, "DESCRIPTION         ");
        assignValue(task.createDate         , r, "CREATE_DATE         ");
        assignValue(task.runDateTime        , r, "RUN_DATETIME        ");
        assignValue(task.nextRunTime        , r, "NEXT_DATETIME       ");
        assignValue(task.nextDateTime       , r, "NEXT_DATETIME       ");
        assignValue(task.regularityMonth    , r, "REGULARITY_MONTH    ");
        assignValue(task.regularityDay      , r, "REGULARITY_DAY      ");
        assignValue(task.regularityHour     , r, "REGULARITY_HOUR     ");
        assignValue(task.attemptLimit       , r, "ATTEMPT_LIMIT       ");
        assignValue(task.attemptCounter     , r, "ATTEMPT_COUNTER     ");
        assignValue(task.attemptInterval    , r, "ATTEMPT_INTERVAL    ");
        assignValue(task.execStatus         , r, "EXEC_STATUS         ");
        assignValue(task.isEnabled          , r, "IS_ENABLED          ");
        assignValue(task.isPublic           , r, "IS_PUBLIC           ");
        assignValue(task.relPeriodDuration  , r, "REL_PERIOD_DURATION ");
        assignValue(task.modelId            , r, "MODEL_ID            ");
        assignValue(task.runChildTask       , r, "RUN_CHILD_TASK      ");
        assignValue(task.skipSync           , r, "SKIP_SYNC           ");
        assignValue(task.parentName         , r, "PARENT_NAME         ");
        assignValue(task.modelName          , r, "MODEL_NAME          ");

        if (isPeriodic)
        {
            if (taskType == TaskType::LearnModel)
            {
                initLearnPeriod(task.period, task.relPeriodDuration);
            }
            else if (taskType == TaskType::ScoreCalc)
            {
                initScorePeriod(task.period);
            }
        }
        else
        {
            assignValue(task.period.begin, r, "PERIOD_BEGIN");
            assignValue(task.period.end  , r, "PERIOD_END  ");
        }

        if (lst::FindResult fr = _tasks.findRef(task.id))
        {
            const data::TaskProgress& progress = _tasks.item(fr.index())->progress();
            task.progressCurrent = progress.current;
            task.progressTotal = progress.total;
        }

        // Если задача присутствует списке синхронизации ...
        if (lst::FindResult fr = syncList.findRef(task.id, {lst::BruteForce::Yes}))
        {
            task.waitPosition = fr.index();
            task.progressCurrent = syncList[fr.index()].current;
            task.progressTotal   = syncList[fr.index()].total;
        }
        list.append(task);
    }
    return true;
}

void Scheduler::command_TaskModelInfo(const Message::Ptr& message)
{
    data::TaskModelInfo taskModelInfo;
    READ_FROM_MESSAGE(message, taskModelInfo);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    task::data::TaskFilter filter;
    filter.id = taskModelInfo.taskId;
    filter.userId = taskModelInfo.userId;
    filter.mode = TaskMode::All;
    filter.showOtherUsers = isAdmin;

    QVector<data::Task> list;
    if (!taskList(TaskType::LearnModel, filter, list))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (list.isEmpty())
    {
        data::MessageError msgerr = error::learnmodel_task_not_found;
        log_error_m << "Get info model task. Id: " << taskModelInfo.taskId
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    data::TaskModelInfoA taskModel;
    static_cast<data::Task&>(taskModel) = list[0];

    writeToJsonMessage(taskModel, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskModelList(const Message::Ptr& message)
{
    data::TaskModelList taskModelLst;
    READ_FROM_MESSAGE(message, taskModelLst);

    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::Task> list;
    if (!taskList(TaskType::LearnModel, taskModelLst.filter, list))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    taskModelLst.items = list;
    writeToJsonMessage(taskModelLst, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskScoreCreate(const Message::Ptr& message)
{
    data::TaskScoreCreate taskScoreCreate;
    READ_FROM_MESSAGE(message, taskScoreCreate);

    Message::Ptr answer = message->cloneForAnswer();

    // Проверить на корректность имени и описания сразу, что бы
    // при неправильных входных данных не делать ничего лишнего.
    if (taskScoreCreate.name.isEmpty()
        || taskScoreCreate.description.isEmpty())
    {
        writeToJsonMessage(error::task_inncorrect_name, answer);
        webCon().send(answer);
        return;
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (taskScoreCreate.isPeriodic())
    {
        int tastCount = periodicTaskCount(q, TaskType::ScoreCalc);
        if (tastCount < 0)
        {
            writeToJsonMessage(error::select_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        if (tastCount > 0)
        {
            writeToJsonMessage(error::scorecalc_periodic_exists, answer);
            webCon().send(answer);
            return;
        }
    }

    taskScoreCreate.id = QUuidEx::createUuid();
    taskScoreCreate.attemptLimit = 5;
    taskScoreCreate.attemptCounter = 0;
    taskScoreCreate.attemptInterval = 15;

    /* Время createDate устанавливается внутри функции mainTaskSqlQuery()
       taskScoreCreate.createDate = QDateTime::currentDateTime(); */

    // Запрос на вставку/обновление данных в основную таблицу
    if (!mainTaskSqlQuery(q, true, TaskType::ScoreCalc, taskScoreCreate))
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    _passTimer = true;
    writeToJsonMessage(taskScoreCreate, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskScoreEdit(const Message::Ptr& message)
{
    data::TaskScoreEdit taskScoreEdit;
    READ_FROM_MESSAGE(message, taskScoreEdit);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != taskScoreEdit.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT NAME, EXEC_STATUS, IS_PERIODIC FROM TASK WHERE ID = ?",
        taskScoreEdit.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (!q.first())
    {
        data::MessageError msgerr = error::scorecalc_task_not_found;
        log_error_m << "Edit score task. Id: " << taskScoreEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    //QString taskName;
    TaskExecStatus execStatus;
    bool isPeriodic = false;

    //assignValue(taskName,   q.record(), "NAME");
    assignValue(execStatus, q.record(), "EXEC_STATUS");
    assignValue(isPeriodic, q.record(), "IS_PERIODIC");

    if (execStatus == TaskExecStatus::Running
        || execStatus == TaskExecStatus::WaitSync
        || _tasks.findRef(taskScoreEdit.id))
    {
        data::MessageError msgerr = error::edit_running_task;
        log_error_m << "Edit score task. Id: " << taskScoreEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Запрет редактирование регулярной задачи для пользователей (не Администраторов)
    if (isPeriodic && !isAdmin)
    {
        data::MessageError msgerr = error::edit_task_denied;
        log_error_m << "Edit model task. Id: " << taskScoreEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Проверить, нет ли попытки поменять тип задачи Разовая/Регулярная
    if (taskScoreEdit.isPeriodic() != isPeriodic)
    {
        data::MessageError msgerr = error::change_run_mode_task;
        log_error_m << "Edit model task. Id: " << taskScoreEdit.id
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    // Запрос на вставку/обновление данных в основную таблицу
    if (!mainTaskSqlQuery(q, false, TaskType::ScoreCalc, taskScoreEdit))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Параметры задачи 'Применение' были изменены";
    log_verbose_m << EventLog(msg)
                  << EventTask(taskScoreEdit.id)
                  << EventUser(message->tag());

    writeToJsonMessage(taskScoreEdit, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskScoreDelete(const Message::Ptr& message)
{
    data::TaskScoreDelete taskScoreDelete;
    READ_FROM_MESSAGE(message, taskScoreDelete);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    if (!isAdmin)
    {
        QUuidEx userId = uright().userId(userHashId);
        if (userId != taskScoreDelete.userId)
        {
            writeToJsonMessage(error::admin_privileges, answer);
            webCon().send(answer);
            return;
        }
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    #define LOG_ERROR_DELETE \
        log_error_m << "Delete score task. Id: " << taskScoreDelete.taskId \
                    << ". " << msgerr.description;

    if (!sql::exec(q,
        "SELECT NAME, EXEC_STATUS FROM TASK WHERE ID = ?", taskScoreDelete.taskId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (!q.first())
    {
        data::MessageError msgerr = error::scorecalc_task_not_found;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    QString taskName;
    assignValue(taskName, q.record(), "NAME");

    TaskExecStatus execStatus;
    assignValue(execStatus, q.record(), "EXEC_STATUS");
    if (execStatus == TaskExecStatus::Running
        || execStatus == TaskExecStatus::WaitSync
        || _tasks.findRef(taskScoreDelete.taskId))
    {
        data::MessageError msgerr = error::delete_running_task;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    int res = taskDelete(q, TaskType::ScoreCalc, taskScoreDelete.taskId,
                                                 taskScoreDelete.userId,
                                                 isAdmin);
    if (res < 0)
    {
        data::MessageError msgerr = error::scorecalc_task_not_found;
        LOG_ERROR_DELETE
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }
    else if (res == 0)
    {
        writeToJsonMessage(error::delete_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    const char* msg = u8"Задача применения '%1' удалена";
    log_verbose_m << EventLog(msg, taskName)
                  << EventTask(taskScoreDelete.taskId)
                  << EventUser(message->tag());

    writeToJsonMessage(taskScoreDelete, answer);
    webCon().send(answer);

    #undef LOG_ERROR_DELETE
}

void Scheduler::command_TaskScoreInfo(const Message::Ptr& message)
{
    data::TaskScoreInfo taskScoreInfo;
    READ_FROM_MESSAGE(message, taskScoreInfo);

    Message::Ptr answer = message->cloneForAnswer();

    quint64 userHashId = message->tag();
    bool isAdmin = uright().isAdmin(userHashId);

    task::data::TaskFilter filter;
    filter.id = taskScoreInfo.taskId;
    filter.userId = taskScoreInfo.userId;
    filter.mode = TaskMode::All;
    filter.showOtherUsers = isAdmin;

    QVector<data::Task> list;
    if (!taskList(TaskType::ScoreCalc, filter, list))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    if (list.isEmpty())
    {
        data::MessageError msgerr = error::scorecalc_task_not_found;
        log_error_m << "Get info score task. Id: " << taskScoreInfo.taskId
                    << ". " << msgerr.description;
        writeToJsonMessage(msgerr, answer);
        webCon().send(answer);
        return;
    }

    data::TaskScoreInfoA taskScore;
    *(static_cast<data::Task*>(&taskScore)) = list[0];

    writeToJsonMessage(taskScore, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskScoreList(const Message::Ptr& message)
{
    data::TaskScoreList taskScoreList;
    READ_FROM_MESSAGE(message, taskScoreList);

    //quint64 userHashId = message->tag();
    Message::Ptr answer = message->cloneForAnswer();

    QVector<data::Task> list;
    if (!taskList(TaskType::ScoreCalc, taskScoreList.filter, list))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    taskScoreList.items = list;
    writeToJsonMessage(taskScoreList, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskReportCreate(const Message::Ptr& message)
{
    data::TaskReportCreate taskReportCreate;
    READ_FROM_MESSAGE(message, taskReportCreate);

    quint64 userHashId = message->tag();
    Message::Ptr answer = message->cloneForAnswer();

    if (taskReportCreate.name.trimmed().isEmpty())
    {
        writeToJsonMessage(error::report_name_undef, answer);
        webCon().send(answer);
        return;
    }
    if (taskReportCreate.type == ReportType::Undefined)
    {
        writeToJsonMessage(error::report_type_undef, answer);
        webCon().send(answer);
        return;
    }
    if (taskReportCreate.ranging == ReportRanging::Undefined)
    {
        writeToJsonMessage(error::report_ranging_undef, answer);
        webCon().send(answer);
        return;
    }
    if (taskReportCreate.present == ReportPresent::Undefined)
    {
        writeToJsonMessage(error::report_present_undef, answer);
        webCon().send(answer);
        return;
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q, "SELECT PERIOD_BEGIN, PERIOD_END FROM SCORE WHERE ID = ?",
                      taskReportCreate.scoreId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::score_not_found)

    TimeRange scorePetiod;
    assignValue(scorePetiod.begin, q.record(), "PERIOD_BEGIN");
    assignValue(scorePetiod.end,   q.record(), "PERIOD_END");

    // Корректируем временной интервал для отчета
    if (taskReportCreate.period.begin.isNull()
        || taskReportCreate.period.begin < scorePetiod.begin)
    {
        taskReportCreate.period.begin = scorePetiod.begin;
        const char* msg = u8"Верхняя граница для отчета '%1' установлена в %2";
        log_warn_m << EventLog(msg, taskReportCreate.name, scorePetiod.begin)
                   << EventUser(userHashId); // Пользователь выполнивший действие
    }
    if (taskReportCreate.period.end.isNull()
        || taskReportCreate.period.end > scorePetiod.end)
    {
        taskReportCreate.period.end = scorePetiod.end;
        const char* msg = u8"Нижняя граница для отчета '%1' установлена в %2";
        log_warn_m << EventLog(msg, taskReportCreate.name, scorePetiod.end)
                   << EventUser(userHashId);
    }

    QUuidEx taskId = QUuidEx::createUuid();

    taskReportCreate.id = QUuidEx::createUuid();
    taskReportCreate.taskId = taskId;

    QString fields =
        "  ID            "
        ", TASK_ID       "
        ", USER_ID       "
        ", SCORE_ID      "
        ", NAME          "
        ", REPORT_TYPE   "
        ", PERIOD_BEGIN  "
        ", PERIOD_END    "
        ", CODE_LPU      "
        ", CODE_MKB      "
        ", VIDMP         "
        ", MED_PROFILE   "
        ", THRESHOLD     "
        ", RECORD_COUNT  "
        ", RANGING_TYPE  "
        ", SORT_TYPE     "
        ", PRESENT_TYPE  ";

    QString sql = sql::INSERT_INTO("REPORT", fields);

    if (!q.prepare(sql))
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    bindValue(q, ":ID           ", taskReportCreate.id            );
    bindValue(q, ":TASK_ID      ", taskReportCreate.taskId        );
    bindValue(q, ":USER_ID      ", taskReportCreate.userId        );
    bindValue(q, ":SCORE_ID     ", taskReportCreate.scoreId       );
    bindValue(q, ":NAME         ", taskReportCreate.name          );
    bindValue(q, ":REPORT_TYPE  ", taskReportCreate.type          );
    bindValue(q, ":PERIOD_BEGIN ", taskReportCreate.period.begin  );
    bindValue(q, ":PERIOD_END   ", taskReportCreate.period.end    );
    bindValue(q, ":CODE_LPU     ", taskReportCreate.lpu           );
    bindValue(q, ":CODE_MKB     ", taskReportCreate.mkb           );
    bindValue(q, ":VIDMP        ", taskReportCreate.vidmp         );
    bindValue(q, ":MED_PROFILE  ", taskReportCreate.medProfile    );
    bindValue(q, ":THRESHOLD    ", taskReportCreate.threshold     );
    bindValue(q, ":RECORD_COUNT ", taskReportCreate.recordCount   );
    bindValue(q, ":RANGING_TYPE ", taskReportCreate.ranging       );
    bindValue(q, ":SORT_TYPE    ", taskReportCreate.sort          );
    bindValue(q, ":PRESENT_TYPE ", taskReportCreate.present       );

    if (!q.exec())
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    // Создаем задачу
    data::Task task;
    task.id = taskId;
    task.userId = taskReportCreate.userId;
    task.name = taskReportCreate.name;

    task.regularityMonth = 0;
    task.regularityDay   = 0;
    task.regularityHour  = 0;

    task.period = taskReportCreate.period;
    task.runDateTime = QDateTime::currentDateTime();

    // Запрос на вставку/обновление данных в основную таблицу
    if (!mainTaskSqlQuery(q, true, TaskType::CreateReport, task))
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    // Время createDate устанавливается внутри функции mainTaskSqlQuery(),
    // это время необходимо вернуть в веб-интерфейс
    taskReportCreate.createDate = task.createDate;

    // Заполняем вспомогательные поля
    q = QSqlQuery(dbcon->createResult());
    if (!sql::exec(q,
        " SELECT                          "
        "   R.ID                          "
        "  ,S.NAME AS SCORE_NAME          "
        " FROM                            "
        "   REPORT R                      "
        " LEFT JOIN                       "
        "   SCORE S ON S.ID = R.SCORE_ID  "
        " WHERE                           "
        "   R.ID = ?                      ", taskReportCreate.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (q.first())
    {
        QSqlRecord r = q.record();
        assignValue(taskReportCreate.scoreName, r, "SCORE_NAME" );
    }

    // Запускаем задачу немедленно
    data::TaskStartNow taskStartNow;
    taskStartNow.taskId = task.id;
    taskStartNow.userId = task.userId;
    Message::Ptr m = createJsonMessage(taskStartNow);
    this->message(m);

    writeToJsonMessage(taskReportCreate, answer);
    webCon().send(answer);
}

static bool reportList(data::PagingInfo& paging, const data::ReportFilter& filter,
                       QVector<data::Report>& reports)
{
    reports.clear();

    QString sql =
        " SELECT                          "
        "   R.ID                          "
        "  ,R.TASK_ID                     "
        "  ,R.USER_ID                     "
        "  ,R.SCORE_ID                    "
        "  ,R.NAME                        "
        "  ,R.REPORT_TYPE                 "
        "  ,R.CREATE_DATE                 "
        "  ,R.PERIOD_BEGIN                "
        "  ,R.PERIOD_END                  "
        "  ,R.CODE_LPU                    "
        "  ,R.CODE_MKB                    "
        "  ,R.VIDMP                       "
        "  ,R.MED_PROFILE                 "
        "  ,R.THRESHOLD                   "
        "  ,R.RECORD_COUNT                "
        "  ,R.RANGING_TYPE                "
        "  ,R.PRESENT_TYPE                "
        "  ,R.SORT_TYPE                   "
        "  ,R.EXEC_STATUS                 "
        "  ,S.NAME AS SCORE_NAME          "
        " FROM                            "
        "   REPORT R                      "
        " LEFT JOIN                       "
        "   SCORE S ON S.ID = R.SCORE_ID  "
        " WHERE (1 = 1)                   ";

    if (!filter.id.isNull())
        sql += " AND R.ID = :ID ";

    if (!filter.userId.isNull())
        sql += " AND R.USER_ID = :USER_ID ";

    if (!filter.name.isEmpty())
        sql += " AND R.NAME CONTAINING :NAME ";

    if (filter.type != ReportType::Undefined)
        sql += " AND R.REPORT_TYPE = :REPORT_TYPE ";

    if (filter.createDate.begin.isValid())
        sql += " AND R.CREATE_DATE >= :CREATE_DATE_BEGIN ";

    if (filter.createDate.end.isValid())
        sql += " AND R.CREATE_DATE <= :CREATE_DATE_END ";

    if (filter.dataPeriod.begin.isValid() && filter.dataPeriod.end.isValid())
        // a.start <= b.end AND a.end >= b.start
        sql += " AND R.PERIOD_BEGIN <= :PERIOD_END AND R.PERIOD_END >= :PERIOD_BEGIN ";

    if (!filter.scoreId.isNull())
        sql += " AND MODEL_ID = :SCORE_ID ";

    sql += " ORDER BY             "
           "   R.CREATE_DATE DESC ";

    pagingPrepare(paging, sql);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
        return false;

    if (!filter.id.isNull())
        bindValue(q, ":ID", filter.id);

    if (!filter.userId.isNull())
        bindValue(q, ":USER_ID", filter.userId);

    if (!filter.name.isEmpty())
        bindValue(q, ":NAME", filter.name);

    if (filter.type != ReportType::Undefined)
        bindValue(q, ":REPORT_TYPE", filter.type);

    if (filter.createDate.begin.isValid())
        bindValue(q, ":CREATE_DATE_BEGIN", filter.createDate.begin);

    if (filter.createDate.end.isValid())
        bindValue(q, ":CREATE_DATE_END", filter.createDate.end);

    if (filter.dataPeriod.begin.isValid() && filter.dataPeriod.end.isValid())
    {
        bindValue(q, ":PERIOD_BEGIN", filter.dataPeriod.begin);
        bindValue(q, ":PERIOD_END", filter.dataPeriod.end);
    }

    if (!filter.scoreId.isNull())
        bindValue(q, ":SCORE_ID", filter.scoreId);

    if (!q.exec())
        return false;

    while (q.next())
    {
        data::Report report;
        QSqlRecord r = q.record();

        assignValue(report.id,            r, "ID           " );
        assignValue(report.taskId,        r, "TASK_ID      " );
        assignValue(report.userId,        r, "USER_ID      " );
        assignValue(report.scoreId,       r, "SCORE_ID     " );
        assignValue(report.type,          r, "REPORT_TYPE  " );
        assignValue(report.name,          r, "NAME         " );
        assignValue(report.createDate,    r, "CREATE_DATE  " );
        assignValue(report.period.begin,  r, "PERIOD_BEGIN " );
        assignValue(report.period.end,    r, "PERIOD_END   " );
        assignValue(report.lpu,           r, "CODE_LPU     " );
        assignValue(report.mkb,           r, "CODE_MKB     " );
        assignValue(report.vidmp,         r, "VIDMP        " );
        assignValue(report.medProfile,    r, "MED_PROFILE  " );
        assignValue(report.threshold,     r, "THRESHOLD    " );
        assignValue(report.recordCount,   r, "RECORD_COUNT " );
        assignValue(report.ranging,       r, "RANGING_TYPE " );
        assignValue(report.present,       r, "PRESENT_TYPE " );
        assignValue(report.sort,          r, "SORT_TYPE    " );
        assignValue(report.execStatus,    r, "EXEC_STATUS  " );
        assignValue(report.scoreName,     r, "SCORE_NAME   " );

        reports.append(report);
    }

    paging.total = q.size();
    return true;
}

void Scheduler::command_ReportInfo(const Message::Ptr& message)
{
    data::ReportInfo reportInfo;
    READ_FROM_MESSAGE(message, reportInfo);

    Message::Ptr answer = message->cloneForAnswer();

    data::PagingInfo paging;
    data::ReportFilter filter;
    filter.id = reportInfo.id;

    QVector<data::Report> reports;
    if (!task::reportList(paging, filter, reports))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (reports.isEmpty())
    {
        writeToJsonMessage(error::report_not_found, answer);
        webCon().send(answer);
        return;
    }
    data::ReportInfoA reportInfoA;
    static_cast<data::Report&>(reportInfoA) = reports[0];

    writeToJsonMessage(reportInfoA, answer);
    webCon().send(answer);
}

void Scheduler::command_ReportList(const Message::Ptr& message)
{
    data::ReportList reportList;
    READ_FROM_MESSAGE(message, reportList);

    Message::Ptr answer = message->cloneForAnswer();

    if (!task::reportList(reportList.paging, reportList.filter, reportList.items))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(reportList, answer);
    webCon().send(answer);
}

void Scheduler::command_ReportData(const Message::Ptr& message)
{
    data::ReportData reportData;
    READ_FROM_MESSAGE(message, reportData);

    Message::Ptr answer = message->cloneForAnswer();

    QString sql =
        " SELECT          "
        "   ID            "
        "  ,REPORT_ID     "
        "  ,RESULT_NAME   "
        "  ,RESULT_CODE   "
        "  ,RESULT_VALUE  "
        "  ,RESULT_COUNT  "
        "  ,CODE_LPU      "
        "  ,NAME_LPU      "
        " FROM            "
        "   REPORT_DATA   "
        " WHERE           "
        "   REPORT_ID = ? "
        " ORDER BY        "
        "   RESULT_VALUE  ";

    sql += (reportData.reportSort == ReportSort::Descending)
           ? " DESC "
           : " ASC ";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, sql, reportData.reportId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    reportData.items.clear();
    while (q.next())
    {
        data::ReportDataItem rdi;
        QSqlRecord r = q.record();

        assignValue(rdi.name,    r, "RESULT_NAME  ");
        assignValue(rdi.code,    r, "RESULT_CODE  ");
        assignValue(rdi.value,   r, "RESULT_VALUE ");
        assignValue(rdi.nameLpu, r, "NAME_LPU     ");
        assignValue(rdi.codeLpu, r, "CODE_LPU     ");

        reportData.items.append(rdi);
    }

    writeToJsonMessage(reportData, answer);
    webCon().send(answer);
}

void Scheduler::command_ReportEdit(const Message::Ptr& message)
{
    data::ReportEdit reportEdit;
    READ_FROM_MESSAGE(message, reportEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "UPDATE REPORT SET NAME = ? WHERE ID = ?",
                      reportEdit.name, reportEdit.id))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(reportEdit, answer);
    webCon().send(answer);
    return;
}

void Scheduler::command_ReportDelete(const Message::Ptr& message)
{
    data::ReportDelete reportDelete;
    READ_FROM_MESSAGE(message, reportDelete);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT TASK_ID, EXEC_STATUS FROM REPORT WHERE ID = ?", reportDelete.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (q.first())
    {
        QUuidEx taskId;
        assignValue(taskId, q.record(), "TASK_ID");

        TaskExecStatus execStatus;
        assignValue(execStatus, q.record(), "EXEC_STATUS");

        if (_tasks.findRef(taskId)
            || execStatus == TaskExecStatus::Running)
        {
            writeToJsonMessage(error::report_task_running.asFailed(), answer);
            webCon().send(answer);
            return;
        }
        if (!sql::exec(q, "DELETE FROM TASK WHERE ID = ?", taskId))
        {
            writeToJsonMessage(error::delete_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        if (!sql::exec(q,
            "DELETE FROM REPORT WHERE ID = ?", reportDelete.id))
        {
            writeToJsonMessage(error::delete_sql_statement, answer);
            webCon().send(answer);
            return;
        }
        if (!sql::exec(q,
            "DELETE FROM REPORT_DATA WHERE REPORT_ID = ?", reportDelete.id))
        {
            writeToJsonMessage(error::delete_sql_statement, answer);
            webCon().send(answer);
            return;
        }
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    log_warn_m << EventLog(u8"Отчет '%1' удален", reportDelete.id)
               << EventUser(reportDelete.userId); // Пользователь выполнивший действие

    writeToJsonMessage(reportDelete, answer);
    webCon().send(answer);
}

static void checkReportFedIndexes()
{
    int currentYear = 0;
    db::settings::getValue("foms.report_fed.current_year", currentYear);

    QDate curDt = QDate::currentDate();
    if (currentYear != curDt.year())
    {
        db::settings::setValue("foms.report_fed.current_year", curDt.year());
        db::settings::setValue("foms.report_fed.rpe_index", 1);
        db::settings::setValue("foms.report_fed.rpes_index", 1);
    }
}

static void getReportFedIndex(data::ReportFedNameA& reportFedNameA)
{
    reportFedNameA.rpeIndex = 0;
    db::settings::getValue("foms.report_fed.rpe_index", reportFedNameA.rpeIndex);

    reportFedNameA.rpesIndex = 0;
    db::settings::getValue("foms.report_fed.rpes_index", reportFedNameA.rpesIndex);

    reportFedNameA.rpeIndex  += 1;
    reportFedNameA.rpesIndex += 1;
}

static void getReportFedName(data::ReportFedNameA& reportFedNameA)
{
    QDate curDt = QDate::currentDate();
    reportFedNameA.rpeName = "RPE" + reportFedNameA.fomsCode + curDt.toString("yy")
                           + QString::number(reportFedNameA.rpeIndex).rightJustified(4, '0');

    reportFedNameA.rpesName = "RPES" + reportFedNameA.fomsCode + curDt.toString("yy")
                            + QString::number(reportFedNameA.rpesIndex).rightJustified(4, '0');
}

void Scheduler::command_TaskReportFedCreate(const Message::Ptr& message)
{
    data::TaskReportFedCreate taskReportFedCreate;
    READ_FROM_MESSAGE(message, taskReportFedCreate);

    quint64 userHashId = message->tag();
    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q, "SELECT PERIOD_BEGIN, PERIOD_END FROM SCORE WHERE ID = ?",
                      taskReportFedCreate.scoreId))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::score_not_found)

    TimeRange scorePetiod;
    assignValue(scorePetiod.begin, q.record(), "PERIOD_BEGIN");
    assignValue(scorePetiod.end,   q.record(), "PERIOD_END");

    QUuidEx taskId = QUuidEx::createUuid();

    taskReportFedCreate.id = QUuidEx::createUuid();
    taskReportFedCreate.taskId = taskId;

    // Корректируем временной интервал для отчета
    if (taskReportFedCreate.period.begin.isNull()
        || taskReportFedCreate.period.begin < scorePetiod.begin)
    {
        taskReportFedCreate.period.begin = scorePetiod.begin;
        const char* msg = u8"Верхняя граница для фед-отчета '%1' установлена в %2";
        log_warn_m << EventLog(msg, taskReportFedCreate.name, scorePetiod.begin)
                   << EventUser(userHashId); // Пользователь выполнивший действие
    }
    if (taskReportFedCreate.period.end.isNull()
        || taskReportFedCreate.period.end > scorePetiod.end)
    {
        taskReportFedCreate.period.end = scorePetiod.end;
        const char* msg = u8"Нижняя граница для фед-отчета '%1' установлена в %2";
        log_warn_m << EventLog(msg, taskReportFedCreate.name, scorePetiod.end)
                   << EventUser(userHashId);
    }

    data::ReportFedNameA reportFedNameA;

    checkReportFedIndexes();
    if (taskReportFedCreate.orderNum != 0)
    {
        reportFedNameA.rpesIndex = taskReportFedCreate.orderNum;
        reportFedNameA.rpeIndex  = taskReportFedCreate.orderNum;
    }
    else
    {
        getReportFedIndex(reportFedNameA);
    }
    getReportFedName(reportFedNameA);

    taskReportFedCreate.name = (taskReportFedCreate.isCorrection)
                               ? reportFedNameA.rpesName
                               : reportFedNameA.rpeName;
    QString fields =
        "  ID            "
        ", TASK_ID       "
        ", USER_ID       "
        ", SCORE_ID      "
        ", NAME          "
        ", PERIOD_BEGIN  "
        ", PERIOD_END    "
        ", IS_PERIODIC   "
        ", IS_CORRECTION "
        ", ORDER_NUM     ";

    QString sql = sql::INSERT_INTO("REPORT_FED", fields);

    if (!q.prepare(sql))
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    bindValue(q, ":ID           ", taskReportFedCreate.id           );
    bindValue(q, ":TASK_ID      ", taskReportFedCreate.taskId       );
    bindValue(q, ":USER_ID      ", taskReportFedCreate.userId       );
    bindValue(q, ":SCORE_ID     ", taskReportFedCreate.scoreId      );
    bindValue(q, ":NAME         ", taskReportFedCreate.name         );
    bindValue(q, ":PERIOD_BEGIN ", taskReportFedCreate.period.begin );
    bindValue(q, ":PERIOD_END   ", taskReportFedCreate.period.end   );
    bindValue(q, ":IS_PERIODIC  ", taskReportFedCreate.isPeriodic   );
    bindValue(q, ":IS_CORRECTION", taskReportFedCreate.isCorrection );
    bindValue(q, ":ORDER_NUM    ", taskReportFedCreate.orderNum     );

    if (!q.exec())
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    // Обновляем счетчики индексов
    if (taskReportFedCreate.isCorrection)
        db::settings::setValue(q, "foms.report_fed.rpes_index", reportFedNameA.rpesIndex);
    else
        db::settings::setValue(q, "foms.report_fed.rpe_index",  reportFedNameA.rpeIndex);

    // Создаем задачу
    data::Task task;
    task.id = taskId;
    task.userId = taskReportFedCreate.userId;
    task.name = taskReportFedCreate.name;

    task.regularityMonth = 0;
    task.regularityDay   = 0;
    task.regularityHour  = 0;

    task.period = taskReportFedCreate.period;
    task.runDateTime = QDateTime::currentDateTime();

    // Запрос на вставку/обновление данных в основную таблицу
    if (!mainTaskSqlQuery(q, true, TaskType::CreateReportFed, task))
    {
        writeToJsonMessage(error::insert_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    // Время createDate устанавливается внутри функции mainTaskSqlQuery(),
    // это время необходимо вернуть в веб-интерфейс
    taskReportFedCreate.createDate = task.createDate;

    // Заполняем вспомогательные поля
    q = QSqlQuery(dbcon->createResult());
    if (!sql::exec(q,
        " SELECT                          "
        "   R.ID                          "
        "  ,S.NAME AS SCORE_NAME          "
        " FROM                            "
        "   REPORT_FED R                  "
        " LEFT JOIN                       "
        "   SCORE S ON S.ID = R.SCORE_ID  "
        " WHERE                           "
        "   R.ID = ?                      ", taskReportFedCreate.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (q.first())
    {
        QSqlRecord r = q.record();
        assignValue(taskReportFedCreate.scoreName, r, "SCORE_NAME" );
    }

    // Запускаем задачу немедленно
    data::TaskStartNow taskStartNow;
    taskStartNow.taskId = task.id;
    taskStartNow.userId = task.userId;
    Message::Ptr m = createJsonMessage(taskStartNow);
    this->message(m);

    writeToJsonMessage(taskReportFedCreate, answer);
    webCon().send(answer);
}

void Scheduler::command_ReportFedEdit(const Message::Ptr& message)
{
    data::ReportFedEdit reportFedEdit;
    READ_FROM_MESSAGE(message, reportFedEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "UPDATE REPORT_FED SET NAME = ? WHERE ID = ?",
                      reportFedEdit.name, reportFedEdit.id))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(reportFedEdit, answer);
    webCon().send(answer);
    return;
}

void Scheduler::command_ReportFedDelete(const Message::Ptr& message)
{
    data::ReportFedDelete reportFedDelete;
    READ_FROM_MESSAGE(message, reportFedDelete);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        writeToJsonMessage(error::begin_transaction, answer);
        webCon().send(answer);
        return;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        "SELECT TASK_ID, EXEC_STATUS FROM REPORT_FED WHERE ID = ?", reportFedDelete.id))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (q.first())
    {
        QUuidEx taskId;
        assignValue(taskId, q.record(), "TASK_ID");

        TaskExecStatus execStatus;
        assignValue(execStatus, q.record(), "EXEC_STATUS");

        if (_tasks.findRef(taskId)
            || execStatus == TaskExecStatus::Running)
        {
            writeToJsonMessage(error::report_task_running.asFailed(), answer);
            webCon().send(answer);
            return;
        }
        if (!sql::exec(q, "DELETE FROM TASK WHERE ID = ?", taskId))
        {
            writeToJsonMessage(error::delete_sql_statement, answer);
            webCon().send(answer);
            return;
        }

        if (!sql::exec(q,
            "DELETE FROM REPORT_FED WHERE ID = ?", reportFedDelete.id))
        {
            writeToJsonMessage(error::delete_sql_statement, answer);
            webCon().send(answer);
            return;
        }
    }

    if (!transact->commit())
    {
        writeToJsonMessage(error::commit_transaction, answer);
        webCon().send(answer);
        return;
    }

    log_warn_m << EventLog(u8"Отчет '%1' удален", reportFedDelete.id)
               << EventUser(reportFedDelete.userId); // Пользователь выполнивший действие

    writeToJsonMessage(reportFedDelete, answer);
    webCon().send(answer);
}

bool Scheduler::reportFedList(data::PagingInfo& paging,
                              const data::ReportFedFilter& filter,
                              QVector<data::ReportFed>& reports)
{
    reports.clear();

    QString sql =
        " SELECT                          "
        "   R.ID                          "
        "  ,R.TASK_ID                     "
        "  ,R.USER_ID                     "
        "  ,R.SCORE_ID                    "
        "  ,R.NAME                        "
        "  ,R.CREATE_DATE                 "
        "  ,R.PERIOD_BEGIN                "
        "  ,R.PERIOD_END                  "
        "  ,R.XML_DATA                    "
        "  ,R.IS_PERIODIC                 "
        "  ,R.IS_CORRECTION               "
        "  ,R.ORDER_NUM                   "
        "  ,R.EXEC_STATUS                 "
        "  ,S.NAME AS SCORE_NAME          "
        " FROM                            "
        "   REPORT_FED R                  "
        " LEFT JOIN                       "
        "   SCORE S ON S.ID = R.SCORE_ID  "
        " WHERE (1 = 1)                   ";

    if (!filter.id.isNull())
        sql += " AND R.ID = :ID ";

    if (!filter.userId.isNull())
        sql += " AND R.USER_ID = :USER_ID ";

    if (!filter.name.isEmpty())
        sql += " AND R.NAME CONTAINING :NAME ";

    if (filter.createDate.begin.isValid())
        sql += " AND R.CREATE_DATE >= :CREATE_DATE_BEGIN ";

    if (filter.createDate.end.isValid())
        sql += " AND R.CREATE_DATE <= :CREATE_DATE_END ";

    if (filter.dataPeriod.begin.isValid() && filter.dataPeriod.end.isValid())
        // a.start <= b.end AND a.end >= b.start
        sql += " AND R.PERIOD_BEGIN <= :PERIOD_END AND R.PERIOD_END >= :PERIOD_BEGIN ";

    if (!filter.scoreId.isNull())
        sql += " AND MODEL_ID = :SCORE_ID ";

    sql += " ORDER BY             "
           "   R.CREATE_DATE DESC ";

    pagingPrepare(paging, sql);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(sql))
        return false;

    if (!filter.id.isNull())
        bindValue(q, ":ID", filter.id);

    if (!filter.userId.isNull())
        bindValue(q, ":USER_ID", filter.userId);

    if (!filter.name.isEmpty())
        bindValue(q, ":NAME", filter.name);

    if (filter.createDate.begin.isValid())
        bindValue(q, ":CREATE_DATE_BEGIN", filter.createDate.begin);

    if (filter.createDate.end.isValid())
        bindValue(q, ":CREATE_DATE_END", filter.createDate.end);

    if (filter.dataPeriod.begin.isValid() && filter.dataPeriod.end.isValid())
    {
        bindValue(q, ":PERIOD_BEGIN", filter.dataPeriod.begin);
        bindValue(q, ":PERIOD_END", filter.dataPeriod.end);
    }

    if (!filter.scoreId.isNull())
        bindValue(q, ":SCORE_ID", filter.scoreId);

    if (!q.exec())
        return false;

    while (q.next())
    {
        data::ReportFed report;
        QSqlRecord r = q.record();

        assignValue(report.id,            r, "ID           " );
        assignValue(report.taskId,        r, "TASK_ID      " );
        assignValue(report.userId,        r, "USER_ID      " );
        assignValue(report.scoreId,       r, "SCORE_ID     " );
        assignValue(report.name,          r, "NAME         " );
        assignValue(report.createDate,    r, "CREATE_DATE  " );
        assignValue(report.period.begin,  r, "PERIOD_BEGIN " );
        assignValue(report.period.end,    r, "PERIOD_END   " );

        QByteArray xmlData;
        assignValue(xmlData,              r, "XML_DATA     " );
        report.xmlData = QString::fromUtf8(xmlData);

        assignValue(report.isPeriodic,    r, "IS_PERIODIC  " );
        assignValue(report.isCorrection,  r, "IS_CORRECTION" );
        assignValue(report.orderNum,      r, "ORDER_NUM    " );
        assignValue(report.execStatus,    r, "EXEC_STATUS  " );
        assignValue(report.scoreName,     r, "SCORE_NAME   " );

        if (lst::FindResult fr = _tasks.findRef(report.taskId))
        {
            const data::TaskProgress& progress = _tasks.item(fr.index())->progress();
            report.progressCurrent = progress.current;
            report.progressTotal = progress.total;
        }

        reports.append(report);
    }

    paging.total = q.size();
    return true;
}

void Scheduler::command_ReportFedInfo(const Message::Ptr& message)
{
    data::ReportFedInfo reportFedInfo;
    READ_FROM_MESSAGE(message, reportFedInfo);

    Message::Ptr answer = message->cloneForAnswer();

    data::PagingInfo paging;
    data::ReportFedFilter filter;
    filter.id = reportFedInfo.id;

    QVector<data::ReportFed> reports;
    if (!this->reportFedList(paging, filter, reports))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    if (reports.isEmpty())
    {
        writeToJsonMessage(error::report_not_found, answer);
        webCon().send(answer);
        return;
    }
    data::ReportFedInfoA reportFedInfoA;
    static_cast<data::ReportFed&>(reportFedInfoA) = reports[0];

    writeToJsonMessage(reportFedInfoA, answer);
    webCon().send(answer);
}

void Scheduler::command_ReportFedList(const Message::Ptr& message)
{
    data::ReportFedList reportFedList;
    READ_FROM_MESSAGE(message, reportFedList);

    Message::Ptr answer = message->cloneForAnswer();

    if (!this->reportFedList(reportFedList.paging,
                             reportFedList.filter,
                             reportFedList.items))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    writeToJsonMessage(reportFedList, answer);
    webCon().send(answer);
}

void Scheduler::command_ReportFedName(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    data::ReportFedNameA reportFedNameA;

    checkReportFedIndexes();
    getReportFedIndex(reportFedNameA);
    getReportFedName(reportFedNameA);

    reportFedNameA.fomsCode = "01";
    config::base().getValue("foms.code", reportFedNameA.fomsCode);

    writeToJsonMessage(reportFedNameA, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskReportFedNow(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    if (lst::FindResult fr = _tasks.findRef(TaskType::ReportFed))
        if (_tasks.item(fr.index())->taskIsRunning())
        {
            writeToJsonMessage(error::task_already_running.asFailed(), answer);
            webCon().send(answer);
            return;
        }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "UPDATE TASK SET FORCED = 1 WHERE TASK_TYPE = ?",
                      getTaskString(TaskType::ReportFed)))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    _passTimer = true;
    webCon().send(answer);
}

void Scheduler::command_TaskReportFedInfo(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        "SELECT FIRST 1 ID, NAME FROM TASK WHERE TASK_TYPE = ? AND IS_PERIODIC = 1",
        getTaskString(TaskType::ScoreCalc)))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::scorecalc_task_not_found)

    data::TaskReportFedInfoA taskReportFedInfoA;

    assignValue(taskReportFedInfoA.parentId,   q.record(), "ID  ");
    assignValue(taskReportFedInfoA.parentName, q.record(), "NAME");

    if (!sql::exec(q,
        " SELECT           "
        "   ID             "
        "  ,RUN_DATETIME   "
        "  ,NEXT_DATETIME  "
        "  ,IS_ENABLED     "
        " FROM             "
        "   TASK           "
        " WHERE            "
        "   TASK_TYPE = ?  ", getTaskString(TaskType::ReportFed)))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::reportfed_task_not_found)

    QSqlRecord r = q.record();
    assignValue(taskReportFedInfoA.runDateTime,  r, "RUN_DATETIME ");
    assignValue(taskReportFedInfoA.nextDateTime, r, "NEXT_DATETIME");
    assignValue(taskReportFedInfoA.isEnabled,    r, "IS_ENABLED   ");

    taskReportFedInfoA.fomsCode = "01";
    config::base().getValue("foms.code", taskReportFedInfoA.fomsCode);

    taskReportFedInfoA.storeDir = "/var/opt/aisexpert/reportfed";
    db::settings::getValue("foms.report_fed.store_dir", taskReportFedInfoA.storeDir);

    writeToJsonMessage(taskReportFedInfoA, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskReportFedEdit(const Message::Ptr& message)
{
    data::TaskReportFedEdit taskReportFedEdit;
    READ_FROM_MESSAGE(message, taskReportFedEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " UPDATE TASK SET         "
        "   REGULARITY_MONTH = 1  "
        "  ,REGULARITY_DAY   = 0  "
        "  ,REGULARITY_HOUR  = 0  "
        "  ,RUN_DATETIME     = ?  "
        "  ,NEXT_DATETIME    = ?  "
        "  ,IS_ENABLED       = ?  "
        " WHERE                   "
        "   TASK_TYPE = ?         ",
        taskReportFedEdit.nextDateTime,
        taskReportFedEdit.nextDateTime,
        taskReportFedEdit.isEnabled,
        getTaskString(TaskType::ReportFed)))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    //db::settings::setValue("foms.code", taskReportFedEdit.fomsCode);
    db::settings::setValue("foms.report_fed.store_dir", taskReportFedEdit.storeDir);

    // Если все нормально, то посылаем пустой ответ
    webCon().send(answer);
}

void Scheduler::command_TaskSyncNsiNow(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    if (lst::FindResult fr = _tasks.findRef(TaskType::SyncNsi))
        if (_tasks.item(fr.index())->taskIsRunning())
        {
            writeToJsonMessage(error::task_already_running.asFailed(), answer);
            webCon().send(answer);
            return;
        }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "UPDATE TASK SET FORCED = 1 WHERE TASK_TYPE = ?",
                      getTaskString(TaskType::SyncNsi)))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        webCon().send(answer);
        return;
    }

    _passTimer = true;
    webCon().send(answer);
}

void Scheduler::command_TaskSyncNsiInfo(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT             "
        "   ID               "
        "  ,RUN_DATETIME     "
        "  ,NEXT_DATETIME    "
        "  ,ATTEMPT_LIMIT    "
        "  ,ATTEMPT_INTERVAL "
        "  ,EXEC_STATUS      "
        "  ,IS_ENABLED       "
        " FROM               "
        "   TASK             "
        " WHERE              "
        "   TASK_TYPE = ?    ", getTaskString(TaskType::SyncNsi)))
    {
        writeToJsonMessage(error::select_sql_statement, answer);
        webCon().send(answer);
        return;
    }
    Q_FIRST_CHECK(error::syncnsi_task_not_found)

    QSqlRecord r = q.record();
    data::TaskSyncNsiInfoA taskSyncNsiInfoA;

    assignValue(taskSyncNsiInfoA.taskId         , r, "ID              ");
    assignValue(taskSyncNsiInfoA.runDateTime    , r, "RUN_DATETIME    ");
    assignValue(taskSyncNsiInfoA.nextDateTime   , r, "NEXT_DATETIME   ");
    assignValue(taskSyncNsiInfoA.attemptLimit   , r, "ATTEMPT_LIMIT   ");
    assignValue(taskSyncNsiInfoA.attemptInterval, r, "ATTEMPT_INTERVAL");
    assignValue(taskSyncNsiInfoA.taskExecStatus , r, "EXEC_STATUS     ");
    assignValue(taskSyncNsiInfoA.isEnabled      , r, "IS_ENABLED      ");

    writeToJsonMessage(taskSyncNsiInfoA, answer);
    webCon().send(answer);
}

void Scheduler::command_TaskSyncNsiEdit(const Message::Ptr& message)
{
    data::TaskSyncNsiEdit taskSyncNsiEdit;
    READ_FROM_MESSAGE(message, taskSyncNsiEdit);

    Message::Ptr answer = message->cloneForAnswer();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!q.prepare(
        " UPDATE TASK SET                        "
        "   REGULARITY_MONTH = :REGULARITY_MONTH "
        "  ,REGULARITY_DAY   = :REGULARITY_DAY   "
        "  ,REGULARITY_HOUR  = :REGULARITY_HOUR  "
        "  ,RUN_DATETIME     = :RUN_DATETIME     "
        "  ,NEXT_DATETIME    = :NEXT_DATETIME    "
        "  ,ATTEMPT_LIMIT    = :ATTEMPT_LIMIT    "
        "  ,ATTEMPT_INTERVAL = :ATTEMPT_INTERVAL "
        "  ,IS_ENABLED       = :IS_ENABLED       "
        " WHERE                                  "
        "   TASK_TYPE        = :TASK_TYPE        "))
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        return;
    }

    bindValue(q, ":REGULARITY_MONTH ", 1                               );
    bindValue(q, ":REGULARITY_DAY   ", 0                               );
    bindValue(q, ":REGULARITY_HOUR  ", 0                               );
    bindValue(q, ":RUN_DATETIME     ", taskSyncNsiEdit.nextDateTime    );
    bindValue(q, ":NEXT_DATETIME    ", taskSyncNsiEdit.nextDateTime    );
    bindValue(q, ":ATTEMPT_LIMIT    ", taskSyncNsiEdit.attemptLimit    );
    bindValue(q, ":ATTEMPT_INTERVAL ", taskSyncNsiEdit.attemptInterval );
    bindValue(q, ":IS_ENABLED       ", taskSyncNsiEdit.isEnabled       );
    bindValue(q, ":TASK_TYPE        ", getTaskString(TaskType::SyncNsi));

    if (!q.exec())
    {
        writeToJsonMessage(error::update_sql_statement, answer);
        return;
    }

    // Если все нормально, то посылаем пустой ответ
    webCon().send(answer);
}

bool fillTaskInfo(const QUuidEx& taskId, Scheduler::TaskInfo& taskInfo)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                 "
        "   ID                   "
        "  ,USER_ID              "
        "  ,TASK_TYPE            "
        "  ,NAME                 "
        "  ,CREATE_DATE          "
        "  ,RUN_DATETIME         "
        "  ,REGULARITY_DAY       "
        "  ,REGULARITY_HOUR      "
        "  ,REGULARITY_MONTH     "
        "  ,NEXT_DATETIME        "
        "  ,ATTEMPT_LIMIT        "
        "  ,ATTEMPT_COUNTER      "
        "  ,ATTEMPT_INTERVAL     "
        "  ,FORCED               "
        " FROM TASK WHERE ID = ? ", taskId))
    {
        return false;
    }
    if (!q.first())
        return false;

    QSqlRecord r = q.record();

    assignValue(taskInfo.id               , r, "ID               ");
    assignValue(taskInfo.userId           , r, "USER_ID          ");
    assignTaskT(taskInfo.taskType         , r                     );
    assignValue(taskInfo.name             , r, "NAME             ");
    assignValue(taskInfo.description      , r, "DESCRIPTION      ");
    assignValue(taskInfo.created          , r, "CREATE_DATE      ");
    assignValue(taskInfo.runDateTime      , r, "RUN_DATETIME     ");
    assignValue(taskInfo.regularityHour   , r, "REGULARITY_HOUR  ");
    assignValue(taskInfo.regularityDay    , r, "REGULARITY_DAY   ");
    assignValue(taskInfo.regularityMonth  , r, "REGULARITY_MONTH ");
    assignValue(taskInfo.nextDateTime     , r, "NEXT_DATETIME    ");
    assignValue(taskInfo.attempt_counter  , r, "ATTEMPT_COUNTER  ");
    assignValue(taskInfo.attempt_interval , r, "ATTEMPT_INTERVAL ");
    assignValue(taskInfo.attempt_limit    , r, "ATTEMPT_LIMIT    ");
    assignValue(taskInfo.forced           , r, "FORCED           ");

    return true;
}

bool calcNextRun(const QUuidEx& taskId, QDateTime& nextRun)
{
    Scheduler::TaskInfo taskInfo;

    if (!fillTaskInfo(taskId, taskInfo))
        return  false;

    const QDateTime now {QDateTime::currentDateTime()};

    if (now < taskInfo.nextDateTime)
    {
        /*
          В случае планового запуска время завершения задачи 'now' всегда больше чем nextDateTime.
          Это происходит потому, что время планового запуска не меняется до тех пор
          пока не выполнится задача.

          Наступление завершения задачи раньше nextDateTime возможно только в том случае,
          если задача была запущена принудительно и её время завершения случилось раньше
          чем время указанное в nextDateTime.

          При наступления такого кейса, плановое время сдвигать нет необходимости. Так
          как оно может убежать вперед если несколько раз запустить задачу принудительно.
        */
        nextRun = taskInfo.nextDateTime;
        return true;
    }

    /*
      Вычисление единицы дискретности запуска, для добавления
      этого значения ко времени запуска "NEXT_DATETIME".

      Примечания:
        Если у задачи стоит дискретность "час", то остальные модификаторы
        день/месяц - не учитываются. Так как запуск задачи каждый час приведет
        к тому, что задача и так уже будет запускаться каждый день/месяц.

        Если у задачи установлена дискретность "день", то модификатор
        месяц не учитывается, так как задача и так будет запускаться каждый
        месяц.

        Если у задачи установлена дискретность "месяц". То задача будет
        запускаться каждый раз, когда проходит месяц с даты предыдущего
        месяца.
    */
    if (taskInfo.regularityHour > 0)
    {
        // Если дискретность запуска часы - то добавить час.
        int secs = taskInfo.regularityHour * 60*60/*1 час*/;
        nextRun = taskInfo.nextDateTime.addSecs(secs);

        // Если задача в ходе выполнения пропустила очередной запуск,
        // то сдвигаем время на следующий запуск
        while (nextRun <= now)
            nextRun = nextRun.addSecs(secs);
    }
    else if (taskInfo.regularityDay > 0)
    {
        // Если дискретность запуска дни - то добавить сутки.
        nextRun = taskInfo.nextDateTime.addDays(taskInfo.regularityDay);

        // Если задача в ходе выполнения пропустила очередной запуск,
        // то сдвигаем время на следующий запуск
        while (nextRun <= now)
            nextRun = nextRun.addDays(taskInfo.regularityDay);

    }
    else if (taskInfo.regularityMonth > 0)
    {
        // Если дискретность запуска месяц - то добавить месяц.
        nextRun = taskInfo.nextDateTime.addMonths(taskInfo.regularityMonth);

        // Если задача в ходе выполнения пропустила очередной запуск,
        // то сдвигаем время на следующий запуск
        while (nextRun <= now)
            nextRun = nextRun.addMonths(taskInfo.regularityMonth);
    }

    return true;
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
