#pragma once

#include "base_task.h"
#include "database/firebird_driver.h"
#include "commands/nsi.h"
#include "commands/task.h"
#include "commands/report_fed.h"

#include "shared/simple_timer.h"
#include "shared/safe_singleton.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/thread/qthreadex.h"
#include "shared/qt/communication/error_sender.h"
#include "shared/qt/communication/func_invoker.h"
#include "shared/qt/communication/message.h"

#include <QtCore>
#include <QSqlRecord>
#include <atomic>

namespace task {

using namespace std;
using namespace communication;
using namespace communication::transport;

// TODO нормально выполнить описание работы планировщика
/**
  За то когда запускать задачи отвечает 3 параметра
  sql:REGULARITY_DAY - период повторного запуска задачи в днях
  sql:REGULARITY_HOUR - период повтора запуска в часах. Срабатывает после времени в RUN_DATETIME
  и действует в течении суток указанных в NEXT_DATETIME
  sql:REGULARITY_MONTH - период повторного запуска в месяцах
  sql:RUN_DATETIME - главный образом хранит расчетное время после которого запускается задача
  Так же при установке sql:FORCED = 1 и если sql:IS_ENABLED = 1, то задача стартует и отрабатывает вне
  расписания, при этом следующий запуск по расписанию будет сохранен. Если задача с sql:FORCED = 1
  завершилась с кодом(retCode) > 1, то повторных запусков по счетчику attempt не предусмотрено.
  Если запуск задачи завершился с кодом > 1, то она будет повторяться sql:ATTEMPT_LIMIT раз с
  периодом sql:ATTEMPT_INTERVAL(минуты) если все попытки будут исчерпаны, то задача переходит в
  статус sql:EXEC_STATUS = ExecStatus::Failed(4) и далее обрабатываться планировщиком не будет пока
  администратор не устранит причины авариного завершения и не сменит sql:EXEC_STATUS отличный от 4.
  Для задачи с sql:EXEC_STATUS = 4, флаг sql:FORCED = 1 не действует.
*/
class Scheduler : public QThreadEx
{
    struct TaskInfo
    {
        QUuidEx     id;
        QUuidEx     userId;
        TaskType    taskType = {TaskType::Undefined};
        QString     name;
        QString     description;
        QDateTime   runDateTime;
        QDateTime   nextDateTime;
        QDateTime   created;
        qint16      regularityDay    = {0};
        qint16      regularityHour   = {0};
        qint16      regularityMonth  = {0};
        qint16      attempt_limit    = {0};
        qint16      attempt_counter  = {0};
        qint16      attempt_interval = {0};
        int         completed        = {0};
        bool        forced           = {false};

        // Признак периодической задачи
        bool isPeriodic() const {
            return regularityMonth || regularityDay || regularityHour;
        }
    };

public:
    bool init();

    //  Пробуждение потока для перечитвания задач
    void awake();

public slots:
    void message(const communication::Message::Ptr&);

private:
    void run() override final;
    void threadStopEstablished() override;

    bool initSingleTasks();

    void loadTasks();
    void notifyFailTask(const QUuidEx&);

    //--- Обработчики команд ---
    void command_TaskSyncDataNow(const Message::Ptr&);
    void command_TaskSyncDataStop(const Message::Ptr&);
    void command_TaskSyncDataInfo(const Message::Ptr&);
    void command_TaskSyncDataEdit(const Message::Ptr&);

    void command_TaskStartNow(const Message::Ptr&);
    void command_TaskProgress(const Message::Ptr&);
    void command_TaskInterrupt(const Message::Ptr&);

    void command_TaskModelCreate(const Message::Ptr&);
    void command_TaskModelEdit(const Message::Ptr&);
    void command_TaskModelDelete(const Message::Ptr&);
    void command_TaskModelInfo(const Message::Ptr&);
    void command_TaskModelList(const Message::Ptr&);

    void command_TaskScoreCreate(const Message::Ptr&);
    void command_TaskScoreEdit(const Message::Ptr&);
    void command_TaskScoreDelete(const Message::Ptr&);
    void command_TaskScoreInfo(const Message::Ptr&);
    void command_TaskScoreList(const Message::Ptr&);

    // Функции по работе с отчетами находятся в планировщике,
    // так как отчеты тесно связаны с создающей их задачей
    void command_TaskReportCreate(const Message::Ptr&);
    void command_ReportEdit(const Message::Ptr&);
    void command_ReportDelete(const Message::Ptr&);
    void command_ReportInfo(const Message::Ptr&);
    void command_ReportData(const Message::Ptr&);
    void command_ReportList(const Message::Ptr&);

    void command_TaskReportFedCreate(const Message::Ptr&);
    void command_ReportFedEdit(const Message::Ptr&);
    void command_ReportFedDelete(const Message::Ptr&);
    void command_ReportFedInfo(const Message::Ptr&);
    void command_ReportFedList(const Message::Ptr&);
    void command_ReportFedName(const Message::Ptr&);

    void command_TaskReportFedNow(const Message::Ptr&);
    void command_TaskReportFedInfo(const Message::Ptr&);
    void command_TaskReportFedEdit(const Message::Ptr&);

    void command_TaskSyncNsiNow(const Message::Ptr&);
    void command_TaskSyncNsiInfo(const Message::Ptr&);
    void command_TaskSyncNsiEdit(const Message::Ptr&);

    bool taskList(TaskType taskType, const data::TaskFilter& filter,
                  QVector<data::Task>& list);

    bool reportFedList(data::PagingInfo& paging,
                       const data::ReportFedFilter& filter,
                       QVector<data::ReportFed>& reports);

private:
    Q_OBJECT
    DISABLE_DEFAULT_COPY(Scheduler)
    Scheduler();

    Message::List _messages;
    FunctionInvoker _funcInvoker;
    //ErrorSenderFunc _errorSenderWeb;

    BaseTask::List _tasks;

    QMutex _threadLock;
    QWaitCondition _threadCond;
    std::atomic_bool _passTimer;

    friend bool fillTaskInfo(const QUuidEx&, Scheduler::TaskInfo&);
    friend bool calcNextRun(const QUuidEx&, QDateTime&);

    template<typename T, int> friend T& ::safe_singleton();
};

bool fillTaskInfo(const QUuidEx&, Scheduler::TaskInfo&);
bool calcNextRun(const QUuidEx&, QDateTime&);

Scheduler& scheduler();

} // namespace task
