#include "score_calc.h"
#include "task_messages.h"
#include "functions.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "commands/error.h"
#include "commands/score.h"
#include "commands/send_score.h"
#include "shared/qt/config/config.h"
#include "xgboost/options.h"
#include "xgboost/autofree.h"
#include "3rdparty/xgboost/include/xgboost/learner.h"
#include "3rdparty/xgboost/include/xgboost/c_api.h"

#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"

#include <unistd.h>
#include <chrono>
#include <string>
#include <QtSql>
#include <QDateTime>
#include <QDate>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "ScoreCalc")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "ScoreCalc")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "ScoreCalc")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "ScoreCalc")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "ScoreCalc")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "ScoreCalc")

namespace task {

using namespace db::firebird;
using namespace sql;

ScoreCalc::ScoreCalc(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTaskThread(TaskType::ScoreCalc, taskId, userId)
{}

void ScoreCalc::run()
{
    #define TASK_NAME u8"Применение"

    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    _interrupted = false;

    log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME);

    if (!_baseInit)
    {
        _retInfo = {RetInfo::Error::Init};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_INIT);
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    _progressCurrent = 0;
    _progressTotal = 100;

    _binaryDump = false;
    config::base().getValue("common.binary_dump", _binaryDump);

    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    try
    {
        // Получить минимальные сведения о задаче, необходимые для её выполнения
        _retInfo = initPeriod();
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_INIT)

        // Выполнение синхронизации данных
        _retInfo = syncCheck(this);
        TASK_CHECK_NEEDSYNC_AND_ERROR(TASK_NAME, TASK_ERR_SYNC)

        // Выполнить применение модели
        _retInfo = makeApply(ScoreType::MEE);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_APPLY_MEE)

        // Выполнить применение модели
        _retInfo = makeApply(ScoreType::EKMP);
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_APPLY_EKMP)

        // Добавить в базу применение модели
        _retInfo = insertScore();
        TASK_CHECK_INT_AND_ERROR(TASK_NAME, TASK_ERR_SCORE_SAVE)
    }
    catch (const std::exception& e)
    {
        _retInfo = {RetInfo::Error::Undef};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_UNHANDLED, TASK_NAME, e.what());
        TASK_CLEAN_AND_RETURN(TASK_NAME);
    }

    data::TaskContentCreate taskContentCreate;
    taskContentCreate.taskType = {TaskType::ScoreCalc};
    taskContentCreate.taskId = id();
    taskContentCreate.userId = userId();
    taskContentCreate.contentId = contentId();
    taskContentCreate.isPeriodic = isPeriodic();

    m = createJsonMessage(taskContentCreate, Message::Type::Event);
    webCon().send(m);

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME);

    #undef TASK_NAME
}

const QUuidEx& ScoreCalc::contentId() const
{
    return _scoreId;
}

RetInfo ScoreCalc::initPeriod()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                "
        "   PERIOD_BEGIN        "
        "  ,PERIOD_END          "
        "  ,MODEL_ID            "
        "  ,PARENT_ID           "
        "  ,REL_PERIOD_DURATION "
        " FROM                  "
        "   TASK                "
        " WHERE                 "
        "   ID = ?              ", id() ))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        return {RetInfo::Error::Sql};
    }

    QSqlRecord r = q.record();

    assignValue(_period.begin,      r, "PERIOD_BEGIN"        );
    assignValue(_period.end,        r, "PERIOD_END"          );
    assignValue(_baseModelId,       r, "MODEL_ID"            );
  //assignValue(_baseTaskId,        r, "PARENT_ID"           );
    assignValue(_relPeriodDuration, r, "REL_PERIOD_DURATION" );


    // Если задача периодическая, интервал необходимо вычислить исходя
    // из текущего месяца.
    if (isPeriodic())
        initScorePeriod(_period);

    // Если задача периодическая, то необходимо заполнить поле базовой модели
    if (isPeriodic())
    {
        if (!sql::exec(q,
            " SELECT FIRST 1    "
            "   ID              "
            " FROM              "
            "   TASK            "
            " WHERE             "
            "   TASK_TYPE = ?   "
            "   AND             "
            "   IS_PERIODIC = 1 "
            " ORDER BY          "
            "   CREATE_DATE ASC ",
            getTaskString(TaskType::LearnModel)))
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::Sql};
        }
        if (!q.first())
        {
            log_error_m << TASK_EVENT_LOG(TASK_MSG_REG_NOTASK, id());
            return {RetInfo::Error::General};
        }

        assignValue(_baseTaskId, q.record(), "ID");

        if (!sql::exec(q,
            " SELECT FIRST 1 ID FROM MODEL                "
            " WHERE TASK_ID = ? ORDER BY CREATE_DATE DESC ", _baseTaskId))
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::Sql};
        }
        if (!q.first())
        {
            log_error_m << TASK_EVENT_LOG(TASK_MSG_REG_NOMODEL, _baseTaskId);
            return {RetInfo::Error::NoModel};
        }
        assignValue(_baseModelId, q.record(), "ID");
    }

    return {RetInfo::Success};
}

RetInfo ScoreCalc::insertScore()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    QString query =
        " SELECT                 "
        "   ID                   "
        "  ,USER_ID              "
        "  ,PARENT_ID            "
        "  ,TASK_TYPE            "
        "  ,NAME                 "
        "  ,DESCRIPTION          "
        "  ,RUN_DATETIME         "
        "  ,REGULARITY_MONTH     "
        "  ,REGULARITY_DAY       "
        "  ,REGULARITY_HOUR      "
        "  ,ATTEMPT_LIMIT        "
        "  ,ATTEMPT_COUNTER      "
        "  ,ATTEMPT_INTERVAL     "
        "  ,IS_ENABLED           "
        "  ,IS_PUBLIC            "
        "  ,PERIOD_BEGIN         "
        "  ,PERIOD_END           "
        "  ,REL_PERIOD_DURATION  "
        "  ,MODEL_ID             "
        " FROM TASK              "
        "   WHERE ID = :ID       ";

    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":ID" , id());

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        return {RetInfo::Error::Sql};
    }

    QSqlRecord r = q.record();

    QString name, decr;
    bool isPublic = false;

    assignValue(name,          r, "NAME"        );
    assignValue(decr,          r, "DESCRIPTION" );
    assignValue(isPublic,      r, "IS_PUBLIC"   );
    // Идентификатор модели уже найден в "Инициализации", поэтому
    // на данном этапе корректно заполнено.
    //assignValue(_baseModelId,  r, "MODEL_ID"    );
    // Перезаписывать нельзя, так как может потеряться введенное ранее
    // правильное значение.
    //assignValue(_baseTaskId,   r, "PARENT_ID"   );

    if (!sql::exec(q, "SELECT COUNT(*) FROM MODEL WHERE ID = ?", _baseModelId))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    q.first();
    if (q.value(0).toInt() == 0)
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_NODATA);
        return {RetInfo::Error::Sql};
    }

    QDateTime create {QDateTime::currentDateTime()};

    QString fields =
        "  ID                "
        ", TASK_ID           "
        ", USER_ID           "
        ", MODEL_ID          "
        ", NAME              "
        ", DESCRIPTION       "
        ", CREATE_DATE       "
        ", PERIOD_BEGIN      "
        ", PERIOD_END        "
      //", PERIOD_BEGIN_REAL "
      //", PERIOD_END_REAL   "
        ", IS_PUBLIC         "
        ", OUTDATED_ROWS     "
        ", SEND_STATUS       ";

    query = sql::INSERT_INTO("SCORE", fields);
    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    // Если задача оценок разовая, то данная оценка получает статус "Создано".
    // Данный статус не подразумевает автоматическую отправку.
    // Для периодической задачи оценок устанавливает стауст "Нужна отправка".
    // Оценка с таким статусом будет отправлена автоматически.
    SendScoreStatus sendScoreStatus = (isPeriodic())
                                      ? SendScoreStatus::NeedSend
                                      : SendScoreStatus::Created;

    bindValue(q, ":ID                " , contentId()       );
    bindValue(q, ":TASK_ID           " , id()              );
    bindValue(q, ":USER_ID           " , userId()          );
    bindValue(q, ":MODEL_ID          " , _baseModelId      );
    bindValue(q, ":NAME              " , name              );
    bindValue(q, ":DESCRIPTION       " , decr              );
    bindValue(q, ":CREATE_DATE       " , create            );
    bindValue(q, ":PERIOD_BEGIN      " , _period.begin     );
    bindValue(q, ":PERIOD_END        " , _period.end       );
  //bindValue(q, ":PERIOD_BEGIN_REAL " , _periodReal.begin );
  //bindValue(q, ":PERIOD_END_REAL   " , _periodReal.end   );
    bindValue(q, ":IS_PUBLIC         " , isPublic          );
    bindValue(q, ":OUTDATED_ROWS     " , 0                 );
    bindValue(q, ":SEND_STATUS       " , sendScoreStatus   );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!sql::exec(q, "DELETE FROM SYNC_PLANNING WHERE TASK_ID = ?", id()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        return {RetInfo::Error::Sql};
    }

    return {RetInfo::Success};
}

RetInfo ScoreCalc::fillArrays(Ret2DArray& data, ScoreType type)
{
    _rowIndexer.clear();

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QString sql =
        " SELECT IIF(EXISTS(            "
        "   SELECT * FROM SYNC_DATA     "
        "   WHERE                       "
        "     DATE_OT_PER >= :PERIOD_BEGIN "
        "     AND                       "
        "     DATE_OT_PER <= :PERIOD_END   "
        "     AND                       "
        "     %1 IS NOT NULL            "
        "   ), 1, 0)                    "
        "   AS DATA_EXISTS              "
        " FROM                          "
        "   RDB$DATABASE                ";

    sql = sql.arg((type == ScoreType::MEE) ? "MEE" : "EKMP");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":PERIOD_BEGIN", _period.begin );
    bindValue(q, ":PERIOD_END",   _period.end   );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    q.first();
    bool dataExists = false;
    assignValue(dataExists, q.record(), "DATA_EXISTS");

    if (!dataExists)
    {
        const char* score = (type == ScoreType::MEE) ? u8"МЭЭ" : u8"ЭКМП";
        log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_CALC_DATA, score);
        return {RetInfo::Error::NoData};
    }

    // Временной периоды для вычисления количества месяцев
    QDate periodBegin = firstDay(_period.begin);
    QDate periodEnd = lastDay(_period.end);

    // В лоб считаем количество месяцев
    int monthCount = 0;
    while (periodBegin <= periodEnd)
    {
        ++monthCount;
        periodBegin = periodBegin.addMonths(1);
    }
    log_debug_m << "Month count: " << monthCount;

    sql = "SELECT REC_COUNT FROM GET_RECORDS_COUNT(:PERIOD_BEGIN, :PERIOD_END, :MEE, :EKMP)";

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":PERIOD_BEGIN ", _period.begin );
    bindValue(q, ":PERIOD_END   ", _period.end   );
    if (type == ScoreType::MEE)
    {
        bindValue(q, ":MEE ", -1 );
        bindValue(q, ":EKMP ", 2 );
    }
    else
    {
        bindValue(q, ":MEE ", 2 );
        bindValue(q, ":EKMP ", -1 );
    }

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    q.first();
    int recCount = q.value(0).toInt();


    if (alog::logger().level() >= alog::Level::Debug)
    {
        log_debug_m << "Filling array for Score: " << contentId()
                    << ", baseModel: " << _baseModelId
                    << ", baseTask:  " << _baseTaskId
                    << ", begin: "     << _period.begin.date()
                    << ", end: "       << _period.end.date()
                    << ", recCount: "  << recCount;
    }

    int step = recCount / 9;
    step = (step == 0) ? 1 : step;

    _progressCurrent = (type == ScoreType::MEE) ? 1 : 51;
    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    QString colNames =
        " MSK_OT_ORD, VID_MP_ORD, USL_OK_ORD, PROFIL_ORD, MKB1_ORD, MKB2_ORD, CODE_USL_ORD,         "
        " CODE_MD_ORD, KOL_USL, KOL_FACT, ISH_MOV_ORD, RES_GOSP_ORD, TARIF_B, TARIF_S, VID_TR_ORD,  "
        " EXTR_ORD, SPEC_MD_ORD, DOMC_TYPE_ORD, CODE_LPU_ORD, VID_SF_ORD, PERSCODE_ORD, VID_KOEFF,  "
        " USL_TMP, SEX, FOR_POM, P_CEL_ORD, DN_ORD, MKB0_ORD, DS_ONK, MKB1SS_ORD, MKB1S_ORD,        "
        " DATE_NPR_OF_WEEK, DATE_NPR_MONTH, DATE_NPR_DIF, DATE_IN_OF_WEEK, DATE_IN_MONTH,           "
        " DATE_OUT_OF_WEEK, DATE_OUT_MONTH, DATE_DIFF, OLD_VAR, KOL_USL_per_Day, KOL_FACT_per_Day,  "
        " DATE_NPR_DIF_RATE,KOL_USLKOL_FACT, TARIF_BTARIF_D, TARIF_BSUM_RUB, KOL_USLKOL_FACT_p,     "
        " TARIF_BSUM_RUB_p, KOL_USL_rate, KOL_FACT_rate, DIF_OT, AGGR_LPU_KOL_DEF,  AGGR_LPU_COUNT, "
        " AGGR_LPU_RATE, AGGR_MSK_OT_KOL_DEF, AGGR_MSK_OT_COUNT, AGGR_MSK_OT_RATE, OKATO_INS_ORD, CODE_MSK_ORD, SUM_PER_DAY ";

    sql =
        " SELECT                                                          "
        "   GKEY,                                                         ";
    for (const QString& column : colNames.split(","))
    {
        sql += column.trimmed() + ",";
    }
    sql += "   KOL_DEF                                                    "
           " FROM                                                         "
           "   GET_APPLY_ROWS(:MODEL_ID, :PERIOD_BEGIN, :PERIOD_END, %1); ";

    sql = sql.arg((type == ScoreType::MEE) ? "0" : "1");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":MODEL_ID     ", _baseModelId );
    bindValue(q, ":PERIOD_BEGIN ", _period.begin.date());
    bindValue(q, ":PERIOD_END   ", _period.end.date()  );

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    const int colCount = colNames.split(",").count();
    QList<QVector<float>> records;
    int limit = (type == ScoreType::MEE) ? 10 : 60;

    while (q.next())
    {
        CHECK_QTHREADEX_STOP

        int iCol = 0;
        QSqlRecord r = q.record();

        QVector<float> rec;
        rec.resize(colCount);

        QUuidEx rowId;
        assignValue(rowId, r, "GKEY");

        for (const QString& column : colNames.split(","))
        {
            rec[iCol++] = toFloat(r, column.trimmed());
        }

#ifdef TEST_MODE
        // Логгирование для проверки данных с БД
        if (alog::logger().level() >= alog::Level::Debug)
        {
            alog::Line logLine = log_debug_m << "Values: ";
            for (int i = 0; i < iCol - 1; ++i)
                logLine << rec[i] << "; ";
        }
        // Логгирование для проверки данных с БД
        if (alog::logger().level() >= alog::Level::Debug)
        {
            alog::Line logLine = log_debug_m << "DefValues: ";
            logLine << recDef[0] << "; ";
        }
#endif
        records.append(rec);
        _rowIndexer.append(rowId);

        if (records.count() % step == 0)
        {
            _progressCurrent = (_progressCurrent < limit) ? (_progressCurrent + 1) : limit;
            m = createJsonMessage(progress(), Message::Type::Event);
            webCon().send(m);
        }
    } // while (q.next())

    if (threadStop())
        return {RetInfo::Error::Interrupt};

    if (records.isEmpty())
    {
        const char* score = (type == ScoreType::MEE) ? u8"МЭЭ" : u8"ЭКМП";
        log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_CALC_DATA, score);
        return {RetInfo::Error::NoData};
    }

    data.alloc(records.count(), colCount);

    for (int i = 0; i < records.count(); ++i)
    {
        float* r1 = (float*)records[i].constData();
        float* r2 = data.ptr() + i * data.columns();
        for (int j = 0; j < colCount; ++j)
            *r2++ = *r1++;
    }

    // Сохранение дампа бинарных данных в файл, имя которого имеет формат
    // [Идентификатор Модели]_[Тип Модели].dump
    if (_binaryDump)
    {
        QString modelId = _scoreId.toString();
        modelId = modelId.remove(0, 1);
        modelId = modelId.remove(modelId.size()-1, 1);

        QString modelType = (type == ScoreType::MEE) ? "mee" : "ekmp";
        QString dataType = "score";
        QString filename = QString(modelId + "_" + modelType + "_" + dataType + ".dump");

        QFile file(filename);
        file.open(QIODevice::WriteOnly);
        QTextStream filestream(&file);

        for (int i = 0; i < records[0].count(); ++i)
        {
            filestream << colNames.split(",").toVector()[i].trimmed() << ";";
        }
        filestream << "KOL_DEF" << endl;

        for (int i = 0; i< records.count(); ++i)
        {
            const QVector<float>& line = records[i];
            for (float item : line)
                filestream << item << ";";

            filestream << "-1" << endl;
        }

        file.close();
    }

    return {RetInfo::Success};
}

RetInfo ScoreCalc::makeApply(ScoreType type)
{
    int res;
    RetInfo retInfo = {RetInfo::Success};

    Ret2DArray inData;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!q.prepare(" SELECT RAW_DATA FROM MODEL_DATA "
                   " WHERE ID = :ID AND KIND = :KIND"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    bindValue(q, ":ID"   , _baseModelId);
    bindValue(q, ":KIND" , (type == ScoreType::MEE) ? 0 : 1);

    if (!q.exec())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_NOMODEL);
        return {RetInfo::Error::NoModel, RetInfo::Critical::Yes};
    }

    QByteArray ba;
    assignValue(ba, q.record(), "RAW_DATA");

    if (!(retInfo = fillArrays(inData, type)))
        return retInfo;

    if (inData.rows() == 0)
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_NODATA);
        return {RetInfo::Error::NoData, RetInfo::Critical::No};
    }

    _progressCurrent = (type == ScoreType::MEE) ? 10 : 60;
    Message::Ptr m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    // create the booster and load some parameters
    BoosterHandle h_booster;

    res = XGBoosterCreate(nullptr, 0, &h_booster);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterCreate()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }
    XGBOOST_HANDLE_AUTO_FREE(h_booster)

    res = XGBoosterLoadModelFromBuffer(h_booster, ba.data(), ba.size());
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterLoadModelFromBuffer()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }

    DMatrixHandle inMatrix;
    res = XGDMatrixCreateFromMat(inData.ptr(), inData.rows(), inData.columns(), 0, &inMatrix);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGDMatrixCreateFromMat()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }
    XGBOOST_MATRIX_AUTO_FREE(inMatrix)

    bst_ulong out_len;
    const float* f;
    res = XGBoosterPredict(h_booster, inMatrix, 0, 0, &out_len, &f);
    if (res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_FUNC_CALL, "XGBoosterPredict()");
        return {RetInfo::Error::Xgboost, RetInfo::Critical::Yes};
    }

    // Проверка предиктов на значения больше 1
    for (int i = 0; i < int(out_len); ++i)
    {
        double score = (double)f[i];
        if (score > 1)
        {
             log_debug2_m << "Predict incorrect for Score: " << contentId()
                          << " because it more than 1";
        }
    }

    QString query;
    if (type == ScoreType::MEE)
    {
        query = " UPDATE OR INSERT INTO SCORE_DATA "
                " ( SCORE_ID, GKEY, MEE )          "
                " VALUES                           "
                " ( :SCORE_ID, :GKEY, :MEE )       "
                " MATCHING (SCORE_ID, GKEY)        ";

    }
    else
    {
        query = " UPDATE OR INSERT INTO SCORE_DATA "
                " ( SCORE_ID, GKEY, EKMP )         "
                " VALUES                           "
                " ( :SCORE_ID, :GKEY, :EKMP )      "
                " MATCHING (SCORE_ID, GKEY)        ";
    }

    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    int step = out_len / 40.;
    step = (step == 0) ? 1 : step;
    int limit = (type == ScoreType::MEE) ? 50 : 100;

    for (int i = 0; i < int(out_len); ++i)
    {
        CHECK_QTHREADEX_STOP

        bindValue(q, ":SCORE_ID",   contentId());
        bindValue(q, ":GKEY    ",   _rowIndexer.at(i));

        double score = (double)f[i];
        if (score > 1)
        {
             log_debug2_m << "Update for predict incorrect for Score: " << contentId()
                          << " because it more than 1. "
                          << " Service id: " << _rowIndexer.at(i);
        }

        if (type == ScoreType::MEE)
        {
            bindValue(q, ":MEE",    score);
//            log_debug2_m << "Score id: " << contentId()
//                         << " MEE Service Identificator: " << _rowIndexer.at(i)
//                         << " has predict: " << score;
        }
        else
        {
            bindValue(q, ":EKMP",   score);
//            log_debug2_m << "Score id: " << contentId()
//                         << " EKMP Service Identificator: " << _rowIndexer.at(i)
//                         << " has predict: " << score;
        }

        if (!q.exec())
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::Sql};
        }

        if ((i + 1) % step == 0)
        {
            _progressCurrent = (_progressCurrent < limit) ? ++_progressCurrent : limit;
            m = createJsonMessage(progress(), Message::Type::Event);
            webCon().send(m);
        }
    }
    if (threadStop())
        return {RetInfo::Error::Interrupt};

    query =
    " UPDATE SCORE_DATA SC                                                        "
    " SET                                                                         "
    "   CODE_LPU = (SELECT SY.CODE_LPU FROM SYNC_DATA SY WHERE SC.GKEY = SY.GKEY) "
    "  ,MKB1     = (SELECT SY.MKB1     FROM SYNC_DATA SY WHERE SC.GKEY = SY.GKEY) "
    "  ,PROFIL   = (SELECT SY.PROFIL   FROM SYNC_DATA SY WHERE SC.GKEY = SY.GKEY) "
    "  ,VID_MP   = (SELECT SY.VID_MP   FROM SYNC_DATA SY WHERE SC.GKEY = SY.GKEY) "
    " WHERE                                                                       "
    "   EXISTS (SELECT 1 FROM SYNC_DATA SY WHERE SC.GKEY = SY.GKEY)               "
    "   AND                                                                       "
    "   SC.SCORE_ID = ?                                                           ";

    if (!sql::exec(q, query, _scoreId))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        return {RetInfo::Error::Sql};
    }

    _progressCurrent = (type == ScoreType::MEE) ? 50 : 100;
    m = createJsonMessage(progress(), Message::Type::Event);
    webCon().send(m);

    return {RetInfo::Success};
}

bool ScoreCalc::loadXgbOptions(QMap<QString, QString>& options)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "SELECT NAME, CONTENT FROM MODEL_XGB WHERE ID = ?", _baseModelId))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return false;
    }

    while (q.next())
    {
        QSqlRecord r = q.record();
        QString name, value;
        assignValue(name,  r, "NAME"  );
        assignValue(value, r, "CONTENT" );
        options[name] = value;
    }

    return true;
}

void ScoreCalc::sanitize()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q, "DELETE FROM SCORE_DATA WHERE SCORE_ID = ?", contentId()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
    }
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
