#include "send_score.h"
#include "functions.h"
#include "task_messages.h"
#include "commands/commands.h"
#include "commands/score.h"
#include "commands/send_score.h"
#include "database/connect.h"
#include "database/sql_func.h"

#include "shared/defmac.h"
#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "SendScore")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "SendScore")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "SendScore")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "SendScore")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "SendScore")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "SendScore")

/*
#define SEND_SCORE_LOG(MSG, ...) \
    EventLog(MSG, ##__VA_ARGS__) << EventUser(_userId)
*/

namespace task {

using namespace communication::transport;
using namespace db::firebird;
using namespace sql;

SendScore& sendScore()
{
    return ::safe_singleton<SendScore>();
}

SendScore::SendScore() : SyncTransfer()
{
    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &SyncTransfer::command_Dummy, (SyncTransfer*)this);

    FUNC_REGISTRATION(SupportCommand)
    FUNC_REGISTRATION(SendScore)
    FUNC_REGISTRATION(SendScoreCheck)
    FUNC_REGISTRATION(SendScoreCrcOk)

    #undef FUNC_REGISTRATION
}

bool SendScore::init()
{
    //if (!databaseInit(_db))
    //    return false;

    return true;
}

void SendScore::run()
{
    log_info_m << "Started";

    _threadId = trd::gettid();

    while (true)
    {
        CHECK_QTHREADEX_STOP

        // Без подключения к ФОМС процесс выгрузки скоров не начинается
        if (!fomsCon().isConnected())
        {
            sleep(1);
            continue;
        }
        if (!fomsCon().isAuthorized())
        {
            sleep(1);
            continue;
        }

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        if (!sql::exec(q,
            " SELECT FIRST 1                          "
            "   S.ID           AS SCORE_ID            "
            "  ,S.NAME         AS SCORE_NAME          "
            "  ,S.DESCRIPTION  AS SCORE_DESCR         "
            "  ,S.CREATE_DATE  AS CREATE_DATE         "
            "  ,S.PERIOD_BEGIN AS SCORE_PERIOD_BEGIN  "
            "  ,S.PERIOD_END   AS SCORE_PERIOD_END    "
            "  ,M.ID           AS MODEL_ID            "
            "  ,M.NAME         AS MODEL_NAME          "
            "  ,M.DESCRIPTION  AS MODEL_DESCR         "
            "  ,M.PERIOD_BEGIN AS MODEL_PERIOD_BEGIN  "
            "  ,M.PERIOD_END   AS MODEL_PERIOD_END    "
            "  ,U.ID           AS USER_ID             "
            "  ,U.NAME         AS USER_NAME           "
            " FROM                                    "
            "   SCORE S                               "
            " LEFT JOIN                               "
            "   MODEL M ON M.ID = S.MODEL_ID          "
            " LEFT JOIN                               "
            "   USERS U ON U.ID = S.USER_ID           "
            " WHERE                                   "
            "   M.ID IS NOT NULL                      "
            "   AND                                   "
            "   S.SEND_STATUS = ?                     "
            "   AND                                   "
            "   S.IS_ARCHIVE = 0                      ",
            SendScoreStatus::NeedSend))
        {
            log_error_m << EventLog(TASK_ERR_SQL_EXEC);
            QMutexLocker locker(&_threadLock); (void) locker;
            _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            continue;
        }
        if (!q.first())
        {
            QMutexLocker locker(&_threadLock); (void) locker;
            _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            continue;
        }

        QSqlRecord r = q.record();

        assignValue(_scoreId,           r, "SCORE_ID           ");
        assignValue(_scoreName,         r, "SCORE_NAME         ");
        assignValue(_scoreDescr,        r, "SCORE_DESCR        ");
        assignValue(_createDate,        r, "CREATE_DATE        ");
        assignValue(_scorePeriod.begin, r, "SCORE_PERIOD_BEGIN ");
        assignValue(_scorePeriod.end,   r, "SCORE_PERIOD_END   ");
        assignValue(_modelId,           r, "MODEL_ID           ");
        assignValue(_modelName,         r, "MODEL_NAME         ");
        assignValue(_modelDescr,        r, "MODEL_DESCR        ");
        assignValue(_modelPeriod.begin, r, "MODEL_PERIOD_BEGIN ");
        assignValue(_modelPeriod.end,   r, "MODEL_PERIOD_END   ");
        assignValue(_userId,            r, "USER_ID            ");
        assignValue(_userName,          r, "USER_NAME          ");

        // Отправка данных скора с идентификатором scoreId.
        if (!sendScore())
        {
            log_error_m << EventLog(u8"Ошибка экспорта применения: %1", _scoreId);
            continue;
        }

        // Присвоить статус "Отправлено" для скора.
        if (!sql::exec(q,
            "UPDATE SCORE SET SEND_STATUS = ? WHERE ID = ?",
            SendScoreStatus::IsSent, _scoreId))
        {
            log_error_m << EventLog(TASK_ERR_SQL_EXEC);
            continue;
        }
    } // while (true)

//    if (_db->isOpen())
//    {
//        log_verbose << "Close send-score connect to database";
//        _db->close();
//    }

    log_info_m << "Stopped";
}

bool SendScore::sendScore()
{
    log_verbose_m << EventLog(u8"Старт экспорта применения: %1", _scoreId);

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << EventLog(TASK_ERR_SQL_BEGIN);
        return false;
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q,
        " SELECT COUNT(*) FROM (                "
        "   SELECT                              "
        "     SY.IDSL                           "
        "   FROM                                "
        "     SCORE_DATA SC                     "
        "   LEFT JOIN                           "
        "     SYNC_DATA SY ON SY.GKEY = SC.GKEY "
        "   WHERE                               "
        "     SC.SCORE_ID = ?                   "
        "   GROUP BY                            "
        "     SY.IDSL                           "
        " ) T                                   ", _scoreId ))
    {
        log_error_m << EventLog(TASK_ERR_SQL_EXEC);
        return false;
    }

    q.first();
    int recItem = 0;
    int recCount = q.value(0).toInt();
    int step = recCount / 20;

    log_verbose_m << EventLog(u8"Всего записей для экспорта: %1. Применение: %2",
                              recCount, _scoreId);

    if (!sql::exec(q,
        " SELECT                              "
        "   SY.IDSL      AS IDSL              "
        "  ,MAX(SC.MEE)  AS MEE               "
        "  ,MAX(SC.EKMP) AS EKMP              "
        " FROM                                "
        "   SCORE_DATA SC                     "
        " LEFT JOIN                           "
        "   SYNC_DATA SY ON SY.GKEY = SC.GKEY "
        " WHERE                               "
        "   SC.SCORE_ID = ?                   "
        " GROUP BY                            "
        "   SY.IDSL                           ", _scoreId ))
    {
        log_error_m << EventLog(TASK_ERR_SQL_EXEC);
        return false;
    }

    quint32 rowNumber = 0;
    Message::Ptr message;
    Message::Ptr answer;

    data::SendScore sendScore;
    sendScore.id = _scoreId;
    sendScore.name = _scoreName;
    sendScore.descript = _scoreDescr;
    sendScore.createDate = _createDate;
    sendScore.period = _scorePeriod;
    sendScore.userId = _userId;
    sendScore.userName = _userName;
    sendScore.modelId = _modelId;
    sendScore.modelName = _modelName;
    sendScore.modelDescr = _modelDescr;
    sendScore.modelPeriod = _modelPeriod;
    sendScore.isFirst = true;

    // Извлечение всех записей, с отправкой через _sendSize строк данных.
    while (q.next())
    {
        if (threadStop())
            return false;

        QSqlRecord r = q.record();
        data::SendScoreItem sendScoreItem;

        assignValue(sendScoreItem.IDSL, r, "IDSL" );
        assignValue(sendScoreItem.MEE,  r, "MEE"  );
        assignValue(sendScoreItem.EKMP, r, "EKMP" );

        sendScore.items.append(sendScoreItem);
        ++rowNumber;

        if (rowNumber == _sendSize)
        {
            message = createJsonMessage(sendScore);
            message->setPriority(Message::Priority::Low);

            if (!send(message).isSuccess())
                return false;

            // В случае получения некорректного ответа прервать экспорт, и
            // повторить попытку
            if (!waitMessage(answer, command::SendScore).isSuccess())
                return false;

            sendScore.items.clear();
            sendScore.isFirst = false;
            rowNumber = 0;
        }

        if (++recItem % step == 0)
        {
            int val = (double(recItem) / recCount) * 100;
            log_verbose_m << EventLog(u8"Экспортировано записей: %1%. Применение: %2",
                                      val, _scoreId);
        }
    }

    sendScore.isLast = true;

    message = createJsonMessage(sendScore);

    message->setPriority(Message::Priority::Low);

    if (!send(message).isSuccess())
        return false;

    // В случае получения некорректного ответа прервать экспорт, и
    // повторить попытку
    if (!waitMessage(answer, command::SendScore).isSuccess())
        return false;

    // Отправить команду на ФОМС сервер для подсчета контрольной суммы оценки,
    // для который выполнялся экспорт.
    data::SendScoreCheck sendScoreCheck;
    sendScoreCheck.id = _scoreId;

    message = createJsonMessage(sendScoreCheck);

    if (!send(message).isSuccess())
        return false;

    if (!waitMessage(answer, command::SendScoreCheck).isSuccess())
        return false;

    SResult res = readFromMessage(answer, sendScoreCheck);
    if (!res)
    {
        log_error_m << EventLog(TASK_ERR_ANSWER_PARSER);
        return false;
    }

    // Подсчет CRC экспортированных оценок
    quint64 crc, count;
    if (!calcCrc(transact, crc, count))
    {
        log_error_m << EventLog(u8"Ошибка вычисления контрольной суммы "
                                u8"для применения: %1", _scoreId);
        return false;
    }

    if (sendScoreCheck.crc != crc)
    {
        log_error_m << EventLog(u8"Неверная контрольная сумма "
                                u8"для применения: %1", _scoreId);
        log_debug_m << "CRC calc: " << crc << ". CRC from FOMS: " << sendScoreCheck.crc
                    << ". ScoreId: " << _scoreId
                    << ". CRC record count: " << count;

        sendCrcStatus(false);

        return false;
    }

    sendCrcStatus(true);

    log_debug_m << "Scores CRC: " << crc;

    log_verbose_m << EventLog(u8"Окончание экспорта применения: %1", _scoreId);
    return true;
}

bool SendScore::sendCrcStatus(bool status)
{
    // Отправка подтверждения о том, что CRC проверка успешна, с пердварительной
    // проверкой того, что команда SendScoreCrcOk поддерживается ФОМС сервером.
    data::SupportCommand supportCommand;
    supportCommand.commandId = command::SendScoreCrcOk;
    Message::Ptr message = createJsonMessage(supportCommand);

    if (!send(message).isSuccess())
        return false;

    Message::Ptr answer;

    if (!waitMessage(answer, command::SupportCommand).isSuccess())
        return false;

    SResult res = readFromMessage(answer, supportCommand);

    if (!res)
    {
        log_error_m << EventLog(TASK_ERR_ANSWER_PARSER);
        return false;
    }

    if (supportCommand.isSupport)
    {
        data::SendScoreCrcOk sendScoreCrcOk;
        sendScoreCrcOk.id = _scoreId;
        sendScoreCrcOk.crcOk = status;

        message = createJsonMessage(sendScoreCrcOk);

        if (!send(message).isSuccess())
            return false;

        if (!waitMessage(answer, command::SendScoreCrcOk).isSuccess())
            return false;
    }

    return true;
}

bool SendScore::calcCrc(Transaction::Ptr transact, quint64& crc, quint64& count)
{
    crc = 0;
    count = 0;
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!sql::exec(q,
        " SELECT                              "
        "   SY.IDSL      AS IDSL              "
        "  ,MAX(SC.MEE)  AS MEE               "
        "  ,MAX(SC.EKMP) AS EKMP              "
        " FROM                                "
        "   SCORE_DATA SC                     "
        " LEFT JOIN                           "
        "   SYNC_DATA SY ON SY.GKEY = SC.GKEY "
        " WHERE                               "
        "   SC.SCORE_ID = ?                   "
        " GROUP BY                            "
        "   SY.IDSL                           "
        " ORDER BY                            "
        "   SY.IDSL ASC                       ",
        _scoreId ))
    {
        log_error_m << EventLog(TASK_ERR_SQL_EXEC);
        return false;
    }

    while (q.next())
    {
        QUuidEx idsl;
        assignValue(idsl,  q.record(), "IDSL");

        quint64 p1 = ((quint64*) &idsl)[0];
        quint64 p2 = ((quint64*) &idsl)[1];
        quint64 pCrc = (p1 ^ p2);

        crc = crc ^ pCrc;

        ++count;
    }
    return true;
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
