#include "sync_base.h"
#include "functions.h"
#include "event_log.h"
#include "scheduler.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "commands/commands.h"
#include "commands/errors.h"
#include "tasks/utils.h"

#include "shared/steady_timer.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/break_point.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/config/config.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"

#include <unistd.h>
#include <chrono>
#include <string>
#include <QtSql>
#include <QCryptographicHash>

#define log_error_m   alog::logger().error_f  (__FILE__, LOGGER_FUNC_NAME, __LINE__, "TaskSync")
#define log_warn_m    alog::logger().warn_f   (__FILE__, LOGGER_FUNC_NAME, __LINE__, "TaskSync")
#define log_info_m    alog::logger().info_f   (__FILE__, LOGGER_FUNC_NAME, __LINE__, "TaskSync")
#define log_verbose_m alog::logger().verbose_f(__FILE__, LOGGER_FUNC_NAME, __LINE__, "TaskSync")
#define log_debug_m   alog::logger().debug_f  (__FILE__, LOGGER_FUNC_NAME, __LINE__, "TaskSync")
#define log_debug2_m  alog::logger().debug2_f (__FILE__, LOGGER_FUNC_NAME, __LINE__, "TaskSync")

namespace task {

using namespace db::firebird;
using namespace sql;

SyncBase::SyncBase(data::TaskType type, const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTask(type, taskId, userId)
{
    chk_connect_d(fomsCon(), SIGNAL(message(communication::Message::Ptr)),
                  this, SLOT(message(communication::Message::Ptr)))

    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &SyncBase::command_Dummy, this);

    FUNC_REGISTRATION(RequestSyncData)
    FUNC_REGISTRATION(GetSyncData)
    FUNC_REGISTRATION(GetSyncData2)
    FUNC_REGISTRATION(BreakDataTransfer)
    FUNC_REGISTRATION(SyncDataCheck)

    #undef FUNC_REGISTRATION
}

bool SyncBase::isEmptyContains(const data::DataItem& dataItem)
{
    return true;
//    return dataItem.VID_MP.isEmpty()
//           || dataItem.USL_OK.isEmpty()
//           || dataItem.NHISTORY.isEmpty()
//           || dataItem.PROFIL.isEmpty()
//           || dataItem.MKB1.isEmpty()
//           || dataItem.CODE_USL.isEmpty()
//           || dataItem.CODE_MD.isEmpty()
//           || dataItem.DATE_IN.isEmpty()
//           || dataItem.DATE_OUT.isEmpty()
//           || dataItem.PERSCODE.isEmpty()
//           || dataItem.CODE_LPU.isEmpty()
//           || dataItem.OT_PER.isEmpty()
//           || dataItem.MSK_OT.isEmpty()
//           || dataItem.VID_PROV.isEmpty();
}

QByteArray SyncBase::makeGkey(const data::DataItem& dataItem)
{
    QByteArray ba;
    {
        QDataStream stream {&ba, QIODevice::WriteOnly};
        stream.setVersion(QDataStream::Qt_4_8);
        stream << dataItem.TIME_MARK    // 0
               << dataItem.OT_PER       // 1
               << dataItem.MSK_OT       // 2
               << dataItem.CODE_MSK     // 3
               << dataItem.VID_MP       // 4
               << dataItem.USL_OK       // 5
               << dataItem.PROFIL       // 6
               << dataItem.MKB1         // 7
               << dataItem.MKB2         // 8
               << dataItem.MKB3         // 9
               << dataItem.CODE_USL     // 10
               << dataItem.CODE_MD      // 11
               << dataItem.KOL_USL      // 12
               << dataItem.KOL_FACT     // 13
               << dataItem.ISH_MOV      // 14
               << dataItem.RES_GOSP     // 15
               << dataItem.TARIF_B      // 16
               << dataItem.TARIF_S      // 17
               << dataItem.TARIF_1K     // 18
               << dataItem.SUM_RUB      // 19
               << dataItem.VID_TR       // 20
               << dataItem.EXTR         // 21
               << dataItem.CODE_OTD     // 22
               << dataItem.SOUF         // 23
               << dataItem.SPEC_MD      // 24
               << dataItem.DOMC_TYPE    // 25
               << dataItem.OKATO_INS    // 26
               << dataItem.NOVOR        // 27
               << dataItem.CODE_LPU     // 28
               << dataItem.VID_SF       // 29
               << dataItem.NHISTORY     // 30
               << dataItem.PERSCODE     // 31
               << dataItem.DATE_IN      // 32
               << dataItem.DATE_OUT     // 33
               << dataItem.TARIF_D      // 34
               << dataItem.VID_KOEFF    // 35
               << dataItem.USL_TMP      // 36
               << dataItem.BIRTHDAY     // 37
               << dataItem.SEX          // 38
               << dataItem.COUNTRY      // 39
               << dataItem.SEX_P        // 40
               << dataItem.BIRTHDAY_P   // 41
               << dataItem.INV          // 42
               << dataItem.DATE_NPR     // 43
               << dataItem.FOR_POM      // 44
               << dataItem.MSE          // 45
               << dataItem.P_CEL        // 46
               << dataItem.DN           // 47
               << dataItem.TAL_P        // 48
               << dataItem.PROFIL_K     // 49
               << dataItem.NAPR_MO      // 50
               << dataItem.MKB0         // 51
               << dataItem.DS_ONK       // 52
               << dataItem.VAL_KOEFF    // 53
               << dataItem.C_ZAB        // 54
               << dataItem.CODE_NOM1    // 55
               << dataItem.CODE_NOM2    // 56
               << dataItem.CODE_NOM3    // 57
               << dataItem.VAL_TMP      // 58
               << dataItem.TIME_FIX     // 59
               << dataItem.TIME_IN      // 60
               << dataItem.TIME_OUT     // 61
               << dataItem.DATE_FIX     // 62
               << dataItem.KATEG_MD     // 63
               << dataItem.POST_MD      // 64
               //" << dataItem.KOL_DEF  // 65
               << dataItem.VID_PROV     // 66
               << dataItem.EKMP         // 67
               << dataItem.MEE          // 68

               << dataItem.MED_AREA     // 69
               << dataItem.TAL_HMP      // 70
               << dataItem.DATE_HMP     // 71
               << dataItem.MCOD_OUT     // 72
               << dataItem.NOM_NPR      // 73
               << dataItem.SERIES       // 74
               << dataItem.NAME_MSK     // 75
               << dataItem.OKATO_NAS    // 76
               << dataItem.PR_LG;       // 77
               //<< dataItem.IS_EKMP;   // 78
    }

    QByteArray gKey = QCryptographicHash::hash(ba, QCryptographicHash::Md5);
    return std::move(gKey);
}

QString SyncBase::makeValues(const data::DataItem& dataItem)
{
    QString gKeyStr;

//    gKeyStr += "OT_PER     : " + dataItem.OT_PER;
//    gKeyStr += "MSK_OT     : " + dataItem.MSK_OT;
//    gKeyStr += "CODE_MSK   : " + dataItem.CODE_MSK;
//    gKeyStr += "VID_MP     : " + dataItem.VID_MP;
//    gKeyStr += "USL_OK     : " + dataItem.USL_OK;
//    gKeyStr += "PROFIL     : " + dataItem.PROFIL;
//    gKeyStr += "MKB1       : " + dataItem.MKB1;
//    gKeyStr += "MKB2       : " + dataItem.MKB2;
//    gKeyStr += "MKB3       : " + dataItem.MKB3;
//    gKeyStr += "CODE_USL   : " + dataItem.CODE_USL;
//    gKeyStr += "CODE_MD    : " + dataItem.CODE_MD;
//    gKeyStr += "KOL_USL    : " + dataItem.KOL_USL;
//    gKeyStr += "KOL_FACT   : " + dataItem.KOL_FACT;
//    gKeyStr += "ISH_MOV    : " + dataItem.ISH_MOV;
//    gKeyStr += "RES_GOSP   : " + dataItem.RES_GOSP;
//    gKeyStr += "TARIF_B    : " + dataItem.TARIF_B;
//    gKeyStr += "TARIF_S    : " + dataItem.TARIF_S;
//    gKeyStr += "TARIF_1K   : " + dataItem.TARIF_1K;
//    gKeyStr += "SUM_RUB    : " + dataItem.SUM_RUB;
//    gKeyStr += "VID_TR     : " + dataItem.VID_TR;
//    gKeyStr += "EXTR       : " + dataItem.EXTR;
//    gKeyStr += "CODE_OTD   : " + dataItem.CODE_OTD;
//    gKeyStr += "SOUF       : " + dataItem.SOUF;
//    gKeyStr += "SPEC_MD    : " + dataItem.SPEC_MD;
//    gKeyStr += "DOMC_TYPE  : " + dataItem.DOMC_TYPE;
//    gKeyStr += "OKATO_INS  : " + dataItem.OKATO_INS;
//    gKeyStr += "NOVOR      : " + dataItem.NOVOR;
//    gKeyStr += "CODE_LPU   : " + dataItem.CODE_LPU;
//    gKeyStr += "VID_SF     : " + dataItem.VID_SF;
//    gKeyStr += "NHISTORY   : " + dataItem.NHISTORY;
//    gKeyStr += "PERSCODE   : " + dataItem.PERSCODE;
//    gKeyStr += "DATE_IN    : " + dataItem.DATE_IN;
//    gKeyStr += "DATE_OUT   : " + dataItem.DATE_OUT;
//    gKeyStr += "TARIF_D    : " + dataItem.TARIF_D;
//    gKeyStr += "VID_KOEFF  : " + dataItem.VID_KOEFF;
//    gKeyStr += "USL_TMP    : " + dataItem.USL_TMP;
//    gKeyStr += "BIRTHDAY   : " + dataItem.BIRTHDAY;
//    gKeyStr += "SEX        : " + dataItem.SEX;
//    gKeyStr += "COUNTRY    : " + dataItem.COUNTRY;
//    gKeyStr += "SEX_P      : " + dataItem.SEX_P;
//    gKeyStr += "BIRTHDAY_P : " + dataItem.BIRTHDAY_P;
//    gKeyStr += "INV        : " + dataItem.INV;
//    gKeyStr += "DATE_NPR   : " + dataItem.DATE_NPR;
//    gKeyStr += "FOR_POM    : " + dataItem.FOR_POM;
//    gKeyStr += "MSE        : " + dataItem.MSE;
//    gKeyStr += "P_CEL      : " + dataItem.P_CEL;
//    gKeyStr += "DN         : " + dataItem.DN;
//    gKeyStr += "TAL_P      : " + dataItem.TAL_P;
//    gKeyStr += "PROFIL_K   : " + dataItem.PROFIL_K;
//    gKeyStr += "NAPR_MO    : " + dataItem.NAPR_MO;
//    gKeyStr += "MKB0       : " + dataItem.MKB0;
//    gKeyStr += "DS_ONK     : " + dataItem.DS_ONK;
//    gKeyStr += "VAL_KOEFF  : " + dataItem.VAL_KOEFF;
//    gKeyStr += "C_ZAB      : " + dataItem.C_ZAB;
//    gKeyStr += "CODE_NOM1  : " + dataItem.CODE_NOM1;
//    gKeyStr += "CODE_NOM2  : " + dataItem.CODE_NOM2;
//    gKeyStr += "CODE_NOM3  : " + dataItem.CODE_NOM3;
//    gKeyStr += "VAL_TMP    : " + dataItem.VAL_TMP;
//    gKeyStr += "TIME_FIX   : " + dataItem.TIME_FIX;
//    gKeyStr += "TIME_IN    : " + dataItem.TIME_IN;
//    gKeyStr += "TIME_OUT   : " + dataItem.TIME_OUT;
//    gKeyStr += "DATE_FIX   : " + dataItem.DATE_FIX;
//    gKeyStr += "KATEG_MD   : " + dataItem.KATEG_MD;
//    gKeyStr += "POST_MD    : " + dataItem.POST_MD;
//    gKeyStr += "VID_PROV   : " + dataItem.VID_PROV;
//    gKeyStr += "EKMP       : " + dataItem.EKMP;
//    gKeyStr += "MEE        : " + dataItem.MEE;

//    gKeyStr += "MED_AREA   : " + dataItem.MED_AREA;
//    gKeyStr += "TAL_HMP    : " + dataItem.TAL_HMP;
//    gKeyStr += "DATE_HMP   : " + dataItem.DATE_HMP;
//    gKeyStr += "MCOD_OUT   : " + dataItem.MCOD_OUT;
//    gKeyStr += "NOM_NPR    : " + dataItem.NOM_NPR;
//    gKeyStr += "SERIES     : " + dataItem.SERIES;
//    gKeyStr += "NAME_MSK   : " + dataItem.NAME_MSK;
//    gKeyStr += "OKATO_NAS  : " + dataItem.OKATO_NAS;
//    gKeyStr += "PR_LG      : " + dataItem.PR_LG;

    return std::move(gKeyStr);
}

RetInfo SyncBase::send(const Message::Ptr& message)
{
    if (fomsCon()->isConnected())
    {
        message->setTag(hashId());
        fomsCon()->send(message);
        return {RetInfo::Error::Ok};
    }
    log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_FOMS);

    return {RetInfo::Error::General};
}

RetInfo SyncBase::saveToDB(const data::GetSyncData2& data, QDateTime& timeMark)
{
    Transaction::Ptr transact = createTransact(maindb());
    QSqlQuery q {_db->createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql};
    }

    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    if (!sql::exec(q, "DELETE FROM SYNC_TMP_DATA;"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QString fields = "  GKEY       "  //
                     ", TIME_MARK  "  //
                     ", OT_PER     "  // 1
                     ", MSK_OT     "  // 2
                     ", CODE_MSK   "  // 3
                     ", VID_MP     "  // 4
                     ", USL_OK     "  // 5
                     ", PROFIL     "  // 6
                     ", MKB1       "  // 7
                     ", MKB2       "  // 8
                     ", MKB3       "  // 9
                     ", CODE_USL   "  // 10
                     ", CODE_MD    "  // 11
                     ", KOL_USL    "  // 12
                     ", KOL_FACT   "  // 13
                     ", ISH_MOV    "  // 14
                     ", RES_GOSP   "  // 15
                     ", TARIF_B    "  // 16
                     ", TARIF_S    "  // 17
                     ", TARIF_1K   "  // 18
                     ", SUM_RUB    "  // 19
                     ", VID_TR     "  // 20
                     ", EXTR       "  // 21
                     ", CODE_OTD   "  // 22
                     ", SOUF       "  // 23
                     ", SPEC_MD    "  // 24
                     ", DOMC_TYPE  "  // 25
                     ", OKATO_INS  "  // 26
                     ", NOVOR      "  // 27
                     ", CODE_LPU   "  // 28
                     ", VID_SF     "  // 29
                     ", NHISTORY   "  // 30
                     ", PERSCODE   "  // 31
                     ", DATE_IN    "  // 32
                     ", DATE_OUT   "  // 33
                     ", TARIF_D    "  // 34
                     ", VID_KOEFF  "  // 35
                     ", USL_TMP    "  // 36
                     ", BIRTHDAY   "  // 37
                     ", SEX        "  // 38
                     ", COUNTRY    "  // 39
                     ", SEX_P      "  // 40
                     ", BIRTHDAY_P "  // 41
                     ", INV        "  // 42
                     ", DATE_NPR   "  // 43
                     ", FOR_POM    "  // 44
                     ", MSE        "  // 45
                     ", P_CEL      "  // 46
                     ", DN         "  // 47
                     ", TAL_P      "  // 48
                     ", PROFIL_K   "  // 49
                     ", NAPR_MO    "  // 50
                     ", MKB0       "  // 51
                     ", DS_ONK     "  // 52
                     ", VAL_KOEFF  "  // 53
                     ", C_ZAB      "  // 54
                     ", CODE_NOM1  "  // 55
                     ", CODE_NOM2  "  // 56
                     ", CODE_NOM3  "  // 57
                     ", VAL_TMP    "  // 58
                     ", TIME_FIX   "  // 59
                     ", TIME_IN    "  // 60
                     ", TIME_OUT   "  // 61
                     ", DATE_FIX   "  // 62
                     ", KATEG_MD   "  // 63
                     ", POST_MD    "  // 64
                     ", KOL_DEF    "  // 65
                     ", VID_PROV   "  // 66
                     ", MED_AREA   "  // 67
                     ", TAL_HMP    "  // 68
                     ", DATE_HMP   "  // 69
                     ", MCOD_OUT   "  // 70
                     ", NOM_NPR    "  // 71
                     ", SERIES     "  // 72
                     ", NAME_MSK   "  // 73
                     ", OKATO_NAS  "  // 74
                     ", PR_LG      "  // 75
                     ", EKMP       "
                     ", MEE        ";

    QString query = sql::UPDATE_OR_INSERT_INTO("SYNC_TMP_DATA", fields, "GKEY");

    if (!q.prepare(query))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    #define VARCHAR(COLUMN)  utl::toVarchar(dataItem.COLUMN)
    #define DOUBLE(COLUMN)   utl::toDouble(dataItem.COLUMN)
    #define SMALLINT(COLUMN) utl::toSmallint(dataItem.COLUMN)
    #define DATE(COLUMN)     utl::toDate(dataItem.COLUMN)
    #define BIGINT(COLUMN)   utl::toBigint(dataItem.COLUMN)

    QString gKeyStr;
    for (int i = 0; i != data.items.size(); i++)
    {
        data::DataItem dataItem = data.items.at(i);
        const QByteArray gKey = makeGkey(dataItem);

        gKeyStr.clear();
        if (alog::logger().level() == alog::Level::Debug2)
            gKeyStr = makeValues(dataItem);

        bindValue(q, ":GKEY", gKey);

        if(dataItem.TIME_MARK > timeMark)
            timeMark = dataItem.TIME_MARK;

        bindValue(q, ":TIME_MARK",  dataItem.TIME_MARK );

        bindValue(q, ":OT_PER",     VARCHAR (OT_PER     )); // 1
        bindValue(q, ":MSK_OT",     VARCHAR (MSK_OT     )); // 2
        bindValue(q, ":CODE_MSK",   VARCHAR (CODE_MSK   )); // 3
        bindValue(q, ":VID_MP",     VARCHAR (VID_MP     )); // 4
        bindValue(q, ":USL_OK",     VARCHAR (USL_OK     )); // 5
        bindValue(q, ":PROFIL",     VARCHAR (PROFIL     )); // 6
        bindValue(q, ":MKB1",       VARCHAR (MKB1       )); // 7
        bindValue(q, ":MKB2",       VARCHAR (MKB2       )); // 8
        bindValue(q, ":MKB3",       VARCHAR (MKB3       )); // 9
        bindValue(q, ":CODE_USL",   VARCHAR (CODE_USL   )); // 10
        bindValue(q, ":CODE_MD",    VARCHAR (CODE_MD    )); // 11
        bindValue(q, ":KOL_USL",    DOUBLE  (KOL_USL    )); // 12
        bindValue(q, ":KOL_FACT",   DOUBLE  (KOL_FACT   )); // 13
        bindValue(q, ":ISH_MOV",    VARCHAR (ISH_MOV    )); // 14
        bindValue(q, ":RES_GOSP",   VARCHAR (RES_GOSP   )); // 15
        bindValue(q, ":TARIF_B",    DOUBLE  (TARIF_B    )); // 16
        bindValue(q, ":TARIF_S",    DOUBLE  (TARIF_S    )); // 17
        bindValue(q, ":TARIF_1K",   DOUBLE  (TARIF_1K   )); // 18
        bindValue(q, ":SUM_RUB",    DOUBLE  (SUM_RUB    )); // 19
        bindValue(q, ":VID_TR",     VARCHAR (VID_TR     )); // 20
        bindValue(q, ":EXTR",       SMALLINT(EXTR       )); // 21
        bindValue(q, ":CODE_OTD",   VARCHAR (CODE_OTD   )); // 22
        bindValue(q, ":SOUF",       SMALLINT(SOUF       )); // 23
        bindValue(q, ":SPEC_MD",    VARCHAR (SPEC_MD    )); // 24
        bindValue(q, ":DOMC_TYPE",  VARCHAR (DOMC_TYPE  )); // 25
        bindValue(q, ":OKATO_INS",  VARCHAR (OKATO_INS  )); // 26
        bindValue(q, ":NOVOR",      VARCHAR (NOVOR      )); // 27
        bindValue(q, ":CODE_LPU",   VARCHAR (CODE_LPU   )); // 28
        bindValue(q, ":VID_SF",     VARCHAR (VID_SF     )); // 29
        bindValue(q, ":NHISTORY",   VARCHAR (NHISTORY   )); // 30
        bindValue(q, ":PERSCODE",   VARCHAR (PERSCODE   )); // 31
        bindValue(q, ":DATE_IN",    DATE    (DATE_IN    )); // 32
        bindValue(q, ":DATE_OUT",   DATE    (DATE_OUT   )); // 33
        bindValue(q, ":TARIF_D",    DOUBLE  (TARIF_D    )); // 34
        bindValue(q, ":VID_KOEFF",  VARCHAR (VID_KOEFF  )); // 35
        bindValue(q, ":USL_TMP",    VARCHAR (USL_TMP    )); // 35
        bindValue(q, ":BIRTHDAY",   DATE    (BIRTHDAY   )); // 36
        bindValue(q, ":SEX",        VARCHAR (SEX        )); // 37
        bindValue(q, ":COUNTRY",    VARCHAR (COUNTRY    )); // 38
        bindValue(q, ":SEX_P",      VARCHAR (SEX_P      )); // 39
        bindValue(q, ":BIRTHDAY_P", DATE    (BIRTHDAY_P )); // 40
        bindValue(q, ":INV",        SMALLINT(INV        )); // 41
        bindValue(q, ":DATE_NPR",   DATE    (DATE_NPR   )); // 42
        bindValue(q, ":FOR_POM",    SMALLINT(FOR_POM    )); // 43
        bindValue(q, ":MSE",        SMALLINT(MSE        )); // 44
        bindValue(q, ":P_CEL",      VARCHAR (P_CEL      )); // 45
        bindValue(q, ":DN",         SMALLINT(DN         )); // 46
        bindValue(q, ":TAL_P",      DATE    (TAL_P      )); // 47
        bindValue(q, ":PROFIL_K",   SMALLINT(PROFIL_K   )); // 49
        bindValue(q, ":NAPR_MO",    VARCHAR (NAPR_MO    )); // 48
        bindValue(q, ":MKB0",       VARCHAR (MKB0       )); // 49
        bindValue(q, ":DS_ONK",     SMALLINT(DS_ONK     )); // 50
        bindValue(q, ":VAL_KOEFF",  DOUBLE  (VAL_KOEFF  )); // 51
        bindValue(q, ":C_ZAB",      SMALLINT(C_ZAB      )); // 52
        bindValue(q, ":CODE_NOM1",  VARCHAR (CODE_NOM1  )); // 53
        bindValue(q, ":CODE_NOM2",  VARCHAR (CODE_NOM2  )); // 54
        bindValue(q, ":CODE_NOM3",  VARCHAR (CODE_NOM3  )); // 55
        bindValue(q, ":VAL_TMP",    DOUBLE  (VAL_TMP    )); // 56
        bindValue(q, ":TIME_FIX",   VARCHAR (TIME_FIX   )); // 57
        bindValue(q, ":TIME_IN",    VARCHAR (TIME_IN    )); // 58
        bindValue(q, ":TIME_OUT",   VARCHAR (TIME_OUT   )); // 59
        bindValue(q, ":DATE_FIX",   DATE    (DATE_FIX   )); // 60
        bindValue(q, ":KATEG_MD",   VARCHAR (KATEG_MD   )); // 61
        bindValue(q, ":POST_MD",    VARCHAR (POST_MD    )); // 62
        bindValue(q, ":KOL_DEF",    DOUBLE  (KOL_DEF    )); // 63
        bindValue(q, ":VID_PROV",   VARCHAR (VID_PROV   )); // 64
        bindValue(q, ":MED_AREA",   VARCHAR (MED_AREA   )); // 65
        bindValue(q, ":TAL_HMP",    VARCHAR (TAL_HMP    )); // 66
        bindValue(q, ":DATE_HMP",   DATE    (DATE_HMP   )); // 67
        bindValue(q, ":MCOD_OUT",   VARCHAR (MCOD_OUT   )); // 68
        bindValue(q, ":NOM_NPR",    BIGINT  (NOM_NPR    )); // 69
        bindValue(q, ":SERIES",     VARCHAR (SERIES     )); // 70
        bindValue(q, ":NAME_MSK",   VARCHAR (NAME_MSK   )); // 71
        bindValue(q, ":OKATO_NAS",  VARCHAR (OKATO_NAS  )); // 72
        bindValue(q, ":PR_LG",      VARCHAR (PR_LG      )); // 73

        bindValue(q, ":EKMP",       DOUBLE  (EKMP       )); //
        bindValue(q, ":MEE",        DOUBLE  (MEE        )); //

        if (!q.exec())
        {
            QString info = u8"Error on insert record";
            info += u8". gKey: " + QUuidEx::fromRfc4122(gKey).toString();

            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_UNIQ, info);
        }

        CHECK_QTHREADEX_STOP
    }

    #undef VARCHAR
    #undef DOUBLE
    #undef SMALLINT
    #undef DATE
    #undef BIGINT

    if (threadStop())
    {
        log_info_m << TASK_EVENT_LOG(TASK_MSG_INT);
        return {RetInfo::Error::Interrupt, RetInfo::Critical::No};
    }

    // Добавление с группировкой
    if (!sql::exec(q, "EXECUTE PROCEDURE UPDATE_SYNC_DATA;"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (!transact->commit())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::Sql};
    }

    return {RetInfo::Error::Ok};
}

RetInfo SyncBase::sync(TimeRange period, const QDateTime& timeMark)
{
    RetInfo ri {RetInfo::Error::Ok};

    // Начало и конец полученного диапазона приводится к типу QDate, так как
    // инорфмация о времени не нужна.
    period.begin = dateOnly(period.begin);
    period.end = dateOnly(period.end);

    // periodBegin устанавливается на первый день запрашиваемого периода.
    period.begin = firstDay(period.begin);

    // periodEnd устанавилвается на последний день запрашиваемого периода
    period.end = lastDay(period.end);

    // current - используется для перемещения по календарю, пока current != periodEnd
    QDateTime current = period.begin;

    QDateTime lastTimeMark = timeMark;
    data::GetSyncData2 getSyncData2;
    getSyncData2.count = _recvSize;
    // Основной цикл "подневной" прокрутки периода. Весь период разбивается на
    // дни, и синхронизация выполняется для каждого дня.
    do // while(current != periodEnd.addDays(1));
    {
        // Цикл получения данных за 1 день отчетного периода
        do // while (getSyncData2.items.count() != 0);
        {
            getSyncData2.period.begin = current;
            getSyncData2.period.end = current;
            getSyncData2.timeMark = lastTimeMark;

            Message::Ptr message = createJsonMessage(getSyncData2);
            message->setPriority(Message::Priority::Low);

            ri = send(message);
            if (ri.isError())
                return ri;

            ri = waitMessage(message);
            if (ri.isError())
                return ri;

            ri = checkMessage(command::GetSyncData2, message);
            if (ri.isError())
                return ri;

            SResult res = readFromMessage(message, getSyncData2);
            if (!res)
            {
                ri = {RetInfo::Error::General};
                log_error_m << TASK_EVENT_LOG(TASK_ERR_BAD_ANSWER);
                return ri;
            }

            if (getSyncData2.items.size() > 0)
            {
                ri = saveToDB(getSyncData2, lastTimeMark);
                if (ri.isError())
                    return ri;
                // "подвинуть" дату времени последнего изменения записи.
                getSyncData2.timeMark = lastTimeMark;
            }

            CHECK_QTHREADEX_STOP
        }
        while (getSyncData2.items.count() != 0);

        if (!checkCount(getSyncData2.period, timeMark))
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SYNC_ROW_COUNT);
            break;
        }

        if (threadStop())
        {
            log_info_m << TASK_EVENT_LOG(TASK_MSG_INT);
            break;
        }

        current = current.addDays(1);
    }
    while (current != period.end.addDays(1));

    // Если задача синхронизации была прервана или перезапущена, необходимо
    // возвратить временные метки дня, на котором прервалась синхронизация.
    if (threadStop())
    {
        QSqlQuery q {_db->createResult()};
        if (!sql::exec(q, " UPDATE          "
                          "   SYNC_DATA     "
                          " SET             "
                          "   TIME_MARK = ? "
                          " WHERE           "
                          "   DATE_OUT >= ? "
                          "   AND           "
                          "   DATE_OUT <= ? ", timeMark, current, current))
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC,
                                          "Не удалось восстановить временные метки");
            return {RetInfo::Error::Sql};
        }
    }

    if (threadStop())
        return {RetInfo::Error::Interrupt};

    return {RetInfo::Error::Ok};
}

bool SyncBase::checkCount(const TimeRange& range, const QDateTime& timeMark)
{
    data::SyncDataCheck syncDataCheck;
    Message::Ptr message = createJsonMessage(syncDataCheck);

    RetInfo ri{RetInfo::Error::Ok};
    // Запросить количество строк диапазона range с временем модификации
    // старше timeMark у ФОМС-сервера.
    ri = send(message);
    if (ri.isError())
        return false;

    ri = waitMessage(message);
    if (ri.isError())
        return false;

    ri = checkMessage(command::SyncDataCheck, message);
    if (ri.isError())
        return false;

    SResult res = readFromMessage(message, syncDataCheck);
    if (!res)
    {
        ri = {RetInfo::Error::General};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_BAD_ANSWER);
        return false;
    }

    // Запросить количество строк диапазона range с временем модификации
    // старше timeMark в Базе Данных АИС.
    QSqlQuery q {_db->createResult()};

    if (!sql::exec(q, " SELECT                            "
                      "   COUNT(*) AS REC_COUNT           "
                      " FROM                              "
                      "   SYNC_DATA                       "
                      " WHERE                             "
                      "   DATE_OUT >= ? AND DATE_OUT <= ? "
                      "   AND                             "
                      "   TIME_MARK >= ?                  ",
                      range.begin, range.end, timeMark))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return false;
    }

    if (!q.first())
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EMPTY);
        return false;
    }

    int recordsCount = 0;
    assignValue(recordsCount, q.record(), "REC_COUNT");

    // Сравнить количество строк данных из Базы Данных ФОМС со значением
    // в Базе Данных АИС.

    //if (syncDataCheck.rowsCount != recordsCount)
    //    return false;

    return true;
}

RetInfo SyncBase::waitMessage(Message::Ptr& message)
{
    steady_timer breakTimer;
    while (true)
    {
        CHECK_QTHREADEX_STOP

        QMutexLocker locker(&_threadLock); (void) locker;
        if (!_messages.empty())
        {
            message = {_messages.release(0), false};
            break;
        }
        if (breakTimer.elapsed() > 60*1000 /*1 мин*/)
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_ANSWER);
            return {RetInfo::Error::General};
        }
        _threadCond.wait(&_threadLock, 100 /*iterationSleepTime*/);
    }

    if (threadStop())
    {
        log_info_m << TASK_EVENT_LOG(TASK_MSG_INT);
        return {RetInfo::Error::Interrupt, RetInfo::Critical::No};
    }

    return {RetInfo::Error::Ok};
}

RetInfo SyncBase::checkMessage(const QUuidEx& command, const Message::Ptr& message)
{
    if (message->command() == command::BreakDataTransfer)
    {
        log_error_m << TASK_EVENT_LOG(TASK_MSG_BREAK);
        return {RetInfo::Error::General, RetInfo::Critical::No};
    }

    if (message->execStatus() == Message::ExecStatus::Failed
        || message->execStatus() == Message::ExecStatus::Error)
    {
        log_error_m << TASK_EVENT_LOG(errorDescription(message));
        return {RetInfo::Error::General, RetInfo::Critical::No};
    }

    if (message->type() != Message::Type::Answer
        || message->command() != command)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_BAD_TYPE);
        return {RetInfo::Error::General, RetInfo::Critical::No};
    }

    return {RetInfo::Error::Ok};
}

void SyncBase::command_Dummy(const Message::Ptr& message)
{
    (void) message;
}

}// namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
