#include "sync_data.h"
#include "task_messages.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "commands/error.h"
#include "commands/task.h"

#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "SyncData")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "SyncData")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "SyncData")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "SyncData")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "SyncData")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "SyncData")

namespace task {

using namespace db::firebird;
using namespace sql;

SyncData::SyncData(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTaskThread(TaskType::SyncData, taskId, userId)
{}

void SyncData::run()
{
    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    _interrupted = false;

    // Инициализация параметра "Периодичность"
    if (!_baseInit)
    {
        _retInfo = {RetInfo::Error::Init};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_INIT);
        TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCDATA);
    }

    // Инициализация периода дат
    _retInfo = initPeriod();
    if (!_retInfo)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_INIT_INFO, TASK_NAME_SYNCDATA);
        TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCDATA);
    }

    // В случае обнаружения не синхронизированных данных, задача получит
    // статус "Ожидание синхронизации"
    _retInfo = syncCheck(this);
    if (_retInfo.correspond(RetInfo::Error::NeedSync))
    {
        // Здесь выводим в лог сообщение о старте задачи
        log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME_SYNCDATA);
        return;
    }
    else if (!_retInfo)
    {
        log_error_m << TASK_EVENT_LOG(u8"Ошибка запуска задачи регулярной синхронизации");
        TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCDATA);
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    // После успешной синхронизации, необходимо удалить запись в таблице
    // "Запланированная синхронизация"
    if (!sql::exec(q, "DELETE FROM SYNC_PLANNING WHERE TASK_ID = ?", id()))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        _retInfo = {RetInfo::Error::Sql};
        return;
    }

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME_SYNCDATA);
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
