#include "sync_nsi.h"
#include "functions.h"
#include "task_messages.h"

#include "commands/error.h"
#include "commands/nsi.h"
#include "database/connect.h"
#include "database/sql_func.h"

#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "SyncNsi")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "SyncNsi")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "SyncNsi")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "SyncNsi")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "SyncNsi")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "SyncNsi")

namespace task {

using namespace communication::transport;
using namespace db::firebird;
using namespace sql;

SyncNsi::SyncNsi(const QUuidEx& taskId, const QUuidEx& userId)
    : BaseTask(TaskType::SyncNsi, taskId, userId)
{
    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &SyncTransfer::command_Dummy, (SyncTransfer*)this);

    FUNC_REGISTRATION(SyncNsiVidmp)
    FUNC_REGISTRATION(SyncNsiProfile)
    FUNC_REGISTRATION(SyncNsiLpu)
    FUNC_REGISTRATION(SyncNsiMkb)

    #undef FUNC_REGISTRATION
}

RetInfo SyncNsi::syncVidmp(db::firebird::Transaction::Ptr transact)
{
    RetInfo retInfo {RetInfo::Success};

    Message::Ptr message = createJsonMessage(command::SyncNsiVidmp);
    message->setPriority(Message::Priority::Low);

    if (!(retInfo = send(message)))
        return retInfo;

    simple_timer timer;
    Message::Ptr answer;

    // Ожидание сообщения. Сообщение должно быть нужного типа
    if (!(retInfo = waitMessage(answer, command::SyncNsiVidmp)))
        return retInfo;

    log_debug2_m << "NSI Vidmp recieved: "
                 << (timer.elapsed() / 1000) << " seconds";

    data::SyncNsiVidmpA syncNsiVidmpA;
    SResult res = readFromMessage(answer, syncNsiVidmpA);
    if (!res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_ANSWER_PARSER);
        return {RetInfo::Error::JsonParse};
    }

    QSqlQuery q {db::firebird::createResult(transact)};

    if (!sql::exec(q, "DELETE FROM NSI_VIDMP"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QString sql = sql::UPDATE_OR_INSERT_INTO("NSI_VIDMP", "ID, VMPNAME", "ID");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    for (const data::NsiVidmp& nsiVidmp : syncNsiVidmpA.items)
    {
        if (threadStop())
            return {RetInfo::Error::Interrupt};

        bindValue(q, ":ID",      nsiVidmp.id);
        bindValue(q, ":VMPNAME", nsiVidmp.name);

        if (!q.exec())
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::Sql};
        }
    }

    return {RetInfo::Success};
}

RetInfo SyncNsi::syncProfile(db::firebird::Transaction::Ptr transact)
{
    RetInfo retInfo {RetInfo::Success};

    Message::Ptr message = createJsonMessage(command::SyncNsiProfile);
    message->setPriority(Message::Priority::Low);

    if (!(retInfo = send(message)))
        return retInfo;

    simple_timer timer;
    Message::Ptr answer;

    // Ожидание сообщения. Сообщение должно быть нужного типа
    if (!(retInfo = waitMessage(answer, command::SyncNsiProfile)))
        return retInfo;

    log_debug2_m << "NSI Profile recieved: "
                 << (timer.elapsed() / 1000) << " seconds";

    data::SyncNsiProfileA syncNsiProfileA;
    SResult res = readFromMessage(answer, syncNsiProfileA);
    if (!res)
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_ANSWER_PARSER);
        return {RetInfo::Error::JsonParse};
    }

    QSqlQuery q {db::firebird::createResult(transact)};

    if (!sql::exec(q, "DELETE FROM NSI_PROFILE"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QString sql = sql::UPDATE_OR_INSERT_INTO("NSI_PROFILE", "ID, PRNAME", "ID");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    for (const data::NsiProfile& nsiProfile : syncNsiProfileA.items)
    {
        if (threadStop())
            return {RetInfo::Error::Interrupt};

        bindValue(q, ":ID",     nsiProfile.id);
        bindValue(q, ":PRNAME", nsiProfile.name);

        if (!q.exec())
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::Sql};
        }
    }

    return {RetInfo::Success};
}

RetInfo SyncNsi::syncLpu(db::firebird::Transaction::Ptr transact)
{
    RetInfo retInfo {RetInfo::Success};
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!sql::exec(q, "DELETE FROM NSI_LPU"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QString sql = sql::UPDATE_OR_INSERT_INTO("NSI_LPU", "CODE, FULL_NAME, SHORT_NAME", "CODE");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    data::SyncNsiLpu syncNsiLpu;
    syncNsiLpu.code = 0;

    while (true)
    {
        if (threadStop())
            return {RetInfo::Error::Interrupt};

        Message::Ptr message = createJsonMessage(syncNsiLpu);
        message->setPriority(Message::Priority::Low);

        if (!(retInfo = send(message)))
            return retInfo;

        simple_timer timer;
        Message::Ptr answer;

        // Ожидание сообщения. Сообщение должно быть нужного типа
        if (!(retInfo = waitMessage(answer, command::SyncNsiLpu)))
            return retInfo;

        log_debug2_m << "NSI Lpu packet recieved: "
                     << (timer.elapsed() / 1000) << " seconds";

        SResult res = readFromMessage(answer, syncNsiLpu);
        if (!res)
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_ANSWER_PARSER);
            return {RetInfo::Error::JsonParse};
        }

        qint32 max = 0;
        timer.reset();

        for (const data::NsiLpu& nsiLpu : syncNsiLpu.items)
        {
            if (threadStop())
                return {RetInfo::Error::Interrupt};

            bindValue(q, ":CODE      ", nsiLpu.code);
            bindValue(q, ":FULL_NAME ", nsiLpu.fullName);
            bindValue(q, ":SHORT_NAME", nsiLpu.shortName);

            if (!q.exec())
            {
                log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
                return {RetInfo::Error::Sql};
            }

            if (nsiLpu.code > max)
                max = nsiLpu.code;
        }

        log_debug2_m << "NSI Lpu packet saved: "
                     << (timer.elapsed() / 1000) << " seconds";

        if (syncNsiLpu.items.count() != syncNsiLpu.count)
            break;

        syncNsiLpu.code = max + 1;
        syncNsiLpu.items.clear();

    } // while (true)

    return {RetInfo::Success};
}

RetInfo SyncNsi::syncMkb(db::firebird::Transaction::Ptr transact)
{
    RetInfo retInfo {RetInfo::Success};
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!sql::exec(q, "DELETE FROM NSI_MKB"))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    QString sql = sql::UPDATE_OR_INSERT_INTO("NSI_MKB", "MKB_CODE, MKB_NAME", "MKB_CODE");

    if (!q.prepare(sql))
    {
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    data::SyncNsiMkb syncNsiMkb;
    syncNsiMkb.mkbCode = QString();

    while (true)
    {
        if (threadStop())
            return {RetInfo::Error::Interrupt};

        Message::Ptr message = createJsonMessage(syncNsiMkb);
        message->setPriority(Message::Priority::Low);

        if (!(retInfo = send(message)))
            return retInfo;

        simple_timer timer;
        Message::Ptr answer;

        // Ожидание сообщения. Сообщение должно быть нужного типа
        if (!(retInfo = waitMessage(answer, command::SyncNsiMkb)))
            return retInfo;

        log_debug2_m << "NSI Mkb packet recieved: "
                     << (timer.elapsed() / 1000) << " seconds";

        SResult res = readFromMessage(answer, syncNsiMkb);
        if (!res)
        {
            log_error_m << TASK_EVENT_LOG(TASK_ERR_ANSWER_PARSER);
            return {RetInfo::Error::JsonParse};
        }

        timer.reset();

        for (const data::NsiMkb& nsiMkb : syncNsiMkb.items)
        {
            if (threadStop())
                return {RetInfo::Error::Interrupt};

            bindValue(q, ":MKB_CODE  ", nsiMkb.mkbCode );
            bindValue(q, ":MKB_NAME  ", nsiMkb.mkbName );

            if (!q.exec())
            {
                log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_EXEC);
                return {RetInfo::Error::Sql};
            }

            syncNsiMkb.mkbCode = nsiMkb.mkbCode;
        }

        log_debug2_m << "NSI Mkb packet saved: "
                     << (timer.elapsed() / 1000) << " seconds";

        if (syncNsiMkb.items.count() != syncNsiMkb.count)
            break;

        syncNsiMkb.items.clear();

    } // while (true)

    return {RetInfo::Success};
}

void SyncNsi::run()
{
    _threadId = trd::gettid();
    _retInfo = {RetInfo::Error::Undef};
    log_info_m << TASK_EVENT_LOG(TASK_START, TASK_NAME_SYNCNSI);

    if (!fomsCon().isAuthorized())
    {
        _retInfo = {RetInfo::Error::Foms};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_NO_FOMS);
        TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCNSI);
    }

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();

    if (!transact->begin())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_BEGIN);
        TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCNSI);
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    #define CHECK_INT_AND_ERROR(NSI_NAME) \
        if (_retInfo.correspond(RetInfo::Error::Interrupt)) { \
            TASK_RETURN_FOR_INT(TASK_NAME_SYNCNSI); \
        } \
        else if (!_retInfo) { \
            TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCNSI); \
        }

    _retInfo = syncVidmp(transact);
    CHECK_INT_AND_ERROR(u8"Vidmp")

    _retInfo = syncProfile(transact);
    CHECK_INT_AND_ERROR(u8"Profile")

    _retInfo = syncLpu(transact);
    CHECK_INT_AND_ERROR(u8"Lpu")

    _retInfo = syncMkb(transact);
    CHECK_INT_AND_ERROR(u8"Mkb")

    if (!transact->commit())
    {
        _retInfo = {RetInfo::Error::Sql};
        log_error_m << TASK_EVENT_LOG(TASK_ERR_SQL_COMMIT);
        TASK_CLEAN_AND_RETURN(TASK_NAME_SYNCNSI);
    }

    _retInfo = {RetInfo::Success};
    TASK_COMPLETE(TASK_NAME_SYNCNSI);
}

void SyncNsi::interrupt()
{
    QThreadEx::stop(0);
    _interrupted = true;
}

bool SyncNsi::taskIsRunning()
{
    return QThreadEx::isRunning();
}

bool SyncNsi::taskIsFinished()
{
    return QThreadEx::isFinished();
}

void SyncNsi::taskStart()
{
    QThreadEx::start();
}

bool SyncNsi::taskStop(unsigned long timeout)
{
    return QThreadEx::stop(timeout);
}

void SyncNsi::taskTerminate()
{
    QThreadEx::terminate();
}

}// namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
