﻿#include "sync_plan.h"

#include "functions.h"
#include "scheduler.h"
#include "task_messages.h"
#include "utils.h"

#include "commands/commands.h"
#include "commands/error.h"
#include "database/connect.h"
#include "database/sql_func.h"

#include "shared/defmac.h"
#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "SyncPlan")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "SyncPlan")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "SyncPlan")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "SyncPlan")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "SyncPlan")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "SyncPlan")

#define TASK_SYNC_LOG(MSG, ...) \
    EventLog(MSG, ##__VA_ARGS__) << EventTask(_taskId) << EventUser(_userId)

namespace task {

using namespace communication::transport;
using namespace db::firebird;
using namespace sql;

SyncPlan& syncPlan()
{
    return ::safe_singleton<SyncPlan>();
}

SyncPlan::SyncPlan() : SyncTransfer()
{
    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &SyncTransfer::command_Dummy, (SyncTransfer*)this);

//    FUNC_REGISTRATION(SupportCommand)
    FUNC_REGISTRATION(GetSyncData)
    FUNC_REGISTRATION(SyncDataCheck)
    FUNC_REGISTRATION(SyncDataCount)
    FUNC_REGISTRATION(KeepWaitCommand)

    #undef FUNC_REGISTRATION
}

bool SyncPlan::init()
{
    return true;
}

bool SyncPlan::sqlList(QSqlQuery& q)
{
    return sql::exec(q,
        " SELECT                                                            "
        "   SP.TASK_ID                                                      "
        "  ,SP.USER_ID                                                      "
        "  ,SP.PERIOD_BEGIN                                                 "
        "  ,SP.PERIOD_END                                                   "
        "  ,SP.PERIOD_CURRENT                                               "
        "  ,COALESCE(SP.PERIOD_CURRENT, SP.PERIOD_BEGIN) as PERIOD_CURRENT2 "
        "  ,SP.IS_COMPLETE                                                  "
        "  ,SP.IS_PERIODIC                                                  "
        "  ,T.NAME           AS TASK_NAME                                   "
        "  ,T.EXEC_STATUS    AS TASK_EXEC_STATUS                            "
        "  ,T.TASK_TYPE      AS TASK_TYPE                                   "
        "  ,T.DESCRIPTION    AS TASK_DESCRIPTION                            "
        "  ,T.CREATE_DATE    AS TASK_CREATE_DATE                            "
        " FROM                                                              "
        "   SYNC_PLANNING SP                                                "
        " LEFT JOIN TASK T ON                                               "
        "   T.ID = SP.TASK_ID                                               "
        " LEFT JOIN                                                         "
        " (                                                                 "
        "   SELECT                                                          "
        "     ID,                                                           "
        "     CASE WHEN TASK_TYPE = ?                                       "
        "       THEN 0                                                      "
        "       ELSE 1                                                      "
        "     END as PRIORITY                                               "
        "   FROM                                                            "
        "     TASK                                                          "
        " ) TASKPRI ON TASKPRI.ID = SP.TASK_ID                              "
        " WHERE                                                             "
        "   SP.IS_COMPLETE = 0                                              "
        "   AND                                                             "
        "   T.EXEC_STATUS = ?                                               "
        " ORDER BY                                                          "
        "   SP.IS_PERIODIC ASC                                              "
        "  ,TASKPRI.PRIORITY DESC                                           "
        "  ,SP.CREATE_DATE ASC                                              ",
        getTaskString(TaskType::SyncData),
        TaskExecStatus::WaitSync);
}

data::SyncInfo::List SyncPlan::syncList()
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    data::SyncInfo::List lst;

    // Формирование списка задач, ожидающих синхронизацию.
    if (!sqlList(q))
        return lst;

    while (q.next())
    {
        QSqlRecord r = q.record();
        data::SyncInfo* si = lst.add();

        assignValue(si->taskId   , r, "TASK_ID  ");
        assignValue(si->userId   , r, "USER_ID  ");
        assignValue(si->taskName , r, "TASK_NAME");
        assignTaskT(si->taskType , r             );

        QDate begin, end, current;
        assignValue(begin   , r, "PERIOD_BEGIN   ");
        assignValue(end     , r, "PERIOD_END     ");
        assignValue(current , r, "PERIOD_CURRENT2");

        int totalDays = begin.daysTo(end);
        int currentDays = begin.daysTo(current);
        int percent = ((float)currentDays / totalDays) * 100;

        si->current = percent;
        si->total = 100;
    }
    return lst;
}

void SyncPlan::run()
{
    log_info_m << "Started";

    _threadId = trd::gettid();

    while (true)
    {
        CHECK_QTHREADEX_STOP

        _taskId = QUuidEx();
        _userId = QUuidEx();

        // В ситуации когда нет подключения к ФОМС серверу данный параметр позволит
        // сохранить работоспособность системы. Если параметр равен TRUE, то допус-
        // кается выполнять обучение моделей и расчет оценок без процесса синхрони-
        // зации данных
        bool workWithoutSync = false;
        config::base().getValue("foms.work_without_sync", workWithoutSync);

        // Параметр позволяет проигнорировать процесс синхронизации данных и перейти
        // к выполнению следующего шага - обучению модели или расчета оценки.
        // При этом периодическая синхронизация данных работающая по расписанию
        // по прежнему будет работать
        bool skipSync = false;
        config::base().getValue("foms.skip_sync", skipSync);

        _recvSize = 5000;
        config::base().getValue("foms.sync_packet_size", _recvSize);

        _bufferSize = 200*1000;
        config::base().getValue("foms.sync_buffer_size", _bufferSize);

        if (_recvSize > _bufferSize)
        {
            _recvSize = _bufferSize;
            log_warn_m << "Config parameter foms.sync_packet_size is incorrect"
                       << ". Parameter set to " << _bufferSize;
        }

        if (!skipSync)
        {
            // Если подключения к ФОМС серверу нет, и работа без ФОМС сервера
            // запрещена, то процесс синхронизации не начнется
            if (!fomsCon().isConnected() && !workWithoutSync)
            {
                sleep(10);
                continue;
            }
            if (!fomsCon().isAuthorized() && !workWithoutSync)
            {
                sleep(10);
                continue;
            }
        }

        _interrupt = false;

        db::firebird::Driver::Ptr dbcon = dbpool().connect();
        QSqlQuery q {dbcon->createResult()};

        // Удаляем "мусорные" записи
        sql::exec(q, " DELETE FROM SYNC_PLANNING WHERE TASK_ID IN ( "
                     "   SELECT SP.TASK_ID                          "
                     "   FROM SYNC_PLANNING SP                      "
                     "   LEFT JOIN TASK T ON SP.TASK_ID = T.ID      "
                     "   WHERE T.ID IS NULL                         "
                     " )                                            ");

        // Читаем список задач на синхронизацию
        if (!sqlList(q))
        {
            log_error_m << EventLog(TASK_ERR_SQL_EXEC);
            if (!threadStop())
            {
                QMutexLocker locker(&_threadLock); (void) locker;
                _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            }
            continue;
        }
        if (!q.first())
        {
            if (!threadStop())
            {
                QMutexLocker locker(&_threadLock); (void) locker;
                _threadCond.wait(&_threadLock, 60*1000 /*1 мин*/);
            }
            continue;
        }

        QString taskName, taskDescr;
        QDate periodBegin, periodEnd, periodCurrent;
        QDateTime taskCreated;
        //bool taskSkipSync = false;

        // Берем первую задачу из списка
        {
            QSqlRecord r = q.record();

            assignValue(_taskId,       r, "TASK_ID"          );
            assignValue(taskName,      r, "TASK_NAME"        );
            assignValue(_userId,       r, "USER_ID"          );
            assignValue(periodBegin,   r, "PERIOD_BEGIN"     );
            assignValue(periodEnd,     r, "PERIOD_END"       );
            assignValue(periodCurrent, r, "PERIOD_CURRENT"   );
            assignValue(_isPeriodic,   r, "IS_PERIODIC"      );
            assignValue(taskDescr,     r, "TASK_DESCRIPTION" );
            assignValue(taskCreated,   r, "TASK_CREATE_DATE" );
          //assignValue(taskSkipSync,  r, "TASK_SKIP_SYNC"   );
            assignTaskT(_taskType,     r                     );
        }

        _progress.taskId = _taskId;
        _progress.userId = _userId;
        _progress.taskExecStatus = TaskExecStatus::WaitSync;
        _progress.isPeriodic = _isPeriodic;

        QString sqlSkip = " UPDATE SYNC_PLANNING SET IS_COMPLETE = 1 "
                          " WHERE TASK_ID = ?";

        if (skipSync && (_taskType != TaskType::SyncData))
        {
            // Примечание: здесь есть риск попасть в мертвый цикл
            if (!sql::exec(q, sqlSkip, _taskId))
                continue;

            log_verbose_m << TASK_SYNC_LOG(u8"Синхронизация пропущена"
                                           u8" (foms.skip_sync = TRUE)"
                                           u8". Задача '%1'", taskName);

            // Запуск планировщика и переход к следующей задаче синхронизации
            scheduler().awake();
            continue;
        }

        if (!fomsCon().isAuthorized() && workWithoutSync)
        {
            // Примечание: здесь есть риск попасть в мертвый цикл
            if (!sql::exec(q, sqlSkip, _taskId))
                continue;

            log_verbose_m << TASK_SYNC_LOG(u8"Синхронизация пропущена"
                                           u8" (foms.work_without_sync = TRUE)"
                                           u8". Задача '%1'", taskName);

            // Запуск планировщика и переход к следующей задаче синхронизации
            scheduler().awake();
            continue;
        }

        // Перед синхронизацией отправить эвент о новой задаче
        data::TaskProgress taskProgress;
        taskProgress.taskType       = _taskType;
        taskProgress.taskId         = _taskId;
        taskProgress.taskName       = taskName;
        taskProgress.taskDescript   = taskDescr;
        taskProgress.taskCreateDate = taskCreated;
        taskProgress.taskExecStatus = TaskExecStatus::WaitSync;
        taskProgress.userId         = _userId;
        taskProgress.isPeriodic     = isPeriodic();
        taskProgress.waitPosition   = 0;
        taskProgress.current        = 0;
        taskProgress.total          = 100;

        Message::Ptr m = createJsonMessage(taskProgress, Message::Type::Event);
        webCon().send(m);

        // Синхронизация периода [periodBegin,periodEnd]
        // periodCurrent - дата успешно синхронизированного периода, перед
        // прерыванием задачи.
        // При возникновении ошибки, переход к выполнению следующей задачи
        if (!sync(periodBegin, periodEnd, periodCurrent))
        {
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SYNC);
            continue;
        }

        // После успешной синхронизации установка флага IS_COMPLETE = TRUE
        if (!sql::exec(q,
            "UPDATE SYNC_PLANNING SET IS_COMPLETE = 1 WHERE TASK_ID = ?", _taskId))
        {
            // В случае ошибки при выполнении запроса, период не будет
            // отмечен как синхронизированный.
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
            continue;
        }

        // Пробуждение планировщика приведет к перечитыванию задач, среди которых
        // будет и только что синхронизированная задача
        scheduler().awake();

    } // while (true);

    log_info_m << "Stopped";
}

bool SyncPlan::sync(QDate periodBegin, QDate periodEnd, QDate startDay)
{
    log_verbose_m << TASK_SYNC_LOG(u8"Начало синхронизации периода: %1 / %2",
                                   periodBegin, periodEnd);

    // если periodCurrent != NULL, значит синхронизация была прервана, и
    // процесс необходимо продолжить с даты указанной в periodCurrent + 1 день
    if (!startDay.isNull())
        periodBegin = startDay.addDays(1);

    // current - используется для перемещения по календарю, пока current != period.end
    QDate current = periodBegin;

    // Проверка данных за весь период [periodBegin, periodEnd] с шагом в 1 месяц
    while (current <= periodEnd)
    {
        if (threadStop())
            return false;

        if (_interrupt)
            return false;

        bool needSync = true;
        QDate border = lastDay(QDateTime(current)); // Шаг синхронизации: 1 месяц

        if (border > periodEnd)
            border = periodEnd;

        RetInfo retInfo = checkPeriod(current, border, needSync);
        if (!retInfo)
        {
            log_warn_m << TASK_SYNC_LOG(u8"Неудачная проверка. Период: %1.%2",
                                        current.month(), current.year());

            // Если ошибка связана с ФОМС сервером, то необходимо подождать
            // перед повтороной попыткой.
            if (retInfo.correspond(RetInfo::Error::Foms))
                sleep(30);

            continue;
        }

        if (!needSync)
        {
            // Если проверка месяца показала, что данные не добавились, то
            // необходимо пропустить этот месяц.

            db::firebird::Driver::Ptr dbcon = dbpool().connect();
            QSqlQuery q {dbcon->createResult()};

            // Диапазон current -> lastDayInMonth не нуждается в синхронизации.
            if (!sql::exec(q,
                "UPDATE SYNC_PLANNING SET PERIOD_CURRENT = ? WHERE TASK_ID = ?",
                border, _taskId))
            {
                log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
                return false;
            }

            // Формирование события о прогрессе
            progressUpdate();
            Message::Ptr m = createJsonMessage(_progress, Message::Type::Event);
            webCon().send(m);

            current = border.addDays(1);
            continue;
        }

        // Основной цикл прокрутки периода с отрезками в месяц. В этом цикле
        // происходит извлечение данных блоками по _bufferSize до тех пор,
        // пока все данные для периода не будут получены.
        while (current <= border)
        {
            if (threadStop())
                return false;

            if (_interrupt)
                return false;

            if (!fomsCon().isConnected())
            {
                sleep(1);
                continue;
            }
            if (!fomsCon().isAuthorized())
            {
                sleep(1);
                continue;
            }

            // lastTimemark - будет содержать последнее время модификации, после
            // работы метода syncIteration.
            // Если строк страше чем lastTimemark нет, то можно считать, что
            // все данные синхронизированы.
            QDateTime lastTimemark = QDateTime(QDate(1900,1,1));

            RetInfo retInfo = syncIteration(current, border, lastTimemark);
            if (!retInfo.isSuccess())
            {
                log_error_m << TASK_SYNC_LOG(u8"Ошибка в итерации синхронизации");
                return false;
            }

            // Если команда с данными последняя, то отправляем запрос
            // на получение количества строк с датой модификации больше
            // либо равно lastTimemark
            data::SyncDataCount syncDataCount;
            syncDataCount.period.begin = QDateTime(current);
            syncDataCount.period.end = QDateTime(border);
            syncDataCount.timeMark = lastTimemark;

            Message::Ptr message = createJsonMessage(syncDataCount);
            message->setPriority(Message::Priority::Low);

            retInfo = send(message);
            if (!retInfo.isSuccess())
                return false;

            // Ожидание сообщения. Сообщение должно быть SyncDataCount типа
            Message::Ptr answer;
            retInfo = waitMessage(answer, command::SyncDataCount);
            if (!retInfo.isSuccess())
                return false;

            SResult res = readFromMessage(answer, syncDataCount);
            if (!res)
            {
                log_error_m << TASK_SYNC_LOG(TASK_ERR_ANSWER_PARSER);
                return false;
            }

            log_debug2_m << "Records remains on FOMS server: "
                         << syncDataCount.count;

            // Если синхронизация удачна, но данные синхронизированы не полностью, то
            // необходимо повторить итерацию синхронизации.
            if (retInfo.isSuccess() && syncDataCount.count > 0)
                continue;

            if (retInfo.isSuccess())
            {
                db::firebird::Driver::Ptr dbcon = dbpool().connect();
                QSqlQuery q {dbcon->createResult()};

                // Если синхронизация завершилась успешно, то теперь необходимо проверить
                // контрольную сумму всего периода. Это актуально в том случае, если записи
                // на ФОМС сервере были удалены. Если crc не совпадёт, то необходимо
                // удалить все строки за указанный период, и провести синхронизацию заново.


                // Если синхронизация удачна, то отмечаем этот день как успешно синхрони-
                // зированный, путём установки PERIOD_CURRENT = current
                if (!sql::exec(q,
                    "UPDATE SYNC_PLANNING SET PERIOD_CURRENT = ? WHERE TASK_ID = ?",
                    border, _taskId))
                {
                    log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
                    return false;
                }
            }
            else if (retInfo.correspond(RetInfo::Error::Interrupt))
            {
                log_verbose_m << TASK_SYNC_LOG(u8"Синхронизация прервана программно");
                return false;
            }
            else
            {
                // Если синхронизация не удачна, то необходимо повторить попытку
                log_warn_m << TASK_SYNC_LOG(u8"Неудачная синхронизация. День: %1", current);

                // Если ошибка связана с ФОМС сервером, то необходимо подождать
                // перед повтороной попыткой.
                if (retInfo.correspond(RetInfo::Error::Foms))
                    sleep(30);

                // Если есть проблемы с БД, необходимо подождать перед
                // следующей попыткой
                if (retInfo.correspond(RetInfo::Error::DBError))
                    sleep(30);

                continue;
            }

            // Добавление 1 дня приведет к переходу на следующий месяц.
            // Так как border всегда имеет значение последнего дня месяца. А current,
            // в свою очеред "развернется" в следующий месяц.
            current = border.addDays(1);

            // Формирование события о прогрессе
            progressUpdate();
            Message::Ptr m = createJsonMessage(_progress, Message::Type::Event);
            webCon().send(m);

        } // while (current <= border) - месяц

        // проверка общего кол-ва

    } // while(current <= periodEnd) - весь период, разделенный на отрезки по 1 месяцу

    log_verbose_m << TASK_SYNC_LOG(u8"Окончание синхронизации периода: %1 / %2",
                                   periodBegin, periodEnd);
    return true;
}

void SyncPlan::progressUpdate()
{
    int totalDays = 0;
    int currentDays = 0;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    sql::exec(q,
        " SELECT                                                     "
        "   PERIOD_BEGIN,                                            "
        "   PERIOD_END,                                              "
        "   COALESCE(PERIOD_CURRENT, PERIOD_BEGIN) as PERIOD_CURRENT "
        " FROM                                                       "
        "   SYNC_PLANNING SP                                         "
        " WHERE                                                      "
        "   SP.TASK_ID = ?                                           ",
        _progress.taskId);

    q.first();
    QDate begin, end, current;

    assignValue(begin  , q.record(), "PERIOD_BEGIN  ");
    assignValue(end    , q.record(), "PERIOD_END    ");
    assignValue(current, q.record(), "PERIOD_CURRENT");

    totalDays = begin.daysTo(end);
    currentDays = begin.daysTo(current);
    int percent = ((float)currentDays / totalDays) * 100;

    sql::exec(q, " SELECT        "
                 "   NAME        "
                 "  ,DESCRIPTION "
                 "  ,CREATE_DATE "
                 "  ,TASK_TYPE   "
                 " FROM          "
                 "   TASK        "
                 " WHERE         "
                 "   ID = ?      ", _progress.taskId);

    if (q.first())
    {
        QSqlRecord r = q.record();
        assignValue(_progress.taskName       , r, "NAME        " );
        assignValue(_progress.taskDescript   , r, "DESCRIPTION " );
        assignValue(_progress.taskCreateDate , r, "CREATE_DATE " );
        assignTaskT(_progress.taskType       , r );
    }

    _progress.waitPosition = 0;
    _progress.current = percent;
    _progress.total = 100;
}

RetInfo SyncPlan::syncIteration(const QDate& dayStart, const QDate& dayEnd, QDateTime& lastTimemark)
{
    log_debug_m << "Begin sync for [" << dayStart << ", " << dayEnd << "]";

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    Transaction::Ptr transact = dbcon->createTransact();
    QSqlQuery q {db::firebird::createResult(transact)};

    if (!transact->begin())
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_BEGIN);
        return {RetInfo::Error::DBError};
    }
    FIREBIRD_AUTOROLLBACK_TRANSACT(transact)

    QDateTime maxTimeMark {QDate(1900, 1, 1)};

    // Проверка наличия записей в таблице SYNC_DATA
    int recCount = 0;
    if (!sql::exec(q,
        " SELECT                 "
        "   COUNT(*) AS RECCOUNT "
        " FROM                   "
        "   SYNC_DATA            "
        " WHERE                  "
        "   DATE_OT_PER >= ?     "
        "   AND                  "
        "   DATE_OT_PER <= ?     ",
        dayStart, dayEnd))
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }
    if (q.first())
        assignValue(recCount, q.record(), "RECCOUNT");

    // Если записи в таблице SYNC_DATA пристутствуют, неоходимо
    // вычислить дату последней модификации записи и начать
    // синхронизацю с этой даты
    if (recCount > 0)
    {
        if (!sql::exec(q,
            " SELECT             "
            "   MAX(TIME_MARK)   "
            " FROM               "
            "   SYNC_DATA        "
            " WHERE              "
            "   DATE_OT_PER >= ? "
            "   AND              "
            "   DATE_OT_PER <= ? ",
            dayStart, dayEnd))
        {
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::Sql};
        }
        if (q.first())
            assignValue(maxTimeMark, q.record(), "MAX");
    }

    log_debug2_m << "Max TimeMark: " << maxTimeMark << " (for period: " << dayStart << " - " << dayEnd << ")";

    if (threadStop())
    {
        // Предотвращает отправку команды GetSyncData в случае остановки потока
        return {RetInfo::Error::General};
    }

    data::GetSyncData getSyncData;
    getSyncData.period.begin = QDateTime(dayStart);
    getSyncData.period.end = QDateTime(dayEnd);
    getSyncData.timeMark = maxTimeMark;
    getSyncData.count = _recvSize;

    simple_timer timer;
    Message::Ptr answer;
    data::GetSyncData getSyncDataAnswer;

    // Текущее количество переданных записей
    qint32 recivedCount = 0;
    // Количество уникальных записей в таблице 'SYNC_TMP_DATA'
    quint64 uniqueCount = 0;
    // Данный цикл отпрвляет команду GetSyncData на ФОМС Сервер, до тех пор пока
    // не будут переданы все данные, либо количество записей превысит значение
    // буфера синхронизации.
    while (true)
    {
        CHECK_QTHREADEX_STOP

        if (_interrupt)
        {
            setIgnore();
            return {RetInfo::Error::Interrupt};
        }

        Message::Ptr message = createJsonMessage(getSyncData);
        message->setPriority(Message::Priority::Low);

        timer.reset();

        // Первое сообщение
        RetInfo retInfo = send(message);
        if (!retInfo)
            return retInfo;

        // Ожидание сообщения. Сообщение должно быть нужного типа
        retInfo = waitMessage(answer, command::GetSyncData);
        if (!retInfo)
            return retInfo;

        log_debug2_m << "Sync packet recieved: "
                     << (timer.elapsed() / 1000) << " seconds";

        SResult res = readFromMessage(answer, getSyncDataAnswer);
        if (!res)
        {
            log_error_m << TASK_SYNC_LOG(TASK_ERR_ANSWER_PARSER);
            return {RetInfo::Error::JsonParse};
        }

        log_debug2_m << "Sync packet recieved: "
                     << getSyncDataAnswer.items.size() << " rows";

        // Если команда вернула запрашиваемое количество данных, то происходит
        // отправка запроса
        QDateTime max {QDate(1900, 1, 1)};
        for (const data::DataItem& di : getSyncDataAnswer.items)
            if (di.TIME_MARK > max)
                max = di.TIME_MARK;

        log_debug2_m << "Shift TimeMark: "
                     << getSyncData.timeMark << " --> " << max;

        // Подсчёт количества полученных строк данных
        recivedCount += getSyncDataAnswer.items.size();

        // Сдвиг временной метки, для получения следующей порции данных
        getSyncData.timeMark = max;
        lastTimemark = max;

        timer.reset();

        // Сохраняем полученные данные в БД
        retInfo = saveToDB(transact, getSyncDataAnswer);
        if (!retInfo.isSuccess())
            return retInfo;

        countRecords(transact, "SYNC_TMP_DATA", uniqueCount);

        log_debug2_m << "Sync rows saved: "
                     << (timer.elapsed() / 1000) << " seconds";

        // В случае заполнение буфера временная метка не увеличиваятся,
        // для возможности загрузки строк с дублями временной метки.
        if (recivedCount >= _bufferSize)
        {
            // Завершение цикла. Так как превышен размер буфера синхронизации.
            // Временная метка не подвергается увеличению, так как на "стыке"
            // буферов, могут быть повторы во временных метках.
            break;
        }
        // В случае если все записи загружены, необходимо увеличить
        // временную метку для завершения цикла синхронизации. Иначе
        // произойдет зацикливание на получении записи с макисмальной
        // временной меткоой.
        if (getSyncDataAnswer.items.size() != getSyncDataAnswer.count)
        {
            // Завершение цикла. Так данные получены в полном объёме.
            lastTimemark = lastTimemark.addMSecs(1);
            break;
        }

        message = createJsonMessage(getSyncData);
        message->setPriority(Message::Priority::Low);

    } // while (true)

    countRecords(transact, "SYNC_TMP_DATA", uniqueCount);

    // Если команда с данными последняя, то отправляем запрос
    // на получение контрольной суммы timeMark
    data::SyncDataCheck syncDataCheck;
    syncDataCheck.period.begin = QDateTime(dayStart);
    syncDataCheck.period.end = QDateTime(dayEnd);
    syncDataCheck.timeMark = maxTimeMark;
    syncDataCheck.count = uniqueCount;

    Message::Ptr message = createJsonMessage(syncDataCheck);
    message->setPriority(Message::Priority::Low);

    RetInfo retInfo = send(message);
    if (!retInfo)
        return retInfo;

    // Ожидание сообщения. Сообщение должно быть нужного типа
    retInfo = waitMessage(answer, command::SyncDataCheck);
    if (!retInfo)
        return retInfo;

    log_debug2_m << "Sync crc recieved: "
                 << (timer.elapsed() / 1000) << " seconds";

    SResult res = readFromMessage(answer, syncDataCheck);
    if (!res)
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_ANSWER_PARSER);
        return {RetInfo::Error::JsonParse};
    }

    // Подсчет CRC в таблице, содержащей полученный строки данных
    // за отчётный период (за 1 день)
    quint64 crc;
    if (!calcCrc(transact, crc))
    {
        log_error_m << TASK_SYNC_LOG(u8"Ошибка вычисления контрольной "
                                     u8"суммы для временных меток");
        return {RetInfo::Error::General};
    }
    if (syncDataCheck.crc != crc)
    {
        log_error_m << TASK_SYNC_LOG(u8"Не сошлась контрольная сумма "
                                     u8"для временных меток. Период: [%1, %2]", dayStart, dayEnd);
        return {RetInfo::Error::General};
    }
    log_debug2_m << "TimeMark CRC: " << crc;

    countRecords(transact, "SYNC_TMP_DATA", uniqueCount);

    log_debug_m << "Records received: " << recivedCount
                << ". Records unique: " << uniqueCount;

    if (uniqueCount > 0)
    {
        timer.reset();
        // Переносим данные из временной таблицы SYNC_TMP_DATA в SYNC_DATA
        if (!sql::exec(q, "EXECUTE PROCEDURE UPDATE_SYNC_DATA;"))
        {
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
            return {RetInfo::Error::General};
        }
        if (!transact->commit())
        {
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_COMMIT);
            return {RetInfo::Error::General};
        }
        log_debug_m << "Sync rows commited: "
                    << (timer.elapsed() / 1000) << " seconds";
    }
    else
    {
        if (!transact->rollback())
        {
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_ROLLBACK);
            return {RetInfo::Error::General};
        }
    }

    log_debug_m << "End sync for period: " << dayStart << " - " << dayEnd
                << ". Rows recieved: " << uniqueCount;

    return {RetInfo::Success};
}

RetInfo SyncPlan::checkPeriod(QDate periodBegin, QDate periodEnd, bool& needSync)
{
    log_debug_m << "Check period: " << periodBegin << " / " << periodEnd;

    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    QDateTime maxTimeMark {QDate(1900, 1, 1)};

    if (!sql::exec(q,
        " SELECT             "
        "   MAX(TIME_MARK)   "
        " FROM               "
        "   SYNC_DATA        "
        " WHERE              "
        "   DATE_OT_PER >= ? "
        "   AND              "
        "   DATE_OT_PER <= ? ", periodBegin, periodEnd))
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
        return {RetInfo::Error::Sql};
    }

    if (q.first())
    {
        assignValue(maxTimeMark, q.record(), "MAX");
        // Увеличение тайм-марки происходит только в том случае, если записи
        // в БД уже есть. Таким образом избегаем повторной загрузки записей
        maxTimeMark = maxTimeMark.addMSecs(1);
    }
    log_debug2_m << "Max TimeMark for period: " << maxTimeMark;

    if (threadStop())
    {
        // Предотвращает отправку команды GetSyncData в случае остановки потока
        return {RetInfo::Error::General};
    }

    data::GetSyncData getSyncData;
    getSyncData.period.begin = QDateTime(periodBegin);
    getSyncData.period.end = QDateTime(periodEnd);
    getSyncData.timeMark = maxTimeMark;

    // Для проверки необходимости синхронизации не нужно возвращать полный
    // набор данных, достаточно будет одной записи
    getSyncData.count = 1;

    Message::Ptr message = createJsonMessage(getSyncData);
    message->setPriority(Message::Priority::Low);

    // Первое сообщение
    RetInfo retInfo = send(message);
    if (!retInfo.isSuccess())
        return retInfo;

    simple_timer timer;
    Message::Ptr answer;

    // Ожидание сообщения. Сообщение должно быть нужного типа
    retInfo = waitMessage(answer, command::GetSyncData);
    if (!retInfo.isSuccess())
        return retInfo;

    log_debug2_m << "Sync packet recieved: "
                 << (timer.elapsed() / 1000) << " seconds";

    data::GetSyncData getSyncDataAnswer;
    SResult res = readFromMessage(answer, getSyncDataAnswer);
    if (!res)
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_ANSWER_PARSER);
        return {RetInfo::Error::JsonParse};
    }

    // Если команда вернула строки данных, значит необходимо выполнить
    // подневную синхронизацию периода
    needSync = (getSyncDataAnswer.items.count() != 0);

    if (needSync)
        log_debug_m << "Need sync period: "
                    << periodBegin << " / " << periodEnd;

    return {RetInfo::Success};
}

bool SyncPlan::countRecords(Transaction::Ptr transact, const QString& tablename, quint64& recCount)
{
    QSqlQuery q {db::firebird::createResult(transact)};

    QString sql = " SELECT              "
                  "   COUNT(*)          "
                  " FROM                "
                  " (                   "
                  "   SELECT DISTINCT   "
                  "     TIME_MARK, GKEY "
                  "   FROM              "
                  "     %1              "
                  " ) T                 ";

    sql = sql.arg(tablename);

    if (!sql::exec(q, sql))
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
        return false;
    }

    recCount = 0;

    if (q.first())
        assignValue(recCount, q.record(), "COUNT");

    log_debug2_m << "Now in table 'SYNC_TMP_DATA' " << recCount << " records.";

    return true;
}

bool SyncPlan::calcCrc(Transaction::Ptr transact, quint64& crc)
{
    crc = 0;
    QSqlQuery q {db::firebird::createResult(transact)};

    QString sql = " SELECT DISTINCT           "
                  "   TIME_MARK, GKEY         "
                  " FROM                      "
                  "   SYNC_TMP_DATA           "
                  " ORDER BY                  "
                  "   TIME_MARK ASC, GKEY ASC ";

    if (!sql::exec(q, sql))
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_EXEC);
        return false;
    }

    int recCount = 0;
    while (q.next())
    {
        ++recCount;
        QDateTime time;
        QUuidEx uuid;
        assignValue(time, q.record(), "TIME_MARK");
        assignValue(uuid, q.record(), "GKEY");
        qint64 msecs = time.toMSecsSinceEpoch();
        log_debug2_m << "uuid " << uuid << " rec number: " << recCount << " msecs " << msecs;
        quint64 value = *((quint64*)&msecs);

        crc = crc ^ value;
    }

    log_debug2_m << "calcCrc result: " << crc;

    return true;
}

RetInfo SyncPlan::saveToDB(Transaction::Ptr transact, const data::GetSyncData& data)
{
    if (data.items.count() == 0)
        return {RetInfo::Success};

    simple_timer timer;
    QSqlQuery q {db::firebird::createResult(transact)};

    QString fields = "  GKEY       "  //
                     ", IDSL       "  //
                     ", TIME_MARK  "  //
                     ", OT_PER     "  // 1
                     ", MSK_OT     "  // 2
                     ", CODE_MSK   "  // 3
                     ", VID_MP     "  // 4
                     ", USL_OK     "  // 5
                     ", PROFIL     "  // 6
                     ", MKB1       "  // 7
                     ", MKB2       "  // 8
                     ", MKB3       "  // 9
                     ", CODE_USL   "  // 10
                     ", CODE_MD    "  // 11
                     ", KOL_USL    "  // 12
                     ", KOL_FACT   "  // 13
                     ", ISH_MOV    "  // 14
                     ", RES_GOSP   "  // 15
                     ", TARIF_B    "  // 16
                     ", TARIF_S    "  // 17
                     ", TARIF_1K   "  // 18
                     ", SUM_RUB    "  // 19
                     ", VID_TR     "  // 20
                     ", EXTR       "  // 21
                     ", CODE_OTD   "  // 22
                     ", SOUF       "  // 23
                     ", SPEC_MD    "  // 24
                     ", DOMC_TYPE  "  // 25
                     ", OKATO_INS  "  // 26
                     ", NOVOR      "  // 27
                     ", CODE_LPU   "  // 28
                     ", VID_SF     "  // 29
                     ", NHISTORY   "  // 30
                     ", PERSCODE   "  // 31
                     ", DATE_IN    "  // 32
                     ", DATE_OUT   "  // 33
                     ", TARIF_D    "  // 34
                     ", VID_KOEFF  "  // 35
                     ", USL_TMP    "  // 36
                     ", BIRTHDAY   "  // 37
                     ", SEX        "  // 38
                     ", COUNTRY    "  // 39
                     ", SEX_P      "  // 40
                     ", BIRTHDAY_P "  // 41
                     ", INV        "  // 42
                     ", DATE_NPR   "  // 43
                     ", FOR_POM    "  // 44
                     ", MSE        "  // 45
                     ", P_CEL      "  // 46
                     ", DN         "  // 47
                     ", TAL_P      "  // 48
                     ", PROFIL_K   "  // 49
                     ", NAPR_MO    "  // 50
                     ", MKB0       "  // 51
                     ", DS_ONK     "  // 52
                     ", VAL_KOEFF  "  // 53
                     ", C_ZAB      "  // 54
                     ", CODE_NOM1  "  // 55
                     ", CODE_NOM2  "  // 56
                     ", CODE_NOM3  "  // 57
                     ", VAL_TMP    "  // 58
                     ", TIME_FIX   "  // 59
                     ", TIME_IN    "  // 60
                     ", TIME_OUT   "  // 61
                     ", DATE_FIX   "  // 62
                     ", KATEG_MD   "  // 63
                     ", POST_MD    "  // 64
                     ", KOL_DEF    "  // 65
                     ", VID_PROV   "  // 66
                     ", MED_AREA   "  // 67
                     ", TAL_HMP    "  // 68
                     ", DATE_HMP   "  // 69
                     ", MCOD_OUT   "  // 70
                     ", NOM_NPR    "  // 71
                     ", SERIES     "  // 72
                     ", NAME_MSK   "  // 73
                     ", OKATO_NAS  "  // 74
                     ", PR_LG      "  // 75
                     ", EKMP       "
                     ", MEE        ";

    QString query = sql::INSERT_INTO("SYNC_TMP_DATA", fields);

    if (!q.prepare(query))
    {
        log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_PREPARE);
        return {RetInfo::Error::Sql};
    }

    #define VARCHAR(COLUMN)  toVarchar(dataItem.COLUMN)
    #define DOUBLE(COLUMN)   toDouble(dataItem.COLUMN)
    #define SMALLINT(COLUMN) toSmallint(dataItem.COLUMN)
    #define DATE(COLUMN)     toDate(dataItem.COLUMN)
    #define BIGINT(COLUMN)   toBigint(dataItem.COLUMN)

    for (int i = 0; i < data.items.count(); ++i)
    {
        if (threadStop())
            return {RetInfo::Error::Interrupt};

        if (_interrupt)
        {
            setIgnore();
            return {RetInfo::Error::Interrupt};
        }

        if (i % 25 == 0)
        {
            QMutexLocker locker(&_threadLock); (void) locker;
            if ((timer.elapsed() * 1.15) > (_waitTimeout * 1000))
            {
                alog::Line logLine = log_debug2_m << "Database increase wait timeout: "
                                                  << _waitTimeout << " -> ";
                _waitTimeout *= 1.25;

                logLine << _waitTimeout;
            }
        }

        const data::DataItem& dataItem = data.items.at(i);

        bindValue(q, ":GKEY",       dataItem.GKEY        );
        bindValue(q, ":IDSL",       dataItem.IDSL        );
        bindValue(q, ":TIME_MARK",  dataItem.TIME_MARK   );
        bindValue(q, ":OT_PER",     VARCHAR (OT_PER     )); // 1
        bindValue(q, ":MSK_OT",     VARCHAR (MSK_OT     )); // 2
        bindValue(q, ":CODE_MSK",   VARCHAR (CODE_MSK   )); // 3
        bindValue(q, ":VID_MP",     VARCHAR (VID_MP     )); // 4
        bindValue(q, ":USL_OK",     VARCHAR (USL_OK     )); // 5
        bindValue(q, ":PROFIL",     VARCHAR (PROFIL     )); // 6
        bindValue(q, ":MKB1",       VARCHAR (MKB1       )); // 7
        bindValue(q, ":MKB2",       VARCHAR (MKB2       )); // 8
        bindValue(q, ":MKB3",       VARCHAR (MKB3       )); // 9
        bindValue(q, ":CODE_USL",   VARCHAR (CODE_USL   )); // 10
        bindValue(q, ":CODE_MD",    VARCHAR (CODE_MD    )); // 11
        bindValue(q, ":KOL_USL",    DOUBLE  (KOL_USL    )); // 12
        bindValue(q, ":KOL_FACT",   DOUBLE  (KOL_FACT   )); // 13
        bindValue(q, ":ISH_MOV",    VARCHAR (ISH_MOV    )); // 14
        bindValue(q, ":RES_GOSP",   VARCHAR (RES_GOSP   )); // 15
        bindValue(q, ":TARIF_B",    DOUBLE  (TARIF_B    )); // 16
        bindValue(q, ":TARIF_S",    DOUBLE  (TARIF_S    )); // 17
        bindValue(q, ":TARIF_1K",   DOUBLE  (TARIF_1K   )); // 18
        bindValue(q, ":SUM_RUB",    DOUBLE  (SUM_RUB    )); // 19
        bindValue(q, ":VID_TR",     VARCHAR (VID_TR     )); // 20
        bindValue(q, ":EXTR",       SMALLINT(EXTR       )); // 21
        bindValue(q, ":CODE_OTD",   VARCHAR (CODE_OTD   )); // 22
        bindValue(q, ":SOUF",       SMALLINT(SOUF       )); // 23
        bindValue(q, ":SPEC_MD",    VARCHAR (SPEC_MD    )); // 24
        bindValue(q, ":DOMC_TYPE",  VARCHAR (DOMC_TYPE  )); // 25
        bindValue(q, ":OKATO_INS",  VARCHAR (OKATO_INS  )); // 26
        bindValue(q, ":NOVOR",      VARCHAR (NOVOR      )); // 27
        bindValue(q, ":CODE_LPU",   VARCHAR (CODE_LPU   )); // 28
        bindValue(q, ":VID_SF",     VARCHAR (VID_SF     )); // 29
        bindValue(q, ":NHISTORY",   VARCHAR (NHISTORY   )); // 30
        bindValue(q, ":PERSCODE",   VARCHAR (PERSCODE   )); // 31
        bindValue(q, ":DATE_IN",    DATE    (DATE_IN    )); // 32
        bindValue(q, ":DATE_OUT",   DATE    (DATE_OUT   )); // 33
        bindValue(q, ":TARIF_D",    DOUBLE  (TARIF_D    )); // 34
        bindValue(q, ":VID_KOEFF",  VARCHAR (VID_KOEFF  )); // 35
        bindValue(q, ":USL_TMP",    VARCHAR (USL_TMP    )); // 35
        bindValue(q, ":BIRTHDAY",   DATE    (BIRTHDAY   )); // 36
        bindValue(q, ":SEX",        VARCHAR (SEX        )); // 37
        bindValue(q, ":COUNTRY",    VARCHAR (COUNTRY    )); // 38
        bindValue(q, ":SEX_P",      VARCHAR (SEX_P      )); // 39
        bindValue(q, ":BIRTHDAY_P", DATE    (BIRTHDAY_P )); // 40
        bindValue(q, ":INV",        SMALLINT(INV        )); // 41
        bindValue(q, ":DATE_NPR",   DATE    (DATE_NPR   )); // 42
        bindValue(q, ":FOR_POM",    SMALLINT(FOR_POM    )); // 43
        bindValue(q, ":MSE",        SMALLINT(MSE        )); // 44
        bindValue(q, ":P_CEL",      VARCHAR (P_CEL      )); // 45
        bindValue(q, ":DN",         SMALLINT(DN         )); // 46
        bindValue(q, ":TAL_P",      DATE    (TAL_P      )); // 47
        bindValue(q, ":PROFIL_K",   SMALLINT(PROFIL_K   )); // 49
        bindValue(q, ":NAPR_MO",    VARCHAR (NAPR_MO    )); // 48
        bindValue(q, ":MKB0",       VARCHAR (MKB0       )); // 49
        bindValue(q, ":DS_ONK",     SMALLINT(DS_ONK     )); // 50
        bindValue(q, ":VAL_KOEFF",  DOUBLE  (VAL_KOEFF  )); // 51
        bindValue(q, ":C_ZAB",      SMALLINT(C_ZAB      )); // 52
        bindValue(q, ":CODE_NOM1",  VARCHAR (CODE_NOM1  )); // 53
        bindValue(q, ":CODE_NOM2",  VARCHAR (CODE_NOM2  )); // 54
        bindValue(q, ":CODE_NOM3",  VARCHAR (CODE_NOM3  )); // 55
        bindValue(q, ":VAL_TMP",    DOUBLE  (VAL_TMP    )); // 56
        bindValue(q, ":TIME_FIX",   VARCHAR (TIME_FIX   )); // 57
        bindValue(q, ":TIME_IN",    VARCHAR (TIME_IN    )); // 58
        bindValue(q, ":TIME_OUT",   VARCHAR (TIME_OUT   )); // 59
        bindValue(q, ":DATE_FIX",   DATE    (DATE_FIX   )); // 60
        bindValue(q, ":KATEG_MD",   VARCHAR (KATEG_MD   )); // 61
        bindValue(q, ":POST_MD",    VARCHAR (POST_MD    )); // 62
        bindValue(q, ":KOL_DEF",    DOUBLE  (KOL_DEF    )); // 63
        bindValue(q, ":VID_PROV",   VARCHAR (VID_PROV   )); // 64
        bindValue(q, ":MED_AREA",   VARCHAR (MED_AREA   )); // 65
        bindValue(q, ":TAL_HMP",    VARCHAR (TAL_HMP    )); // 66
        bindValue(q, ":DATE_HMP",   DATE    (DATE_HMP   )); // 67
        bindValue(q, ":MCOD_OUT",   VARCHAR (MCOD_OUT   )); // 68
        bindValue(q, ":NOM_NPR",    BIGINT  (NOM_NPR    )); // 69
        bindValue(q, ":SERIES",     VARCHAR (SERIES     )); // 70
        bindValue(q, ":NAME_MSK",   VARCHAR (NAME_MSK   )); // 71
        bindValue(q, ":OKATO_NAS",  VARCHAR (OKATO_NAS  )); // 72
        bindValue(q, ":PR_LG",      VARCHAR (PR_LG      )); // 73

        bindValue(q, ":EKMP",       DOUBLE  (EKMP       )); // 74
        bindValue(q, ":MEE",        DOUBLE  (MEE        )); // 75

        if (!q.exec())
        {
            QString info = "GKEY: " + dataItem.GKEY.toString();
            log_error_m << TASK_SYNC_LOG(TASK_ERR_SQL_INSERT, info);
            return {RetInfo::Error::DBError};
        }
    }

    #undef VARCHAR
    #undef DOUBLE
    #undef SMALLINT
    #undef DATE
    #undef BIGINT

    return {RetInfo::Success};
}

} // namespace task

#undef TASK_SYNC_LOG

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
