#pragma once

#include "sync_transfer.h"
#include "commands/task.h"
#include "commands/time_range.h"
#include "commands/sync_data.h"
#include "database/firebird_driver.h"
#include "shared/safe_singleton.h"

#include <atomic>
#include <QtSql>

namespace task {

using namespace communication;
using namespace db::firebird;

class SyncPlan : public SyncTransfer
{
public:
    bool init();
    QUuidEx taskId()  const {return _taskId;}
    bool isPeriodic() const {return _isPeriodic;}
    bool isSystem()   const {return (_taskType == TaskType::SyncData);}

    // Список планов синхронизации
    static bool sqlList(QSqlQuery& q);

    // Возвращает список текущих прогрессов синхронизации
    static data::SyncInfo::List syncList();

private:
    Q_OBJECT
    DISABLE_DEFAULT_COPY(SyncPlan)
    SyncPlan();

    void run() override;

    // Обновление прогресса выполнения текущей задачи синхронизации
    void progressUpdate();

    /**
      @brief Синхронизация периода данных
      @param taskId - идентификатор задачи, для которой выполняется синхронизация
      @param periodBegin - правая границы выборки
      @param periodEnd - левая границы выборки
      @param periodCurrent - день периода, с которого необходимо начать синхронизацию
      @param timeMark - дата последнего изменения
      @return
    */
    bool sync(QDate periodBegin, QDate periodEnd, QDate startDay);

    /**
      @brief Синхронизация одного дня
      @param taskId - идентификатор задачи, для которой выполняется синхронизация
      @param day - день для которого выполняется синхронизация
      @return
    */
    RetInfo syncIteration(const QDate& dayStart, const QDate& dayEnd, QDateTime& lastTimemark);

    /**
      @brief Проверка актуальности данных для периода checkPeriod.
      @param period   - период данных
      @param needSync - флаг, получающий значение true в том случае если данные
                        требуют обновления. false в противном случае.
      @return
    */
    RetInfo checkPeriod(QDate periodBegin, QDate periodEnd, bool& needSync);

    // Вычисление контрольной суммы и количества строк полученных от ФОМС
    // сервера
    bool calcCrc(Transaction::Ptr transact, quint64& crc/*, quint64 count*/);

    /**
      @brief Подсчёт количества записей в таблице 'tablename'
      @param transact - текущая транзакция
      @return
    */
    bool countRecords(Transaction::Ptr transact, const QString& tablename, quint64& recCount);

    /**
      @brief Сохранение строк данных data во временную таблицу.
      @param transact
      @param data
      @return
    */
    RetInfo saveToDB(Transaction::Ptr transact, const data::GetSyncData& data);

private:
    // Идентификаторы текущего плана
    QUuidEx _taskId;
    QUuidEx _userId;

    // Признак периодичности для текущего плана
    bool _isPeriodic = {false};

    // Признак системной задачи синхронизации
    TaskType _taskType = {TaskType::Undefined};

    // Количество строк данных на прием
    qint32 _recvSize;

    // Количество строк данных, получаемых за одну итерацию синхронизации
    qint32 _bufferSize;

    data::TaskProgress _progress;

    template<typename T, int> friend T& ::safe_singleton();
};

SyncPlan& syncPlan();

} // namespace task
