#include "sync_transfer.h"
#include "functions.h"
#include "task_messages.h"
#include "commands/commands.h"
#include "database/connect.h"

#include "shared/defmac.h"
#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "SyncTrans")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "SyncTrans")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "SyncTrans")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "SyncTrans")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "SyncTrans")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "SyncTrans")

namespace task {

using namespace communication;
using namespace communication::transport;

SyncTransfer::SyncTransfer()
{
    chk_connect_d(fomsCon().socket().get(), SIGNAL(message(communication::Message::Ptr)),
                  this, SLOT(message(communication::Message::Ptr)))

    _waitTimeout = _defaultWaitTimeout;
}

void SyncTransfer::awake()
{
    QMutexLocker locker(&_threadLock); (void) locker;
    _threadCond.wakeAll();
}

void SyncTransfer::interrupt()
{
    _interrupt = true;
    awake();
}

void SyncTransfer::threadStopEstablished()
{
    dbpool().abortOperation(_threadId);
    awake();
}

void SyncTransfer::message(const communication::Message::Ptr& message)
{
    if (message->processed())
        return;

    { //Block for QMutexLocker
        QMutexLocker locker(&_threadLock); (void) locker;
        for (const QUuidEx id : _messageIgnores)
            if (message->id() == id)
            {
                log_debug2_m << "Message ignored"
                             << ". Id: " << message->id()
                             << ". Command: " << CommandNameLog(message->command());
                return;
            }
    }

    if (message->command() == command::KeepWaitCommand)
    {
        data::KeepWaitCommand keepWaitCommand;
        readFromMessage(message, keepWaitCommand);

        if (_funcInvoker.containsCommand(keepWaitCommand.commandId))
        {
            if (_messageId == keepWaitCommand.messageId)
            {
                QMutexLocker locker(&_threadLock); (void) locker;
                log_debug2_m << "FOMS increase wait timeout: "
                             << _waitTimeout << " -> "
                             << (_waitTimeout + keepWaitCommand.timeToAdd)
                             << " for command "
                             << command::pool().commandName(keepWaitCommand.commandId)
                             << " (" << _messageId << ")";
                _waitTimeout += keepWaitCommand.timeToAdd;
            }
        }
        return;
    }

    if (_funcInvoker.containsCommand(message->command()))
    {
        if (!command::pool().commandIsMultiproc(message->command()))
            message->markAsProcessed();

        QMutexLocker locker(&_threadLock); (void) locker;
        if (_messageId.isNull())
        {
            // Если идентификатор сообщения не определен, значит мы не ждем
            // прихода сообщений
            return;
        }
        _messageAnswer = message;
        _threadCond.wakeAll();
    }
}

RetInfo SyncTransfer::send(const Message::Ptr& message)
{
    QMutexLocker locker(&_threadLock); (void) locker;
    _messageAnswer.reset();
    _messageId = QUuidEx();
    _interrupt = false;

    message->setMaxTimeLife(_defaultWaitTimeout - 2);

    if (!fomsCon().send(message))
    {
        log_error_m << EventLog(TASK_ERR_NO_FOMS);
        return {RetInfo::Error::Foms};
    }

    log_debug2_m << "Message sent"
                 << ". Id: " << message->id()
                 << ". Command: " << CommandNameLog(message->command());

    _messageId = message->id();
    _messageTimer.reset();
    _waitTimeout = _defaultWaitTimeout;
    return {RetInfo::Success};
}

void SyncTransfer::setIgnore()
{
    QMutexLocker locker(&_threadLock); (void) locker;
    _messageIgnores.append(_messageId);

    while (_messageIgnores.count() > 50)
        _messageIgnores.removeAt(0);
}

RetInfo SyncTransfer::waitMessage(Message::Ptr& answer, const QUuidEx& expectCommand)
{
    answer.reset();

    while (true)
    {
        CHECK_QTHREADEX_STOP

        if (!fomsCon().isAuthorized())
        {
            QMutexLocker locker(&_threadLock); (void) locker;
            _messageId = QUuidEx();

            emit commandProcessed(false);
            return {RetInfo::Error::Foms};
        }

        if (_interrupt)
        {
            // Аналог вызова setIgnore()
            QMutexLocker locker(&_threadLock); (void) locker;
            _messageIgnores.append(_messageId);
            _messageId = QUuidEx();

            log_debug_m << "WaitMessage interrupted, beacuse task was interrupted";
            break;
        }

        QUuidEx messageId;
        { //Block for QMutexLocker
            QMutexLocker locker(&_threadLock); (void) locker;
            int64_t elapsed = _messageTimer.elapsed();
            if (elapsed > (_waitTimeout * 1000))
            {
                log_debug2_m << "Message timer elapsed: " << elapsed
                             << ". Wait timeout: " << _waitTimeout * 1000 << " ms";

                QString commandName = command::pool().commandName(expectCommand);
                log_error_m << EventLog(TASK_ERR_ANSWER_TIMEOUT, commandName, _messageId);

                _messageId = QUuidEx();

                emit commandProcessed(false);
                return {RetInfo::Error::Foms};
            }

            if (_messageAnswer.empty())
            {
                _threadCond.wait(&_threadLock, 1000);
                continue;
            }
            answer = std::move(_messageAnswer);
            messageId = _messageId;
            _messageId = QUuidEx();
        }

        log_debug2_m << "Message recieved"
                     << ". Id: " << answer->id()
                     << ". Command: " << CommandNameLog(answer->command());

        if (answer->command() != expectCommand)
        {
            QString answerName = command::pool().commandName(answer->command());
            QString commandName = command::pool().commandName(expectCommand);
            log_error_m << EventLog(TASK_ERR_ANSWER_COMMAND, commandName, answerName);

            emit commandProcessed(false);
            return {RetInfo::Error::Foms};
        }
        if (answer->id() != messageId)
        {
            log_error_m << EventLog(TASK_ERR_ANSWER_ID, messageId, answer->id());

            emit commandProcessed(false);
            return {RetInfo::Error::Foms};
        }
        if (answer->type() != Message::Type::Answer)
        {
            QString answerName = command::pool().commandName(answer->command());
            log_error_m << EventLog(TASK_ERR_ANSWER_TYPE, answerName, answer->id());

            emit commandProcessed(false);
            return {RetInfo::Error::Foms};
        }

        if (answer->execStatus() == Message::ExecStatus::Failed)
        {
            QString msg = errorDescription(answer);
            log_error_m << EventLog(TASK_ERR_ANSWER_FAILED, msg);

            emit commandProcessed(false);
            return {RetInfo::Error::Foms};
        }

        // Обработка ошибок отправленных с ФОМС сервера
        if (answer->execStatus() == Message::ExecStatus::Error)
        {
            data::MessageError error;
            readFromMessage(answer, error);
            log_error_m << EventLog(TASK_ERR_ANSWER_FAILED,  error.description);

            emit commandProcessed(false);
            return {RetInfo::Error::Foms};
        }

        emit commandProcessed(true);
        return {RetInfo::Success};

    } // while (true)

    return {RetInfo::Error::Interrupt};
}

void SyncTransfer::command_Dummy(const Message::Ptr& message)
{
    (void) message;
}

} // namespace task

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
