#pragma once

#include "ret_info.h"

#include "shared/simple_timer.h"
#include "shared/thread/thread_utils.h"
#include "shared/qt/communication/message.h"
#include "shared/qt/communication/func_invoker.h"
#include "shared/qt/thread/qthreadex.h"

#include <QtCore>

namespace task {

using namespace communication;

/**
  В классе реализован бызовый механизм для синхронной передачи/приема сообщений
*/
class SyncTransfer : public QThreadEx
{
public:
    /**
      Пробуждение потока
    */
    void awake();

    /**
      Прерывание процесса синхронизации, с последующием перечитыванием плана.
    */
    void interrupt();

    /**
      Фиктивный обработчик сообщений, нужен для механизма регистрации команд
    */
    void command_Dummy(const Message::Ptr&);

signals:
    // Сигнал эмитируется со значением TRUE, когда последняя команда обработана
    // успешно, в противном случае значение будет FALSE
    void commandProcessed(bool);

public slots:
    void message(const communication::Message::Ptr&);

protected:
    SyncTransfer();

    void threadStopEstablished() override;

    /**
      Отправка сообщения с проверкой подключения
    */
    RetInfo send(const Message::Ptr& message);

    /**
      Когда текущее ожидаемое сообщение становится более не актуальным, его
      можно проигнорировать
    */
    void setIgnore();

    /**
      Ожидание сообщения
      answer - полученное сообщение
      command - ожидаемая команда

      Если сообщение было получено в течении заданного таймаута, то сообщение
      помещается в answer, и метод возвращает TRUE.
      Если сообщение не было получени в течении заданного таймаута, то метод
      возвращает FALSE.
      Метод возвращает FALSE в случае, если идентификатор ответного сообщения
      и идентификатор отвеной команды не совпдают.
    */
    RetInfo waitMessage(Message::Ptr& answer, const QUuidEx& expectCommand);

protected:
    QMutex _threadLock;
    QWaitCondition _threadCond;

    // Регистратор обрабатываемых команд
    FunctionInvoker _funcInvoker;

    // Ожидаемое сообщение-ответ
    Message::Ptr _messageAnswer;

    // Идентификатор отправленного сообщения, используется для контроля правиль-
    // ности ожидаемого ответа
    QUuidEx _messageId;

    // Таймер контролирует время ожидания ответа
    simple_timer _messageTimer;

    // Список идентификаторов сообщений, которое необходимо проигнорировать,
    // так как ответ уже не нужен
    QList<QUuidEx> _messageIgnores;

    // Для прерывания в режиме ожидания сообщения
    std::atomic_bool _interrupt = {false};

    int _waitTimeout = {0};
    int _defaultWaitTimeout = {60};

    // Идентификатор потока
    pid_t _threadId = {0};

private:
    Q_OBJECT
    DISABLE_DEFAULT_COPY(SyncTransfer)
};

} // namespace task

