#include "user_rights.h"
#include "functions.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "shared/simple_ptr.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "RightsPool")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "RightsPool")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "RightsPool")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "RightsPool")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "RightsPool")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "RightsPool")

using namespace sql;

UserRight& uright()
{
    return ::safe_singleton<UserRight>();
}

uint qHash(UserRight::Permission type)
{
    return static_cast<uint>(type);
}

bool UserRight::init()
{
    return updatePermissions();
}

bool UserRight::updatePermissions()
{
    QHashEx<Permission, Level> defaultPermiss;
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT LEARN_MODEL, APPLY_MODEL, REPORT, MONITORING "
        " FROM GROUPS WHERE IS_DEFAULT = 1"))
    {
        return false;
    }
    if (q.first())
    {
        User dummy;
        assignPermiss(&dummy, q.record());
        defaultPermiss = dummy.permissions;
    }
    else
    {
        defaultPermiss[Permission::LearnModel] = Level::Disable;
        defaultPermiss[Permission::ApplyModel] = Level::Disable;
        defaultPermiss[Permission::Reports]    = Level::Disable;
        defaultPermiss[Permission::Monitoring] = Level::Disable;
    }

    if (!sql::exec(q,
        " SELECT                          "
        "   U.ID                          "
        "  ,U.HASH_ID                     "
        "  ,U.LOGIN                       "
        "  ,U.NAME                        "
        "  ,U.IS_ADMIN                    "
        "  ,U.IS_ACTIVE                   "
        "  ,U.GROUP_ID                    "
        "  ,G.LEARN_MODEL                 "
        "  ,G.APPLY_MODEL                 "
        "  ,G.REPORT                      "
        "  ,G.MONITORING                  "
        " FROM                            "
        "   USERS U                       "
        " LEFT JOIN                       "
        "   GROUPS G ON U.GROUP_ID = G.ID "))
    {
        return false;
    }

    User::List users;
    while (q.next())
    {
        User* usr = users.add();
        QSqlRecord r = q.record();

        assignValue(usr->id,       r, "ID       ");
        assignValue(usr->hashId,   r, "HASH_ID  ");
        assignValue(usr->login,    r, "LOGIN    ");
        assignValue(usr->name,     r, "NAME     ");
        assignValue(usr->isAdmin,  r, "IS_ADMIN ");
        assignValue(usr->isActive, r, "IS_ACTIVE");

        if (r.field("GROUP_ID").isNull())
            usr->permissions = defaultPermiss;
        else
            assignPermiss(usr, r);
    }
    users.sort();

    QMutexLocker locker(&_lock); (void) locker;
    _users.swap(users);
    return true;
}

bool UserRight::updatePermissions(quint64 userHashId)
{
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    if (!sql::exec(q,
        " SELECT                          "
        "   U.ID                          "
        "  ,U.HASH_ID                     "
        "  ,U.LOGIN                       "
        "  ,U.NAME                        "
        "  ,U.IS_ADMIN                    "
        "  ,U.IS_ACTIVE                   "
        "  ,U.GROUP_ID                    "
        "  ,G.LEARN_MODEL                 "
        "  ,G.APPLY_MODEL                 "
        "  ,G.REPORT                      "
        "  ,G.MONITORING                  "
        " FROM                            "
        "   USERS U                       "
        " LEFT JOIN                       "
        "   GROUPS G ON U.GROUP_ID = G.ID "
        " WHERE                           "
        "   U.HASH_ID = ?                 ", userHashId))
    {
        return false;
    }
    if (q.first())
    {
        simple_ptr<User> usr {new User};

        QSqlRecord r = q.record();
        assignValue(usr->id,       r, "ID       ");
        assignValue(usr->hashId,   r, "HASH_ID  ");
        assignValue(usr->login,    r, "LOGIN    ");
        assignValue(usr->name,     r, "NAME     ");
        assignValue(usr->isAdmin,  r, "IS_ADMIN ");
        assignValue(usr->isActive, r, "IS_ACTIVE");

        assignPermiss(usr.get(), r);

        QMutexLocker locker(&_lock); (void) locker;
        lst::FindResult fr = _users.findRef(userHashId);
        if (fr.success())
            _users.replace(fr.index(), usr.release(), true);
        else
            _users.addInSort(usr.release(), fr);
    }
    else
    {
        QMutexLocker locker(&_lock); (void) locker;
        if (lst::FindResult fr = _users.findRef(userHashId))
            _users.remove(fr.index());
    }
    return true;
}

void UserRight::assignPermiss(User* usr, const QSqlRecord& record)
{
    short val = 0;
    assignValue(val, record, "LEARN_MODEL");
    usr->permissions[Permission::LearnModel] = permissLevel(val);

    val = 0;
    assignValue(val, record, "APPLY_MODEL");
    usr->permissions[Permission::ApplyModel] = permissLevel(val);

    val = 0;
    assignValue(val, record, "REPORT");
    usr->permissions[Permission::Reports] = permissLevel(val);

    val = 0;
    assignValue(val, record, "MONITORING");
    usr->permissions[Permission::Monitoring] = permissLevel(val);
}

UserRight::Level UserRight::permissLevel(short val)
{
    Level level = Level::Disable;
    if (val == 1)
        level = Level::Read;
    else if (val == 2)
        level = Level::Write;

    return level;
}

bool UserRight::isAdmin(quint64 userHashId) const
{
    //break_point

    QMutexLocker locker(&_lock); (void) locker;
    if (_webHashId && (_webHashId == userHashId))
        return true;

    if (lst::FindResult fr = _users.findRef(userHashId))
        return _users.item(fr.index())->isAdmin;

    return false;
}

bool UserRight::isAdmin(const QUuidEx& userId) const
{
    return isAdmin(hash64(userId));
}

bool UserRight::isWebAppl(quint64 userHashId) const
{
    QMutexLocker locker(&_lock); (void) locker;
    if (_webHashId && (_webHashId == userHashId))
        return true;

    return false;
}

quint64 UserRight::webHashId() const
{
    QMutexLocker locker(&_lock); (void) locker;
    return _webHashId;
}

void UserRight::setWebHashId(quint64 val)
{
    QMutexLocker locker(&_lock); (void) locker;
    _webHashId = val;
}

QUuidEx UserRight::userId(quint64 userHashId) const
{
    QMutexLocker locker(&_lock); (void) locker;
    if (lst::FindResult fr = _users.findRef(userHashId))
        return _users.item(fr.index())->id;

    return {};
}

QString UserRight::userLogin(quint64 userHashId) const
{
    QMutexLocker locker(&_lock); (void) locker;
    if (lst::FindResult fr = _users.findRef(userHashId))
        return _users.item(fr.index())->login;

    return {};
}

QString UserRight::userName(quint64 userHashId) const
{
    QMutexLocker locker(&_lock); (void) locker;
    if (lst::FindResult fr = _users.findRef(userHashId))
        return _users.item(fr.index())->name;

    return {};
}

QString UserRight::userLogin(const QUuidEx& userId) const
{
    return userLogin(hash64(userId));
}

QString UserRight::userName (const QUuidEx& userId) const
{
    return userName(hash64(userId));
}

bool UserRight::checkPermission(quint64 userHashId, Permission type, Level level) const
{
    break_point

    QMutexLocker locker(&_lock); (void) locker;
    if (lst::FindResult fr = _users.findRef(userHashId))
    {
        User* u = _users.item(fr.index());
        if (!u->isActive)
            return false;

        Level lvl = u->permissions[type];
        return (lvl >= level);
    }
    return false;
}

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
