#include "options.h"
#include "event_log.h"
#include "database/connect.h"
#include "database/sql_func.h"
#include "shared/qt/config/config.h"
#include "qt/logger/logger_operators.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "XGB")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "XGB")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "XGB")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "XGB")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "XGB")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "XGB")

namespace xgb { // XGBoost

/**
  @brief Первоначальная инициализация настроек XGBoost
  @return true - если ошибок не возникло, false - в противном случае.
  @details Данная функция производит чтение настроек из конфиг-файла.
           Все значения настроек, которые пристуствуют в конфиг-файле
           перезапишут значения которые сейчас находятся в БД.
*/
bool init()
{
    // Прочитать default-настройки из конфиг файла
    QMap<QString, QString> defOptions;
    YamlConfig::Func loadFunc = [&](YamlConfig* conf, YAML::Node& xgbOptions, bool)
    {
        using namespace std;
        for (auto it = xgbOptions.begin(); it != xgbOptions.end(); ++it)
        {
            string name = it->first.as<string>("");
            if (name.empty())
                continue;

            QString content;
            conf->getValue(xgbOptions, name, content, true);
            if (content.isEmpty())
                continue;

            if (name == "nthread")
            {
                int threadCount = QThread::idealThreadCount();
                int nt = qMin(content.toInt(), threadCount - 2);
                if (nt <= 0)
                    nt = qMax(1, threadCount - 2);

                content = QString::number(nt);
                log_debug_m << "Param 'nthread' set to " << content;
            }

            defOptions.insert(QString::fromStdString(name), content);
        }
        return true;
    };
    config::base().getValue("xgboost.ekmp", loadFunc);

    // Перезаписать default-профиль в БД настройками прочитанными из файла
    db::firebird::Driver::Ptr dbcon = dbpool().connect();
    QSqlQuery q {dbcon->createResult()};

    for (const QString& key : defOptions.keys())
    {
        QString content = defOptions.value(key);

        if (!sql::exec(q,
            " UPDATE OR INSERT INTO MODEL_XGB "
            " (ID, KIND, NAME, CONTENT, ENABLED)    "
            " VALUES                          "
            " (?, ?, ?, ?, ?)                    "
            " MATCHING (ID, KIND, NAME)             ",
            QUuidEx(), 1, key, content, true))
        {
            return false;
        }
    }

    defOptions.clear();
    config::base().getValue("xgboost.mee", loadFunc);

    for (const QString& key : defOptions.keys())
    {
        QString content = defOptions.value(key);

        if (!sql::exec(q,
            " UPDATE OR INSERT INTO MODEL_XGB "
            " (ID, KIND, NAME, CONTENT, ENABLED)    "
            " VALUES                          "
            " (?, ?, ?, ?, ?)                    "
            " MATCHING (ID, KIND, NAME)             ",
            QUuidEx(), 0, key, content, true))
        {
            return false;
        }
    }
    return true;
}
/**
  @brief Применение настроек к хендлу XGBoost
  @param bh - хэндл XGBoost
  @param options - список настроек для применения
  @return
*/
task::RetInfo applyOptions(BoosterHandle bh, const QMap<QString, QString>& options)
{
    task::RetInfo retInfo {task::RetInfo::Success};

    for (const QString& key : options.keys())
    {
        //QString value = options.value(key);
        int res = XGBoosterSetParam(bh,
                                    key.toStdString().c_str(),
                                    options.value(key).toStdString().c_str());

        if (res)
        {
            log_error_m << EventLog(u8"Ошибка вызова функции XGBoosterSetParam(). Key '%1'", key);
            retInfo = {task::RetInfo::Error::Xgboost,
                       //"XGBoosterSetParam(" + key + ") error: " + QString::number(res),
                       //"XGBoosterSetParam(%1) error: %2", key, res,
                       task::RetInfo::Critical::Yes};
        }
    }
    return retInfo;
}

} // namespace xgb

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
