/*****************************************************************************
  В модуле представлен список идентификаторов команд для коммуникации между
  клиентской и серверной частями приложения.
  В данном модуле представлен список команд персональный для этого приложения.

  Требование надежности коммуникаций: однажды назначенный идентификатор коман-
  ды не должен более меняться.
*****************************************************************************/

#pragma once

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/** WEB
  Команда добавляет роли пользователей
*/
extern const QUuidEx GroupCreate;

/** WEB
  Команда обновляет роли пользователей
*/
extern const QUuidEx GroupEdit;

/** WEB
    Удаление роли пользователей
*/
extern const QUuidEx GroupDelete;

/** WEB
  Получение роли по id
*/
extern const QUuidEx GroupInfo;

/** WEB
    Список ролей пользователей
*/
extern const QUuidEx GroupList;

/** WEB
    Привязка пользователя к группе
*/
extern const QUuidEx AssignGroup;

} // namespace command


//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct Group
{
    enum class Level : quint32
    {
        Disable = 0,
        Read    = 1,
        Write   = 2
    };

    QUuidEx id;
    QString name;
    Level   canLearnModel = Level::Disable;
    Level   canApplyModel = Level::Disable;
    Level   canRWReport   = Level::Disable;
    Level   canReportFed  = Level::Disable;
    Level   canMonitoring = Level::Disable;
    bool    isDefault     = false;
    quint16 userCount     = 0;

    //Group() = default;
    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id            )
        J_SERIALIZE_ITEM( name          )
        J_SERIALIZE_ITEM( canLearnModel )
        J_SERIALIZE_ITEM( canApplyModel )
        J_SERIALIZE_ITEM( canRWReport   )
        J_SERIALIZE_OPT ( canReportFed  )
        J_SERIALIZE_ITEM( canMonitoring )
        J_SERIALIZE_ITEM( isDefault     )
        J_SERIALIZE_ITEM( userCount     )
    J_SERIALIZE_END
};

struct GroupCreate : Group, Data<&command::GroupCreate,
                                  Message::Type::Command,
                                  Message::Type::Answer>
{};

struct GroupEdit : Group, Data<&command::GroupEdit,
                                Message::Type::Command,
                                Message::Type::Answer>
{};

struct GroupDelete : Data<&command::GroupDelete,
                           Message::Type::Command,
                           Message::Type::Answer>
{
    QUuidEx id;
    J_SERIALIZE_ONE( id )
};

struct GroupDeleteFail : MessageFailed,
                         Data<&command::GroupDelete,
                               Message::Type::Answer>
{
    QUuidEx id;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_BASE( MessageFailed )
        J_SERIALIZE_OPT ( id )
    J_SERIALIZE_END
};

struct GroupInfo : Data<&command::GroupInfo,
                         Message::Type::Command>
{
    QUuidEx id; // id роли, по которой запрашивается информация
    J_SERIALIZE_ONE( id )
};

struct GroupInfoA : Group, Data<&command::GroupInfo,
                                 Message::Type::Answer>
{};

struct GroupList : Data<&command::GroupList,
                         Message::Type::Command,
                         Message::Type::Answer>
{
    QVector<Group> items;
    J_SERIALIZE_ONE( items )
};

struct AssignGroup : Data<&command::AssignGroup,
                           Message::Type::Command,
                           Message::Type::Answer>
{
    QUuidEx groupId;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( groupId )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


