#pragma once

#include "filter.h"
#include "paging.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------


/** WEB
  Команда может редактировать только описание модели
*/
extern const QUuidEx ModelEdit;

/** WEB
*/
extern const QUuidEx ModelDelete;

/** WEB
*/
extern const QUuidEx ModelInfo;

/** WEB
*/
extern const QUuidEx ModelList;

} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct Model
{
    // Идентификатор модели
    QUuidEx id;

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя
    QUuidEx userId;

    // Имя модели
    QString name;

    // Описание модели
    QString description;

    // Время создания модели
    QDateTime createDate = {QDateTime::currentDateTime()};

    // Левая и правая граница интервала выборки данных для обучения модели.
    // Значение актуально только для разовой задачи
    TimeRange period;

    // TODO удалить
    // Фактическая левая и правая граница интервала выборки данных для обучения
    // модели
    TimeRange periodReal;

    // Признак определяет, что модель будет доступна всем пользователям
    bool isPublic = {false};

    // Признак архивной модели, если равно TRUE, то модель не будет отображается
    // в интерфейсе пользователя
    bool isArchive = {false};

    // Количество строк, изменившихся в процессе синхронизации после создания
    // модели
    qint32 rowsOutdated = 0;

    //--- Вспомогательные поля для WEB-интерфейса ---
    // Признак определяет, что модель была создана периодической задачей
    bool isPeriodic = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id          )
        J_SERIALIZE_ITEM( taskId      )
        J_SERIALIZE_ITEM( userId      )
        J_SERIALIZE_ITEM( name        )
        J_SERIALIZE_ITEM( description )
        J_SERIALIZE_ITEM( createDate  )
        J_SERIALIZE_ITEM( period      )
        J_SERIALIZE_OPT ( periodReal  )
        J_SERIALIZE_ITEM( isPublic    )
        J_SERIALIZE_OPT ( isArchive   )
        J_SERIALIZE_OPT ( isPeriodic  )
    J_SERIALIZE_END
};

struct ModelFilter
{
    // Идентификатор модели
    QUuidEx id;

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя (владельца модели)
    QUuidEx userId;

    // Полное или частичное имя модели
    QString name;

    // Отображать публичные модели других пользователей
    filter::Show show = {filter::Show::Self};

    // Отображать архивные модели (по умолчанию - не отображать)
    filter::Archive archive = {filter::Archive::No};

    // Идентификаторы дополнительной фильтрации по пользователям, используются
    // совместно с filter::Show
    QVector<QUuidEx> otherUsers;

    // Фильтр "Дата создания"
    TimeRange createDate;

    // Фильтр "Период данных"
    TimeRange dataPeriod;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id         )
        J_SERIALIZE_ITEM( taskId     )
        J_SERIALIZE_ITEM( userId     )
        J_SERIALIZE_ITEM( name       )
        J_SERIALIZE_ITEM( show       )
        J_SERIALIZE_OPT ( archive    )
        J_SERIALIZE_ITEM( otherUsers )
        J_SERIALIZE_ITEM( createDate )
        J_SERIALIZE_ITEM( dataPeriod )
    J_SERIALIZE_END
};

struct ModelEdit : Data<&command::ModelEdit,
                         Message::Type::Command,
                         Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;
    QString name;
    QString description;
    bool    isPublic = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id          )
        J_SERIALIZE_ITEM( userId      )
        J_SERIALIZE_ITEM( name        )
        J_SERIALIZE_ITEM( description )
        J_SERIALIZE_ITEM( isPublic    )
    J_SERIALIZE_END
};

struct ModelDelete : Data<&command::ModelDelete,
                           Message::Type::Command,
                           Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;

    // Признак принудительного удаления модели, если равен TRUE, то модель
    // будет удалена даже если у нее есть зависимости
    bool force = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
        J_SERIALIZE_OPT ( force  )
    J_SERIALIZE_END
};

struct ModelInfo : Data<&command::ModelInfo,
                         Message::Type::Command>
{
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct ModelInfoA : Model, Data<&command::ModelInfo,
                                 Message::Type::Answer>
{
};

struct ModelList : Data<&command::ModelList,
                         Message::Type::Command,
                         Message::Type::Answer>
{
    // Фильтр выбора данных
    ModelFilter filter;

    // Фильтр для постраничной выборки данных
    PagingInfo paging;

    // Результирующая выборка
    QVector<Model> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( paging )
        J_SERIALIZE_ITEM( items  )
    J_SERIALIZE_END
};


} // namespace data
} // namespace communication


