#pragma once

#include "sync_data.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

/** Описание механизма экспорта данных **

  В блоке представлен список идентификаторов команд для коммуникации между
  клиентской (АИС Эксперт) и серверной (ФОМС Сервер) частями приложения.

  Коммуникация между Клиентом и Сервером состоит из
  1) Получение данных от ФОМС Сервра в АИС Эксперт - синхронизация данных.
  2) Отправка данных от АИС Эксперт на ФОМС Сервер - экспорт оценок.

  Экспорт оценок.

  Экспорт оценок реализуется командой SendScore.

  Процесс экспорт оценок состоит из цикла отправки АИС команды SendScore с
  заполненными параметрами:
    id          - идентификатор оценки
    name        - название оценки
    descript    - описание оценки
    createDate  - дата создания оценки
    period      - период данных для оценки
    userId      - идентификатор пользователя, создавшего оценку
    userName    - имя пользователя, создавшего оценку
    modelId     - идентификатор модели, на базе которой создана оценка
    modelName   - название модели, на базе которой создана оценка
    modelDescr  - описание модели, на базе которой создана оценка
    modelPeriod - период данных, на котором обучена модель
    isFirst     - флаг-признак 'первый блок'
    isLast      - флаг-признак 'последний блок'
  АИС производит выборку всех данных оценки,  для которой выполняется синхрони-
  зация. Далее  выполняется  последовательно  заполнение  и  отправка  команды
  SendScore на ФОМС сервер. Параметр 'isLast' последнего блока устанавливается
  в значение TRUE.

  После экспорта всех данных происходит  проверка  отправленных  данных,  путем
  вычисления контрольных сумм на стороне АИС и на стороне ФОМС. В случае совпа-
  дения вычисленных контрольных сумм экспорт оценок считается успешным.
  При несовпадении контрольных сумм процедура экспорта оценок  будет  выполнена
  повторно. Вычисление и проверка контрольной суммы отправленных  данных реали-
  зуется командой SendScoreCheck.

  Команда SendScoreCheck содержит параметры:
    id    - Идентификатор оценки
    count - Количество строк данных;
    crc   - Контрольная сумма

  При получении данной команды, ФОМС Сервер вычисляет контрольную сумму записей
  оценки,  которая  была экспортирована  из  системы  АИС Эксперт.  Вычисленная
  контрольная сумма записывается в 'crc' и отправляется в АИС.
  Для подсчёта контрольной суммы необходимо взять все записи  с идентификатором
  экспортированного применения  оценок - данный  идентификатор  АИС  отправляет
  в поле id. Далее необходимо отсортировать  полученные  записи  по возрастанию
  используя идентификатор случая.
  После необходимо выполнить расчёт crc согласно алгоритму:
    uint64 crc = 0;
    while (query.next()) {
      QUuidEx idsl;
      assignValue(idsl, query.record(), "IDSL");

      uint64 p1 = ((quint64*) &idsl)[0];
      uint64 p2 = ((quint64*) &idsl)[1];
      uint64 pCrc = (p1 XOR p2);
      crc = crc XOR pCrc;
    }

  В том случае, если crc отправленная в АИС из ФОМС сервера совпадает со значе-
  нием вычисленным АИС, то в ФОМС сервер отправляется  команда  SendScoreCrcOk.
  Данная команда подтверждает тот факт, что контрольные  суммы  на стороне  АИС
  и ФОМС совпадают.
  Команда SendScoreCrcOk заполняется следующим образом:
    id    - Идентификатор оценки, для которой выполнялся экспорт
    crcOk - Результат сравнения crc. Если crc АИС и ФОМС совпадают, то данное
            поле равно TRUE, в противном случае - FALSE.

  Команда SendScoreCrcOk, отправляется только в том случае, если данная команда
  поддерживается ФОМС сервером.

  Проверка поддержки команды  на ФОМС  сервере  реализуется  отправкой  команды
  SupportCommand. Команда  SupportCommand  содержит  идентификатор  проверяемой
  команды.  В том случае, если  команда  поддерживается,  необходимо  отправить
  ответное сообщение, присвоив параметру isSupport значение TRUE.
  Если команда не поддерживается, то параметр isSupport должен быть равен FALSE.
*/

//----------------------------- Список команд --------------------------------

/** WEB
  Команда используется для запуска механизма экспорта данных по оценке
*/
extern const QUuidEx NeedSendScore;

/** FOMS
  Команда используется для отправки в ФОМС данных по оценке.  Команда содержит
  все сведения об оценке,  модели и пользователе.  Эта  информация  передается
  каждый раз при отправке блока с оценками. Если ФОМС присылает  положительный
  ответ на эту команду, то АИС отправляет следующий блок. При отправке послед-
  него блока данных поле isLast равно TRUE - таким образом принимающая сторона
  уведомляется, что данное сообщение последнее,  и на  этом  пересылка  данных
  завершена
*/
extern const QUuidEx SendScore;

/** FOMS
  Проверка корректности отправки оценок на ФОМС сервер. Данной командой
  проверяется количество переданных строк(оценок) и контрольная сумма оценок.
*/
extern const QUuidEx SendScoreCheck;

/** FOMS
  Уведомление ФОМС сервера о том, что проверка контрольной суммы была успешной.
  После отправки данной команды, попыток отправки скоров не будет. За исключением
  ситуации, когда отправка будет инициирована пользователем вручную.
*/
extern const QUuidEx SendScoreCrcOk;

} // namespace command

namespace data {

struct NeedSendScore : Data<&command::NeedSendScore,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    // Идентификатор оценки
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct SendScoreItem
{
    // Идентификатор "под"-случая
    QUuidEx IDSL;
    // Оценка по МЭЭ
    double MEE;
    // Оценка по ЭКМП
    double EKMP;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( IDSL )
        J_SERIALIZE_ITEM( MEE  )
        J_SERIALIZE_ITEM( EKMP )
    J_SERIALIZE_END
};

struct SendScore : Data<&command::SendScore,
                         Message::Type::Command>
{
    // Идентификатор оценки
    QUuidEx id;

    // Название оценки
    QString name;

    // Описание оценки
    QString descript;

    // Время создания оценки
    QDateTime createDate;

    // Левая и правая граница интервала выборки данных для оценки
    TimeRange period;

    // Идентификатор пользователя
    QUuidEx userId;
    QString userName;

    // Идентификатор модели
    QUuidEx    modelId;
    QString    modelName;
    QString    modelDescr;
    TimeRange  modelPeriod;

    // Признак первого набора данных
    bool isFirst = {false};

    // Признак последнего набора данных
    bool isLast = {false};

    // Список элементов
    QVector<SendScoreItem> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id          )
        J_SERIALIZE_ITEM( name        )
        J_SERIALIZE_ITEM( descript    )
        J_SERIALIZE_ITEM( createDate  )
        J_SERIALIZE_ITEM( period      )
        J_SERIALIZE_ITEM( userId      )
        J_SERIALIZE_ITEM( userName    )
        J_SERIALIZE_ITEM( modelId     )
        J_SERIALIZE_ITEM( modelName   )
        J_SERIALIZE_ITEM( modelDescr  )
        J_SERIALIZE_ITEM( modelPeriod )
        J_SERIALIZE_OPT ( isFirst     )
        J_SERIALIZE_ITEM( isLast      )
        J_SERIALIZE_ITEM( items       )
    J_SERIALIZE_END
};

struct SendScoreA : Data<&command::SendScore,
                          Message::Type::Answer>
{
    // Идентификатор оценки
    QUuidEx id;
    J_SERIALIZE_ONE( id )
};

struct SendScoreCheck : Data<&command::SendScoreCheck,
                              Message::Type::Command,
                              Message::Type::Answer>
{
    // Идентификатор оценки
    QUuidEx id;

    // Количество строк данных
    quint64 count = {0};

    // Контрольная сумма, вычисляется как hash64
    // от идентификаторов случаев (IDSL)
    quint64 crc = {0};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id    )
        J_SERIALIZE_ITEM( count )
        J_SERIALIZE_ITEM( crc   )
    J_SERIALIZE_END
};

struct SendScoreCrcOk : Data<&command::SendScoreCrcOk,
                              Message::Type::Command,
                              Message::Type::Answer>
{
    // Идентификатор оценки
    QUuidEx id;

    // Контрольная сумма, вычисляется как hash64
    // от идентификаторов случаев (IDSL)
    bool crcOk = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id    )
        J_SERIALIZE_ITEM( crcOk )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


