#pragma once

#include "task.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

/** Описание механизма синхронизации данных **

  Синхронизация данных состоит из двух этапов:
    1. Получение данных;
    2. Сверка контрольной суммы для полученных данных.

  1) Получение данных осуществляется посредством отправки команды GetSyncData.
  Команда отправляется циклически на ФОМС сервер до тех пор пока  необходимый
  объем данных не будет получен.
  Команда содержит следующие поля:
    period   - период запрашиваемых данных (begin/end). На стороне ФОМС период
               соответствует полю OT_PER. Так как в БД поле OT_PER является
               строковым типом, формата MMYYYY (например, 092019), данное
               значение необходимо преобразовать в дату, где день заполнить
               первым числом месяца. То есть '092019' конверитируется в дату
               '01.09.2019'. То есть период для одного месяца будет задан
               следующим образом:
                 period.begin: 01.09.2019,
                 period.end: 30.09.2019.
               Данный период в этом случае включает дату содеражующуюся
               в OT_PER, так как OT_PER будет иметь значение 01.09.2019.

               Несмотря на то, что период  задается  с точностью  до дня,  поля
               period.begin, period.end  являются  типами  QDateTime  (точность
               хранения времени до миллисекунды). Это значит,  что при сериали-
               зации в JSON они будут представлены в формате int64.
               Нулевое значение соответствует началу эпохи (1970.01.01 00:00:00);
    timeMark - временная метка, задается с точностью до миллисекунды, сериали-
               зуется в int64. Временная метка должна быть у каждой записи, она
               определяем время последней модификации данных в записи.
               Если временная метка содержит дату/время раньше чем начало эпохи
               (1970.01.01 00:00:00), то она будет  представлена  отрицательным
               числом.
               При формировании датасета в него  должны  быть  включены  записи
               у которых временная метка больше или равна  значению  указанному
               в этом поле;
    count    - количество записей, которое нужно включить в ответное  сообщение
               (поле items). Если записей меньше,  чем запрашивается,  то вклю-
               чается все что есть.

  Порядок заполнения ответного сообщения (поле items).
  Из исходного набора данных выбираются записи удовлетворяющие следующим
  условиям:
    1. Записи находятся внутри диапазона [period.begin:period.end];
    2. Временные метки записей должны быть больше или равны значению указанному
       в поле timeMark.
  Далее полученная выборка данных сортируется по возрастанию по временной метке
  и по столбцу GKEY (идентификатор усулги). Из полученного набора отбираются
  первые count записей. Если набор  содержит записей меньше чем count, то
  берутся  все  записи  набора. Если  нет  данных удовлетворяющих условиям выше,
  то возвращается пустой набор данных.

  Команда GetSyncData повторяется до тех пор, пока соблюдается условие:
    items.count = count
  Как только items.count будет меньше count - считаем, что пришел последний
  набор данных, и приступаем к проверке контрольной суммы.

  2) Получение контрольной суммы выполняется при помощи команды SyncDataCheck.
  Команда содержит следующие поля:
    period   - см. описание для команды GetSyncData;
    timeMark - см. описание для команды GetSyncData;
    count    - количество записей для которых выполнить подсчёт контрольной
               суммы. То есть count, это количество первых записей в выборке,
               которая отсортирована по timeMark и GKEY (идентификатор услуги)
    crc      - контрольная сумма (тип uint64).

  Порядок вычисления контрольной суммы.
  Из исходного набора данных выбираются записи удовлетворяющие следующим
  условиям:
    1. Записи находятся внутри диапазона [period.begin:period.end];
    2. Временные метки записей должны быть больше или равны значению указанному
       в поле timeMark;
    3. Записи должны быть уникальны по полям:
         "идентификатор записи" + "временная метка" (поля: GKEY + TIME_MARK)
       Это значит, что все повторяющиеся записи должны быть отброшены.
    Важное замечание: если База Данных ФОМС содержит несколько записей, которые
    хранят историю изменения дефекта для услуги, то ФОМС должен возвращать статус
    МЭЭ или ЭКМП, а также другие данные, для записи с максимальным timeMark.

  Далее полученная выборка данных сортируется по возрастанию по временной
  метке (timeMark) и идентификатору услуги (GKEY), после чего вычисляется
  контрольная сумма по следующему алгоритму:
    uint64 crc = 0;
    for (i = 0; i < records.count; ++i) {
      record = records[i];
      int64 msecs = record[TIME_MARK].toMillisecond;
      uint64 value = convert_to_uint64(msecs);
      crc = crc XOR value;
    }


  Команда KeepWaitCommand.
  Команда содержит следующие поля:
    commandId - идентификатор команды;
    messageId - идентификатор сообщение;
    timeToAdd - время (в секундах), на которое необходимо продлить ожидание

  Описание команды: время на обработку запроса для ФОМС регламентировано. Данное
  время задается программно, и равняется 60-2 секундам (это значение  содержится
  в сообщении в поле maxTimeLife). В случае, если ФОМС  не укладывается  в отве-
  денный таймаут, то он может продлить время  ожидания  отправив  в АИС  команду
  KeepWaitCommand.  Команду можно отправлять неограниченное  количество раз, но
  желательно придерживаться разумных временных пределов.
  Если времени на обработку целевой команды все же окажется недостаточно и ответ
  на команду придет уже после того  как  время  на обработку  истекло,  то такой
  ответ считается устаревшим и будет отброшен.
*/

//----------------------------- Список команд --------------------------------

/** WEB
  Выполнить задачу синхронизации немедленно. Команда отправляется из WEB-интер-
  фейса без параметров
*/
extern const QUuidEx TaskSyncDataNow;

/** WEB
  Остановить задачу синхронизации. Команда отправляется из WEB-интерфейса без
  параметров
*/
extern const QUuidEx TaskSyncDataStop;

/** WEB
  Команда возвращает информацию по настройкам синхронизации
*/
extern const QUuidEx TaskSyncDataInfo;

/** WEB
  Команда устанавливает настройки синхронизации
*/
extern const QUuidEx TaskSyncDataEdit;

/** AIS
  Внутренняя служебная команда АИС, используется для планирования работы задачи
  синхронизации данных
*/
//extern const QUuidEx CreateSyncPlanning;

/** AIS
  Отладочная команда, используется для тестирования работы механизма синхрониза-
  ции данных
*/
//extern const QUuidEx CreateSyncPlanningDebug;

/** FOMS
  Команда используется для упрощенного механизма синхронизации. Команда выполняет
  передачу пакета данных (сообщения) от ФОМС Сервера к АИС Эксперт.
*/
extern const QUuidEx GetSyncData;

/** FOMS
  Команда возвращает контрольную сумму записей на стороне ФОМС для заданного
  периода, дата и время модификации которых, больше либо переданных даты и
  времени. Контрольная сумма посчитана по значению поля TIME_MARK.
*/
extern const QUuidEx SyncDataCheck;

/** FOMS
  Команда возвращает количество записей для заданного периода, время модификации
  которых больше либо равно указанного значения даты и времени.
*/
extern const QUuidEx SyncDataCount;

} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct TaskSyncData
{
    // Идентификатор задачи (не редактируется)
    QUuidEx taskId;

    // Признак, что задача будет запущена (включение/отключение задачи)
    bool isEnabled = {false};

    // Расчётное время запуска задачи
    QDateTime runDateTime = {QDateTime::currentDateTime()};

    // Время следующего запуска задачи
    QDateTime nextDateTime = {QDateTime::currentDateTime()};

    // Количество попыток повторных запусков после первого неудачного старта
    qint16 attemptLimit = {0};

    // Интервал между повторными запусками задачи (в минутах)
    qint16 attemptInterval = {15};

    // Статус выполнения задачи (не редактируется)
    TaskExecStatus taskExecStatus = {TaskExecStatus::NotRun};

    // Длительность периода используемого для синхронизации/обучения/применения
    qint16 relPeriodDuration = {0};

    // Размещение полей current и total в данной структуре сделано для удобства
    // обращения из web интерфейса. В этом случае одной командой можно сразу
    // получить прогресс задачи синхронизации. Без этих полей потребовалось бы
    // дополнительно запрашивать структуру Task.

    // Текущее значение прогресса. Обычно  это значение  находится в диапазоне
    // значений 0-100,  когда речь идет  о процентном  отображении  прогресса,
    // но может содержать и абсолютные значения обработанных item-ов, например
    // для процесса синхронизации данных.  Если  задача  не может  возвратить
    // текущий прогресс выполнения, то это значение будет равно -1
    qint32 current = {-1};

    // Максимальное значение прогресса для текущей задачи.  Обычно это значение
    // равно 100 (если значения возвращаются в процентах), но может содержать и
    // максимальное значение item-ов для обрабатываемой задачи
    qint32 total = {-1};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_OPT ( taskId            )
        J_SERIALIZE_ITEM( isEnabled         )
        J_SERIALIZE_ITEM( runDateTime       )
        J_SERIALIZE_ITEM( nextDateTime      )
        J_SERIALIZE_ITEM( attemptLimit      )
        J_SERIALIZE_ITEM( attemptInterval   )
        J_SERIALIZE_OPT ( taskExecStatus    )
        J_SERIALIZE_OPT ( relPeriodDuration )
    J_SERIALIZE_END
};

struct TaskSyncDataInfoA : TaskSyncData, Data<&command::TaskSyncDataInfo,
                                               Message::Type::Answer>
{};

struct TaskSyncDataEdit : TaskSyncData, Data<&command::TaskSyncDataEdit,
                                              Message::Type::Command>
{};

struct DataItem
{
    DataItem() = default;

    QUuidEx   GKEY;
    QUuidEx   IDSL;
    QDateTime TIME_MARK;
    QString   OT_PER;
    QString   MSK_OT;
    QString   CODE_MSK;
    QString   VID_MP;
    QString   USL_OK;
    QString   PROFIL ;
    QString   MKB1;
    QString   MKB2;
    QString   MKB3;
    QString   CODE_USL;
    QString   CODE_MD;
    double    KOL_USL = 0;
    double    KOL_FACT = 0;
    QString   ISH_MOV;
    QString   RES_GOSP;
    double    TARIF_B = 0;
    double    TARIF_S = 0;
    double    TARIF_1K = 0;
    double    SUM_RUB = 0;
    QString   VID_TR;
    qint16    EXTR;
    QString   CODE_OTD;
    qint16    SOUF;
    QString   SPEC_MD;
    QString   DOMC_TYPE;
    QString   OKATO_INS;
    QString   NOVOR;
    QString   CODE_LPU;
    QString   VID_SF;
    QString   NHISTORY;
    QString   PERSCODE;
    qint32    DATE_IN = 0;
    qint32    DATE_OUT = 0;
    double    TARIF_D = 0;
    QString   VID_KOEFF;
    QString   USL_TMP;
    qint32    BIRTHDAY = 0;
    QString   SEX;
    QString   COUNTRY;
    QString   SEX_P;
    qint32    BIRTHDAY_P = 0;
    qint16    INV;
    qint32    DATE_NPR = 0;
    qint16    FOR_POM = 0;
    qint16    MSE;
    QString   P_CEL;
    qint16    DN;
    qint32    TAL_P = 0;
    qint16    PROFIL_K;
    QString   NAPR_MO;
    QString   MKB0;
    qint16    DS_ONK = 0;
    double    VAL_KOEFF = 0;
    qint16    C_ZAB;
    QString   CODE_NOM1;    // TODO удалить, не используется
    QString   CODE_NOM2;    // TODO удалить, не используется
    QString   CODE_NOM3;    // TODO удалить, не используется
    QString   VAL_TMP;      // TODO удалить, не используется
    QString   TIME_FIX;     // TODO удалить, не используется
    QString   TIME_IN;      // TODO удалить, не используется
    QString   TIME_OUT;     // TODO удалить, не используется
    QString   DATE_FIX;     // TODO удалить, не используется
    QString   KATEG_MD;     // TODO удалить, не используется
    QString   POST_MD;      // TODO удалить, не используется
    QString   KOL_DEF;      // TODO удалить, не используется
    QString   VID_PROV;     // TODO удалить, не используется

    QString   MED_AREA;     // TODO удалить, не используется
    QString   TAL_HMP;      // TODO удалить, не используется
    qint32    DATE_HMP = 0; // TODO удалить, не используется
    QString   MCOD_OUT;     // TODO удалить, не используется
    QString   NOM_NPR;      // TODO удалить, не используется
    QString   SERIES;       // TODO удалить, не используется
    QString   NAME_MSK;     // TODO удалить, не используется
    QString   OKATO_NAS;    // TODO удалить, не используется
    QString   PR_LG;        // TODO удалить, не используется

    double    EKMP = 0;
    double    MEE = 0;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( GKEY       )
        J_SERIALIZE_ITEM( IDSL       )
        J_SERIALIZE_ITEM( TIME_MARK  )
        J_SERIALIZE_ITEM( OT_PER     )
        J_SERIALIZE_ITEM( MSK_OT     )
        J_SERIALIZE_ITEM( CODE_MSK   )
        J_SERIALIZE_ITEM( VID_MP     )
        J_SERIALIZE_ITEM( USL_OK     )
        J_SERIALIZE_ITEM( PROFIL     )
        J_SERIALIZE_ITEM( MKB1       )
        J_SERIALIZE_ITEM( MKB2       )
        J_SERIALIZE_ITEM( MKB3       )
        J_SERIALIZE_ITEM( CODE_USL   )
        J_SERIALIZE_ITEM( CODE_MD    )
        J_SERIALIZE_ITEM( KOL_USL    )
        J_SERIALIZE_ITEM( KOL_FACT   )
        J_SERIALIZE_ITEM( ISH_MOV    )
        J_SERIALIZE_ITEM( RES_GOSP   )
        J_SERIALIZE_ITEM( TARIF_B    )
        J_SERIALIZE_ITEM( TARIF_S    )
        J_SERIALIZE_ITEM( TARIF_1K   )
        J_SERIALIZE_ITEM( SUM_RUB    )
        J_SERIALIZE_ITEM( VID_TR     )
        J_SERIALIZE_ITEM( EXTR       )
        J_SERIALIZE_ITEM( CODE_OTD   )
        J_SERIALIZE_ITEM( SOUF       )
        J_SERIALIZE_ITEM( SPEC_MD    )
        J_SERIALIZE_ITEM( DOMC_TYPE  )
        J_SERIALIZE_ITEM( OKATO_INS  )
        J_SERIALIZE_ITEM( NOVOR      )
        J_SERIALIZE_ITEM( CODE_LPU   )
        J_SERIALIZE_ITEM( VID_SF     )
        J_SERIALIZE_ITEM( NHISTORY   )
        J_SERIALIZE_ITEM( PERSCODE   )
        J_SERIALIZE_ITEM( DATE_IN    )
        J_SERIALIZE_ITEM( DATE_OUT   )
        J_SERIALIZE_ITEM( TARIF_D    )
        J_SERIALIZE_ITEM( VID_KOEFF  )
        J_SERIALIZE_ITEM( USL_TMP    )
        J_SERIALIZE_ITEM( BIRTHDAY   )
        J_SERIALIZE_ITEM( SEX        )
        J_SERIALIZE_ITEM( COUNTRY    )
        J_SERIALIZE_ITEM( SEX_P      )
        J_SERIALIZE_ITEM( BIRTHDAY_P )
        J_SERIALIZE_ITEM( INV        )
        J_SERIALIZE_ITEM( DATE_NPR   )
        J_SERIALIZE_ITEM( FOR_POM    )
        J_SERIALIZE_ITEM( MSE        )
        J_SERIALIZE_ITEM( P_CEL      )
        J_SERIALIZE_ITEM( DN         )
        J_SERIALIZE_ITEM( TAL_P      )
        J_SERIALIZE_ITEM( PROFIL_K   )
        J_SERIALIZE_ITEM( NAPR_MO    )
        J_SERIALIZE_ITEM( MKB0       )
        J_SERIALIZE_ITEM( DS_ONK     )
        J_SERIALIZE_ITEM( VAL_KOEFF  )
        J_SERIALIZE_ITEM( C_ZAB      )
        J_SERIALIZE_OPT ( CODE_NOM1  )
        J_SERIALIZE_OPT ( CODE_NOM2  )
        J_SERIALIZE_OPT ( CODE_NOM3  )
        J_SERIALIZE_OPT ( VAL_TMP    )
        J_SERIALIZE_OPT ( TIME_FIX   )
        J_SERIALIZE_OPT ( TIME_IN    )
        J_SERIALIZE_OPT ( TIME_OUT   )
        J_SERIALIZE_OPT ( DATE_FIX   )
        J_SERIALIZE_OPT ( KATEG_MD   )
        J_SERIALIZE_OPT ( POST_MD    )
        J_SERIALIZE_OPT ( KOL_DEF    )
        J_SERIALIZE_OPT ( VID_PROV   )
        J_SERIALIZE_OPT ( MED_AREA   )
        J_SERIALIZE_OPT ( TAL_HMP    )
        J_SERIALIZE_OPT ( DATE_HMP   )
        J_SERIALIZE_OPT ( MCOD_OUT   )
        J_SERIALIZE_OPT ( NOM_NPR    )
        J_SERIALIZE_OPT ( SERIES     )
        J_SERIALIZE_OPT ( NAME_MSK   )
        J_SERIALIZE_OPT ( OKATO_NAS  )
        J_SERIALIZE_OPT ( PR_LG      )

        J_SERIALIZE_ITEM( EKMP       )
        J_SERIALIZE_ITEM( MEE        )
    J_SERIALIZE_END
};

struct GetSyncData : Data<&command::GetSyncData,
                           Message::Type::Command,
                           Message::Type::Answer>
{
    // Левая и правая границы выборки
    TimeRange period;

    // Самая старшая дата записи (с точностью до миллисекунды)
    QDateTime timeMark;

    // Количество запрашиваемых строк данных
    qint32 count = {5000};

    // Строки данных
    QVector<data::DataItem> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( period   )
        J_SERIALIZE_ITEM( timeMark )
        J_SERIALIZE_ITEM( count    )
        J_SERIALIZE_ITEM( items    )
    J_SERIALIZE_END
};

struct SyncDataCheck : Data<&command::SyncDataCheck,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    // Левая и правая границы выборки
    TimeRange period;

    // Самая старшая дата записи (с точностью до миллисекунды)
    QDateTime timeMark;

    // Количество записей для которого необходимо посчитать crc
    quint64 count = {0};

    // Контрольная сумма записей для периода 'period', дата и время изменения
    // которых больше либо равно 'timeMark'. Заполняется ФОМС сервером.
    quint64 crc = {0};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( period   )
        J_SERIALIZE_ITEM( timeMark )
        J_SERIALIZE_ITEM( count    )
        J_SERIALIZE_ITEM( crc      )
    J_SERIALIZE_END
};

struct SyncDataCount : Data<&command::SyncDataCount,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    // Левая и правая границы выборки
    TimeRange period;

    // Самая старшая дата записи (с точностью до миллисекунды)
    QDateTime timeMark;

    // Количество записей для периода 'period', дата и время изменения которых
    // больше либо равно 'timeMark'
    quint64 count = 0;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( period   )
        J_SERIALIZE_ITEM( timeMark )
        J_SERIALIZE_ITEM( count    )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


