
#pragma once

#include "paging.h"
#include "time_range.h"

#include "shared/list.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/** WEB
  Команда запускает задачу немедленно (принудительно).
  Примечание: если принудительно запущенная задача завершилась с кодом ошибки
    большим 1 (retCode > 1),  то повторных  запусков этой задачи  по счетчику
    попыток (счетчик attempt) не предусмотрено.
*/
extern const QUuidEx TaskStartNow;

/** WEB
  Команда возвращает статус выполнения задачи
*/
extern const QUuidEx TaskProgress;

/** WEB
  Прерывает задачу
*/
extern const QUuidEx TaskInterrupt;

/** WEB
  Оповещение о завершении задачи
*/
extern const QUuidEx TaskInterruptDone;

/** WEB
  Команда создает задачу по обучению модели
*/
extern const QUuidEx TaskModelCreate;

/** WEB
  Команда редактирует задачу по обучению модели
*/
extern const QUuidEx TaskModelEdit;

/** WEB
  Команда удаляет задачу по обучению модели
*/
extern const QUuidEx TaskModelDelete;

/** WEB
  Возвращает информацию по задаче по обучению модели
*/
extern const QUuidEx TaskModelInfo;

/** WEB
  Возвращает список задач по обучению модели
*/
extern const QUuidEx TaskModelList;

/** WEB
  Команда создает задачу для расчета оценки
*/
extern const QUuidEx TaskScoreCreate;

/** WEB
  Команда редактирует задачу по расчету оценки
*/
extern const QUuidEx TaskScoreEdit;

/** WEB
  Команда удаляет задачу по расчету оценки
*/
extern const QUuidEx TaskScoreDelete;

/** WEB
  Возвращает информацию по задаче по расчету оценки
*/
extern const QUuidEx TaskScoreInfo;

/** WEB
  Возвращает список задач по расчету оценки
*/
extern const QUuidEx TaskScoreList;

/** WEB
  Событие эмитируется когда запущенная задача создает в базе данных запись
  о создаваемом контенте (модель, оценка, отчет)
*/
extern const QUuidEx TaskContentCreate;

/** WEB
  Событие эмитируется когда запущенная задача сообщает об ошибке или неком
  событии во время выполнения.
*/
extern const QUuidEx TaskAlert;

/** WEB
  Событие эмитируется когда web-интерфейс запросил статус синхронизации задачи.
*/
extern const QUuidEx SyncProgress;

} // namespace command

// Фильтр периодичности задачи
enum class TaskMode : quint32
{
    All      = 0, // Все задачи
    OneTime  = 1, // Только разовые
    Periodic = 2, // Только периодичные
};

// Результат попытки прерывания задачи
enum class TaskStopResult : quint32
{
    Success = 0, // Прерывание задачи успешно
    Error   = 1  // Задача, возможно, не прервана, так как возникли ошибки
};

// Тип задачи
enum class TaskType : quint32
{
    Undefined       = 0,
    SyncData        = 1, // Синхронизация (регулярная)
    SyncPlan        = 2, // Синхронизация (пользовательская)
    SendScore       = 3, // Отправка отчета в ФОМС
    LearnModel      = 4, // Обучение модели
    ScoreCalc       = 5, // Вычисление оценки
    CreateReport    = 6, // Создание отчета
    CreateReportFed = 7, // Создание федерального отчета
    ReportFed       = 8, // Создание федерального отчета по расписанию. Регуляр-
                         // ная задача-синглтон для запуска  'create_report_fed'
    SyncNsi         = 9  // Синхронизация справочников НСИ.  Регулярная  задача-
                         // синглтон

};

inline TaskType getTaskType(QString taskString)
{
    taskString = taskString.trimmed();
    if      (taskString == "sync_data"        ) return TaskType::SyncData;
    else if (taskString == "sync_plan"        ) return TaskType::SyncPlan;
    else if (taskString == "send_score"       ) return TaskType::SendScore;
    else if (taskString == "learn_model"      ) return TaskType::LearnModel;
    else if (taskString == "score_calc"       ) return TaskType::ScoreCalc;
    else if (taskString == "create_report"    ) return TaskType::CreateReport;
    else if (taskString == "create_report_fed") return TaskType::CreateReportFed;
    else if (taskString == "report_fed"       ) return TaskType::ReportFed;
    else if (taskString == "sync_nsi"         ) return TaskType::SyncNsi;
    else                                        return TaskType::Undefined;
}

inline QString getTaskString(TaskType type)
{
    if      (type == TaskType::SyncData       ) return "sync_data";
    else if (type == TaskType::SyncPlan       ) return "sync_plan";
    else if (type == TaskType::SendScore      ) return "send_score";
    else if (type == TaskType::LearnModel     ) return "learn_model";
    else if (type == TaskType::ScoreCalc      ) return "score_calc";
    else if (type == TaskType::CreateReport   ) return "create_report";
    else if (type == TaskType::CreateReportFed) return "create_report_fed";
    else if (type == TaskType::ReportFed      ) return "report_fed";
    else if (type == TaskType::SyncNsi        ) return "sync_nsi";
    else                                        return "";
}

// Текущий статус выполнения задачи
enum class TaskExecStatus : quint32
{
    NotRun      = 0,  // Задача еще не запущена
    Running     = 1,  // Задача в процессе выполнения
    Success     = 2,  // Задача успешно завершена
    Attempt     = 3,  // Первое выполнение было неудачно, задача перешла
                      // в состояние ожидания повторного запуска.
    Failed      = 4,  // Задача завершена с ошибками,  все попытки повторного
                      // запуска исчерпаны задача больше не может запускаться
                      // по расписанию,  пользователь  должен  быть уведомлен
                      // об этом по почте.
    Deffered    = 5,  // Задача отложена по причине ограничения на количество
                      // одновременно выполняемых задач планировщиком
    Interrupted = 6,  // Задача была прервана пользователем
    WaitSync    = 7   // Задача ожидает синхронизацию данных
};

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct TaskStartNow : Data<&command::TaskStartNow,
                            Message::Type::Command,
                            Message::Type::Answer>
{
    QUuidEx taskId; // Идентификатор из таблицы TASKS
    QUuidEx userId;
    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct TaskProgressC : Data<&command::TaskProgress,
                             Message::Type::Command>
{
    // Идентификатор задачи
    QUuidEx id;
    J_SERIALIZE_ONE( id )
};

struct TaskProgress : Data<&command::TaskProgress,
                            Message::Type::Answer,
                            Message::Type::Event>
{
    TaskType taskType = {TaskType::Undefined};

    // Идентификатор задачи
    QUuidEx taskId;

    // Поля добавлены по просьбе web-разработчиков
    QString taskName;
    QString taskDescript;
    QDateTime taskCreateDate;

    // Статус выполнения задачи
    TaskExecStatus taskExecStatus = {TaskExecStatus::NotRun};

    // Идентификатор пользователя
    QUuidEx userId;

    // TODO: поле deprecated. Решить после показа 12-го что делать с этим полем,
    // веберам оно больше не нужно.
    // Идентификатор создаваемого контента (модели, оценки, отчета),
    // соотносится с TaskType
    //QUuidEx contentId;

    // Признак периодической задачи
    bool isPeriodic = {false};

    // Текущее значение прогресса. Обычно  это значение  находится в диапазоне
    // значений 0-100,  когда речь идет  о процентном  отображении  прогресса,
    // но может содержать и абсолютные значения обработанных item-ов, например
    // для процесса синхронизации данных.  Если  задача  не может  возвратить
    // текущий прогресс выполнения, то это значение будет равно -1
    qint32 current = {-1};

    // Максимальное значение прогресса для текущей задачи.  Обычно это значение
    // равно 100 (если значения возвращаются в процентах), но может содержать и
    // максимальное значение item-ов для обрабатываемой задачи
    qint32 total = {-1};

    // Для стауса синхронизации. Позиция текущей задачи в очереди на
    // синхронизацию.
    qint32 waitPosition = {-1};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskType       )
        J_SERIALIZE_ITEM( taskId         )
        J_SERIALIZE_ITEM( taskName       )
        J_SERIALIZE_ITEM( taskDescript   )
        J_SERIALIZE_ITEM( taskCreateDate )
        J_SERIALIZE_ITEM( taskExecStatus )
        J_SERIALIZE_ITEM( userId         )
//      J_SERIALIZE_ITEM( contentId      )
        J_SERIALIZE_OPT ( isPeriodic     )
        J_SERIALIZE_ITEM( current        )
        J_SERIALIZE_ITEM( total          )
        J_SERIALIZE_ITEM( waitPosition   )
    J_SERIALIZE_END
};

struct TaskInterrupt : Data<&command::TaskInterrupt,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    QUuidEx  taskId;
    QUuidEx  userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId   )
        J_SERIALIZE_ITEM( userId   )
    J_SERIALIZE_END
};

struct TaskInterruptA : Data<&command::TaskInterrupt,
                              Message::Type::Answer>
{
    QUuidEx taskId;
    J_SERIALIZE_ONE( taskId )
};

struct TaskContentCreate : Data<&command::TaskContentCreate,
                                 Message::Type::Event>
{
    TaskType taskType = {TaskType::Undefined};

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя создающего контент
    QUuidEx userId;

    // Идентификатор контента (модели, оценки, отчета), соотносится с TaskType
    QUuidEx contentId;

    // Признак периодической задачи
    bool isPeriodic = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskType   )
        J_SERIALIZE_ITEM( taskId     )
        J_SERIALIZE_ITEM( userId     )
        J_SERIALIZE_ITEM( contentId  )
        J_SERIALIZE_OPT ( isPeriodic )
    J_SERIALIZE_END
};

struct TaskAlert : Data<&command::TaskAlert,
                         Message::Type::Event>
{
    // Идентификатор задачи
    QUuidEx taskId;

    // Сообщение, которое будет отображено пользователю.
    QString message;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId  )
        J_SERIALIZE_ITEM( message )
    J_SERIALIZE_END
};

struct SyncInfo
{
    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя
    QUuidEx userId;

    // Название задачи
    QString taskName;

    TaskType taskType = {TaskType::Undefined};

    // true - задача в процессе синхронизации, false - задача ожидает синхронизацию
    bool syncStatus = {false};

    // Текущее значение прогресса. Обычно  это значение  находится в диапазоне
    // значений 0-100,  когда речь идет  о процентном  отображении  прогресса,
    // но может содержать и абсолютные значения обработанных item-ов, например
    // для процесса синхронизации данных.  Если  задача  не может  возвратить
    // текущий прогресс выполнения, то это значение будет равно -1
    qint32 current = {-1};

    // Максимальное значение прогресса для текущей задачи.  Обычно это значение
    // равно 100 (если значения возвращаются в процентах), но может содержать и
    // максимальное значение item-ов для обрабатываемой задачи
    qint32 total = {-1};

    // Стратегия поиска по taskId
    struct Find
    {
        int operator() (const QUuidEx* taskId, const SyncInfo* si, void*) const
            {return QUuidEx::compare(*taskId, si->taskId);}
    };
    typedef lst::List<SyncInfo, SyncInfo::Find> List;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId     )
        J_SERIALIZE_ITEM( userId     )
        J_SERIALIZE_ITEM( taskName   )
        J_SERIALIZE_ITEM( taskType   )
        J_SERIALIZE_ITEM( syncStatus )
        J_SERIALIZE_ITEM( current    )
        J_SERIALIZE_ITEM( total      )
    J_SERIALIZE_END
};

struct SyncProgress : Data<&command::SyncProgress,
                            Message::Type::Answer,
                            Message::Type::Event>
{
    SyncInfo::List items;
    J_SERIALIZE_ONE( items )
};

/**
  Структура для описания задачи
*/
struct Task
{
    // Идентификатор задачи (задается при создании задачи в АисЭксперт)
    QUuidEx id;

    // Идентификатор пользователя
    QUuidEx userId;

    // Идентификатор связной задачи, от которой зависит текущая задача
    QUuidEx parentId;

    // Имя задачи
    QString name;

    // Описание задачи
    QString description;

    // Время создания задачи (параметр игнорируется при создании задачи
    // и не редактируется)
    QDateTime createDate = {QDateTime::currentDateTime()};

    // Расчётное время запуска задачи, поле хранит  расчетное  время  первого
    // запуска задачи.
    // В случае принудительного запуска  задачи  (вызов команды TaskStartNow)
    // значение этого поля не измениться,  т.е. принудительный  старт  задачи
    // не оказывают влияния на запуски по расписанию.
    QDateTime runDateTime = {QDateTime::currentDateTime()};

    // TODO поле перименовано в nextDateTime, в будущем удалить
    // Время следующего запуска задачи (параметр игнорируется при создании
    // задачи и не редактируется)
    QDateTime nextRunTime = {QDateTime::currentDateTime()};

    // Время следующего запуска задачи
    QDateTime nextDateTime = {QDateTime::currentDateTime()};

    //--- Параметры для периодического запуска задач ---
    // За периодический запуск задачи отвечают три поля: regularityMonth,
    // regularityDay, regularityHour.
    //
    // 1) regularityMonth - период повторного запуска в месяцах.
    // 2) regularityDay - период повторного запуска задачи в днях,
    //    после runDateTime.
    //    Например runDateTime = '2019-01-01 12:00' regularityDay = 2.
    //    В этом случае задача будет запускаться:
    //    '2019-01-01 12:00' -> '2019-01-03 12:00' -> '2019-01-05 12:00' -> ...
    // 3) regularityHour - период повтора запуска задачи в часах, срабатывает
    //    после времени в runDateTime и действует в течении суток указанных
    //    в nextRunTime.
    //    Например runDateTime = '2019-01-01 12:00' regularityHour = 2.
    //    В этом случае задача будет запускаться
    //   '2019-01-01 12:00' -> '2019-01-01 14:00' -> '2019-01-01 16:00' -> ...
    //   -> '2019-01-01 22:00'
    qint16 regularityMonth = {0};
    qint16 regularityDay   = {0};
    qint16 regularityHour  = {0};

    // Количество попыток повторных запусков после первого неудачного старта
    qint16 attemptLimit = {0};

    // Счетчик для неудачных попыток запуска задачи (параметр игнорируется
    // при создании задачи и не редактируется)
    qint16 attemptCounter = {5};

    // Интервал между повторными запусками задачи (в минутах)
    qint16 attemptInterval = {15};

    // Статус выполнения задачи
    TaskExecStatus execStatus = {TaskExecStatus::NotRun};

    // Признак, что задача может быть запущена (включение/отключение задачи)
    bool isEnabled = {true};

    // Признак определяет, что модель будет доступна всем пользователям
    bool isPublic = {false};

    // Левая и правая  абсолютная граница интервала выборки данных для обучения
    // модели
    TimeRange period;

    // TODO написать более понятный комментарий, имя параметров?
    // Правая (ближняя текущая) относительная граница интервала выборки данных
    // для обучения модели. Задается в месяцах
    quint16 relPeriodBegin = {0};

    // Длина интервала от relPeriodBegin в прошлое (т.е. установка левой точки
    // временного отрезка)
    quint16 relPeriodDuration = {0};

    // Идентификатор модели на основе которой создана оценка
    QUuidEx modelId;
    
    // TODO удалить при очередной стыковке команд с веберами
    // Идентификатор отчета который создается данной задачей
    //QUuidEx reportId;

    // Текущее значение прогресса
    qint32 progressCurrent = {-1};

    // Максимальное значение прогресса для текущей задачи
    qint32 progressTotal = {-1};

    // Признак определяет, будет ли задача обучения запускать дочернюю задачу
    // применения
    bool runChildTask = {false};

    //--- Вспомогательные поля для WEB-интерфейса ---
    QString parentName;
    QString modelName;

    // Позиция задачи в очереди на синхронизацию
    qint32 waitPosition = {-1};

    // Признак определяет, что можно пропустить этап синхронизации
    // при выполнении задачи
    bool skipSync = {false};

    // Признак периодической задачи
    bool isPeriodic() const {
        return regularityMonth || regularityDay || regularityHour;
    }

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id                )
        J_SERIALIZE_ITEM( userId            )
        J_SERIALIZE_ITEM( parentId          )
        J_SERIALIZE_ITEM( name              )
        J_SERIALIZE_ITEM( description       )
        J_SERIALIZE_ITEM( createDate        )
        J_SERIALIZE_ITEM( runDateTime       )
        J_SERIALIZE_ITEM( nextRunTime       )
        J_SERIALIZE_OPT ( nextDateTime      )
        J_SERIALIZE_ITEM( regularityMonth   )
        J_SERIALIZE_ITEM( regularityDay     )
        J_SERIALIZE_ITEM( regularityHour    )
        J_SERIALIZE_ITEM( attemptLimit      )
        J_SERIALIZE_ITEM( attemptCounter    )
        J_SERIALIZE_ITEM( attemptInterval   )
        J_SERIALIZE_ITEM( execStatus        )
        J_SERIALIZE_ITEM( isEnabled         )
        J_SERIALIZE_ITEM( isPublic          )
        J_SERIALIZE_ITEM( period            )
        J_SERIALIZE_ITEM( relPeriodBegin    )
        J_SERIALIZE_ITEM( relPeriodDuration )
        J_SERIALIZE_ITEM( modelId           )
        J_SERIALIZE_ITEM( progressCurrent   )
        J_SERIALIZE_ITEM( progressTotal     )
        J_SERIALIZE_ITEM( runChildTask      )
        J_SERIALIZE_ITEM( parentName        )
        J_SERIALIZE_ITEM( modelName         )
        J_SERIALIZE_OPT ( waitPosition      )
        J_SERIALIZE_OPT ( skipSync          )
    J_SERIALIZE_END
};

struct TaskFilter
{
    // Идентификатор задачи
    QUuidEx id;

    // Идентификатор пользователя
    QUuidEx userId;

    // Вид задачи: разовая или регулярная
    TaskMode mode = {TaskMode::All};

    // Используется для создания списка разовых задач применения, которые
    // зависят от модели с идентфикатором modelId.
    // Поле modelId соответствует столбцу MODEL_ID в БД.
    QUuidEx modelId;

    // Используется для создания списка регулярных задач применения, которые
    // зависят от задачи с идентфикатором parentId.
    // Поле parentId соответствует столбцу PARENT_ID в БД.
    QUuidEx parentId;

    // Если данный флаг имеет значение true, то для пользователя с правами
    // Администратора, будут показаны все задачи, включая задачи других
    // пользователей.
    bool showOtherUsers = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id             )
        J_SERIALIZE_ITEM( userId         )
        J_SERIALIZE_ITEM( mode           )
        J_SERIALIZE_ITEM( modelId        )
        J_SERIALIZE_ITEM( parentId       )
        J_SERIALIZE_OPT ( showOtherUsers )
    J_SERIALIZE_END
};

struct TaskModelCreate : Task, Data<&command::TaskModelCreate,
                                     Message::Type::Command,
                                     Message::Type::Answer>
{};

struct TaskModelEdit : Task, Data<&command::TaskModelEdit,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{};

struct TaskModelDelete : Data<&command::TaskModelDelete,
                               Message::Type::Command,
                               Message::Type::Answer>
{
    QUuidEx taskId;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct TaskModelInfo : Data<&command::TaskModelInfo,
                             Message::Type::Command>
{
    QUuidEx taskId;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct TaskModelInfoA : Task, Data<&command::TaskModelInfo,
                                    Message::Type::Answer>
{};

struct TaskModelList : Data<&command::TaskModelList,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    // Фильтр выбора данных
    TaskFilter filter;

    QVector<Task> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( items   )
    J_SERIALIZE_END
};

struct TaskScoreCreate : Task, Data<&command::TaskScoreCreate,
                                     Message::Type::Command,
                                     Message::Type::Answer>
{};

struct TaskScoreEdit : Task, Data<&command::TaskScoreEdit,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{};

struct TaskScoreDelete : Data<&command::TaskScoreDelete,
                               Message::Type::Command,
                               Message::Type::Answer>
{
    QUuidEx taskId;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct TaskScoreInfo : Data<&command::TaskScoreInfo,
                             Message::Type::Command>
{
    QUuidEx taskId;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( taskId )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct TaskScoreInfoA : Task, Data<&command::TaskScoreInfo,
                                    Message::Type::Answer>
{};

struct TaskScoreList : Data<&command::TaskScoreList,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    // Фильтр выбора данных
    TaskFilter filter;

    QVector<Task> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( items  )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


