
#pragma once

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/**
  Команда отправляет список пользователей AisExpert на сервис UsersPool. Список
  отправляется сразу после подтверждения авторизации сервисов. В ответ приходит
  событие UPoolChangesNotify с информацией о том, какие параметры пользователей
  изменились в Windows Active Directory
*/
extern const QUuidEx UPoolValidUsers; // TODO: поменять коментаррий

/**
  Событие рассылает список пользователей из Windows Active Directory. Событие
  эмитируется когда в кеше пользователей произошли изменения
*/
extern const QUuidEx UPoolChangesNotify;

/**
  Выполняет аутентификацию пользователя АИС Эксперт
*/
extern const QUuidEx UPoolUserAuth;

/**
  Возвращает список пользователей системы
*/
extern const QUuidEx UPoolUserList;

} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct UPoolUserInfo
{
    QString sid;
    QString login;
    QString name;
    bool    isAdmin;
    bool    isValid;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( sid     )
        J_SERIALIZE_ITEM( login   )
        J_SERIALIZE_ITEM( name    )
        J_SERIALIZE_ITEM( isAdmin )
        J_SERIALIZE_ITEM( isValid )
    J_SERIALIZE_END
};

struct UPoolUserList : Data<&command::UPoolUserList,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    QVector<UPoolUserInfo> items;
    J_SERIALIZE_ONE( items )
};

struct UPoolUserAuth : Data<&command::UPoolUserAuth,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    QString login;
    QString password;

    // ФИО пользователя
    QString name;

    // Идентификатор пользователя из системы Windows Active Directory
    QString sid;

    // Идентификатор пользователя, используется для связи с базой данных
    QUuidEx id;

    // Хеш id пользователя, используется для идентификации  пользователя при
    // пересылке коммуникационных сообщений. Значение этого параметра так же
    // передается во всех сообщениях относящихся  к работе  с пользователем
    // в поле Message::tag
    quint64 hashId = {0};

    // Признак, что пользователь является администратором системы АИС Эксперт.
    // Признак администратора определяется во внешней системе авторизации
    bool isAdmin = {false};

    // Статус аутентификации. Если аутентификация прошла успешно значение поля
    // выставляется в TRUE
    bool status = {false};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( login    )
        J_SERIALIZE_ITEM( password )
        J_SERIALIZE_ITEM( name     )
        J_SERIALIZE_ITEM( sid      )
        J_SERIALIZE_ITEM( id       )
        J_SERIALIZE_ITEM( hashId   )
        J_SERIALIZE_ITEM( isAdmin  )
        J_SERIALIZE_OPT ( status   )
    J_SERIALIZE_END
};

struct UPoolValidUsers : Data<&command::UPoolValidUsers,
                               Message::Type::Command>
{
    QList<UPoolUserInfo> items;
    J_SERIALIZE_ONE( items )
};

struct UPoolChangesNotify : Data<&command::UPoolChangesNotify,
                                  Message::Type::Event>
{
    QList<UPoolUserInfo>   changeUsers; // Список пользователей в учетных записях
                                        // которых произошли изменения
    QList<UPoolUserInfo>   newUsers;    // Список добавленных пользователей
    QList<QString /*SID*/> deleteUsers; // Список SID-ов удаленных пользователей
    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( changeUsers )
        J_SERIALIZE_ITEM( newUsers    )
        J_SERIALIZE_ITEM( deleteUsers )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication


