#include "main_window.h"
//#include "../../aisexpert/database/sql_func.h"

#include "commands/monitor.h"
#include "commands/send_score.h"

#include "commands/authorization.h"
#include "shared/defmac.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/config/config.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/logger_operators.h"

#include <QtSql>
#include <QDomDocument>
#include <QMessageBox>
#include <QPalette>
#include <QColor>
#include <unistd.h>

#define QTEXT(MSG) QString::fromUtf8(u8##MSG)

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);
    setWindowTitle(qApp->applicationName());

    #define FUNC_REGISTRATION(COMMAND) \
        _funcInvoker.registration(command:: COMMAND, &MainWindow::command_##COMMAND, this);

    FUNC_REGISTRATION(WebAuthorization)
    FUNC_REGISTRATION(Monitoring)
    FUNC_REGISTRATION(SpeedTest)
    FUNC_REGISTRATION(Error)
    FUNC_REGISTRATION(UPoolUserList)
    FUNC_REGISTRATION(UPoolUserAuth)
    FUNC_REGISTRATION(UserInfo)
    FUNC_REGISTRATION(UserList)
    FUNC_REGISTRATION(TaskSyncDataInfo)

    FUNC_REGISTRATION(GroupInfo)
    FUNC_REGISTRATION(GroupList)
    FUNC_REGISTRATION(GroupDelete)

    FUNC_REGISTRATION(TaskModelCreate)
    FUNC_REGISTRATION(TaskModelEdit)
    FUNC_REGISTRATION(TaskModelDelete)
    FUNC_REGISTRATION(TaskModelInfo)
    FUNC_REGISTRATION(TaskModelList)

    FUNC_REGISTRATION(TaskScoreCreate)
    FUNC_REGISTRATION(TaskScoreEdit)
    FUNC_REGISTRATION(TaskScoreDelete)
    FUNC_REGISTRATION(TaskScoreInfo)
    FUNC_REGISTRATION(TaskScoreList)
    FUNC_REGISTRATION(TaskAlert)

    FUNC_REGISTRATION(ModelEdit)
    FUNC_REGISTRATION(ModelDelete)
    FUNC_REGISTRATION(ModelInfo)
    FUNC_REGISTRATION(ModelList)

    FUNC_REGISTRATION(TaskInterrupt)
    FUNC_REGISTRATION(TaskProgress)

    FUNC_REGISTRATION(ScoreEdit)
    FUNC_REGISTRATION(ScoreDelete)
    FUNC_REGISTRATION(ScoreInfo)
    FUNC_REGISTRATION(ScoreList)

    FUNC_REGISTRATION(TaskReportCreate)
    FUNC_REGISTRATION(ReportList)
    FUNC_REGISTRATION(ReportInfo)
    FUNC_REGISTRATION(ReportData)

    FUNC_REGISTRATION(TaskReportFedCreate)
    FUNC_REGISTRATION(ReportFedList)

    FUNC_REGISTRATION(TaskContentCreate)

    FUNC_REGISTRATION(ModelXgbDelete)
    FUNC_REGISTRATION(ModelXgbInfo)
    FUNC_REGISTRATION(ModelXgbEdit)
    FUNC_REGISTRATION(ModelXgbOption)

    FUNC_REGISTRATION(EventLogList)
    FUNC_REGISTRATION(SyncProgress)

    #undef FUNC_REGISTRATION

    _labelConnectStatus = new QLabel(tr("Disconnected"), this);
    ui->statusBar->addWidget(_labelConnectStatus);
    ui->wgtSettings->setVisible(false);
}

namespace
{
    QMap<QString, QString> xgbEditOption;
}

MainWindow::~MainWindow()
{
    delete ui;
}

bool MainWindow::init(const tcp::Socket::Ptr& socket)
{
    _socket = socket;

    chk_connect_q(_socket.get(), SIGNAL(message(communication::Message::Ptr)),
                  this, SLOT(message(communication::Message::Ptr)))
    chk_connect_q(_socket.get(), SIGNAL(connected(communication::SocketDescriptor)),
                  this, SLOT(socketConnected(communication::SocketDescriptor)))
    chk_connect_q(_socket.get(), SIGNAL(disconnected(communication::SocketDescriptor)),
                  this, SLOT(socketDisconnected(communication::SocketDescriptor)))

    //chk_connect_q(ui->listTypeTasks, SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
    //                      this, SLOT(currentTaskTypeChanged(QListWidgetItem *)))
    //chk_connect_q(ui->listTasks, SIGNAL(currentItemChanged(QListWidgetItem *, QListWidgetItem *)),
    //                      this, SLOT(currentTaskChanged(QListWidgetItem *)))

    chk_connect_q(ui->lstGroups, SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem*)),
                  this, SLOT(currentGroupChanged(QListWidgetItem*)))

    chk_connect_q(ui->listUsers, SIGNAL(currentItemChanged(QListWidgetItem*, QListWidgetItem*)),
                  this, SLOT(currentAisUserChanged(QListWidgetItem*)))

    QMetaObject::invokeMethod(this, "checkConnections", Qt::QueuedConnection);

    chk_connect_a(&_connectTimer, SIGNAL(timeout()), this, SLOT(checkConnections()))
    _connectTimer.start(15*1000);

    chk_connect_a(&_timerMonitoringUpdate, SIGNAL(timeout()), this, SLOT(getMonitoring()));

    return true;
}

void MainWindow::deinit()
{
    _connectTimer.stop();
    //db::firebird::driver()->close();
}

void MainWindow::saveGeometry()
{
    QPoint p = pos();
    std::vector<int> v {p.x(), p.y(), width(), height()};
    config::state().setValue("windows.main_window.geometry", v);
}

void MainWindow::loadGeometry()
{
    std::vector<int> v {0, 0, 800, 600};
    config::state().getValue("windows.main_window.geometry", v);
    move(v[0], v[1]);
    resize(v[2], v[3]);
}

void MainWindow::saveSettings()
{
}

void MainWindow::loadSettings()
{
}

void MainWindow::message(const Message::Ptr& message)
{
    if (message->processed())
        return;

    if (lst::FindResult fr = _funcInvoker.findCommand(message->command()))
    {
        if (message->type() == Message::Type::Answer)
        {
            if (message->execStatus() == Message::ExecStatus::Error)
            {
                data::MessageError data;
                readFromMessage(message, data);
                QMessageBox::critical(this, qApp->applicationName(), data.description);
                return;
            }
        }
        if (!command::pool().commandIsMultiproc(message->command()))
            message->markAsProcessed();
        _funcInvoker.call(message, fr);
    }
}

void MainWindow::socketConnected(SocketDescriptor)
{
    QString password;
    config::state().getValue("ais_authorization.password", password);

    if (!password.isEmpty())
    {
        data::WebAuthorization webAuthorization;
        webAuthorization.password = password;

        Message::Ptr m = createJsonMessage(webAuthorization);
        _socket->send(m);
    }
    _labelConnectStatus->setText("Connected");
}

void MainWindow::socketDisconnected(SocketDescriptor)
{
    _webHashId = 0;
    _labelConnectStatus->setText("Disconnected");
}

void MainWindow::command_WebAuthorization(const Message::Ptr& message)
{
    if (message->type() == Message::Type::Answer)
    {
        if (message->execStatus() == Message::ExecStatus::Success)
        {
            _webHashId = message->tag();
            log_verbose << "'Ais Expert' server authorization success"
                        << ". Access hashId: " << _webHashId;
            // После успешной авторизации, проверяем список пользователей из БД
            //validateUsers();
            //event_log::log(event_log::Type::User, QTEXT("Аутентификация успешна"));

            //getTaskTypes();
            //getTasksLog();
            getGroups();
            getAisUsers();
            getTaskModels();
            getTaskScores(QUuidEx());

            startMonitoring();

            Message::Ptr m = createJsonMessage(command::TaskSyncDataInfo);
            _socket->send(m);

            _labelConnectStatus->setText("Connected (Authorized)");
        }
        else
        {
            log_verbose << "'Ais Expert' server authorization failed";

            QString msg = errorDescription(message);
            QMessageBox::critical(this, qApp->applicationName(), msg);
        }
    }
}

void MainWindow::command_SpeedTest(const Message::Ptr& message)
{
    if (_testFirst)
        return;

    data::SpeedTest speedTest;
    readFromMessage(message, speedTest);

    if (_testTimer.elapsed() < 10*1000)
    {
        ++_testCount;

        //if (_testCount % 1000 == 0)
        //{
        //    for (int i = 0; i <= 1000; ++i)
        //    {
                Message::Ptr m = createJsonMessage(command::SpeedTest);
                m->setPriority(Message::Priority::High);
                _socket->send(m);
        //    }
        //}
        return;
    }

    _testFirst = true;
    QString msg = "SpeedTest count: %1";
    QMessageBox::information(this, qApp->applicationName(), msg.arg(_testCount));
}

void MainWindow::startMonitoring()
{
    getMonitoring();

    if (!_timerMonitoringUpdate.isActive()) {
        _timerMonitoringUpdate.start(3*1000);
    }
}

void MainWindow::getMonitoring()
{
    Message::Ptr m = createJsonMessage(command::Monitoring);
    _socket->send(m);
}

void MainWindow::command_Monitoring(const Message::Ptr& message)
{
    data::Monitoring monitoring;
    readFromMessage(message, monitoring);

    ui->lblCpuTemp->setText(QString::number(monitoring.hardware.cpuMaxCoreTemp, 'f', 1) + QChar(176) + "C");
    ui->barCPU->setValue(monitoring.hardware.cpu.system);

    ui->barRam->setValue(monitoring.hardware.ram.system / double(monitoring.hardware.ram.total) * 100.0);
    ui->lblRamUsedTotal->setText(QString::number(monitoring.hardware.ram.system) + " MB / "
                                 + QString::number(monitoring.hardware.ram.total) + " MB");

    ui->lblHddUsedTotal->setText(QString::number(monitoring.hardware.hdd.system)
                                 + " MB / " + QString::number(monitoring.hardware.hdd.total) + " MB");

    ui->barHdd->setValue(monitoring.hardware.hdd.system / double(monitoring.hardware.hdd.total) * 100.0);

// Установка цветности индикаторов ЦПУ и ОЗУ:
    static const QString normal  = "QProgressBar {text-align: right; border: 1px solid grey; border-radius: 5px;} QProgressBar::chunk {background-color: green;}";
    static const QString warning = "QProgressBar {text-align: right; border: 1px solid grey; border-radius: 5px;} QProgressBar::chunk {background-color: yellow;}";
    static const QString alarm   = "QProgressBar {text-align: right; border: 1px solid grey; border-radius: 5px;} QProgressBar::chunk {background-color: red;}";

    if (ui->barCPU->value() < 45) {
        ui->barCPU->setStyleSheet(normal);
    } else if (ui->barCPU->value() < 75) {
        ui->barCPU->setStyleSheet(warning);
    } else {
        ui->barCPU->setStyleSheet(alarm);
    }

    if (ui->barRam->value() < 45) {
        ui->barRam->setStyleSheet(normal);
    } else if (ui->barRam->value() < 75) {
        ui->barRam->setStyleSheet(warning);
    } else {
        ui->barRam->setStyleSheet(alarm);
    }

    if (ui->barHdd->value() < 45) {
        ui->barHdd->setStyleSheet(normal);
    } else if (ui->barHdd->value() < 75) {
        ui->barHdd->setStyleSheet(warning);
    } else {
        ui->barHdd->setStyleSheet(alarm);
    }

    ui->tabProcessMonitoring->setItem(0,0,new QTableWidgetItem(monitoring.hardware.processName));
    ui->tabProcessMonitoring->setItem(0,1,new QTableWidgetItem(QString::number(monitoring.hardware.PID)));
    ui->tabProcessMonitoring->setItem(0,2,new QTableWidgetItem(QString::number(monitoring.hardware.cpu.process) + " % " ));
    ui->tabProcessMonitoring->setItem(0,3,new QTableWidgetItem(QString::number(monitoring.hardware.nThreads)));
    ui->tabProcessMonitoring->setItem(0,4,new QTableWidgetItem(QString::number(monitoring.hardware.ram.process) + " MB" ));
    ui->tabProcessMonitoring->setItem(0,5,new QTableWidgetItem(QString::number(monitoring.hardware.hdd.process) + " MB" ));

    if (monitoring.software.fomsIsActive)
        ui->labelFomsEcho->setText("FomsIsActive: true");
    else
        ui->labelFomsEcho->setText("FomsIsActive: false");

}

void MainWindow::command_Error(const Message::Ptr& message)
{
    data::Error error;
    readFromMessage(message, error);

    if (error.code  == error::MessageContentParse)
    {
        QString msg = error.description;
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    if (error.commandId == command::UPoolUserAuth)
    {
        QString msg = "'Users Pool' service inaccessible";
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }
}

void MainWindow::command_UPoolUserList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::UPoolUserList upoolUserList;
        readFromMessage(message, upoolUserList);
        //for (auto u : upoolUserList.users)
        //    qDebug() << u;

//        ui->cboxUsers->blockSignals(true);
//        for (const data::UPoolUserInfo& u : upoolUserList.items)
//        {
//            ui->cboxUsers->addItem(u.login);
//            //ui->web_model_userId->addItem(u.login);
//        }
//        ui->cboxUsers->setCurrentIndex(0);
//        ui->cboxUsers->blockSignals(false);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

/*void MainWindow::updateLabels(const QListWidgetItem &current)
{

}*/

//void MainWindow::currentTaskTypeChanged(QListWidgetItem *item)
//{
//    if (!item) return;

//    for (auto t : _taskTypes)
//    {
//        auto uuid = qvariant_cast<QUuidEx>(item->data(Qt::UserRole));
//        if (t.id == uuid)
//        {
//            ui->editTypeTask->setText(t.name);
//            ui->editClassOfType->setText(t.classTypeTask);
//            _selectedTaskType = t;
//            break;
//        }
//    }
//}

//void MainWindow::currentTaskChanged(QListWidgetItem *item)
//{
//    if (!item) return;

//    for (auto t : _tasks)
//    {
//        auto uuid = qvariant_cast<QUuidEx>(item->data(Qt::UserRole));
//        if (t.id == uuid)
//        {
//            ui->editTaskName->setText(t.name);
//            ui->dateEditTaskStart->setDate(t.runDate);
//            ui->timeEditTimeStart->setTime(t.runTime);
//            ui->editTaskName->setText(t.name);
//            ui->sbRegularityDay->setValue(t.regularityDay);
//            ui->sbRegularityHour->setValue(t.regularityHour);
//            _selectedTask = t;
//            break;
//        }
//    }
//}

void MainWindow::command_UPoolUserAuth(const Message::Ptr& message)
{
    if (message->type() == Message::Type::Answer)
    {
        if (message->execStatus() != Message::ExecStatus::Success)
        {
            QString msg = errorDescription(message);
            QMessageBox::critical(this, qApp->applicationName(), msg);
        }
        else
        {
            data::UPoolUserAuth upoolUserAuth;
            readFromMessage(message, upoolUserAuth);

            QString msg = "Success user authorization\nUser: %1\n"
                          "name: %2\n"
                          "sid: %3\n"
                          "is admin: %4\n"
                          "id: %5\n"
                          "hash id: %6";
            msg = msg.arg(upoolUserAuth.login)
                     .arg(upoolUserAuth.name)
                     .arg(upoolUserAuth.sid)
                     .arg(upoolUserAuth.isAdmin)
                     .arg(upoolUserAuth.id.toString())
                     .arg(upoolUserAuth.hashId);

            QMessageBox::information(this, qApp->applicationName(), msg);
        }
    }
}

void MainWindow::currentGroupChanged(QListWidgetItem *item)
{
    if (!item) return;

    //QListWidgetItem* item = ui->lstGrpup...   TaskModels->currentItem();
    QVariant v = item->data(Qt::UserRole);
    data::Group group = v.value<data::Group>();

    ui->editGroupName->setText(group.name);
    ui->cbCanApplyModel->setCurrentIndex(static_cast<int>(group.canApplyModel));
    ui->cbCanLearnModel->setCurrentIndex(static_cast<int>(group.canLearnModel));
    ui->cbCanReports->setCurrentIndex(static_cast<int>(group.canRWReport));
    ui->cbCanMonitoring->setCurrentIndex(static_cast<int>(group.canMonitoring));
    ui->cbIsDefault->setCurrentIndex(group.isDefault ? 1 : 0);

//    for (auto r : _groups)
//    {
//        auto uuid = qvariant_cast<QUuidEx>(item->data(Qt::UserRole));
//        if (r.id == uuid)
//        {
//            ui->editGroupName->setText(r.name);
//            ui->cbCanApplyModel->setCurrentIndex(static_cast<int>(r.canApplyModel));
//            ui->cbCanLearnModel->setCurrentIndex(static_cast<int>(r.canLearnModel));
//            ui->cbCanReports->setCurrentIndex(static_cast<int>(r.canRWReport));
//            ui->cbCanMonitoring->setCurrentIndex(static_cast<int>(r.canMonitoring));
//            ui->cbIsDefault->setCurrentIndex(r.isDefault ? 1 : 0);
//            _selectedGroup = r;
//            break;
//        }
//    }
}

void MainWindow::currentAisUserChanged(QListWidgetItem *item)
{
    if (!item) return;

//    for (auto u : _users)
//    {
//        auto uuid = qvariant_cast<QUuidEx>(item->data(Qt::UserRole));
//        if (u.id == uuid)
//        {
//            _selectedUser = u;
//            break;
//        }
//    }
}

void MainWindow::command_UPoolUserInvalidated(const Message::Ptr& message)
{
    Message::Ptr answer = message->cloneForAnswer();
    _socket->send(answer);
}

void MainWindow::on_btnSpeedTest_clicked(bool)
{
    _testCount = 0;
    _testTimer.reset();
    _testFirst = false;

    //for (int i = 0; i <= 1000; ++i)
    //{
        Message::Ptr m = createJsonMessage(command::SpeedTest);
        m->setPriority(Message::Priority::High);
        _socket->send(m);
    //}
}

void MainWindow::on_btnSendMailDemo_clicked(bool)
{
    data::SendMailDebug sendMailDebug;
    sendMailDebug.addresses.append("hkarel@yandex.ru");
    sendMailDebug.addresses.append("hkarel@rambler.ru");
    sendMailDebug.addresses.append("karelin@tochka.ai");
    sendMailDebug.subject = QTEXT("Тестовое уведомление 3");
    sendMailDebug.body = QTEXT("Тестовое сообщение 3 \nДобавочное сообщение");

    Message::Ptr m = createJsonMessage(sendMailDebug);
    m->setTag(_webHashId);
    _socket->send(m);
}

void MainWindow::on_btnAisConnect_clicked(bool)
{
}

void MainWindow::on_btnAisDisconnect_clicked(bool)
{
    _socket->disconnect();
}

void MainWindow::on_btnGetUsers_clicked(bool)
{
    ui->cboxUsers->clear();

    Message::Ptr m = createJsonMessage(command::UserList);
    _socket->send(m);
}

void MainWindow::on_btnUserInfo_clicked(bool)
{
    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::UserInfo userInfo;
    userInfo.id = user.id;

    Message::Ptr m = createJsonMessage(userInfo);
    _socket->send(m);
}

void MainWindow::on_btnCheckPassw_clicked(bool)
{
    data::UPoolUserAuth upoolUserAuth;
    upoolUserAuth.login = ui->cboxUsers->currentText();
    upoolUserAuth.password = ui->lineUserPassw->text();

    Message::Ptr m = createJsonMessage(upoolUserAuth);
    _socket->send(m);
}

void MainWindow::on_btnDbConnect_clicked(bool)
{
//    bool res = db::firebird::driver()->open("/var/opt/aisexpert/database/main.fdb",
//                                            "SYSDBA",
//                                            "masterkey",
//                                            //"127.0.0.1",
//                                            "hkarel.noip.me",
//                                            3050);

//    if (!res)
//        QMessageBox::information(this, qApp->applicationName(),
//                                 "Failed connect to DB");

//    QStringList sl;
//    if (res)
//    {
//        sl = db::firebird::driver().tables(QSql::SystemTables);
//        qDebug() << sl;

//        QSqlRecord sr = db::firebird::driver().record("TEST1");
//        for (int i = 0; i < sr.count(); ++i)
//        {
//            qDebug() << sr.field(i).name();

//        }
//    }
}

void MainWindow::on_btnDbDisconnect_clicked(bool)
{
//    db::firebird::driver()->close();
}

void MainWindow::on_btnDbSelect_clicked(bool)
{
//    QSqlQuery q {db::firebird::driver()->createResult()};
//    //q.prepare("select ID2 from TEST1");
//    //q.prepare("select LOGIN from USERS where LOGIN in (:LOGINS)");
//    //q.prepare("select LOGIN from USERS where LOGIN in (? , ?)");
//    q.prepare("select LOGIN from USERS where LOGIN in (select LOGIN from USERS where NAME = ?)");
//    if (q.lastError().isValid())
//        return;

//    QStringList logins;
//    //QVector<QVariant> logins;
//    logins << "user1" << "user2";
//    //bindValue(q, ":LOGINS", logins);
//    //q.bindValue(":LOGINS", logins);
//    q.addBindValue("user1");
//    //q.addBindValue("user2");

//    q.exec();
//    if (q.lastError().isValid())
//    {
//        QMessageBox::information(this, qApp->applicationName(),
//                                 "Failed exec SQL");
//        return;
//    }

//    while (q.next())
//    {
//        QSqlRecord r = q.record();
//        alog::Line logLine = log_info << "rec: ";
//        for (int i = 0; i < r.count(); ++i)
//        {
//            if (r.field(i).value().canConvert<QUuidEx>())
//                logLine << r.field(i).value().value<QUuidEx>() << " | ";
//            else
//                logLine << r.field(i).value() << " | ";
//        }
//    }
//    log_info << " ";
}

void MainWindow::on_btnDbInsert_clicked(bool)
{
//    QSqlQuery q {db::firebird::driver()->createResult()};

//    q.prepare("insert into TEST1 (FIELD1, FIELD2, ID2) values (:FIELD1, :FIELD2, :ID22)");
//    if (q.lastError().isValid())
//        return;

////    q.bindValue(":FIELD1", QTEXT("ИМЯ 1"));
////    q.bindValue(":FIELD2", QTEXT("ЗНАЧЕНИЕ 1"));
////    q.exec();

////    q.bindValue(":FIELD1", QTEXT("ИМЯ 2"));
////    q.bindValue(":FIELD2", QTEXT("ЗНАЧЕНИЕ 2"));
////    q.exec();

////    q.bindValue(":FIELD1", QTEXT("ИМЯ 3"));
////    q.bindValue(":FIELD2", QTEXT("ЗНАЧЕНИЕ 3"));

//    //QUuidEx uuid ("773c5587-f7a2-47e4-86a1-a4932b880fcb");
//    //QUuidEx uuid ("17209083-6908-4111-8c11-fbbd2e3a3362");
//    QUuidEx  uuid ("13792be6-9df6-47c0-beeb-77d3e2aa36aa");

//    bindValue(q, ":FIELD1", QTEXT("ИМЯ 1"));
//    bindValue(q, ":ID22",   uuid);
//    bindValue(q, ":FIELD2", QTEXT("ЗНАЧЕНИЕ 1"));
//    q.exec();
}

void MainWindow::on_btnDbGTransact_clicked(bool)
{
//    db::firebird::Transaction::Ptr transact = db::firebird::createTransact();

//    transact->begin();

//    QSqlQuery q {db::firebird::createResult(transact)};
//    q.prepare("insert into TEST1 (FIELD11, FIELD2) values (:FIELD1, :FIELD2)");
//    if (q.lastError().isValid())
//    {
//        transact->rollback();
//        return;
//    }

//    q.bindValue(":FIELD1", QTEXT("GTr 1"));
//    q.bindValue(":FIELD2", QTEXT("ЗНАЧЕНИЕ 1"));
//    q.exec();

//    q.bindValue(":FIELD1", QTEXT("GTr 2"));
//    q.bindValue(":FIELD2", QTEXT("ЗНАЧЕНИЕ 2"));
//    q.exec();

//    q.bindValue(":FIELD1", QTEXT("GTr 3"));
//    q.bindValue(":FIELD2", QTEXT("ЗНАЧЕНИЕ 3"));
//    q.exec();

//    if (q.lastError().isValid())
//    {
//        transact->rollback();
//        return;
//    }
//    transact->commit();
//    //transact->rollback();
}

void MainWindow::on_btnDbDelete_clicked(bool)
{
//    QSqlQuery q {db::firebird::driver()->createResult()};
//    sql::exec(q, "delete from TEST1 where ID >= ?", 4);

////    q.prepare("delete from TEST1 where ID >= ?");
////    q.bindValue(":ID", 4);
//    //q.exec();
}

void MainWindow::on_btnSynData_clicked(bool)
{
//    data::SyncData session;
//    Message::Ptr session_message = createJsonMessage(session);
//    _socket->send(session_message);
//    log_info << "SyncData data retrive session was sended";
    // Get sessionId
}

//void MainWindow::on_btnCalcRating_clicked(bool)
//{
//    data::CreateModel session;
//    // Put sessionId fo sa
//    Message::Ptr session_message = createJsonMessage(session);
//    _socket->send(session_message);
//    log_info << "CreateModel data retrive session was sended";
//}

void MainWindow::on_btnSendData_clicked(bool)
{
//    data::SendData session;
//    // Put sessionId fo sa
//    Message::Ptr session_message = createJsonMessage(session);
//    _socket->send(session_message);
//    log_info << "SendData data retrive session was sended";
}

void MainWindow::on_btnSyncDataNow_clicked(bool)
{
    Message::Ptr m = createJsonMessage(command::TaskSyncDataNow);
    _socket->send(m);
}

void MainWindow::on_btnSyncDataInfo_clicked(bool)
{
    Message::Ptr m = createJsonMessage(command::TaskSyncDataInfo);
    _socket->send(m);

   log_info << "on_btnSyncDataInfo_clicked";
}

void MainWindow::on_btnSyncDataSet_clicked(bool)
{
    data::TaskSyncDataEdit taskSyncDataEdit;
    taskSyncDataEdit.isEnabled = ui->chkSyncDataEnabled->isChecked();
    //taskSyncDataEdit.regularityDay = ui->sboxSyncDataRDay->value();
    //taskSyncDataEdit.regularityHour = ui->sboxSyncDataRHour->value();

    Message::Ptr m = createJsonMessage(taskSyncDataEdit);
    _socket->send(m);

   log_info << "on_btnSyncDataSet_clicked";
}

void MainWindow::on_btnCreateSyncPlanningDebug_clicked(bool)
{
//    data::CreateSyncPlanningDebug createSyncPlanningDebug;
//    createSyncPlanningDebug.period.begin = QDateTime(QDate(2019, 4, 1));
//    createSyncPlanningDebug.period.end = QDateTime(QDate(2019, 4, 3));
//    createSyncPlanningDebug.timeMark = QDateTime(QDate(1970, 1, 1));

//    Message::Ptr m = createJsonMessage(createSyncPlanningDebug);
//    _socket->send(m);

//   log_info << "on_btnStartSyncPlanning_clicked";
}

void MainWindow::on_btnKeepWaitCommand_clicked(bool)
{
    data::KeepWaitCommand keepWaitCommand;
    keepWaitCommand.commandId = QUuidEx("D74DB0E9-872C-4A46-B4ED-DB666F9EB894");
    keepWaitCommand.messageId = QUuidEx("661C8497-EA83-48DB-A40C-171BF1B5C09F");
    keepWaitCommand.timeToAdd = 10;

    Message::Ptr m = createJsonMessage(keepWaitCommand);
    _socket->send(m);

    log_info << "on_btnKeepWaitCommand_clicked";
}

void MainWindow::on_btnSyncIterrupt_clicked(bool)
{
//    data::TaskInterrupt taskInterrupt;
//    taskInterrupt.taskId = QUuidEx();
//    taskInterrupt.taskType = TaskType::SyncData;

//    Message::Ptr m = createJsonMessage(taskInterrupt);

    Message::Ptr m = createJsonMessage(command::TaskSyncDataStop);
    _socket->send(m);

   log_info << "on_btnSyncIterrupt_clicked";
}

void MainWindow::on_chkSyncDataMute_clicked(bool)
{
//    data::SyncDataMute syncDataMute;
//    syncDataMute.value = ui->chkSyncDataMute->isChecked();

//    Message::Ptr m = createJsonMessage(syncDataMute);
//    _socket->send(m);

    log_info << "on_chkSyncDataMute_clicked";
}

void MainWindow::on_btnTaskSyncProgress_clicked(bool)
{
    Message::Ptr m = createJsonMessage(command::SyncProgress);
    _socket->send(m);

    log_info << "on_btnTaskSyncProgress_clicked";
}

void MainWindow::on_btnSyncNsiNow_clicked(bool)
{
    Message::Ptr m = createJsonMessage(command::TaskSyncNsiNow);
    _socket->send(m);
}

void MainWindow::on_btnTaskModelCreate_clicked(bool)
{
    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::TaskModelCreate taskModelCreate;

    taskModelCreate.userId = user.id;
    taskModelCreate.isEnabled = ui->web_model_isEnabled->isChecked();
    taskModelCreate.name = ui->web_model_name->text();
    taskModelCreate.description = ui->web_model_description->text();
    taskModelCreate.runDateTime =  ui->web_model_runDateTime->dateTime();
    taskModelCreate.regularityMonth = ui->web_model_regularityMonth->value();
    taskModelCreate.regularityDay = ui->web_model_regularityDay->value();
    taskModelCreate.regularityHour = ui->web_model_regularityHour->value();
    taskModelCreate.period.begin = ui->web_model_periodBegin->dateTime();
    taskModelCreate.period.end = ui->web_model_periodEnd->dateTime();
    taskModelCreate.isPublic = ui->web_model_isPublic->isChecked();

    Message::Ptr m = createJsonMessage(taskModelCreate);
    _socket->send(m);
    log_info << "on_btnTaskModelCreate_clicked";
}

void MainWindow::on_btnTaskModelEdit_clicked(bool)
{
    if (!ui->lstTaskModels->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskModelEdit taskModelEdit;
    *((data::Task*)&taskModelEdit) = task;

    //taskModelEdit.id = task.id;
    //taskModelEdit.userId = user.id;
    taskModelEdit.isEnabled = ui->web_model_isEnabled->isChecked();
    taskModelEdit.name = ui->web_model_name->text();
    taskModelEdit.description = ui->web_model_description->text();
    taskModelEdit.runDateTime =  ui->web_model_runDateTime->dateTime();
    taskModelEdit.regularityMonth = ui->web_model_regularityMonth->value();
    taskModelEdit.regularityDay = ui->web_model_regularityDay->value();
    taskModelEdit.regularityHour = ui->web_model_regularityHour->value();
    taskModelEdit.relPeriodBegin = ui->web_model_offsetMonthBegin->value();
    taskModelEdit.relPeriodDuration = ui->web_model_offsetMonthEnd->value();
    taskModelEdit.period.begin = ui->web_model_periodBegin->dateTime();
    taskModelEdit.period.end = ui->web_model_periodEnd->dateTime();
    taskModelEdit.isPublic = ui->web_model_isPublic->isChecked();

    Message::Ptr m = createJsonMessage(taskModelEdit);
    m->setTag(hash64(user.id));
    _socket->send(m);
    log_info << "on_btnTaskModelEdit_clicked";
}

void MainWindow::on_btnTaskModelDelete_clicked(bool)
{
    if (!ui->lstTaskModels->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskModelDelete taskModelDelete;
    taskModelDelete.taskId = task.id;
    taskModelDelete.userId = user.id;

    Message::Ptr m = createJsonMessage(taskModelDelete);
    m->setTag(user.hashId);
    _socket->send(m);
    log_info << "on_btnTaskModelDelete_clicked";
}

void MainWindow::on_btnTaskModelInfo_clicked(bool)
{
    if (!ui->lstTaskModels->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskModelInfo taskModelInfo;
    taskModelInfo.taskId = task.id;
    taskModelInfo.userId = user.id;

    Message::Ptr m = createJsonMessage(taskModelInfo);
    m->setTag(user.hashId);
    _socket->send(m);
    log_info << "on_btnTaskModelInfo_clicked";
}

void MainWindow::on_btnTaskModelList_clicked(bool)
{
    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::TaskModelList taskModelList;
    taskModelList.filter.userId = user.id;
    taskModelList.filter.mode = TaskMode::All;

    Message::Ptr m = createJsonMessage(taskModelList);
    m->setTag(user.hashId);
    _socket->send(m);

    log_info << "on_btnTaskModelList_clicked";
}

void MainWindow::on_btnTaskScoreStartNow_clicked(bool)
{
    if (!ui->lstTaskScores->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskScore not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskStartNow taskStartNow;
    taskStartNow.taskId = task.id;
    taskStartNow.userId = user.id;

    Message::Ptr m = createJsonMessage(taskStartNow);
    _socket->send(m);
    log_info << "on_btnTaskScoreStartNow_clicked";
}

void MainWindow::on_btnTaskScoreStopNow_clicked(bool)
{
    if (!ui->lstTaskScores->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskScore not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskInterrupt taskInterrupt;
    taskInterrupt.taskId = task.id;
    taskInterrupt.userId = user.id;

    Message::Ptr m = createJsonMessage(taskInterrupt);
    _socket->send(m);

    log_info << "on_btnTaskScoreStopNow_clicked";
}

void MainWindow::on_btnTaskScoreCreate_clicked(bool)
{
    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    if (ui->sboxTaskScoreRegularityMonth->value() == 0
        && ui->sboxTaskScoreRegularityDay->value() == 0
        && ui->sboxTaskScoreRegularityHour->value() == 0)
    {
        if (ui->cboxScoreModels->currentIndex() == -1)
        {
            QMessageBox::critical(this, qApp->applicationName(), "Model not selected!");
            return;
        }
    }
    else
    {
        if (ui->cboxScoreTaskModels->currentIndex() == -1)
        {
            QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
            return;
        }
    }


    v = ui->cboxScoreModels->itemData(ui->cboxScoreModels->currentIndex());
    data::Model model = v.value<data::Model>();

    v = ui->cboxScoreTaskModels->itemData(ui->cboxScoreTaskModels->currentIndex());
    data::Task task = v.value<data::Task>();

    data::TaskScoreCreate taskScoreCreate;
    taskScoreCreate.userId = user.id;
    taskScoreCreate.modelId = model.id;
    taskScoreCreate.parentId = task.id;

    taskScoreCreate.isEnabled        = ui->cboxTaskScoreEnabled->isChecked();
    taskScoreCreate.name             = ui->lineTaskScoreName->text();
    taskScoreCreate.description      = ui->lineTaskScoreDescr->text();
    taskScoreCreate.runDateTime      = ui->lineTaskScoreRunDT->dateTime();
    taskScoreCreate.regularityMonth  = ui->sboxTaskScoreRegularityMonth->value();
    taskScoreCreate.regularityDay    = ui->sboxTaskScoreRegularityDay->value();
    taskScoreCreate.regularityHour   = ui->sboxTaskScoreRegularityHour->value();
    taskScoreCreate.relPeriodBegin   = ui->sboxTaskScoreOffsetMonthBegin->value();
    taskScoreCreate.relPeriodDuration= ui->sboxTaskScoreOffsetMonthEnd->value();

    taskScoreCreate.period.begin     = ui->sboxTaskScorePeriodBegin->dateTime();
    taskScoreCreate.period.end       = ui->sboxTaskScorePeriodEnd->dateTime();
    taskScoreCreate.isPublic         = ui->chkTaskScorePublic->isChecked();

    Message::Ptr m = createJsonMessage(taskScoreCreate);
    _socket->send(m);
    log_info << "on_btnTaskScoreCreate_clicked";
}

void MainWindow::on_btnTaskScoreEdit_clicked(bool)
{
    if (!ui->lstTaskScores->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskScore not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    QUuidEx uuid;
    if (task.isPeriodic())
    {
        v = ui->cboxScoreTaskModels->itemData(ui->cboxScoreTaskModels->currentIndex());
        data::Task task = v.value<data::Task>();
        uuid = task.id;
    }
    else
    {
        v = ui->cboxScoreModels->itemData(ui->cboxScoreModels->currentIndex());
        data::Model model = v.value<data::Model>();
        uuid = model.id;
    }

    data::TaskScoreEdit taskScoreEdit;
    taskScoreEdit.id = task.id;
    taskScoreEdit.userId = user.id;
    taskScoreEdit.modelId = uuid;

    taskScoreEdit.isEnabled       = ui->cboxTaskScoreEnabled->isChecked();
    taskScoreEdit.name            = ui->lineTaskScoreName->text();
    taskScoreEdit.description     = ui->lineTaskScoreDescr->text();
    taskScoreEdit.runDateTime     = ui->lineTaskScoreRunDT->dateTime();
    taskScoreEdit.regularityMonth = ui->sboxTaskScoreRegularityMonth->value();
    taskScoreEdit.regularityDay   = ui->sboxTaskScoreRegularityDay->value();
    taskScoreEdit.regularityHour  = ui->sboxTaskScoreRegularityHour->value();

    taskScoreEdit.period.begin    = ui->sboxTaskScorePeriodBegin->dateTime();
    taskScoreEdit.period.end      = ui->sboxTaskScorePeriodEnd->dateTime();
    taskScoreEdit.isPublic        = ui->chkTaskScorePublic->isChecked();

    Message::Ptr m = createJsonMessage(taskScoreEdit);
    _socket->send(m);
    log_info << "on_btnTaskScoreEdit_clicked";
}

void MainWindow::on_btnTaskScoreDelete_clicked(bool)
{
    if (!ui->lstTaskScores->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskScoreDelete taskScoreDelete;
    taskScoreDelete.taskId = task.id;
    taskScoreDelete.userId = user.id;

    Message::Ptr m = createJsonMessage(taskScoreDelete);
    m->setTag(user.hashId);
    _socket->send(m);
    log_info << "on_btnTaskScoreDelete_clicked";
}

void MainWindow::on_btnTaskScoreInfo_clicked(bool)
{
    if (!ui->lstTaskScores->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskScoreInfo taskScoreInfo;
    taskScoreInfo.taskId = task.id;
    taskScoreInfo.userId = user.id;

    Message::Ptr m = createJsonMessage(taskScoreInfo);
    m->setTag(user.hashId);
    _socket->send(m);
    log_info << "on_btnTaskScoreInfo_clicked";
}

void MainWindow::on_btnTaskScoreList_clicked(bool)
{
    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::TaskScoreList taskScoreList;
    taskScoreList.filter.userId = user.id;

    Message::Ptr m = createJsonMessage(taskScoreList);
    m->setTag(user.hashId);
    _socket->send(m);

    log_info << "on_btnTaskScoreList_clicked";
}

void MainWindow::on_btnTaskModelStartNow_clicked(bool)
{
    if (!ui->lstTaskModels->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskStartNow taskStartNow;
    taskStartNow.taskId = task.id;
    taskStartNow.userId = user.id;

    Message::Ptr m = createJsonMessage(taskStartNow);
    _socket->send(m);
    log_info << "on_btnTaskModelStartNow_clicked";
}

void MainWindow::on_btnTaskModelStopNow_clicked(bool)
{
    if (!ui->lstTaskModels->currentItem())
    {
        QMessageBox::critical(this, qApp->applicationName(), "TaskModel not selected!");
        return;
    }

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstTaskModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    data::TaskInterrupt taskInterrupt;
    taskInterrupt.taskId = task.id;
    taskInterrupt.userId = user.id;

    Message::Ptr m = createJsonMessage(taskInterrupt);
    _socket->send(m);

    log_info << "on_btnTaskModelStopNow_clicked";
}

void MainWindow::on_btnModelEdit_clicked(bool)
{
    if (!ui->lstModels->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Model model = v.value<data::Model>();

    data::ModelEdit modelEdit;
    modelEdit.id = model.id;
    modelEdit.userId = user.id;

    modelEdit.name = ui->web_model_name_text->text();
    modelEdit.description = ui->web_model_descr_text->text();
    modelEdit.isPublic = ui->web_model_is_public->isChecked();

    Message::Ptr m = createJsonMessage(modelEdit);
    m->setTag(hash64(user.id));
    _socket->send(m);

    log_info << "on_web_modelEditButton_clicked";
}

void MainWindow::on_btnModelDelete_clicked(bool)
{
    if (!ui->lstModels->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Model model = v.value<data::Model>();

    data::ModelDelete modelDelete;
    modelDelete.id = model.id;
    modelDelete.userId = user.id;

    Message::Ptr m = createJsonMessage(modelDelete);
    _socket->send(m);

    log_info << "on_web_modelDeleteButton_clicked";
}

void MainWindow::on_btnModelInfo_clicked(bool)
{
    if (!ui->lstModels->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstModels->currentItem();
    v = item->data(Qt::UserRole);
    data::Model model = v.value<data::Model>();

    data::ModelInfo modelInfo;
    modelInfo.id = model.id;
    modelInfo.userId = user.id;

    Message::Ptr m = createJsonMessage(modelInfo);
    _socket->send(m);

    log_info << "on_web_modelInfoButton_clicked";
}

void MainWindow::on_btnModelList_clicked(bool)
{
    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::ModelList modelList;
    modelList.filter.userId = user.id;
    //modelList.filter.showPublic = true;
    //modelList.filter.name = QString::fromUtf8(u8"регуляр");

    Message::Ptr m = createJsonMessage(modelList);
    _socket->send(m);
    log_info << "on_web_updateModelsButton_clicked";
}

void MainWindow::on_btnScoreEdit_clicked(bool)
{
    if (!ui->lstScores->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Score score = v.value<data::Score>();

    data::ScoreEdit scoreEdit;
    scoreEdit.id = score.id;
    scoreEdit.userId = user.id;

    scoreEdit.name = ui->lineScoreName->text();
    scoreEdit.description = ui->lineScoreDescr->text();
    scoreEdit.isPublic = ui->cboxScorePublic->isChecked();

    Message::Ptr m = createJsonMessage(scoreEdit);
    _socket->send(m);

    log_info << "on_btnScoreEdit_clicked";
}

void MainWindow::on_btnScoreDelete_clicked(bool)
{
    if (!ui->lstScores->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Score score = v.value<data::Score>();

    data::ScoreDelete scoreDelete;
    scoreDelete.id = score.id;
    scoreDelete.userId = user.id;

    Message::Ptr m = createJsonMessage(scoreDelete);
    _socket->send(m);

    log_info << "on_btnScoreDelete_clicked";
}

void MainWindow::on_btnScoreInfo_clicked(bool)
{
    if (!ui->lstScores->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Score score = v.value<data::Score>();

    data::ScoreInfo scoreInfo;
    scoreInfo.id = score.id;
    scoreInfo.userId = user.id;

    Message::Ptr m = createJsonMessage(scoreInfo);
    _socket->send(m);

    log_info << "on_btnScoreInfo_clicked";
}

void MainWindow::on_btnScoreList_clicked(bool)
{
    ui->lstScores->clear();

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::ScoreList scoreList;
    scoreList.filter.userId = user.id;

    Message::Ptr m = createJsonMessage(scoreList);
    _socket->send(m);
    log_info << "on_btnScoreList_clicked";
}

void MainWindow::on_btnXgbBaseOptionsList_clicked(bool)
{
    data::ModelXgbInfo modelXgbInfo;
    modelXgbInfo.id = QUuidEx{};

    Message::Ptr m = createJsonMessage(modelXgbInfo);
    _socket->send(m);
    log_info << "on_btnXgbBaseOptionsList_clicked";
}

void MainWindow::on_btnAddXgbOptSet_clicked(bool)
{
    QString name = ui->xgbOptName->text();
    QString value = ui->xgbOptValue->text();

    xgbEditOption[name] = value;

    ui->xgbOptionsSetList->blockSignals(true);
    ui->xgbOptionsSetList->append(name + ": " + value);

    log_info << "on_btnAddXgbOptSet_clicked";
}

void MainWindow::on_btnClearXgbOptSet_clicked(bool)
{
    ui->xgbOptionsSetList->blockSignals(true);
    ui->xgbOptionsSetList->clear();
    xgbEditOption.clear();

    log_info << "on_btnAddXgbOptSet_clicked";
}

void MainWindow::on_btnXgbEditOptions_clicked(bool)
{
    data::ModelXgbEdit modelXgbEdit;
    modelXgbEdit.id = QUuidEx{0};

    for (const QString& key : xgbEditOption.keys())
    {
        modelXgbEdit.items.append(data::XgbParam{key, xgbEditOption.value(key), true});
    }

    Message::Ptr m = createJsonMessage(modelXgbEdit);
    _socket->send(m);

    log_info << "on_btnXgbEditOptions_clicked";
}

void MainWindow::on_btnXgbDelOptions_clicked(bool)
{
    data::ModelXgbDelete modelXgbDelete;
    modelXgbDelete.id = QUuidEx{0};

    for (const QString& key : xgbEditOption.keys())
    {
        modelXgbDelete.items.append(data::XgbParam{key, xgbEditOption.value(key)});
    }

    Message::Ptr m = createJsonMessage(modelXgbDelete);
    _socket->send(m);

    log_info << "on_btnXgbDelOptions_clicked";
}

void MainWindow::on_btnEnableXgbOptSet_clicked(bool)
{
    data::ModelXgbOption modelXgbOption;
    modelXgbOption.id = QUuidEx{0};

    modelXgbOption.name = ui->xgbOptName->text();
    modelXgbOption.enable = true;
    Message::Ptr m = createJsonMessage(modelXgbOption);
    _socket->send(m);
    log_info << "on_btnEnableXgbOptSet_clicked";
}

void MainWindow::on_btnDisableXgbOptSet_clicked(bool)
{
    data::ModelXgbOption modelXgbOption;
    modelXgbOption.id = QUuidEx{0};

    modelXgbOption.name = ui->xgbOptName->text();
    modelXgbOption.enable = false;
    Message::Ptr m = createJsonMessage(modelXgbOption);
    _socket->send(m);
    log_info << "on_btnDisableXgbOptSet_clicked";
}

void MainWindow::checkConnections()
{
    if (!_socket->isConnected())
    {
        _socket->connect();
        // Ждем подключения 6 секунд
        int sleepCount = 0;
        while (sleepCount++ < 1*10)
        {
            usleep(100*1000);
            qApp->processEvents();
            if (_socket->isConnected())
                break;
        }
        if (!_socket->isConnected())
        {
            log_error << "Time expired for connection to 'Ais Expert' server"
                      << ". Connection point: " << _socket->peerPoint();

            //QString msg = "Time expired for connection";
            //QMessageBox::information(this, qApp->applicationName(), msg);

        }
    }
}

void MainWindow::on_btnAddTypeTask_clicked(bool)
{
//    data::AddTypeTask request;
//    request.typeTask.name = ui->editTypeTask->text();
//    request.typeTask.classTypeTask = ui->editClassOfType->text();
//    Message::Ptr request_msg = createJsonMessage(request);
//    _socket->send(request_msg);

//    getTaskTypes();
}

void MainWindow::on_btnUpdateTypeTask_clicked(bool)
{
//    data::UpdateTypeTask request;

//    request.typeTask =_selectedTaskType;
//    request.typeTask.name = ui->editTypeTask->text();
//    request.typeTask.classTypeTask = ui->editClassOfType->text();
//    Message::Ptr request_msg = createJsonMessage(request);
//    _socket->send(request_msg);
}

void MainWindow::on_btnDeleteTypeTask_clicked(bool)
{
//    if (_selectedTaskType.id.isNull()) return;
//    data::DeleteTypeTask request;
//    request.idTypeTask = _selectedTaskType.id;
//    Message::Ptr request_msg = createJsonMessage(request);
//    _socket->send(request_msg);
}

void MainWindow::on_btnAddTask_clicked(bool)
{
//    data::AddTask request;
//    request.task.regularityDay = ui->sbRegularityDay->value();
//    request.task.idType = _selectedTaskType.id;
//    request.task.name = ui->editTaskName->text();
//    request.task.regularityHour = ui->sbRegularityHour->value();
//    request.task.runDate = ui->dateEditTaskStart->date();
//    request.task.runTime = ui->timeEditTimeStart->time();

//    Message::Ptr request_msg = createJsonMessage(request);
//    //request_msg->setTag(_selectedHashUser);
//    _socket->send(request_msg);

//    getTaskTypes();
}

void MainWindow::on_btnUpdateTask_clicked(bool)
{
//    data::UpdateTask request;

//    request.task.regularityDay = ui->sbRegularityDay->value();
//    request.task.idType = _selectedTaskType.id;
//    request.task.id = _selectedTask.id;
//    request.task.name = ui->editTaskName->text();
//    request.task.regularityHour = ui->sbRegularityHour->value();
//    request.task.runDate = ui->dateEditTaskStart->date();
//    request.task.runTime = ui->timeEditTimeStart->time();

//    Message::Ptr request_msg = createJsonMessage(request);
//    _socket->send(request_msg);
}

void MainWindow::on_btnDeleteTask_clicked(bool)
{
//    data::DeleteTask request;
//    request.id = _selectedTask.id;
//    Message::Ptr request_msg = createJsonMessage(request);
//    _socket->send(request_msg);
}

void MainWindow::on_btnGroupAdd_clicked(bool)
{
    data::GroupCreate request;
    request.name = ui->editGroupName->text();
    request.canApplyModel = static_cast<data::Group::Level>(ui->cbCanApplyModel->currentIndex()) ;
    request.canLearnModel = static_cast<data::Group::Level>(ui->cbCanLearnModel->currentIndex());
    request.canRWReport =   static_cast<data::Group::Level>(ui->cbCanReports->currentIndex());
    request.canMonitoring = static_cast<data::Group::Level>(ui->cbCanMonitoring->currentIndex());
    request.isDefault = ui->cbIsDefault->currentIndex() != 0;

    Message::Ptr request_msg = createJsonMessage(request);
    _socket->send(request_msg);

    //getTaskTypes();
}

void MainWindow::on_btnGroupEdit_clicked(bool)
{
    QListWidgetItem* item = ui->lstGroups->currentItem();
    if (!item) return;

    QVariant v = item->data(Qt::UserRole);
    data::Group group = v.value<data::Group>();

    data::GroupEdit groupEdit;
    groupEdit.id = group.id;
    groupEdit.name = ui->editGroupName->text();
    groupEdit.canApplyModel = static_cast<data::Group::Level>(ui->cbCanApplyModel->currentIndex()) ;
    groupEdit.canLearnModel = static_cast<data::Group::Level>(ui->cbCanLearnModel->currentIndex());
    groupEdit.canRWReport =   static_cast<data::Group::Level>(ui->cbCanReports->currentIndex());
    groupEdit.canMonitoring = static_cast<data::Group::Level>(ui->cbCanMonitoring->currentIndex());
    groupEdit.isDefault = ui->cbIsDefault->currentIndex() != 0;

    Message::Ptr request_msg = createJsonMessage(groupEdit);
    _socket->send(request_msg);
}

void MainWindow::on_btnGroupDelete_clicked(bool)
{
    QListWidgetItem* item = ui->lstGroups->currentItem();
    if (!item) return;

    QVariant v = item->data(Qt::UserRole);
    data::Group group = v.value<data::Group>();

    data::GroupDelete groupDelete;
    groupDelete.id = group.id;

    Message::Ptr m = createJsonMessage(groupDelete);
    _socket->send(m);
}

void MainWindow::on_btnGroupInfo_clicked(bool)
{
    QListWidgetItem* item = ui->lstGroups->currentItem();
    if (!item) return;

    QVariant v = item->data(Qt::UserRole);
    data::Group group = v.value<data::Group>();

    data::GroupInfo groupInfo;
    groupInfo.id = group.id;

    Message::Ptr m = createJsonMessage(groupInfo);
    _socket->send(m);
}

void MainWindow::on_btnAssignGroupUser_clicked(bool)
{
    QListWidgetItem* item = ui->lstGroups->currentItem();
    if (!item) return;

    QVariant v = item->data(Qt::UserRole);
    data::Group group = v.value<data::Group>();

    v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::AssignGroup assignGroup;
    assignGroup.groupId = group.id;
    assignGroup.userId = user.id;

    Message::Ptr request_msg = createJsonMessage(assignGroup);
    _socket->send(request_msg);
}

//void MainWindow::getTaskTypes()
//{
//    ui->listTypeTasks->clear();
//    Message::Ptr m = createJsonMessage(command::GetTypesTask);
//    _socket->send(m);
//}

//void MainWindow::getTasks()
//{
//    ui->listTasks->clear();
//    Message::Ptr m = createJsonMessage(command::GetTasks);
//    _socket->send(m);
//}

void MainWindow::getGroups()
{
    ui->lstGroups->clear();
    Message::Ptr m = createJsonMessage(command::GroupList);
    _socket->send(m);
}

//void MainWindow::getTasksLog()
//{
//    ui->listTasksLog->clear();
//    data::TaskLogList taskLogList;
//    taskLogList.paging.limit  = 20;
//    taskLogList.paging.offset = 20;
//    Message::Ptr m = createJsonMessage(taskLogList);
//    _socket->send(m);
//}

void MainWindow::getAisUsers()
{
    Message::Ptr m = createJsonMessage(command::UserList);
    _socket->send(m);
}

void MainWindow::getTaskModels()
{
    data::TaskModelList taskModelList;
    Message::Ptr m = createJsonMessage(taskModelList);
    _socket->send(m);
}

void MainWindow::getTaskScores(const QUuidEx& userId)
{
    data::TaskScoreList taskScoreList;
    taskScoreList.filter.userId = userId;
    Message::Ptr m = createJsonMessage(taskScoreList);
    _socket->send(m);
}

void MainWindow::getModels()
{
    ui->lstModels->clear();
    ui->cboxScoreModels->clear();

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::ModelList modelList;
    modelList.filter.userId = user.id;

    Message::Ptr m = createJsonMessage(modelList);
    _socket->send(m);
}

void MainWindow::getNotices()
{
//    ui->lstNotices->clear();

//    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
//    data::User user = v.value<data::User>();

//    data::NoticeList noticeList;
//    noticeList.userId = user.id;

//    Message::Ptr m = createJsonMessage(noticeList);
//    _socket->send(m);
}

//void MainWindow::command_GetTypesTask(const Message::Ptr& message)
//{
//    if (message->type() != Message::Type::Answer)
//        return;

//    if (message->execStatus() == Message::ExecStatus::Success)
//    {
//        data::GetTypesTask typesTask;
//        readFromMessage(message, typesTask);

//        _taskTypes = typesTask.tasksTypes;
//        ui->listTypeTasks->blockSignals(true);
//        for (auto t : typesTask.tasksTypes)
//        {
//            auto *item = new QListWidgetItem(t.name);
//            QVariant v;
//            v.setValue(t.id);
//            item->setData(Qt::UserRole, v);
//            ui->listTypeTasks->addItem(item);
//        }
//        ui->listTypeTasks->blockSignals(false);
//    }
//    else
//    {
//        QString msg = errorDescription(message);
//        QMessageBox::critical(this, qApp->applicationName(), msg);
//    }
//}

//void MainWindow::command_GetTasks(const Message::Ptr& message)
//{
//    if (message->type() != Message::Type::Answer)
//        return;

//    if (message->execStatus() == Message::ExecStatus::Success)
//    {
//        data::GetTasks tasks;
//        readFromMessage(message, tasks);
//        _tasks = tasks.tasks;
//        ui->listTasks->blockSignals(true);
//        for (auto t : _tasks)
//        {
//            auto *item = new QListWidgetItem(t.name);
//            QVariant v;
//            v.setValue(t.id);
//            item->setData(Qt::UserRole, v);
//            ui->listTasks->addItem(item);
//        }
//        ui->listTasks->blockSignals(false);
//    }
//    else
//    {
//        QString msg = errorDescription(message);
//        QMessageBox::critical(this, qApp->applicationName(), msg);
//    }
//}

void MainWindow::command_GroupList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::GroupList groupList;
        readFromMessage(message, groupList);

        ui->lstGroups->blockSignals(true);
        for (const data::Group& group : groupList.items)
        {
            QVariant v = QVariant::fromValue(group);
            QString str = /*group.id.toString() + " " + */ group.name;

            QListWidgetItem* item = new QListWidgetItem;
            item->setText(str);
            item->setData(Qt::UserRole, v);
            ui->lstGroups->addItem(item);
        }
        ui->lstGroups->blockSignals(false);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_GroupDelete(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        //data::MessageError data;
        data::GroupDeleteFail err;
        readFromMessage(message, err);
        QString msg = err.description;
        msg += "\nId: " + err.id.toString();
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }

}

void MainWindow::command_GroupInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::GroupInfoA groupInfoA;
        readFromMessage(message, groupInfoA);

        QString msg = "UserInfo\n"
                      "id             %1\n"
                      "name           %2\n"
                      "canLearnModel  %3\n"
                      "canApplyModel  %4\n"
                      "canRWReport    %5\n"
                      "canMonitoring  %6\n"
                      "isDefault      %7\n"
                      "userCount      %8\n";

        msg = msg.arg(groupInfoA.id.toString()            );
        msg = msg.arg(groupInfoA.name                     );
        msg = msg.arg(static_cast<qint32>(groupInfoA.canLearnModel));
        msg = msg.arg(static_cast<qint32>(groupInfoA.canApplyModel));
        msg = msg.arg(static_cast<qint32>(groupInfoA.canRWReport  ));
        msg = msg.arg(static_cast<qint32>(groupInfoA.canMonitoring));
        msg = msg.arg(groupInfoA.isDefault                );
        msg = msg.arg(groupInfoA.userCount                );

        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_UserInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::UserInfoA userInfoA;
        readFromMessage(message, userInfoA);

        QString msg = "UserInfo\n"
                      "id         %1\n"
                      "hashId     %2\n"
                      "login      %3\n"
                      "name       %4\n"
                      "isActive   %5\n"
                      "isValid    %6\n"
                      "isAdmin    %7\n"
                      "groupId    %8\n";

        msg = msg.arg(userInfoA.id.toString()      );
        msg = msg.arg(userInfoA.hashId             );
        msg = msg.arg(userInfoA.login              );
        msg = msg.arg(userInfoA.name               );
        msg = msg.arg(userInfoA.isActive           );
        msg = msg.arg(userInfoA.isValid            );
        msg = msg.arg(userInfoA.isAdmin            );
        msg = msg.arg(userInfoA.groupId.toString() );

        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_UserList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::UserList users;
        readFromMessage(message, users);

        ui->cboxUsers->blockSignals(true);
        ui->cboxUsers->clear();
        //_users = users.items;

        ui->listUsers->blockSignals(true);
        ui->listUsers->clear();

        for (const data::User& u : users.items)
        {
            QVariant v;
            v.setValue(u.id);

            ui->cboxUsers->addItem(u.login, QVariant::fromValue(u));

            auto *item = new QListWidgetItem(u.login);
            item->setData(Qt::UserRole, v);
            ui->listUsers->addItem(item);
        }
        ui->listUsers->blockSignals(false);
        ui->cboxUsers->blockSignals(false);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_TaskInterrupt(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::information(this, qApp->applicationName(), msg);
    }

    data::TaskInterrupt taskInterrupt;
    readFromMessage(message, taskInterrupt);

    QString msg = "Task %1 interrupted\n";
    msg = msg.arg(taskInterrupt.taskId.toString());
    QMessageBox::information(this, qApp->applicationName(), msg);

}

void MainWindow::command_TaskSyncDataInfo(const Message::Ptr& message)
{
//    data::SyncDataInfo syncDataInfo;
//    readFromMessage(message, syncDataInfo);

//    ui->chkSyncDataMute->setChecked(syncDataInfo.isMute);
//    ui->sboxSyncDataRDay->setValue(syncDataInfo.regularityDay);
//    ui->sboxSyncDataRHour->setValue(syncDataInfo.regularityHour);
//    ui->labelSyncDataInProgr->setText(syncDataInfo.inProgress ? "true" : "false");
}

void MainWindow::command_TaskModelCreate(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskModelCreate taskModelCreate;
        readFromMessage(message, taskModelCreate);

        QString msg = "Success TaskModel, ID: %1\n";
        msg = msg.arg(taskModelCreate.id.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }

    getTaskModels();
}

void MainWindow::command_TaskModelEdit(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskModelEdit taskModelEdit;
        readFromMessage(message, taskModelEdit);

        QString msg = "Update TaskModel success , ID: %1\n";
        msg = msg.arg(taskModelEdit.id.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
    getTaskModels();
}

void MainWindow::command_TaskModelDelete(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskModelDelete taskModelDelete;
        readFromMessage(message, taskModelDelete);

        QString msg = "Delete TaskModel success , ID: %1\n";
        msg = msg.arg(taskModelDelete.taskId.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
    getTaskModels();
}

void MainWindow::command_TaskModelInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskModelInfoA taskModelInfo;
        readFromMessage(message, taskModelInfo);

        QString msg = "Success TaskModel\n "
                      "taskId            %1\n"
                      "userId            %2\n"
                      "parentId          %3\n"
                      "isEnabled         %4\n"
                      "name              %5\n"
                      "description       %6\n"
                      "regularityMonth   %7\n"
                      "regularityDay     %8\n"
                      "regularityHour    %9\n"
                      "isPublic          %10\n";

        msg = msg.arg(taskModelInfo.id.toString()       );
        msg = msg.arg(taskModelInfo.userId.toString()   );
        msg = msg.arg(taskModelInfo.parentId.toString() );
        msg = msg.arg(taskModelInfo.isEnabled           );
        msg = msg.arg(taskModelInfo.name                );
        msg = msg.arg(taskModelInfo.description         );
        msg = msg.arg(taskModelInfo.regularityMonth     );
        msg = msg.arg(taskModelInfo.regularityDay       );
        msg = msg.arg(taskModelInfo.regularityHour      );
        msg = msg.arg(taskModelInfo.isPublic            );

        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_TaskModelList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::TaskModelList taskModelList;
    readFromMessage(message, taskModelList);

    ui->lstTaskModels->blockSignals(true);
    ui->lstTaskModels->clear();

    for (const data::Task& model : taskModelList.items)
    {
        QVariant v = QVariant::fromValue(model);
        QString str = model.id.toString() + " " + model.name;

        QListWidgetItem* item = new QListWidgetItem;
        item->setText(str);
        item->setData(Qt::UserRole, v);
        ui->lstTaskModels->addItem(item);
        ui->cboxScoreTaskModels->addItem(str, v);
    }

    ui->lstTaskModels->blockSignals(false);
}

void MainWindow::command_TaskScoreCreate(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskScoreCreate taskScoreCreate;
        readFromMessage(message, taskScoreCreate);

        QString msg = "Success TaskScore, ID: %1\n";
        msg = msg.arg(taskScoreCreate.id.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);

        getTaskScores(taskScoreCreate.userId);
        return;
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }

    getTaskScores(QUuidEx());
}

void MainWindow::command_TaskScoreEdit(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskScoreEdit taskScoreEdit;
        readFromMessage(message, taskScoreEdit);

        QString msg = "Update TaskScore success , ID: %1\n";
        msg = msg.arg(taskScoreEdit.id.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);

        getTaskScores(taskScoreEdit.userId);
        return;
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
    getTaskScores(QUuidEx());
}

void MainWindow::command_TaskScoreDelete(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskScoreDelete taskScoreDelete;
        readFromMessage(message, taskScoreDelete);

        QString msg = "Delete TaskScore success , ID: %1\n";
        msg = msg.arg(taskScoreDelete.taskId.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);

        QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
        data::User user = v.value<data::User>();

        getTaskScores(user.id);
        return;
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
    getTaskScores(QUuidEx());
}

void MainWindow::command_TaskScoreInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskScoreInfoA taskScoreInfo;
        readFromMessage(message, taskScoreInfo);

        QString msg = "Success TaskScore\n"
                      "taskId            %1\n"
                      "userId            %2\n"
                      "parentId          %3\n"
                      "isEnabled         %4\n"
                      "name              %5\n"
                      "description       %6\n"
                      "regularityMonth   %7\n"
                      "regularityDay     %8\n"
                      "regularityHour    %9\n"
                      "isPublic          %10\n"
                      "modelId           %11\n";

        msg = msg.arg(taskScoreInfo.id.toString()       );
        msg = msg.arg(taskScoreInfo.userId.toString()   );
        msg = msg.arg(taskScoreInfo.parentId.toString() );
        msg = msg.arg(taskScoreInfo.isEnabled           );
        msg = msg.arg(taskScoreInfo.name                );
        msg = msg.arg(taskScoreInfo.description         );
        msg = msg.arg(taskScoreInfo.regularityMonth     );
        msg = msg.arg(taskScoreInfo.regularityDay       );
        msg = msg.arg(taskScoreInfo.regularityHour      );
        msg = msg.arg(taskScoreInfo.isPublic            );
        msg = msg.arg(taskScoreInfo.modelId.toString()  );

        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_TaskScoreList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskScoreList taskScoreList;
        readFromMessage(message, taskScoreList);

        ui->lstTaskScores->blockSignals(true);
        ui->lstTaskScores->clear();

        for (const data::Task& score : taskScoreList.items)
        {
            QVariant v = QVariant::fromValue(score);
            QString str = score.id.toString() + " " + score.name;

            QListWidgetItem* item = new QListWidgetItem;
            item->setText(str);
            item->setData(Qt::UserRole, v);
            ui->lstTaskScores->addItem(item);
        }
        ui->lstTaskScores->blockSignals(false);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_TaskAlert(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Event)
        return;

    data::TaskAlert taskAlert;
    readFromMessage(message, taskAlert);

    QMessageBox::critical(this, qApp->applicationName(), taskAlert.message);
}

void MainWindow::command_ModelEdit(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    QMessageBox::information(this, qApp->applicationName(), QString("ModelEdit OK"));
}

void MainWindow::command_ModelDelete(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }
    
    QMessageBox::information(this, qApp->applicationName(), QString("ModelDelete OK"));
}

void MainWindow::command_ModelInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::ModelInfoA modelInfo;
        readFromMessage(message, modelInfo);

        QString msg = "Success Model\n "
                      "id                %1\n"
                      "taskId            %2\n"
                      "userId            %3\n"
                      "name              %4\n"
                      "description       %5\n"
                      "createDate        %6\n"
                      "periodBegin       %7\n"
                      "periodEnd         %8\n"
                      "periodBeginReal   %9\n"
                      "periodEndReal     %10\n"
                      "isPublic          %11\n"
                      "execStatus        %12\n";


        msg = msg.arg(modelInfo.id.toString()               );
        msg = msg.arg(modelInfo.taskId.toString()           );
        msg = msg.arg(modelInfo.userId.toString()           );
        msg = msg.arg(modelInfo.name                        );
        msg = msg.arg(modelInfo.description                 );
        msg = msg.arg(modelInfo.createDate.toString()       );
        msg = msg.arg(modelInfo.period.begin.toString()     );
        msg = msg.arg(modelInfo.period.end.toString()       );
//        msg = msg.arg(modelInfo.periodReal.begin.toString() );
//        msg = msg.arg(modelInfo.periodReal.end.toString()   );
        msg = msg.arg(modelInfo.isPublic                    );

        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_ModelList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::ModelList modelList;
    readFromMessage(message, modelList);

    ui->lstModels->blockSignals(true);
    ui->lstModels->clear();

    ui->cboxScoreModels->blockSignals(true);
    ui->cboxScoreModels->clear();

    for (data::Model model : modelList.items)
    {
        QVariant v = QVariant::fromValue(model);
        QString str = model.id.toString() + " " + model.name;

        QListWidgetItem* item = new QListWidgetItem;
        item->setText(str);
        item->setData(Qt::UserRole, v);
        ui->lstModels->addItem(item);
        ui->cboxScoreModels->addItem(str, v);
    }
    ui->lstModels->blockSignals(false);
    ui->cboxScoreModels->blockSignals(false);
}

//    for (QUuidEx scoreId : childScoreList.items)
//    {
//    }
void MainWindow::command_ScoreEdit(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    QMessageBox::information(this, qApp->applicationName(), QString("ScoreEdit OK"));
}

void MainWindow::command_ScoreDelete(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    QMessageBox::information(this, qApp->applicationName(), QString("ScoreDelete OK"));
}

void MainWindow::command_ScoreInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::ScoreInfoA scoreInfo;
        readFromMessage(message, scoreInfo);

        QString msg = "Success Scores\n"
                      "id                %1\n"
                      "taskId            %2\n"
                      "userId            %3\n"
                      "name              %4\n"
                      "description       %5\n"
                      "createDate        %6\n"
                      "periodBegin       %7\n"
                      "periodEnd         %8\n"
                      "periodBeginReal   %9\n"
                      "periodEndReal     %10\n"
                      "isPublic          %11\n"
                      "execStatus        %12\n";

        msg = msg.arg(scoreInfo.id.toString());
        msg = msg.arg(scoreInfo.taskId.toString());
        msg = msg.arg(scoreInfo.userId.toString());
        msg = msg.arg(scoreInfo.name);
        msg = msg.arg(scoreInfo.description);
        msg = msg.arg(scoreInfo.createDate.toString());
        msg = msg.arg(scoreInfo.period.begin.toString());
        msg = msg.arg(scoreInfo.period.end.toString());
//        msg = msg.arg(scoreInfo.periodReal.begin.toString());
//        msg = msg.arg(scoreInfo.periodReal.end.toString());
        msg = msg.arg(scoreInfo.isPublic);

        QMessageBox::information(this, qApp->applicationName(), msg);
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_ScoreList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::ScoreList scoreList;
    readFromMessage(message, scoreList);

    ui->lstScores->blockSignals(true);
    ui->lstScores->clear();

    for (data::Score score : scoreList.items)
    {
        QVariant v = QVariant::fromValue(score);
        QString str = score.id.toString() + " " + score.name;

        QListWidgetItem* item = new QListWidgetItem;
        item->setText(str);
        item->setData(Qt::UserRole, v);
        ui->lstScores->addItem(item);        
        ui->score_list->addItem(str, v);

    }
    ui->lstScores->blockSignals(false);
}

//    for (QUuidEx scoreId : childReportList.items)
//    {
//    }
// Для работы прогресс бара должна быть выделена модель, которая сейчас
// считается.
void MainWindow::command_TaskProgress(const Message::Ptr& message)
{
    data::TaskProgress taskProgress;
    readFromMessage(message, taskProgress);

    if (taskProgress.taskType == TaskType::LearnModel)
    {
        // Если в UI выбрана какая-либо задача
        if (!ui->lstTaskModels->currentIndex().isValid())
            return;

        QUuidEx taskId = taskProgress.taskId; // ID задачи из события
        //QUuidEx dispId = QUuidEx(ui->lstTaskModels->currentItem()->text());

        QListWidgetItem* item = ui->lstTaskModels->currentItem();
        QVariant v = item->data(Qt::UserRole);
        data::Task task = v.value<data::Task>();

        if (taskId == task.id) // Если ID задачи совпадает с ID выбранной в UI задачи
        {
            ui->web_progressBar->setMinimum(0);
            ui->web_progressBar->setMaximum(taskProgress.total);
            ui->web_progressBar->setValue(taskProgress.current);
        }
    }
    else if (taskProgress.taskType == TaskType::ScoreCalc)
    {
        // Если в UI выбрана какая-либо задача
        if (!ui->lstTaskScores->currentIndex().isValid())
            return;

        QListWidgetItem* item = ui->lstTaskScores->currentItem();
        QVariant v = item->data(Qt::UserRole);
        data::Task task = v.value<data::Task>();

        QUuidEx taskId = taskProgress.taskId; // ID задачи из события

        if (taskId == task.id)
        {
            ui->pbarScoreCalc->setMinimum(0);
            ui->pbarScoreCalc->setMaximum(taskProgress.total);
            ui->pbarScoreCalc->setValue(taskProgress.current);
        }
    }

}

void MainWindow::on_lstTaskModels_itemClicked(QListWidgetItem *item)
{
    QVariant v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    ui->web_model_isEnabled->setChecked(task.isEnabled);
    ui->web_model_name->setText(task.name);
    ui->web_model_description->setText(task.description);
    ui->web_model_runDateTime->setDateTime(task.runDateTime);
    ui->web_model_regularityMonth->setValue(task.regularityMonth);
    ui->web_model_regularityDay->setValue(task.regularityDay);
    ui->web_model_regularityHour->setValue(task.regularityHour);
    ui->web_model_offsetMonthBegin->setValue(task.relPeriodBegin);
    ui->web_model_offsetMonthEnd->setValue(task.relPeriodDuration);
    ui->web_model_periodBegin->setDateTime(task.period.begin);
    ui->web_model_periodEnd->setDateTime(task.period.end);
    ui->web_model_isPublic->setChecked(task.isPublic);


    ui->web_modelsListInfo->clear();
    ui->web_modelsListInfo->append("taskId: "
                                   + task.id.toString() );
    ui->web_modelsListInfo->append("userId: "
                                   + task.userId.toString() );
    ui->web_modelsListInfo->append("parentId: "
                                   + task.parentId.toString() );
    ui->web_modelsListInfo->append("isEnabled: "
                                   + QString::number((qint32)task.isEnabled) );
    ui->web_modelsListInfo->append("name: "
                                   + task.name );
    ui->web_modelsListInfo->append("description: "
                                   + task.description );
    ui->web_modelsListInfo->append("createDate: "
                                   + task.createDate.toString() );
    ui->web_modelsListInfo->append("runDateTime: "
                                   + task.runDateTime.toString() );
    ui->web_modelsListInfo->append("regularityMonth: "
                                   + QString::number(task.regularityMonth) );
    ui->web_modelsListInfo->append("regularityDay: "
                                   + QString::number(task.regularityDay ) );
    ui->web_modelsListInfo->append("regularityHour: "
                                   + QString::number(task.regularityHour) );
    ui->web_modelsListInfo->append("nextRunTime: "
                                   + task.nextDateTime.toString() );
    ui->web_modelsListInfo->append("attemptLimit: "
                                   + QString::number(task.attemptLimit) );
    ui->web_modelsListInfo->append("attemptCounter: "
                                   + QString::number(task.attemptCounter) );
    ui->web_modelsListInfo->append("attemptInterval: "
                                   + QString::number(task.attemptInterval) );
    ui->web_modelsListInfo->append("execStatus: "
                                   + QString::number((qint32)task.execStatus) );
    ui->web_modelsListInfo->append("relPeriodBegin: "
                                   + QString::number(task.relPeriodBegin ) );
    ui->web_modelsListInfo->append("relPeriodDuration: "
                                   + QString::number(task.relPeriodDuration) );
    ui->web_modelsListInfo->append("periodBegin: "
                                   + QString::number((quint64)task.period.begin.toMSecsSinceEpoch()) );
    ui->web_modelsListInfo->append("periodEnd: "
                                   + QString::number((quint64)task.period.end.toMSecsSinceEpoch()) );
    ui->web_modelsListInfo->append("isPublic: "
                                   + QString::number((qint32)task.isPublic) );
}

void MainWindow::on_lstTaskScores_itemClicked(QListWidgetItem *item)
{
    QVariant v = item->data(Qt::UserRole);
    data::Task task = v.value<data::Task>();

    ui->cboxTaskScoreEnabled->setChecked(         task.isEnabled       );
    ui->lineTaskScoreName->setText(               task.name            );
    ui->lineTaskScoreDescr->setText(              task.description     );
    ui->lineTaskScoreRunDT->setDateTime(          task.runDateTime     );
    ui->sboxTaskScoreRegularityMonth->setValue(   task.regularityMonth );
    ui->sboxTaskScoreRegularityDay->setValue(     task.regularityDay   );
    ui->sboxTaskScoreRegularityHour->setValue(    task.regularityHour  );
    ui->sboxTaskScorePeriodBegin->setDateTime(    task.period.begin    );
    ui->sboxTaskScorePeriodEnd->setDateTime(      task.period.end      );
    ui->sboxTaskScoreOffsetMonthBegin->setValue(  task.relPeriodBegin);
    ui->sboxTaskScoreOffsetMonthEnd->setValue(    task.relPeriodDuration  );
    ui->chkTaskScorePublic->setChecked(           task.isPublic        );


    ui->textScoreInfo->clear();
    ui->textScoreInfo->append("taskId: "
                                   + task.id.toString() );
    ui->textScoreInfo->append("userId: "
                                   + task.userId.toString() );
    ui->textScoreInfo->append("parentId: "
                                   + task.parentId.toString() );
    ui->textScoreInfo->append("isEnabled: "
                                   + QString::number((qint32)task.isEnabled) );
    ui->textScoreInfo->append("name: "
                                   + task.name );
    ui->textScoreInfo->append("description: "
                                   + task.description );
    ui->textScoreInfo->append("createDate: "
                                   + task.createDate.toString() );
    ui->textScoreInfo->append("runDateTime: "
                                   + task.runDateTime.toString() );
    ui->textScoreInfo->append("regularityMonth: "
                                   + QString::number(task.regularityMonth) );
    ui->textScoreInfo->append("regularityDay: "
                                   + QString::number(task.regularityDay ) );
    ui->textScoreInfo->append("regularityHour: "
                                   + QString::number(task.regularityHour) );
    ui->textScoreInfo->append("nextRunTime: "
                                   + task.nextDateTime.toString() );
    ui->textScoreInfo->append("attemptLimit: "
                                   + QString::number(task.attemptLimit) );
    ui->textScoreInfo->append("attemptCounter: "
                                   + QString::number(task.attemptCounter) );
    ui->textScoreInfo->append("attemptInterval: "
                                   + QString::number(task.attemptInterval) );
    ui->textScoreInfo->append("execStatus: "
                                   + QString::number((qint32)task.execStatus) );
    ui->textScoreInfo->append("periodBegin: "
                                   + QString::number((quint64)task.period.begin.toMSecsSinceEpoch()) );
    ui->textScoreInfo->append("periodEnd: "
                                   + QString::number((quint64)task.period.end.toMSecsSinceEpoch()) );
    ui->textScoreInfo->append("relPeriodBegin: "
                                   + QString::number(task.relPeriodBegin) );
    ui->textScoreInfo->append("relPeriodDuration: "
                                   + QString::number(task.relPeriodDuration) );
    ui->textScoreInfo->append("isPublic: "
                                   + QString::number((qint32)task.isPublic) );
    ui->textScoreInfo->append("modelId: "
                                   + task.modelId.toString() );

    ui->cboxScoreModels->blockSignals(true);
    ui->cboxScoreModels->clear();
    ui->cboxScoreModels->blockSignals(false);

    ui->cboxScoreTaskModels->blockSignals(true);
    ui->cboxScoreTaskModels->clear();
    ui->cboxScoreTaskModels->blockSignals(false);

    QVariant uv = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = uv.value<data::User>();

    Message::Ptr m{nullptr};
    // Если применение периодическое то необходимо получить список
    // список периодических задач.
    if (task.isPeriodic())
    {
        data::TaskModelList taskModelList;
        taskModelList.filter.userId = user.id;
        taskModelList.filter.mode = TaskMode::Periodic;
        m = createJsonMessage(taskModelList);
    }
    // Если применение разовое, то необходимо получить список моделей.
    else
    {
        data::ModelList modelList;
        modelList.filter.userId = user.id;
        m = createJsonMessage(modelList);
    }

    m->setTag(user.hashId);
    _socket->send(m);

}

void MainWindow::on_lstModels_itemClicked(QListWidgetItem *item)
{
    QVariant v = item->data(Qt::UserRole);
    data::Model model = v.value<data::Model>();

    ui->web_modelsDetails->clear();

    // Подробная информация о модели
    ui->web_modelsDetails->append( model.id.toString()          );
    ui->web_modelsDetails->append( model.taskId.toString()      );
    ui->web_modelsDetails->append( model.userId.toString()      );
    ui->web_modelsDetails->append( model.name                   );
    ui->web_modelsDetails->append( model.description            );
    ui->web_modelsDetails->append( model.createDate.toString()  );
    ui->web_modelsDetails->append( model.period.begin.toString() );
    ui->web_modelsDetails->append( model.period.end.toString()   );
    ui->web_modelsDetails->append( QString::number((qint32)model.isPublic)    );
}

void MainWindow::on_lstScores_itemClicked(QListWidgetItem *item)
{
    QVariant v = item->data(Qt::UserRole);
    data::Score score = v.value<data::Score>();

    ui->txtScoreDetails->clear();

    // Подробная информация о модели
    ui->txtScoreDetails->append( score.id.toString()          );
    ui->txtScoreDetails->append( score.taskId.toString()      );
    ui->txtScoreDetails->append( score.userId.toString()      );
    ui->txtScoreDetails->append( score.modelId.toString()     );
    ui->txtScoreDetails->append( score.name                   );
    ui->txtScoreDetails->append( score.description            );
    ui->txtScoreDetails->append( score.createDate.toString()  );
    ui->txtScoreDetails->append( score.period.begin.toString());
    ui->txtScoreDetails->append( score.period.end.toString()  );
    ui->txtScoreDetails->append( QString::number((qint32)score.isPublic)    );
}

//QDate MainWindow::GetReportEndDate()
//{
//    QDate date = ui->endDate->date();
//    return date.day() == 1 && date.month() == 1 && date.year() == 2000 ? QDate() : date;
//}

//int MainWindow::GetId()
//{
//    return (ui->idCheck->value());
//}

data::TaskReportCreate MainWindow::GetReportFromUI()
{
    using namespace  communication::data;

    data::TaskReportCreate taskReport;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    taskReport.userId = user.id;
    taskReport.type = static_cast<ReportType>(ui->report_type->currentIndex() + 1);

    taskReport.name = ui->report_name_line->text();

    taskReport.lpu= ui->clinic_code_line->text();
    taskReport.mkb = ui->diagnosis_code_line->text();
    taskReport.vidmp = ui->kind_of_help_code_line->text();
    taskReport.medProfile = ui->profile_code_line->text();
    taskReport.threshold = ui->min_score_range->value();
    taskReport.recordCount = ui->record_count->value();

    int index = ui->score_list->currentIndex();
    v = ui->score_list->itemData(index);
    data::Score score = v.value<data::Score>();

    taskReport.scoreId = score.id;
    taskReport.period = score.period;
    taskReport.createDate = QDateTime::currentDateTime();

    taskReport.ranging = static_cast<ReportRanging>(ui->range_type->currentIndex() + 1);
    taskReport.present = static_cast<ReportPresent>(ui->diagram_type->currentIndex() + 1);
    taskReport.sort    = static_cast<ReportSort>(ui->sort_type->currentIndex());

    // TODO  Маша, уверена, что здесь не будет больших накладных расходов?!
    return taskReport;
}

bool MainWindow::CheckReportData(const communication::data::TaskReportCreate& reportData)
{
    //TODO написать проверку
    return true;
}

void MainWindow::on_btnCreateReport_clicked(bool)
{
    using namespace communication;
    data::TaskReportCreate reportModel = GetReportFromUI();
    if (CheckReportData(reportModel))
    {
        Message::Ptr request_msg = createJsonMessage(reportModel);
        _socket->send(request_msg);
    }
}

void MainWindow::on_btnReportList_clicked(bool)
{
    ui->lstReports->clear();

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::ReportList reportList;
    reportList.filter.userId = user.id;

    Message::Ptr m = createJsonMessage(reportList);
    _socket->send(m);
}

void MainWindow::command_ReportList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::ReportList reportList;
    readFromMessage(message, reportList);

    ui->lstReports->blockSignals(true);
    ui->lstReports->clear();

    for (const data::Report& report : reportList.items)
    {
        QVariant v = QVariant::fromValue(report);
        QString str = report.id.toString() + " " + report.name;

        QListWidgetItem* item = new QListWidgetItem;
        item->setText(str);
        item->setData(Qt::UserRole, v);
        ui->lstReports->addItem(item);

    }
    ui->lstReports->blockSignals(false);
}

void MainWindow::command_ReportInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::ReportInfoA ieportInfoA;
    readFromMessage(message, ieportInfoA);


    ui->txtReportInfo->clear();

    //ui->txtReportInfo


}

void MainWindow::on_btnReportInfo_clicked(bool)
{
    if (!ui->lstReports->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstReports->currentItem();
    v = item->data(Qt::UserRole);
    data::Report report = v.value<data::Report>();

    data::ReportInfo reportInfo;
    reportInfo.id = report.id;
    reportInfo.userId = user.id;

    Message::Ptr m = createJsonMessage(reportInfo);
    _socket->send(m);

    log_info << "on_btnReportInfo_clicked";
}


void MainWindow::command_ReportData(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::ReportData reportData;
        readFromMessage(message, reportData);

        ui->textReportData->clear();

        // Здесь должна быть подробная информация об отчете
        //ui->textReportInfo->append(reportData.reportParam.id.toString());
        //ui->textReportInfo->append(reportData.reportParam.name);
        //ui->newReportName->setText(reportData.reportParam.name);

        for(data::ReportDataItem item : reportData.items)
        {
            QString result = item.name + " " + item.code + " " + QString::number(item.value);
            ui->textReportData->append( result );
        }

    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_TaskReportFedCreate(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskReportFedCreate taskReportFedCreate;
        readFromMessage(message, taskReportFedCreate);

        QString msg = "Success TaskReportFedCreate, ID: %1\n";
        msg = msg.arg(taskReportFedCreate.id.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);

        return;
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_ReportFedList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::ReportFedList reportFedList;
    readFromMessage(message, reportFedList);

    ui->lstReportsFed->blockSignals(true);
    ui->lstReportsFed->clear();

    for (const data::ReportFed& report : reportFedList.items)
    {
        QVariant v = QVariant::fromValue(report);
        QString str = report.id.toString() + " " + report.name;

        QListWidgetItem* item = new QListWidgetItem;
        item->setText(str);
        item->setData(Qt::UserRole, v);
        ui->lstReportsFed->addItem(item);

    }
    ui->lstReportsFed->blockSignals(false);
}

void MainWindow::command_ModelXgbDelete(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    QMessageBox::information(this, qApp->applicationName(), QString("ModelXgbDelete OK"));
}

void MainWindow::command_ModelXgbInfo(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::ModelXgbInfoA optionsList;
    readFromMessage(message, optionsList);

    ui->xgbOptionsList->blockSignals(true);
    ui->xgbOptionsList->clear();

    for (const data::XgbParam& param : optionsList.items)
    {
        ui->xgbOptionsList->append(param.name + ": " + param.value);
    }
    ui->lstModels->blockSignals(false);
    ui->cboxScoreModels->blockSignals(false);
}

void MainWindow::command_ModelXgbEdit(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    QMessageBox::information(this, qApp->applicationName(), QString("ModelXgbEdit OK"));
}

void MainWindow::command_ModelXgbOption(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    QMessageBox::information(this, qApp->applicationName(), QString("ModelXgbOption OK"));
}

void MainWindow::command_EventLogList(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() != Message::ExecStatus::Success)
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
        return;
    }

    data::EventLogList eventLogList;
    readFromMessage(message, eventLogList);

    ui->txtEventLogs->clear();
    for (const data::EventLog& itemLog : eventLogList.items)
    {
        QString str = itemLog.userId.toString()
                    + " | " + itemLog.taskId.toString()
                    + " | " + itemLog.createDate.toString()
                    + " | " + itemLog.description;

        ui->txtEventLogs->append(str);
    }
}

void MainWindow::command_SyncProgress(const Message::Ptr& message)
{
    data::SyncProgress taskSyncProgress;
    readFromMessage(message, taskSyncProgress);

    return;
}

void MainWindow::on_btnEditReport_clicked(bool)
{
    if (!ui->lstReports->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstReports->currentItem();
    v = item->data(Qt::UserRole);
    data::Report report = v.value<data::Report>();

    data::ReportEdit reportEdit;
    reportEdit.id = report.id;
    reportEdit.name = report.name; // ui->newReportName->text();

    Message::Ptr m = createJsonMessage(reportEdit);
    _socket->send(m);

    log_info << "on_btnReportInfo_clicked";
}

void MainWindow::on_btnReportDelete_clicked(bool)
{
    if (!ui->lstReports->currentIndex().isValid())
        return;

    QVariant v;
    //v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    //data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstReports->currentItem();
    v = item->data(Qt::UserRole);
    data::Report report = v.value<data::Report>();

    data::ReportDelete reportDelete;
    reportDelete.id = report.id;
    reportDelete.userId = report.userId;

    Message::Ptr m = createJsonMessage(reportDelete);
    _socket->send(m);
}

void MainWindow::on_btnEventLogsClear_clicked(bool)
{
    ui->txtEventLogs->clear();
}

void MainWindow::on_btnEventLogsUpdate_clicked(bool)
{
    data::EventLogList eventLogList;
    eventLogList.filter.createDate.begin = ui->dtEventLogBegin->dateTime();
    eventLogList.filter.createDate.end = ui->dtEventLogEnd->dateTime();
    //eventLogList.filter.descendSort = true;
    eventLogList.paging.offset = 1000;
    eventLogList.paging.limit = 1000;

    if (!ui->lineEventUserId->text().trimmed().isEmpty())
        eventLogList.filter.userId = QUuidEx(ui->lineEventUserId->text().trimmed());

    if (!ui->lineEventTaskId->text().trimmed().isEmpty())
        eventLogList.filter.taskId = QUuidEx(ui->lineEventTaskId->text().trimmed());

    eventLogList.filter.selectAll = ui->chkEventAll->isChecked();

    Message::Ptr m = createJsonMessage(eventLogList);
    _socket->send(m);

    log_info << "on_btnEventLogsUpdate_clicked";
}

void MainWindow::on_btnSendScore_clicked(bool)
{
    if (!ui->lstScores->currentIndex().isValid())
        return;

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstScores->currentItem();
    v = item->data(Qt::UserRole);
    data::Score score = v.value<data::Score>();

    data::NeedSendScore needSendScore;
    needSendScore.id = score.id;

    Message::Ptr m = createJsonMessage(needSendScore);
    _socket->send(m);

    log_info << "on_btnSendScore_clicked";
}

void MainWindow::on_btnXmlCreate_clicked(bool)
{
    QDomDocument doc;
    QDomElement root = doc.createElement("RP_EXP");
    doc.appendChild(root);

    QDomElement zglv = doc.createElement("ZGLV");
    root.appendChild(zglv);

    QDomElement data = doc.createElement("DATA");
    zglv.appendChild(data);

    QDomText text = doc.createTextNode("2018-06-22");
    data.appendChild(text);

    QDomElement fileName = doc.createElement("FILENAME");
    zglv.appendChild(fileName);

    text = doc.createTextNode("RPEXXXXX");
    fileName.appendChild(text);

    QString xml = doc.toString(2);
    xml = xml.prepend("\n");
    xml = xml.prepend(R"(<?xml version="1.0" encoding="UTF-8"?>)");

    QFile file {"/tmp/test.xml"};
    file.open(QIODevice::WriteOnly);
    file.write(xml.toUtf8());
    file.close();
}

void MainWindow::on_btnCreateReportFed_clicked(bool)
{
    data::TaskReportFedCreate taskReportFedCreate;

    taskReportFedCreate.userId  = QUuidEx("{45540F38-D1FA-4CF2-914F-78F9BC11E8D1}");
    taskReportFedCreate.scoreId = QUuidEx("{E016629B-BC40-49D2-BC2D-3430F24BC780}");
    taskReportFedCreate.name = "Report Fed";

    Message::Ptr m = createJsonMessage(taskReportFedCreate);
    _socket->send(m);
}

void MainWindow::on_btnTaskReportFedNow_clicked(bool)
{
    Message::Ptr m = createJsonMessage(command::TaskReportFedNow);
    _socket->send(m);
}

void MainWindow::on_btnTaskReportFedEdit_clicked(bool)
{
    data::TaskReportFedEdit taskReportFedEdit;

    taskReportFedEdit.fomsCode = "01";
    taskReportFedEdit.storeDir = "/123/123";

    Message::Ptr m = createJsonMessage(taskReportFedEdit);
    _socket->send(m);
}

void MainWindow::on_btnReportFedList_clicked(bool)
{
    ui->lstReportsFed->clear();

    QVariant v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    data::User user = v.value<data::User>();

    data::ReportFedList reportFedList;
    reportFedList.filter.userId = user.id;

    Message::Ptr m = createJsonMessage(reportFedList);
    _socket->send(m);
}

void MainWindow::on_btnReportFedDelete_clicked(bool)
{
    if (!ui->lstReportsFed->currentIndex().isValid())
        return;

    QVariant v;
    //v = ui->cboxUsers->itemData(ui->cboxUsers->currentIndex());
    //data::User user = v.value<data::User>();

    QListWidgetItem* item = ui->lstReportsFed->currentItem();
    v = item->data(Qt::UserRole);
    data::ReportFed report = v.value<data::ReportFed>();

    data::ReportFedDelete reportFedDelete;
    reportFedDelete.id = report.id;
    reportFedDelete.userId = report.userId;

    Message::Ptr m = createJsonMessage(reportFedDelete);
    _socket->send(m);
}

void MainWindow::on_btnMonitorThresholdEdit_clicked(bool)
{
    data::MonitorThresholdEdit monitorThresholdEdit;
    monitorThresholdEdit.cputmp.warn = 10;
    monitorThresholdEdit.cputmp.crit = 15;

    Message::Ptr m = createJsonMessage(monitorThresholdEdit);
    _socket->send(m);
}

void MainWindow::command_TaskReportCreate(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Answer)
        return;

    if (message->execStatus() == Message::ExecStatus::Success)
    {
        data::TaskReportCreate taskReportCreate;
        readFromMessage(message, taskReportCreate);

        QString msg = "Success TaskReportCreate, ID: %1\n";
        msg = msg.arg(taskReportCreate.id.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);

        return;
    }
    else
    {
        QString msg = errorDescription(message);
        QMessageBox::critical(this, qApp->applicationName(), msg);
    }
}

void MainWindow::command_TaskContentCreate(const Message::Ptr& message)
{
    if (message->type() != Message::Type::Event)
        return;

    data::TaskContentCreate taskContentCreate;
    readFromMessage(message, taskContentCreate);

    if (taskContentCreate.taskType == TaskType::CreateReport)
    {
        QString msg = "Report created, ContentId: %1\n";
        msg = msg.arg(taskContentCreate.contentId.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);
    }

    if (taskContentCreate.taskType == TaskType::CreateReportFed)
    {
        QString msg = "Report FED created, ContentId: %1\n";
        msg = msg.arg(taskContentCreate.contentId.toString());
        QMessageBox::information(this, qApp->applicationName(), msg);
    }
}


