#include "shared/defmac.h"
#include "shared/utils.h"
#include "shared/logger/logger.h"
#include "shared/logger/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/message.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/config/config.h"
#include "shared/qt/version/version_number.h"
#include "main_window.h"

#include <QtCore>
#include <QApplication>
#include <QNetworkProxy>
#include <signal.h>
#include <unistd.h>

using namespace std;
using namespace communication;
using namespace communication::transport;

namespace {

void stopProgram()
{
    log_info << "'Web Emulator' is stopped";
    alog::logger().stop();
}

//void helpInfo(/*const char * binary*/)
//{
//    alog::logger().clearSavers();
//    alog::logger().addSaverStdOut(alog::Level::Info, true);

//    log_info << "'Web Emulator' service"
//             << " (version: " << productVersion().toString()
//             << "; protocol version: "
//             << BPROTOCOL_VERSION_LOW << "-" << BPROTOCOL_VERSION_HIGH
//             << "; gitrev: " << GIT_REVISION << ")";
//    log_info << "Usage: web-emulator";
//    log_info << "  -n do not daemonize";
//    log_info << "  -h this help";
//    alog::logger().flush();
//}

} // names

int main(int argc, char *argv[])
{
    // Устанавливаем в качестве разделителя целой и дробной части символ '.',
    // если этого не сделать - функции преобразования строк в числа (std::atof)
    // буду неправильно работать.
    qputenv("LC_NUMERIC", "C");

    int ret = 0;
    try
    {
        //google::InitGoogleLogging(argv[0]);
        alog::logger().start();

#ifdef NDEBUG
        alog::logger().addSaverStdOut(alog::Level::Info, true);
#else
        alog::logger().addSaverStdOut(alog::Level::Debug);
#endif
        //signal(SIGTERM, &stopProgramHandler);
        //signal(SIGINT,  &stopProgramHandler);

        QDir homeDir = QDir::home();
        if (!homeDir.exists())
        {
            log_error << "Home dir " << homeDir.path() << " not exists";
            alog::logger().stop();
            return 1;
        }

#ifdef MINGW
        QString configFile = config::dir() + "/../state/web-emulator.state";
#else
        QString configFile = "/var/opt/aisexpert/state/web-emulator.state";
#endif
        config::state().readFile(configFile.toStdString());

        QString logFile;
        if (!config::state().getValue("logger.file", logFile))
        {
#ifdef MINGW
        logFile = config::dir() + "/../log/web-emulator.log";
#else
        logFile = "/var/opt/aisexpert/log/web-emulator.log";
#endif
            config::state().setValue("logger.file", logFile);
            config::state().save();
        }

        QFileInfo logFileInfo {logFile};
        QString logFileDir = logFileInfo.absolutePath();
        if (!QDir(logFileDir).exists())
            if (!QDir().mkpath(logFileDir))
            {
                log_error << "Failed create log directory: " << logFileDir;
                return 1;
            }

        // Создаем дефолтный сэйвер для логгера
        {
            std::string logLevelStr = "debug";
            if (!config::state().getValue("logger.level", logLevelStr))
                config::state().setValue("logger.level", logLevelStr);

            bool logContinue = false;
            if (!config::state().getValue("logger.continue", logContinue))
                config::state().setValue("logger.continue", false);

            alog::Level logLevel = alog::levelFromString(logLevelStr);
            alog::SaverPtr saver {new alog::SaverFile("default",
                                                      logFile.toStdString(),
                                                      logLevel,
                                                      logContinue)};
            alog::logger().addSaver(saver);
        }
        log_info << "'Web Emulator' is running"
                 << " (version " << productVersion().toString() << ")";
        alog::logger().flush();

        if (!communication::command::checkUnique())
        {
            stopProgram();
            return 1;
        }

        QApplication appl {argc, argv};
        QApplication::setApplicationName("Web Emulator " + productVersion().toString());
        //QApplication::setQuitOnLastWindowClosed(false);
        //QApplication::setWindowIcon(QIcon("://resources/toxphone.png"));

        QNetworkProxy::setApplicationProxy(QNetworkProxy::NoProxy);

        tcp::Socket::Ptr socket {new tcp::Socket};
        if (!socket->init({QHostAddress("127.0.0.1"), 62062}))
        {
            stopProgram();
            return 1;
        }
        socket->setCompressionLevel(0);
        socket->setCheckProtocolCompatibility(false);
        socket->setMessageFormat(SerializationFormat::Json);

        MainWindow mw;
        if (!mw.init(socket))
        {
            //QMessageBox::critical(0, qApp->applicationName(), errMessage);
            stopProgram();
            return 1;
        }
        mw.loadGeometry();
        mw.loadSettings();
        mw.show();

        alog::logger().removeSaverStdOut();
        alog::logger().removeSaverStdErr();

        ret = appl.exec();

        mw.saveGeometry();
        mw.saveSettings();
        mw.deinit();

        socket->disconnect();
        config::state().save();
    }
    catch (std::exception& e)
    {
        log_error << "Failed initialization. Detail: " << e.what();
        ret = 1;
    }
    catch (...)
    {
        log_error << "Failed initialization. Unknown error";
        ret = 1;
    }

    stopProgram();
    return ret;
}
