#include "shared/qt/config/config.h"
#include "account_manager.h"
#include <QVector>
#include <QtCore>
#ifdef MINGW
#include "active_directory.h"
#endif
#include "users_cache.h"
#include "ldap_proto.h"

/// Включение/Отключение режима эмуляции
extern bool emulationMode;

namespace account
{

/**
 * @brief Проверка пользователя на валидность
 * @details Проверка происходит по следующим критериям:
 *          1) Аутентификация пары "login" + "password"
 *          2) Устанавливается приндлежность пользователя к группам
 *             "АИС Эксперт" или "АИС Эксперт Админ"
 * @param login - логин пользователя
 * @param password - пароль пользователя
 * @param sid - SID пользователя, в случае успешной валидации
 * @return true - если валидация успешна, false - в обратном случае.
 */
bool checkUser(const QString& login, const QString& password, QString& sid)
{
    QString root_group, user_group;
    /*
      В режиме эмуляции названия групп в конфиге берутся из секции
      "active_directory".
    */
#if defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)
    config::base().getValue("active_directory.root_group", root_group);
    config::base().getValue("active_directory.user_group", user_group);
#elif defined(WINDOWS_ACTIVE_DIRECTORY_LDAP)
    config::base().getValue("ldap.root_group", root_group);
    config::base().getValue("ldap.user_group", user_group);
#else
    config::base().getValue("active_directory.root_group", root_group);
    config::base().getValue("active_directory.user_group", user_group);
#endif

    if (authenticate(login, password)
        && (inGroup(root_group, login) || inGroup(user_group, login)))
    {
        getUserSID(login, sid);
        return true;
    }
    else
        return false;
}
/**
 * @brief Получить SID пользователя с логином "login"
 * @param login - логин пользователя
 * @param userSid - SID, который соответствует пользователю "login"
 */
void getUserSID(const QString& login, QString& userSid)
{
    if (!emulationMode)
    {

#if defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)
        domain::UserSID(login, userSid);
#elif defined(WINDOWS_ACTIVE_DIRECTORY_LDAP)
        ldap_proto::UserSID(login, userSid);
#else
        log_error << "Unsupported Windows Active Directory access. Run program with -e option";
#endif
    }
    else
    {
        YamlConfig::Func loadFunc =
            [&login, &userSid] (YamlConfig* conf, YAML::Node& users, bool logWarn)
        {
            for (const YAML::Node& user : users)
            {
                QString confLogin, confSid;

                conf->getValue(user, "login", confLogin, logWarn);
                conf->getValue(user, "sid", confSid, logWarn);

                if (confLogin == login)
                {
                    userSid = confSid;
                    return true;
                }
            }
            return false;
        };
        config::base().getValue("emulation_users", loadFunc);
    }
}
/**
 * @brief Получение списка пользователей для групп
 * @details Получение списка пользователей групп "АИС Эксперт" и
 *          "АИС Эксперт Админ". Список представляет собой словар usersList
 *          с парами "SID" - "Информация пользователя"
 * @param usersList - словарь для заполнения
 */
void getUsers(user::Record::Map& users)
{
#if (defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)) || defined(WINDOWS_ACTIVE_DIRECTORY_LDAP)
    QString rootGroup, userGroup;
    user::Record::Map domainUsers;
#endif
    if (!emulationMode)
    {
#if defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)
        domain::GetUsers(domainUsers);

        config::base().getValue("active_directory.root_group", rootGroup);
        config::base().getValue("active_directory.user_group", userGroup);
#elif defined(WINDOWS_ACTIVE_DIRECTORY_LDAP)
        ldap_proto::GetUsers(domainUsers);

        config::base().getValue("ldap.root_group", rootGroup);
        config::base().getValue("ldap.user_group", userGroup);
#else
        log_error << "Unsupported Windows Active Directory access. Run program with -e option";
#endif
    }
    else
    {
        config::base().getValue("active_directory.root_group", rootGroup);
        config::base().getValue("active_directory.user_group", userGroup);

        YamlConfig::Func loadFunc =
            [&domainUsers] (YamlConfig* conf, YAML::Node& usersNode, bool logWarn)
        {
            for (const YAML::Node& user : usersNode)
            {
                QString sid;
                conf->getValue(user, "sid", sid, logWarn);

                user::Record usr;
                conf->getValue(user, "login", usr.login, logWarn);
                conf->getValue(user, "name",  usr.name, logWarn);
                conf->getValue(user, "admin", usr.isAdmin, logWarn);
                usr.isValid = true;

                domainUsers[sid] = usr;
            }
            return true;
        };
        config::base().getValue("emulation_users", loadFunc);
    }

    for (auto it = domainUsers.constBegin(); it != domainUsers.constEnd(); ++it)
    {
        user::Record user = it.value();
        bool isAdmin = inGroup(rootGroup, user.login);
        bool isAisUser = isAdmin || inGroup(userGroup, user.login);
        if (isAisUser)
        {
            user.isAdmin = isAdmin;
            QString sid = it.key();
            users[sid] = user;
        }
    }
}
/**
 * @brief Аутентифицировать пару "login" и "password"
 * @param login - логин пользователя
 * @param password - пароль пользователя
 * @return true, если аутентификация успешна
 */
bool authenticate(const QString& login, const QString& password)
{
    bool res = {false};
    if (!emulationMode)
    {
#if defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)
        QString machine, domain;
        domain::GetMachineName(machine);
        domain::GetDomainName(machine, domain);
        res = domain::Login(login, domain, password);
#elif defined(WINDOWS_ACTIVE_DIRECTORY_LDAP)

        QString domain;
        config::base().getValue("ldap.domain", domain);

        res = ldap_proto::Login(login, domain, password);
#else
        log_error << "Unsupported Windows Active Directory access. Run program with -e option";
        return res;
#endif
    }
    else
    {
        YamlConfig::Func loadFunc =
            [&login, &password] (YamlConfig* conf, YAML::Node& users, bool logWarn)
        {
            for (const YAML::Node& user : users)
            {
                QString confLogin, confPassword;
                conf->getValue(user, "login", confLogin, logWarn);
                conf->getValue(user, "passw", confPassword, logWarn);

                if (confLogin == login && confPassword == password)
                    return true;
            }
            return false;
        };
        res = config::base().getValue("emulation_users", loadFunc);
    }
    return  res;
}
/**
 * @brief Состоит ли пользователь "login" в группе "group"
 * @param group - группа
 * @param login - логин пользователя
 * @return true, если пользователь "login" состоит в группе "groups"
 */
bool inGroup(const QString& group, const QString& login)
{
    bool res = {false};
    if (!emulationMode)
    {
#if defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)
        res = domain::InGroup(login, group);
#elif defined(WINDOWS_ACTIVE_DIRECTORY_LDAP)
        res = ldap_proto::InGroup(login, group);
#else
        log_error << "Unsupported Windows Active Directory access. Run program with -e option";
        return res;
#endif
    }
    else
    {
        YamlConfig::Func loadFunc =
            [&login, &group] (YamlConfig* conf, YAML::Node& users, bool logWarn)
        {
            for (const YAML::Node& user : users)
            {
                QString confLogin, confGroup;
                conf->getValue(user, "login", confLogin, logWarn);
                conf->getValue(user, "group", confGroup, logWarn);

                if (confLogin == login && confGroup == group)
                    return true;
            }
            return false;
        };
        res = config::base().getValue("emulation_users", loadFunc);
    }
    return res;
}

} // namespace users
