#include "shared/defmac.h"
#if defined(MINGW) && defined(WINDOWS_ACTIVE_DIRECTORY)
#include "active_directory.h"
#include <windows.h>
#include <string>
#include <lmwksta.h>
#include <lm.h>
#include <algorithm>
#include <activeds.h>
#include <sddl.h>
#include <QtCore>

namespace domain
{

using namespace  std;

HRESULT FindUsers(IDirectorySearch *pContainerToSearch, LPOLESTR szFilter,
                  BOOL bIsVerbose, user::Record::Map& users );

bool GetMachineName(QString& machine)
{
    wchar_t buffer[INFO_BUFFER_SIZE];
    DWORD bufCount = INFO_BUFFER_SIZE;

    std::wstring wtmp;
    if (!GetComputerNameW(buffer, &bufCount))
        return false;

    machine = QString::fromStdWString(buffer);

    return true;
}

bool GetCurrentUser(QString& user)
{
    wchar_t buffer[INFO_BUFFER_SIZE];
    DWORD  bufCount = INFO_BUFFER_SIZE;

    if (!GetUserNameW(buffer, &bufCount ))
        return false;
    user = QString::fromStdWString(buffer);

    return true;
}

bool GetDomainName(const QString& machine, QString& domain )
{
    LPWSTR domain_name = const_cast<wchar_t*>( machine.toStdWString().c_str() );
    LPWKSTA_INFO_102 info = nullptr;

    if (NERR_Success != NetWkstaGetInfo(domain_name, 102, reinterpret_cast<LPBYTE *>(&info)) )
        return false;

    domain = QString::fromStdWString(info->wki102_langroup);

    return true;
}

bool Login(const QString& login, const QString& domain, const QString& passw)
{
    HANDLE hToken = nullptr;
    if (!LogonUserW(login.toStdWString().c_str(),
                    domain.toStdWString().c_str(),
                    passw.toStdWString().c_str(),
                    LOGON32_LOGON_NETWORK,
                    LOGON32_PROVIDER_DEFAULT,
                    &hToken))
    {
        CloseHandle(hToken);
        return FALSE;
    }
    CloseHandle(hToken);
    return TRUE;
}
/**
 * @brief Проверить членство login в группе group.
 * @param login - логин пользователя
 * @param group - группа
 * @return true, если пользователь с логином "login" состоит в группе AD group.
 */
bool InGroup(const QString& login, const QString& group)
{
    std::vector<wstring> local_groups;
    std::vector<wstring> global_groups;
    LPBYTE buffer;
    DWORD entries, total_entries;

    NetUserGetLocalGroups(nullptr,
                          login.toStdWString().c_str(),
                          0,
                          LG_INCLUDE_INDIRECT,
                          &buffer,
                          MAX_PREFERRED_LENGTH,
                          &entries,
                          &total_entries);

    LOCALGROUP_USERS_INFO_0 *groups = reinterpret_cast<LOCALGROUP_USERS_INFO_0*>(buffer);
    for (int i = 0; i < static_cast<int>(entries); i++)
        local_groups.push_back(groups[i].lgrui0_name);
    NetApiBufferFree(buffer);

    NetUserGetGroups(nullptr,
                     login.toStdWString().c_str(),
                     0, &buffer,
                     MAX_PREFERRED_LENGTH,
                     &entries,
                     &total_entries);

    GROUP_USERS_INFO_0 *ggroups = reinterpret_cast<GROUP_USERS_INFO_0*>(buffer);
    for (int i = 0; i < static_cast<int>(entries); i++)
        global_groups.push_back(ggroups[i].grui0_name);

    NetApiBufferFree(buffer);

    std::vector<wstring>::iterator res = find(global_groups.begin(), global_groups.begin() + static_cast<std::ptrdiff_t>( global_groups.size()), group);
    if (res != global_groups.end())
        return true;
    else
        return false;
}
/**
 * @brief Получить SID для пользвателя login
 * @param login - логин пользователя
 * @param sid - SID пользователя
 * @return true, если процедура получения SID прошла без ошибок.
 */
bool UserSID(const QString& login, QString& sid)
{
    LPWSTR user = const_cast<wchar_t *>(login.toStdWString().c_str());
    LPWSTR wszDomainName = static_cast<LPWSTR>(GlobalAlloc(GPTR, sizeof(wchar_t) * 1024));
    DWORD cchDomainName = 1024;
    SID_NAME_USE eSidType;
    LPWSTR sidString;
    char sid_buffer[1024];
    DWORD cbSid = 1024;
    SID* sidValue = reinterpret_cast<SID *>(sid_buffer);

    if (!LookupAccountNameW(nullptr, user, sid_buffer, &cbSid, wszDomainName, &cchDomainName, &eSidType))
        return false;

    if (!ConvertSidToStringSidW(sidValue, &sidString))
        return false;

    sid = QString::fromStdWString(sidString);

    return true;
}

bool GetUsers(user::Record::Map& users)
{
    //  Handle the command line arguments.
    DWORD dwLength = MAX_PATH*2;
    LPOLESTR pszBuffer = new OLECHAR[dwLength];
    wcsncpy(pszBuffer, L"", dwLength);
    BOOL bReturnVerbose = FALSE;

    CoInitialize(nullptr);
    HRESULT hr = S_OK;
    //  Get rootDSE and the current user domain container distinguished name.
    IADs *pObject = nullptr;
    IDirectorySearch *pContainerToSearch = nullptr;
    LPOLESTR szPath = new OLECHAR[MAX_PATH];
    VARIANT var;
    hr = ADsOpenObject(L"LDAP://rootDSE",
                       nullptr,
                       nullptr,
                       ADS_SECURE_AUTHENTICATION, // Use Secure Authentication.
                       IID_IADs , // __uuidof(IADs)
                       reinterpret_cast<void**>(&pObject));
    if (FAILED(hr))
    {
        // "Cannot execute query. Cannot bind to LDAP://rootDSE.\n"
        if (pObject)
            pObject->Release();

        return false;
    }

    if (SUCCEEDED(hr))
    {
        hr = pObject->Get(BSTR(L"defaultNamingContext"), &var);
        if (SUCCEEDED(hr))
        {
            //  Build path to the domain container.
            wcsncpy(szPath, L"LDAP://", MAX_PATH);
            wcsncat(szPath, var.bstrVal, MAX_PATH - wcslen(szPath));
            hr = ADsOpenObject(szPath,
                               nullptr,
                               nullptr,
                               ADS_SECURE_AUTHENTICATION, //  Use Secure Authentication.
                               IID_IDirectorySearch,
                               reinterpret_cast<void**>(&pContainerToSearch));

            if (SUCCEEDED(hr))
            {
                hr = FindUsers(pContainerToSearch, //  IDirectorySearch pointer to domainDNS container.
                               pszBuffer,
                               bReturnVerbose,
                               users
                             );
                if (SUCCEEDED(hr))
                {
                    if (hr == S_FALSE)
                        wprintf(L"User object cannot be found.\n");
                }
                else if (E_ADS_INVALID_FILTER==hr)
                    wprintf(L"Cannot execute query. Invalid filter was specified.\n");
                else
                    wprintf(L"Query failed to run. HRESULT: %x\n",hr);
            }
            else
            {
                // "Cannot execute query. Cannot bind to the container.\n"
            }
            if (pContainerToSearch)
                pContainerToSearch->Release();
        }
        else
            wprintf(L"\nSome errror defaultNamingContext...1\n\n");
        VariantClear(&var);
    }
    else
        wprintf(L"\nSome errror defaultNamingContext...2\n\n");

    if (pObject)
        pObject->Release();

    CoUninitialize();
    delete [] szPath;
    delete [] pszBuffer;
    return true;
}

HRESULT FindUsers(IDirectorySearch *pContainerToSearch,  //  IDirectorySearch pointer to the container to search.
                  LPOLESTR szFilter, //  Filter to find specific users.s
                  //  NULL returns all set properties.
                  BOOL bIsVerbose, //  TRUE indicates that display all properties for the found objects.
                  //  FALSE indicates that only the RDN.
                  user::Record::Map& users
    )
{
    if (!pContainerToSearch)
        return E_POINTER;

    DWORD dwLength = MAX_PATH*2;
    // Create search filter.
    LPOLESTR pszSearchFilter = new OLECHAR[dwLength];

    //  Add the filter.
    swprintf_s(pszSearchFilter, dwLength, L"(&(objectClass=user)(objectCategory=person)%s)",szFilter);

    //  Specify subtree search.
    ADS_SEARCHPREF_INFO SearchPrefs;
    SearchPrefs.dwSearchPref = ADS_SEARCHPREF_SEARCH_SCOPE;
    SearchPrefs.vValue.dwType = ADSTYPE_INTEGER;
    SearchPrefs.vValue.Integer = ADS_SCOPE_SUBTREE;
    DWORD dwNumPrefs = 1;

    //  COL for iterations.
    LPOLESTR pszColumn = nullptr;
    ADS_SEARCH_COLUMN col;
    HRESULT hr = S_OK;

    //  Search handle.
    ADS_SEARCH_HANDLE hSearch = nullptr;

    //  Set search preference.
    hr = pContainerToSearch->SetSearchPreference(&SearchPrefs, dwNumPrefs);
    if (FAILED(hr))
        return hr;

    LPOLESTR szDSGUID = new WCHAR [39];
    VARIANT varDate;
    LPOLESTR pszNonVerboseList[] = { L"name", L"sn", L"givenName", L"initials" };

    LPOLESTR szName = new OLECHAR[MAX_PATH];
    LPOLESTR szDN = new OLECHAR[MAX_PATH];

    VariantInit(&varDate);

    int iCount = 0;

    hr = pContainerToSearch->ExecuteSearch(pszSearchFilter, nullptr, static_cast<DWORD>(-1), &hSearch);

    if (SUCCEEDED(hr))
    {
        //  Call IDirectorySearch::GetNextRow() to retrieve the next data row.
        hr = pContainerToSearch->GetFirstRow(hSearch);
        if (SUCCEEDED(hr))
        {
            // чтение строк данных
            while(hr != S_ADS_NOMORE_ROWS)
            {
                //  Keep track of count.
                iCount++;
                std::wstring fullName;
                std::wstring uname, sn, givenName, initials;
                std::wstring objectGUID;
                QString objectSid, account, userPrincipalName;
                // чтение столбцовs
                while(pContainerToSearch->GetNextColumnName(hSearch, &pszColumn) != S_ADS_NOMORE_COLUMNS)
                {
                    hr = pContainerToSearch->GetColumn(hSearch, pszColumn, &col);
                    if (SUCCEEDED(hr))
                    {
                        if (wcscmp(L"sAMAccountName", pszColumn) == 0)
                        {
                            account = QString::fromStdWString(col.pADsValues->CaseIgnoreString);
                        }
                        if (wcscmp(L"userPrincipalName", pszColumn) == 0)
                        {
                            userPrincipalName = QString::fromStdWString(col.pADsValues->CaseIgnoreString);
                        }
                        if (wcscmp(L"name", pszColumn) == 0)
                        {
                            uname = col.pADsValues->CaseIgnoreString;
                        }
                        if (wcscmp(L"sn", pszColumn) == 0)
                        {
                            sn = col.pADsValues->CaseIgnoreString;
                        }
                        if (wcscmp(L"givenName", pszColumn) == 0)
                        {
                            givenName = col.pADsValues->CaseIgnoreString;
                        }
                        if (wcscmp(L"initials", pszColumn) == 0)
                        {
                            initials = col.pADsValues->CaseIgnoreString;
                        }
                        if (wcscmp(L"objectSid", pszColumn) == 0)
                        {
                            UserSID(account, objectSid);
                        }
                        pContainerToSearch->FreeColumn(&col);
                    }
                    FreeADsMem(pszColumn);
                }
                UserSID(account, objectSid);
                UserSID(userPrincipalName, objectSid);
                fullName = sn + L" " + givenName + L" " + initials;
                user::Record curRow = { account, QString::fromStdWString(uname), false, true };
                users.insert(objectSid, curRow);
                //  Get the next row.
                hr = pContainerToSearch->GetNextRow(hSearch);
            }

        }
        //  Close the search handle to cleanup.
        pContainerToSearch->CloseSearchHandle(hSearch);
    }
    if (SUCCEEDED(hr) && 0==iCount)
        hr = S_FALSE;

    delete [] szName;
    delete [] szDN;
    delete [] szDSGUID;
    delete [] pszSearchFilter;
    return hr;
}

} // namespace domain

#endif
