#ifdef WINDOWS_ACTIVE_DIRECTORY_LDAP
#include "function.h"
#include "ldap_error.h"
#include "ldap_proto.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "ldap.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "LdapProto")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "LdapProto")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "LdapProto")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "LdapProto")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "LdapProto")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "LdapProto")

#define LDAP_INIT                                                        \
            QString ldapAddress;                                         \
            config::base().getValue("ldap.address", ldapAddress);        \
            ldapAddress = "ldap://" + ldapAddress;                       \
            if (ldap_initialize(&ld, ldapAddress.toStdString().c_str())) \
            {                                                            \
                log_error_m << "ldap_initialize failed";                 \
                                                                         \
                throw LdapError("ldap_initialize failed");               \
            }                                                            \

#define LDAP_BIND(username, cred, servcred)                              \
            do                                                           \
            {                                                            \
                rc = ldap_sasl_bind_s(ld,                                \
                                      username.toUtf8().data(),          \
                                      NULL,                              \
                                      &cred,                             \
                                      NULL,                              \
                                      NULL,                              \
                                      &servcred);                        \
            }                                                            \
            while (rc == LDAP_SASL_BIND_IN_PROGRESS);                    \

#define LDAP_ANSWER(message)                                             \
           if (rc != LDAP_SUCCESS)                                       \
           {                                                             \
               log_error_m << message                                    \
                           << ": "                                       \
                           << ldap_err2string(rc);                       \
               throw LdapError(ldap_err2string(rc));                     \
           }                                                             \

namespace ldap_proto {

using namespace  std;

LDAPMessageFree::LDAPMessageFree(LDAPMessage* result)
    : hResult(result)
{}

LDAPMessageFree::~LDAPMessageFree()
{
    //log_debug2_m << "LDAPMessageFree destructor";

    if (hResult)
        ldap_msgfree(hResult);
}

LDAPUnBind::LDAPUnBind(LDAP* ld)
    : ld(ld)
{}

LDAPUnBind::~LDAPUnBind()
{
    //log_debug2_m << "LDAPUnBind destructor";

    if (ld)
        ldap_unbind_ext_s(ld, NULL, NULL);
}

/**
  @brief Проверка логина и пароля пользователя в домене.
  @details Принцип проверки построен на том, что пользователю разрешено
           выполнять ldap запросы. В противном случае аутентификация будет
           провалена, и считаем что пара логин и пароль некорректна.
  @param login - логин пользователя.
  @param domain - домен пользователя.
  @param passw - пароль пользователя.
  @return true - если логин и пароль пользователя корректны, иначе false
*/
bool Login(const QString& login, const QString& domain, const QString& passw)
{
    LDAP* ld = NULL;
    int rc = 0;

    LDAP_INIT

    struct berval credential;
    struct berval* servcred;
    /*
      Заполнение структуры с паролем LDAP-пользователя.
    */
    std::shared_ptr<char> pPassw(new char[passw.size()], std::default_delete<char[]>());
    strcpy(pPassw.get(), passw.toUtf8().data());
    credential.bv_val = pPassw.get();
    credential.bv_len = strlen(pPassw.get());

    int version = LDAP_VERSION3;

    ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version);

    QString ldapUsername = QString(login/* + "@" + domain*/);

    LDAP_BIND(ldapUsername, credential, servcred)

    if (rc != LDAP_SUCCESS)
    {
        ldap_unbind_ext_s(ld, NULL, NULL);
        return false;
    }

    ldap_unbind_ext_s(ld, NULL, NULL);
    return true;
}
/**
 * @brief Проверить членство login в группе group.
 * @param login - логин пользователя
 * @param group - группа
 * @return true, если пользователь с логином "login" состоит в группе AD group.
 */
bool InGroup(const QString& login, const QString& group)
{
    LDAP* ld = NULL;
    int rc = 0;

    LDAP_INIT

    struct berval credential;
    struct berval* servcred;
    /*
      Домен, логин, пароля - для доступа по LDAP.
    */
    QString ldapLogin, ldapPassw, ldapDomain;
    config::base().getValue("ldap.domain",   ldapDomain);
    config::base().getValue("ldap.login",    ldapLogin);
    config::base().getValue("ldap.password", ldapPassw);
    /*
      RDN групп "АИС Эксперт Пользователи" и "АИС Эксперт Администраторы"
    */
    QString rdnGroup;
    QVector<QString> rdnUsers;
    config::base().getValue("ldap.rdn_users",  rdnUsers);
    config::base().getValue("ldap.rdn_group",  rdnGroup);
    /*
      Название групп АИС Эксперт.
    */
    QString userGroup, rootGroup;
    config::base().getValue("ldap.user_group",  userGroup);
    config::base().getValue("ldap.root_group",  rootGroup);
    /*
      Заполнение структуры с паролем LDAP-пользователя.
    */
    std::shared_ptr<char> pPassw(new char[ldapPassw.size()], std::default_delete<char[]>());
    strcpy(pPassw.get(), ldapPassw.toUtf8().data());
    credential.bv_val = pPassw.get();
    credential.bv_len = strlen(pPassw.get());

    int version = LDAP_VERSION3;

    ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version);

    /*
      Полное доменное имя пользователя - пользователь@домен
    */
    QString ldapUsername = QString(ldapLogin/* + "@" + ldapDomain*/);

    LDAP_BIND(ldapUsername, credential, servcred)

    LDAP_ANSWER("Authentication failed")

    LDAP_RESULT_AUTO_UNBIND(ld)
    /*
      DN для групп AisExpertUsers и AisExpertAdmins
    */
    QString groupDn = "CN=" + group + "," + rdnGroup;
    appendDc(groupDn, ldapDomain);
    /*
      Фильтр поиска пользователя в группе.
    */
    QString shortLogin = makeShortLogin(login);

    QString filter = "(&(objectClass=user)"
                     "(sAMAccountName=" + shortLogin + ")"
                     "(memberOf=" + groupDn + "))";

    LDAPMessage* result;

    /*
      Проверка каждого RDN из конфиг-файла, в котором будет произведен
      поиск пользователей
    */
    for (const QString& item : rdnUsers)
    {

        QString base = "";
        base = item;
        appendDc(base, ldapDomain);

        rc = ldap_search_ext_s(ld,
                               base.toUtf8().data(),
                               LDAP_SCOPE_SUB,
                               filter.toUtf8().data(),
                               NULL,
                               0,
                               NULL,
                               NULL,
                               NULL,
                               LDAP_NO_LIMIT,
                               &result);

        if ( rc != LDAP_SUCCESS)
        {
            log_error_m << "ldap_search_ext_s: " << ldap_err2string(rc);
            throw LdapError(ldap_err2string(rc));
        }

        LDAP_RESULT_AUTO_FREE(result)

        char* pAttrName = NULL;
        BerElement* pBer = NULL;
        LDAPMessage* entry;

        for (entry = ldap_first_entry(ld, result); entry != NULL; entry = ldap_next_entry(ld, entry))
        {
            pAttrName = ldap_first_attribute(ld, entry, &pBer);

            if (!pAttrName)
                continue;

            do
            {
                struct berval** res = NULL;

                res = ldap_get_values_len(ld, entry, pAttrName);

                QString attr = QString(pAttrName);
                QString val = QString((*res)->bv_val);

                if ((attr == "sAMAccountName") && (val == shortLogin))
                    return true;
            }
            while ((pAttrName = ldap_next_attribute(ld, entry, pBer)) != NULL);
        }
    }
    return false;
}
/**
 * @brief Получить SID для пользвателя login
 * @param login - логин пользователя
 * @param sid - SID пользователя
 * @return true, если процедура получения SID прошла без ошибок.
 */
bool UserSID(const QString& login, QString& sid)
{
    LDAP* ld = NULL;
    int rc = 0;

    LDAP_INIT

    struct berval credential;
    struct berval* servcred;
    /*
      Домен, логин, пароля - для доступа по LDAP.
    */
    QString ldapLogin, ldapPassw, ldapDomain;
    config::base().getValue("ldap.domain",   ldapDomain);
    config::base().getValue("ldap.login",    ldapLogin);
    config::base().getValue("ldap.password", ldapPassw);
    /*
      RDN групп "АИС Эксперт Пользователи" и "АИС Эксперт Администраторы"
    */
    QString rdnGroup;
    QVector<QString> rdnUsers;
    config::base().getValue("ldap.rdn_users",  rdnUsers);
    config::base().getValue("ldap.rdn_group",  rdnGroup);
    /*
      Название групп АИС Эксперт.
    */
    QString userGroup, rootGroup;
    config::base().getValue("ldap.user_group",  userGroup);
    config::base().getValue("ldap.root_group",  rootGroup);
    /*
      Заполнение структуры с паролем LDAP-пользователя.
    */
    std::shared_ptr<char> pPassw(new char[ldapPassw.size()], std::default_delete<char[]>());
    strcpy(pPassw.get(), ldapPassw.toUtf8().data());
    credential.bv_val = pPassw.get();
    credential.bv_len = strlen(pPassw.get());

    int version = LDAP_VERSION3;

    ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version);

    /*
      Полное доменное имя пользователяя - пользователь@домен
    */
    QString ldapUsername = QString(ldapLogin/* + "@" + ldapDomain*/);

    LDAP_BIND(ldapUsername, credential, servcred)

    LDAP_ANSWER("Authentication failed")

    LDAP_RESULT_AUTO_UNBIND(ld)

    /*
      Фильтр поиска пользователя по логину.
    */
    QString shortLogin = makeShortLogin(login);

    QString filter = "(&(objectClass=user)"
                     "(sAMAccountName=" + shortLogin + "))";

    LDAPMessage* result;

    /*
      RDN для групп AisExpertUsers и AisExpertAdmins
    */
    QString usersPath = "CN=" + userGroup + "," + rdnGroup;
    QString adminPath = "CN=" + rootGroup + "," + rdnGroup;
    /*
      Добавление вычисленного RDN к base-root.
    */
    appendDc(usersPath, ldapDomain);
    appendDc(adminPath, ldapDomain);

    /*
      Проверка каждого RDN из конфиг-файла, в котором будет произведен
      поиск пользователей
    */
    for (const QString& item : rdnUsers)
    {
        QString base = "";
        base = item;
        appendDc(base, ldapDomain);

        rc = ldap_search_ext_s(ld,
                               base.toUtf8().data(),
                               LDAP_SCOPE_SUB,
                               filter.toUtf8().data(),
                               NULL,
                               0,
                               NULL,
                               NULL,
                               NULL,
                               LDAP_NO_LIMIT,
                               &result);

        if (rc != LDAP_SUCCESS)
        {
            log_error_m << "ldap_search_ext_s: " << ldap_err2string(rc);
            throw LdapError(ldap_err2string(rc));
        }

        LDAP_RESULT_AUTO_FREE(result)

        char* pAttrName = NULL;
        BerElement* pBer = NULL;
        LDAPMessage* entry;

        for (entry = ldap_first_entry(ld, result); entry != NULL; entry = ldap_next_entry(ld, entry))
        {
            pAttrName = ldap_first_attribute(ld, entry, &pBer);

            if (!pAttrName)
                continue;

            do
            {
                struct berval** res = NULL;

                res = ldap_get_values_len(ld, entry, pAttrName);

                QString attr = QString(pAttrName);
                QString val = QString((*res)->bv_val);

                if (attr == "objectGUID")
                {
                    QByteArray ba = QByteArray::fromRawData((*res)->bv_val, (*res)->bv_len);
                    QUuid gid = QUuid::fromRfc4122(ba);
                    sid = gid.toString();
                }

                if ((attr == "sAMAccountName") && (val == shortLogin))
                    return true;
            }
            while ((pAttrName = ldap_next_attribute(ld, entry, pBer)) != NULL);
        }
    }

    return false;
}
/**
  @brief Заполнение списка пользователей users данными из AD посредством LDAP
  @param users - список для заполнения.
  @return true - при отсутствии ошибок, иначе - false.
*/
void GetUsers(user::Record::Map& users)
{
    LDAP* ld = NULL;
    int rc = 0;

    LDAP_INIT

    struct berval credential;
    struct berval *servcred;
    /*
      Домен, логин, пароля - для доступа по LDAP.
    */
    QString ldapLogin, ldapPassw, ldapDomain;
    config::base().getValue("ldap.domain",   ldapDomain);
    config::base().getValue("ldap.login",    ldapLogin);
    config::base().getValue("ldap.password", ldapPassw);
    /*
      RDN групп "АИС Эксперт Пользователи" и "АИС Эксперт Администраторы"
    */
    QString rdnGroup;
    QVector<QString> rdnUsers;
    config::base().getValue("ldap.rdn_users",  rdnUsers);
    config::base().getValue("ldap.rdn_group",  rdnGroup);
    /*
      Название групп АИС Эксперт.
    */
    QString userGroup, rootGroup;
    config::base().getValue("ldap.user_group",  userGroup);
    config::base().getValue("ldap.root_group",  rootGroup);
    /*
      Заполнение структуры с паролем LDAP-пользователя.
    */
    std::shared_ptr<char> pPassw(new char[ldapPassw.size()], std::default_delete<char[]>());
    strcpy(pPassw.get(), ldapPassw.toUtf8().data());
    credential.bv_val = pPassw.get();
    credential.bv_len = strlen(pPassw.get());

    int version = LDAP_VERSION3;

    ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version);

    QString ldapUsername = QString(ldapLogin/* + "@" + ldapDomain*/);

    LDAP_BIND(ldapLogin, credential, servcred)

    LDAP_ANSWER("Authentication failed")

    LDAP_RESULT_AUTO_UNBIND(ld)

    LDAPMessage* result;
    /*
      RDN для групп AisExpertUsers и AisExpertAdmins
    */
    QString usersPath = "CN=" + userGroup + "," + rdnGroup;
    QString adminPath = "CN=" + rootGroup + "," + rdnGroup;
    /*
      Добавление вычисленного RDN к base-root.
    */
    appendDc(usersPath, ldapDomain);
    appendDc(adminPath, ldapDomain);
    /*
      Фильтр поиска пользователя в группах AisExpertUsers и AisExpertAdmins.
    */
    QString filter = "(&(objectClass=user)(|(memberOf=" + usersPath + ")(memberOf=" + adminPath + ")))";
    /*
      Проверка каждого RDN из конфиг-файла, в котором будет произведен
      поиск пользователей
    */
    for (const QString& item : rdnUsers)
    {

        QString base = "";
        base = item;
        appendDc(base, ldapDomain);

        rc = ldap_search_ext_s(ld,
                               base.toUtf8().data(),
                               LDAP_SCOPE_SUB,
                               filter.toStdString().c_str(),
                               NULL,
                               0,
                               NULL,
                               NULL,
                               NULL,
                               LDAP_NO_LIMIT,
                               &result);

        LDAP_RESULT_AUTO_FREE(result)

        if (rc != LDAP_SUCCESS)
        {
            log_error_m << "ldap_search_ext_s: " << ldap_err2string(rc);
            throw LdapError(ldap_err2string(rc));
        }

        char* pAttrName = NULL;
        BerElement* pBer = NULL;

        LDAPMessage* entry;

        for (entry = ldap_first_entry(ld, result); entry != NULL; entry = ldap_next_entry(ld, entry))
        {
            pAttrName = ldap_first_attribute(ld, entry, &pBer);

            if (!pAttrName)
                continue;

            QString objectGuid; // GID пользователя
            QString account; // Аккаунт
            QString uname;   // Имя пользователя

            do
            {
                struct berval** res = NULL;

                res = ldap_get_values_len(ld, entry, pAttrName);

                QString attr = QString(pAttrName);
                QString val = QString((*res)->bv_val);

                if (attr == "objectGUID")
                {
                    QByteArray ba = QByteArray::fromRawData((*res)->bv_val, (*res)->bv_len);
                    QUuid gid = QUuid::fromRfc4122(ba);
                    objectGuid = gid.toString();
                }

                if (attr == "sAMAccountName")
                    account = QString::fromUtf8((*res)->bv_val);

                if (attr == "name")
                    uname = QString::fromUtf8((*res)->bv_val);

            }
            while ((pAttrName = ldap_next_attribute(ld, entry, pBer)) != NULL);

            bool isAdmin = InGroup(account, "AisExpertAdmins");

            user::Record curRow = { account, uname, isAdmin, true };
            users.insert(objectGuid, curRow);
        }
    }
}

void appendDc(QString& base, const QString& domain) noexcept
{
    bool isEmpty = base.isEmpty() ? true : false;

    QStringList items = domain.split(".");

    int pos = 0;

    for (const QString& item : items)
    {
        if ((pos == 0) && isEmpty)
            base.append("dc=" + item);
        else
            base.append(",dc=" + item);

        ++pos;
    }
}

} // namespace ldap_proto

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m

#endif // #ifdef WINDOWS_ACTIVE_DIRECTORY_LDAP
