#include "function.h"
#include "ldap_error.h"
#include "users_cache.h"
#include "account_manager.h"
#include "ldap_proto.h"
#include "shared/safe_singleton.h"
#include "shared/logger/logger.h"
#include "shared/qt/config/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"

//#include "commands/commands.h"
//#include "commands/datamodel_commands.h"
#include <map>

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "UsersCache")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "UsersCache")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "UsersCache")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "UsersCache")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "UsersCache")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "UsersCache")

namespace user {

using namespace communication::transport;

Cache& cache()
{
    return ::safe_singleton<Cache>();
}

Cache::Cache()
{
    config::base().getValue("active_directory.root_group", _rootGroup);
    config::base().getValue("active_directory.user_group", _userGroup);
}

QVector<UPoolUserInfo> Cache::upoolUsers() const
{
    QVector<UPoolUserInfo> res;
    for (const QString& sid : _users.keys())
    {
        const user::Record& u = _users[sid];
        UPoolUserInfo userInfo;
        userInfo.sid = sid;
        userInfo.login = u.login;
        userInfo.name = u.name;
        userInfo.isAdmin = u.isAdmin;
        userInfo.isValid = u.isValid;
        res.append(userInfo);
    }
    return res;
}
/**
 * @brief Является ли пользователь "login" администратором.
 * @param login - логин пользователя.
 * @return true, если пользователь является администратором.
 */
bool Cache::isAdmin(const QString& login)
{
    QString shortLogin = makeShortLogin(login);

    for (const QString& sid : _users.keys())
    {
        // TODO: в будущем на проверку в реальном времени
        const user::Record& u = _users[sid];
        if (u.login == shortLogin)
            return u.isAdmin;
    }
    return false;
}

/**
 * @brief Синхронизация пользователей из AD и кэшем пользователей.
 */
void Cache::actualize()
{
    // Чтение списка пользователей
    Record::Map users;

    try
    {
        account::getUsers(users);
    }
    catch (const LdapError& e)
    {
        log_error_m << "Listing users failed. Detail: "
                    << QString::fromUtf8(e.what());
        return;
    }

    if (_first) // "первый" запуск, инициализация кеша
    {
        _users = users;
        _first = false;
    }
    else // обновление/синхронизация кеша с AD
    {
        // Перед сохранением в кеш, найти изменения
        if (validateUsers(users))
        {
            // Актуализировать кеш новым списком
            applyChanges(users);
        }
    }

    log_debug_m << "UsersCache was actualized. Current users list is ...";

    { //Block for alog::Line
        alog::Line logLine = log_debug_m << "Administrators: ";
        for (const auto& item : users)
        {
            if (item.isAdmin)
                logLine << item.login << "; ";
        }
    }
    { //Block for alog::Line
        alog::Line logLine = log_debug_m << "Users: ";
        for (const auto& item : users)
        {
            if (!item.isAdmin)
                logLine << item.login << "; ";
        }
    }
}
/**
 * @brief Создание сообщения с уведомлением, о произошедших изменениях
 * @param changed - список пользователей подвергихся изменениям
 * @param inserted - список новых пользователей
 * @param deleted - список удаленных пользователей
 */
void Cache::notifyChanges(const QList<UPoolUserInfo>& changeUsers,
                          const QList<UPoolUserInfo>& newUsers,
                          const QList<QString>& deleteUsers) const
{
    UPoolChangesNotify upoolChangesNotify;
    upoolChangesNotify.changeUsers = changeUsers;
    upoolChangesNotify.newUsers    = newUsers;
    upoolChangesNotify.deleteUsers = deleteUsers;

    Message::Ptr m = createJsonMessage(upoolChangesNotify, Message::Type::Event);
    writeToJsonMessage(upoolChangesNotify, m);
    tcp::listener().send(m);
}
/**
 * @brief Синхронизировать кеш со списоком пользователей users.
 * @param users - список пользователей, для изменения кеша.
 */
void Cache::applyChanges(const Record::Map& users)
{
    for (const QString& sid : users.keys())
    {
        const Record& usr = users[sid];
        if (!_users.contains(sid))
        {
            // пользователя нет в кеше
            _users[sid] = usr;
        }
        else
        {
            // пользователь есть в кеше
            const Record& current = _users[sid];
            if (current != usr)
                _users[sid] = usr;
        }
    }
    // Проверить и удалить
    for (const QString& sid : _users.keys())
    {
        if (!users.contains(sid))
            _users.remove(sid);
    }
}
/**
 * @brief Проверить всех пользователей "newestUsers" на валидность/изменения.
 * @details Метод для проверки пользователей из AD
 * @param newestUsers - список пользователей с атрибутами из AD.
 */
bool Cache::validateUsers(const Record::Map& aisUsers)
{
    QList<UPoolUserInfo>   changeUsers;
    QList<UPoolUserInfo>   newUsers;
    QList<QString /*SID*/> deleteUsers;

    // Изменённые записи
    for (const QString& sid : aisUsers.keys())
    {
        if (_users.contains(sid)) // sid есть в кеше
        {
            Record aisUser = aisUsers[sid];
            Record cacheUser = _users[sid];
            if (cacheUser != aisUser)
            {
                UPoolUserInfo userInfo;
                userInfo.sid = sid;
                userInfo.login = aisUser.login;
                userInfo.name = aisUser.name;
                userInfo.isAdmin = aisUser.isAdmin;
                userInfo.isValid = true;
                changeUsers.append(userInfo);
            }
        }
        else // Новые пользователи в Active Directory
        {
            Record aisUser = aisUsers[sid];
            UPoolUserInfo userInfo;
            userInfo.sid = sid;
            userInfo.login = aisUser.login;
            userInfo.name = aisUser.name;
            userInfo.isAdmin = aisUser.isAdmin;
            userInfo.isValid = true;
            newUsers.append(userInfo);
        }
    }

    // Удаленные из Active Directory
    for (const QString& sid : _users.keys())
    {
        if (!aisUsers.contains(sid))
            deleteUsers.append(sid);
    }

    if (changeUsers.size() || newUsers.size() || deleteUsers.size())
    {
        notifyChanges(changeUsers, newUsers, deleteUsers);
        return true;
    }
    return false;
}
/**
 * @brief Проверить всех пользователей "isValidUsers" на валидность/изменения.
 * @details Метод для проверки пользователей из АИС
 * @param isValidUsers - список пользователей с атрибутами из АИС.
 */
void Cache::validateSids(const QList<UPoolUserInfo>& aisUsers)
{
    QList<UPoolUserInfo>   changeUsers;
    QList<UPoolUserInfo>   newUsers;
    QList<QString /*SID*/> deleteUsers;

    // Изменённые записи
    for (const UPoolUserInfo& user : aisUsers)
    {
        if (_users.contains(user.sid)) // sid есть в кеше
        {
            const Record& cachedUser = _users[user.sid];
            // Необходимо проверить изменение login, name, isAdmin, и в случае
            // найденных изменений поправить значения.
            // Значение isValid будет установлено в 'true', так как в кеше
            // userspool содержаться пользователи, которые находятся в группах
            // AisExpertUsers или AisExpertAdmins
            if (cachedUser != user)
            {
                UPoolUserInfo userInfo; // изменения для отправки
                userInfo.sid = user.sid;
                userInfo.login = cachedUser.login;
                userInfo.name = cachedUser.name;
                userInfo.isAdmin = cachedUser.isAdmin;
                userInfo.isValid = true;
                changeUsers.append(userInfo);
            }
        }
    }

    // В кеше нет пользователя из АИС
    for (const UPoolUserInfo& user : aisUsers)
    {
        if (!_users.contains(user.sid))
            deleteUsers.append(user.sid);
    }

    // В кеше пользователь, которго нет в АИС
    for (const QString& sid : _users.keys())
    {
        // Найти в списке newest элемент с заданным SIDом
        auto it = std::find_if(aisUsers.begin(), aisUsers.end(),
                               [sid](const UPoolUserInfo& upoolUser) {
                                   return (upoolUser.sid == sid);
                               });
        // Если пользователя из кеша в списке "newestUsers" не найдено,
        // значит он удален.
        if (it == aisUsers.end())
        {
            UPoolUserInfo userInfo;
            userInfo.sid = sid;
            const Record& u = _users[sid];
            userInfo.login = u.login;
            userInfo.name = u.name;
            userInfo.isAdmin = u.isAdmin;
            userInfo.isValid = true;
            newUsers.append(userInfo);
        }
    }

    if (changeUsers.size() || newUsers.size() || deleteUsers.size())
    {
        notifyChanges(changeUsers, newUsers, deleteUsers);
    }
}

} // namespace upoolCache

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
