#include "userspool_appl.h"
#include "shared/defmac.h"
#include "shared/utils.h"
#include "shared/logger/logger.h"
#include "shared/logger/config.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/commands_pool.h"
#include "shared/qt/communication/message.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/config/config.h"
#include "shared/qt/version/version_number.h"
#include "account_manager/users_cache.h"

#include <QNetworkProxy>

#ifdef MINGW
#include <windows.h>
#else
#include <signal.h>
#endif
#include <unistd.h>

using namespace std;
using namespace communication::transport;

bool emulationMode = false;

/**
  Используется для уведомления основного потока о завершении работы программы.
*/
#ifdef MINGW
BOOL WINAPI stopProgramHandler(DWORD sig)
{
    if ((sig == CTRL_C_EVENT) || (sig == CTRL_BREAK_EVENT))
    {
        const char* sigName = (sig == CTRL_C_EVENT) ? "CTRL_C_EVENT" : "CTRL_BREAK_EVENT";
        log_verbose << "Signal " << sigName << " is received. Program will be stopped";
        Application::stop();
    }
    else
        log_verbose << "Signal " << sig << " is received";
    return TRUE;
}
#else
void stopProgramHandler(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
    {
        const char* sigName = (sig == SIGTERM) ? "SIGTERM" : "SIGINT";
        log_verbose << "Signal " << sigName << " is received. Program will be stopped";
        Application::stop();
    }
    else
        log_verbose << "Signal " << sig << " is received";
}
#endif // #ifdef MINGW

void stopLog()
{
    alog::logger().flush();
    alog::logger().waitingFlush();
    alog::logger().stop();
}

void stopProgram()
{
    #define STOP_THREAD(THREAD_FUNC, NAME, TIMEOUT) \
        if (!THREAD_FUNC.stop(TIMEOUT * 1000)) { \
            log_info << "Thread '" NAME "': Timeout expired, thread will be terminated"; \
            THREAD_FUNC.terminate(); \
        }

//    STOP_THREAD(videoCapture(),         "VideoCapture",       15)
//    STOP_THREAD(detectFaces1(),         "DetectFaces1",       15)

    #undef STOP_THREAD

    tcp::listener().close();

    log_info << "'Users Pool' service is stopped";
    stopLog();
}

void helpInfo(/*const char * binary*/)
{
    alog::logger().clearSavers();
    alog::logger().addSaverStdOut(alog::Level::Info, true);

    log_info << "'Users Pool' service"
             << " (version: " << productVersion().toString()
             << "; protocol version: "
             << BPROTOCOL_VERSION_LOW << "-" << BPROTOCOL_VERSION_HIGH
             << "; gitrev: " << GIT_REVISION << ")";
    log_info << "Usage: userspool";
    log_info << "  -e emulation users list";
    log_info << "  -h this help";
    alog::logger().flush();
}

int main(int argc, char *argv[])
{
    int ret = 0;
    try
    {
        //google::InitGoogleLogging(argv[0]);
        alog::logger().start();

#ifdef NDEBUG
        alog::logger().addSaverStdOut(alog::Level::Info, true);
#else
        alog::logger().addSaverStdOut(alog::Level::Debug2);
#endif
#ifdef MINGW
        if (!SetConsoleCtrlHandler(stopProgramHandler, TRUE))
        {
            log_error << "Could not set control handler";
            stopLog();
            return 1;
        }
#else
        signal(SIGTERM, &stopProgramHandler);
        signal(SIGINT,  &stopProgramHandler);
#endif

        QDir homeDir = QDir::home();
        if (!homeDir.exists())
        {
            log_error << "Home dir " << homeDir.path() << " not exists";
            stopLog();
            return 1;
        }

        int c;
        bool isDaemon = true;
        while ((c = getopt(argc, argv, "ehn")) != EOF)
        {
            switch (c)
            {
                case 'h':
                    helpInfo();
                    stopLog();
                    exit(0);
                case 'e':
                    emulationMode = true;
                    break;
                case 'n':
                    isDaemon = false;
                    break;
                case '?':
                    log_error << "Invalid option";
                    stopLog();
                    return 1;
            }
        }

        // Путь к основному конфиг-файлу
        QString configFile = config::dir() + "/userspool.conf";
        if (!QFile::exists(configFile))
        {
            log_error << "Config file " << configFile << " not exists";
            stopLog();
            return 1;
        }

        config::base().setReadOnly(true);
        config::base().setSaveDisabled(true);
        if (!config::base().readFile(configFile.toStdString()))
        {
            stopLog();
            return 1;
        }

        QString logFile;
#ifdef MINGW
        config::base().getValue("logger.file_win", logFile);
#else
        config::base().getValue("logger.file", logFile);
#endif
        config::dirExpansion(logFile);

        QFileInfo logFileInfo {logFile};
        QString logFileDir = logFileInfo.absolutePath();
        if (!QDir(logFileDir).exists())
            if (!QDir().mkpath(logFileDir))
            {
                log_error << "Failed create log directory: " << logFileDir;
                stopLog();
                return 1;
            }

        // Создаем дефолтный сэйвер для логгера
        {
            std::string logLevelStr = "info";
            config::base().getValue("logger.level", logLevelStr);

            bool logContinue = true;
            config::base().getValue("logger.continue", logContinue);

            alog::Level logLevel = alog::levelFromString(logLevelStr);
            alog::SaverPtr saver {new alog::SaverFile("default",
                                                      logFile.toStdString(),
                                                      logLevel,
                                                      logContinue)};
            alog::logger().addSaver(saver);
        }
        {// Block for alog::Line
            alog::Line logLine = log_info
                << "'Users Pool' service is running"
                << " (version " << productVersion().toString() ;
            if (emulationMode)
                logLine << ", emulation mode";
            logLine << ")";
        }
        alog::logger().flush();

#ifndef MINGW
        if (isDaemon)
        {
            stopLog();
            alog::logger().removeSaverStdOut();
            alog::logger().removeSaverStdErr();

            if (daemon(1, 0) != 0)
                return 0;

            alog::logger().start();
            log_verbose << "Demonization success";
        }
#endif
        alog::logger().removeSaverStdOut();
        alog::logger().removeSaverStdErr();

        alog::printSaversInfo();

        if (!communication::command::checkUnique())
        {
            stopProgram();
            return 1;
        }

        Application appl {argc, argv};
        appl.init();

        // Устанавливаем текущую директорию. Эта конструкция работает только
        // когда создан экземпляр QCoreApplication.
        if (QDir::setCurrent(QCoreApplication::applicationDirPath()))
        {
            log_debug << "Set work directory: " << QCoreApplication::applicationDirPath();
        }
        else
        {
            log_error << "Failed set work directory";
            stopProgram();
            return 1;
        }

        QNetworkProxy::setApplicationProxy(QNetworkProxy::NoProxy);

        // Инициализация communication::listener::tcp
        QHostAddress hostAddress = QHostAddress::Any;
        QString hostAddressStr;
        if (config::base().getValue("listener.address", hostAddressStr))
        {
            if (hostAddressStr.toLower().trimmed() == "localhost")
                hostAddress = QHostAddress::LocalHost;
            else if (hostAddressStr.toLower().trimmed() == "any")
                hostAddress = QHostAddress::Any;
            else
                hostAddress = QHostAddress(hostAddressStr);
        }
        int port = 62063;
        config::base().getValue("listener.port", port);
        if (!tcp::listener().init({hostAddress, port}))
        {
            stopProgram();
            return 1;
        }
        tcp::listener().setCheckProtocolCompatibility(false);

        chk_connect_q(&tcp::listener(), SIGNAL(message(communication::Message::Ptr)),
                      &appl, SLOT(message(communication::Message::Ptr)))

        chk_connect_q(&tcp::listener(), SIGNAL(socketConnected(communication::SocketDescriptor)),
                      &appl, SLOT(socketConnected(communication::SocketDescriptor)))

        chk_connect_q(&tcp::listener(), SIGNAL(socketDisconnected(communication::SocketDescriptor)),
                      &appl, SLOT(socketDisconnected(communication::SocketDescriptor)))

        if (Application::isStopped())
            return 0;

        QMetaObject::invokeMethod(&appl, "updateUserList", Qt::QueuedConnection);

        ret = appl.exec();
    }
    catch (std::exception& e)
    {
        log_error << "Failed initialization. Detail: " << e.what();
        ret = 1;
    }
    catch (...)
    {
        log_error << "Failed initialization. Unknown error";
        ret = 1;
    }

    stopProgram();
    return ret;
}
