#include "userspool_appl.h"
#include "commands/authorization.h"
#include "commands/commands.h"
#include "commands/group.h"
#include "commands/upool.h"
#include "commands/user.h"
#include "commands/error.h"
#include "shared/break_point.h"
#include "shared/logger/logger.h"
#include "shared/qt/logger/logger_operators.h"
#include "shared/qt/config/config.h"
#include "shared/qt/communication/functions.h"
#include "shared/qt/communication/logger_operators.h"
#include "shared/qt/communication/transport/tcp.h"
#include "shared/qt/communication/commands_pool.h"
#include "account_manager/account_manager.h"
#include "account_manager/users_cache.h"

#define log_error_m   alog::logger().error  (__FILE__, __func__, __LINE__, "Application")
#define log_warn_m    alog::logger().warn   (__FILE__, __func__, __LINE__, "Application")
#define log_info_m    alog::logger().info   (__FILE__, __func__, __LINE__, "Application")
#define log_verbose_m alog::logger().verbose(__FILE__, __func__, __LINE__, "Application")
#define log_debug_m   alog::logger().debug  (__FILE__, __func__, __LINE__, "Application")
#define log_debug2_m  alog::logger().debug2 (__FILE__, __func__, __LINE__, "Application")

#define READ_FROM_MESSAGE(MESSAGE, DATA) { \
    SResult res = readFromMessage(MESSAGE, DATA); \
    if (!res) { \
        communication::data::MessageError err {error::json_parse_error}; \
        err.description = res.description(); \
        communication::Message::Ptr answer = MESSAGE->cloneForAnswer(); \
        writeToJsonMessage(err, answer); \
        tcp::listener().send(answer); \
        return; \
    }}

using namespace communication::data;

volatile bool Application::_stop = false;
std::atomic_int Application::_exitCode = {0};
extern bool emulationMode;

Application::Application(int& argc, char** argv)
    : QCoreApplication(argc, argv)
{
    _stopTimerId = startTimer(1000);
    _userListTimerId = startTimer(15*1000  /*1 мин*/);
}

bool Application::init()
{
    return true;
}

void Application::timerEvent(QTimerEvent* event)
{
    if (event->timerId() == _stopTimerId)
    {
        if (_stop)
        {
            killTimer(_stopTimerId);
            killTimer(_userListTimerId);
            exit(_exitCode);
            return;
        }

        // Закрываем сокеты не прошедшие авторизацию за отведенное время
        std::time_t curTime = std::time(nullptr);
        for (int i = 0; i < _waitingAuthorizSockets.count(); ++i)
            if (_waitingAuthorizSockets[i].second < curTime)
            {
                tcp::Socket::Ptr socket = tcp::listener()
                                          .socketByDescriptor(_waitingAuthorizSockets[i].first)
                                          .dynamic_cast_to<tcp::Socket::Ptr>();
                if (socket)
                {
                    data::CloseConnection closeConnection;
                    closeConnection.description = "Authorization timeout expired";

                    Message::Ptr m = createJsonMessage(closeConnection);
                    m->destinationSocketDescriptors().insert(socket->socketDescriptor());
                    tcp::listener().send(m);

                    log_error_m << "Authorization timeout expired"
                                << ". Remote host: " << socket->peerPoint()
                                << ". Connection will be closed";

                    _waitingAuthorizSockets[i].second = curTime + 3;
                    _waitingCloseSockets.append(_waitingAuthorizSockets[i]);
                }
                _waitingAuthorizSockets.remove(i--);
            }

        for (int i = 0; i < _waitingCloseSockets.count(); ++i)
            if (_waitingCloseSockets[i].second < curTime)
            {
                tcp::Socket::Ptr socket = tcp::listener()
                                          .socketByDescriptor(_waitingCloseSockets[i].first)
                                          .dynamic_cast_to<tcp::Socket::Ptr>();
                if (socket)
                {

                    log_error_m << "Force close of connection"
                                << ". Remote host: " << socket->peerPoint();
                    socket->stop();
                }
                _waitingCloseSockets.remove(i--);
            }
    }
    else if (event->timerId() == _userListTimerId)
    {
        updateUserList();
    }
}

void Application::stop(int exitCode)
{
    _exitCode = exitCode;
    stop();
}

void Application::message(const Message::Ptr& message)
{
    using namespace communication::data;

    if (message->command() == command::Error)
    {
        data::Error error;
        readFromMessage(message, error);
        log_error_m << "Error message"
                    << ". Command: " << CommandNameLog(error.commandId)
                    << ". Message id: " << error.messageId
                    << ". Detail: " << error.description;
        return;
    }

    if (message->command() == command::UPoolAuthorization)
    {
        data::UPoolAuthorization upoolAuthorization;
        READ_FROM_MESSAGE(message, upoolAuthorization)

        QString password;
        config::base().rereadFile();
        config::base().getValue("listener.password", password);

        if (password == upoolAuthorization.password)
        {
            for (int i = 0; i < _waitingAuthorizSockets.count(); ++i)
                if (_waitingAuthorizSockets[i].first == message->socketDescriptor())
                    _waitingAuthorizSockets.remove(i--);

            // Отправляем подтверждение об успешной операции: пустое сообщение
            Message::Ptr answer = message->cloneForAnswer();
            tcp::listener().send(answer);

            log_info_m << "Success authorization"
                       << ". Remote host: " << message->sourcePoint();
        }
        else
        {
            data::MessageFailed failed;
            failed.description = "Failed authorization";

            Message::Ptr answer = message->cloneForAnswer();
            writeToJsonMessage(failed, answer);
            tcp::listener().send(answer);

            data::CloseConnection closeConnection;
            closeConnection.description = failed.description;

            Message::Ptr m = createJsonMessage(closeConnection);
            m->destinationSocketDescriptors().insert(message->socketDescriptor());
            tcp::listener().send(m);

            for (int i = 0; i < _waitingAuthorizSockets.count(); ++i)
                if (_waitingAuthorizSockets[i].first == message->socketDescriptor())
                {
                    _waitingAuthorizSockets[i].second = std::time(nullptr) + 3;
                    _waitingCloseSockets.append(_waitingAuthorizSockets[i]);
                    _waitingAuthorizSockets.remove(i--);
                }

            log_error_m << "Failed authorization"
                        << ". Remote host: " << message->sourcePoint()
                        << ". Connection will be closed";
        }
        return;
    }

    for (int i = 0; i < _waitingAuthorizSockets.count(); ++i)
        if (_waitingAuthorizSockets[i].first == message->socketDescriptor())
        {
            // Не обрабатываем сообщения от сокетов не прошедших авторизацию
            return;
        }

    if (message->command() == command::UPoolUserList)
    {
        data::UPoolUserList upoolUserList;
        upoolUserList.items = user::cache().upoolUsers();

        Message::Ptr answer = message->cloneForAnswer();
        writeToJsonMessage(upoolUserList, answer);
        tcp::listener().send(answer);
    }
    else if (message->command() == command::UPoolUserAuth)
    {
        data::UPoolUserAuth upoolUserAuth;
        READ_FROM_MESSAGE(message, upoolUserAuth)

        Message::Ptr answer = message->cloneForAnswer();

        try
        {
            if (account::checkUser(upoolUserAuth.login, upoolUserAuth.password, upoolUserAuth.sid))
            {
                upoolUserAuth.isAdmin = user::cache().isAdmin(upoolUserAuth.login);
                upoolUserAuth.status = true;
            }
            else
            {
                upoolUserAuth.password.clear();
                upoolUserAuth.sid.clear();
                upoolUserAuth.isAdmin = false;
                upoolUserAuth.status = false;
            }

            writeToJsonMessage(upoolUserAuth, answer);

        }
        catch (const std::exception& e)
        {
            writeToJsonMessage(error::ldap_answer, answer);
        }

        tcp::listener().send(answer);
    }
    else if (message->command() == command::UPoolValidUsers)
    {
        // Чтение списка пользователей от АИС
        data::UPoolValidUsers upoolValidUsers;
        READ_FROM_MESSAGE(message, upoolValidUsers)

        // Валидация пользователей
        user::cache().validateSids(upoolValidUsers.items);
    }
}

void Application::socketConnected(SocketDescriptor socketDescr)
{
    _waitingAuthorizSockets.append(qMakePair(socketDescr, std::time(nullptr) + 5));
}

void Application::socketDisconnected(SocketDescriptor socketDescr)
{
    for (int i = 0; i < _waitingAuthorizSockets.count(); ++i)
        if (_waitingAuthorizSockets[i].first == socketDescr)
            _waitingAuthorizSockets.remove(i--);

    for (int i = 0; i < _waitingCloseSockets.count(); ++i)
        if (_waitingCloseSockets[i].first == socketDescr)
            _waitingCloseSockets.remove(i--);
}
/**
 * @brief Синхронизировать кеш пользователей
 */
void Application::updateUserList()
{
    if (emulationMode)
        config::base().rereadFile();

    user::cache().actualize();
}

#undef log_error_m
#undef log_warn_m
#undef log_info_m
#undef log_verbose_m
#undef log_debug_m
#undef log_debug2_m
