#pragma once

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

/** WEB
  Команда создает элемент почтовой рассылки
*/
extern const QUuidEx NotifyAddressCreate;

/** WEB
  Команда редактирует элемент почтовой рассылки
*/
extern const QUuidEx NotifyAddressEdit;

/** WEB
  Команда удаляет элемент почтовой рассылки
*/
extern const QUuidEx NotifyAddressDelete;

/** WEB
  Возвращает информацию по элементу почтовой рассылки
*/
extern const QUuidEx NotifyAddressInfo;

/** WEB
  Возвращает список элементов почтовой рассылки
*/
extern const QUuidEx NotifyAddressList;

/** WEB
  Команда создает триггер почтового уведомления
*/
extern const QUuidEx NotifyTriggerCreate;

/** WEB
  Команда редактирует триггер почтового уведомления
*/
extern const QUuidEx NotifyTriggerEdit;

/** WEB
  Команда удаляет триггер почтового уведомления
*/
extern const QUuidEx NotifyTriggerDelete;

/** WEB
  Возвращает информацию по триггеру
*/
extern const QUuidEx NotifyTriggerInfo;

/** WEB
  Возвращает список триггеров
*/
extern const QUuidEx NotifyTriggerList;



} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct NotifyAddress
{
    // Идентификатор адресата
    QUuidEx id;

    // Наименование адресата
    QString name;

    // Электронный адрес
    QString email;

    // Список идентификаторов триггеров с которыми связан адресат
    QVector<QUuidEx> triggers;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id       )
        J_SERIALIZE_ITEM( name     )
        J_SERIALIZE_ITEM( email    )
        J_SERIALIZE_ITEM( triggers )
    J_SERIALIZE_END
};

struct NotifyAddressCreate : NotifyAddress,
                             Data<&command::NotifyAddressCreate,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{};

struct NotifyAddressEdit : NotifyAddress,
                           Data<&command::NotifyAddressEdit,
                                 Message::Type::Command,
                                 Message::Type::Answer>
{};

struct NotifyAddressDelete : Data<&command::NotifyAddressDelete,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{
    QUuidEx id;
    J_SERIALIZE_ONE( id )
};

struct NotifyAddressInfo : Data<&command::NotifyAddressInfo,
                                 Message::Type::Command>
{
    QUuidEx id; // id адресата, по которому запрашивается информация
    J_SERIALIZE_ONE( id )
};

struct NotifyAddressInfoA : NotifyAddress,
                            Data<&command::NotifyAddressInfo,
                                  Message::Type::Answer>
{};

struct NotifyAddressList : Data<&command::NotifyAddressList,
                                 Message::Type::Command,
                                 Message::Type::Answer>
{
    QVector<NotifyAddress> items;
    J_SERIALIZE_ONE( items )
};


struct NotifyTrigger
{
    // Идентификатор триггера
    QUuidEx id;

    // Короткое наименование триггера (30 символов)
    QString name;

    // Тип триггера. Допускаются следующие типы:
    //   'cpu'    - триггер для процессора (в процентах);
    //   'ram'    - триггер для оперативной памяти (в процентах);
    //   'hdd'    - триггер для жесткого диска (в процентах);
    //   'cputmp' - триггер для температуры процессора (в градусах)
    //   'foms'   - триггер для статуса активности ФОМС (да/нет)
    //   'evlog'  - триггер для отправки лог-сообщений (да/нет)
    QString type;

    // Порог срабатывания (проценты или абсолютные величины)
    qint16 threshold = {50};

    // Интервал отправки повторного сообщения (в минутах)
    qint16 emissionInterval = {30};

    // Тема почтового сообщения (100 символов)
    QString subject;

    // Тело почтового сообщения (1000 символов)
    QString messageTemplate;


    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id               )
        J_SERIALIZE_ITEM( name             )
        J_SERIALIZE_ITEM( type             )
        J_SERIALIZE_ITEM( threshold        )
        J_SERIALIZE_ITEM( emissionInterval )
        J_SERIALIZE_ITEM( subject          )
        J_SERIALIZE_ITEM( messageTemplate  )
    J_SERIALIZE_END
};

struct NotifyTriggerCreate : NotifyTrigger,
                             Data<&command::NotifyTriggerCreate,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{};

struct NotifyTriggerEdit : NotifyTrigger,
                           Data<&command::NotifyTriggerEdit,
                                 Message::Type::Command,
                                 Message::Type::Answer>
{};

struct NotifyTriggerDelete : Data<&command::NotifyTriggerDelete,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{
    QUuidEx id;
    J_SERIALIZE_ONE( id )
};

struct NotifyTriggerInfo : Data<&command::NotifyTriggerInfo,
                                 Message::Type::Command>
{
    QUuidEx id; // id триггера, по которому запрашивается информация
    J_SERIALIZE_ONE( id )
};

struct NotifyTriggerInfoA : NotifyTrigger,
                            Data<&command::NotifyTriggerInfo,
                                  Message::Type::Answer>
{};

struct NotifyTriggerList : Data<&command::NotifyTriggerList,
                                 Message::Type::Command,
                                 Message::Type::Answer>
{
    QVector<NotifyTrigger> items;
    J_SERIALIZE_ONE( items )
};

} // namespace data
} // namespace communication
