#pragma once

#include "paging.h"
#include "task.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/** WEB
  Данные для справочника "Вид медицинской помощи"
*/
extern const QUuidEx NsiVidmpList;

/** WEB
  Данные для справочника "Профиль медицинской помощи"
*/
extern const QUuidEx NsiProfileList;

/** WEB
  Данные для справочника ЛПУ (Лечебно-профилактические учреждения)
*/
extern const QUuidEx NsiLpuList;

/** WEB
  Данные для справочника МКБ (Международный классификтор болезней)
*/
extern const QUuidEx NsiMkbList;

/** WEB
  Запустить периодическую задачу по синхронизации НСИ немедленно
*/
extern const QUuidEx TaskSyncNsiNow;

/** WEB
  Возвращает настройки для периодической задачи по синхронизации НСИ
*/
extern const QUuidEx TaskSyncNsiInfo;

/** WEB
  Устанавливает настройки для периодической задачи по синхронизации НСИ
*/
extern const QUuidEx TaskSyncNsiEdit;

/** FOMS
  Запрос на обновление данных в справочнике Vidmp. Ввиду малого  объема данных
  в справочнике он обновляется за одну итерацию.  На ФОМС Сервер  отправляется
  пустая команда (без контента),  в  ответ  приходит  структура  SyncNsiVidmpA
  заполнненная данными справочника
*/
extern const QUuidEx SyncNsiVidmp;

/** FOMS
  Запрос на обновление данных в справочнике Profile. Ввиду малого объема данных
  в справочнике он обновляется за одну итерацию.  На  ФОМС Сервер  отправляется
  пустая команда (без контента),  в  ответ  приходит  структура SyncNsiProfileA
  заполнненная данными справочника
*/
extern const QUuidEx SyncNsiProfile;

/** FOMS
  Запрос на обновление данных в справочнике Lpu. Ввиду большого  объема  данных
  в справочнике его обновление выполняется итерационно.  Механизм  работы анало-
  гичен системе синхронизации данных (см. описание команды GetSyncData).
  Команда  отправляет  структуру  данных  SyncNsiLpu со следующими  заполенными
  полями:
    code  - Код ЛПУ, с которого нужно начать заполнение элементов;
    count - Количество запрашиваемых строк данных

  В ответ приходит запрашиваемый набор данных. Итерации продолжаются до тех пор
  пока не будут получены все данные.  Вычисление  контрольной  суммы для сверки
  данных не выполняется. Обработка команды KeepWaitCommand - поддерживается
*/
extern const QUuidEx SyncNsiLpu;

/** FOMS
  Запрос на обновление данных в справочнике Mkb. Ввиду большого  объема  данных
  в справочнике его обновление выполняется итерационно.  Механизм  работы анало-
  гичен системе синхронизации данных (см. описание команды GetSyncData).
  Команда  отправляет  структуру  данных  SyncNsiMkb со следующими  заполенными
  полями:
    id    - Идентификатор МКБ, с которого нужно начать заполнение элементов;
    count - Количество запрашиваемых строк данных

  В ответ приходит запрашиваемый набор данных. Итерации продолжаются до тех пор
  пока не будут получены все данные.  Вычисление  контрольной  суммы для сверки
  данных не выполняется. Обработка команды KeepWaitCommand - поддерживается

*/
extern const QUuidEx SyncNsiMkb;

} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct NsiVidmp
{
    qint32  id;
    QString name; // Наименование вида мед. помощи

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id   )
        J_SERIALIZE_ITEM( name )
    J_SERIALIZE_END
};

struct NsiVidmpList : Data<&command::NsiVidmpList,
                            Message::Type::Command,
                            Message::Type::Answer>
{
    QUuidEx scoreId;
    QVector<NsiVidmp> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( scoreId )
        J_SERIALIZE_ITEM( items   )
    J_SERIALIZE_END
};

struct NsiProfile
{
    qint32  id;
    QString name; // Наименование профиля мед. помощи

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id   )
        J_SERIALIZE_ITEM( name )
    J_SERIALIZE_END
};

struct NsiProfileList : Data<&command::NsiProfileList,
                              Message::Type::Command,
                              Message::Type::Answer>
{
    QUuidEx scoreId;
    QVector<NsiProfile> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( scoreId )
        J_SERIALIZE_ITEM( items   )
    J_SERIALIZE_END
};

struct NsiLpu
{
    qint32  code;      // Код ЛПУ
    QString fullName;  // Полное наименование ЛПУ
    QString shortName; // Сокращенное наименование ЛПУ

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( code      )
        J_SERIALIZE_ITEM( fullName  )
        J_SERIALIZE_ITEM( shortName )
    J_SERIALIZE_END
};

struct NsiLpuFilter
{
    QUuidEx scoreId;

    //QString fullName;  // Часть полного наименование ЛПУ
    //QString shortName; // Часть сокращенного наименование ЛПУ

    // Выбирать записи по CODE или FULL_NAME
    QString lpuSearch;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( scoreId   )
        J_SERIALIZE_ITEM( lpuSearch )
    J_SERIALIZE_END
};

struct NsiLpuList : Data<&command::NsiLpuList,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    // Фильтр выбора данных
    NsiLpuFilter filter;

    // Результирующая выборка
    QVector<NsiLpu> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( items  )
    J_SERIALIZE_END
};

struct NsiMkb
{
//    qint32 id = {0}; // TODO: удалить. В БД сделать MKB_CODE первичным ключом.
    QString mkbCode;
    QString mkbName;

    J_SERIALIZE_BEGIN
//        J_SERIALIZE_ITEM( id      ) // TODO: удалить
        J_SERIALIZE_ITEM( mkbCode )
        J_SERIALIZE_ITEM( mkbName )
    J_SERIALIZE_END
};

struct NsiMkbFilter
{
    QUuidEx scoreId;
    qint32  parentId = {-1};

    // Выбирать только те записи, у которых есть МКБ код
    bool onlyMkbCode = {false};

    // Поиск по МКБ коду и по МКБ имени
    QString mkbSearch;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( scoreId     )
        J_SERIALIZE_ITEM( parentId    )
        J_SERIALIZE_ITEM( onlyMkbCode )
        J_SERIALIZE_ITEM( mkbSearch   )
    J_SERIALIZE_END
};

struct NsiMkbList : Data<&command::NsiMkbList,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    // Фильтр выбора данных
    NsiMkbFilter filter;

    // Результирующая выборка
    QVector<NsiMkb> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( items  )
    J_SERIALIZE_END
};

struct TaskSyncNsi
{
    // Идентификатор задачи (не редактируется)
    QUuidEx taskId;

    // Признак, что задача будет запущена (включение/отключение задачи)
    bool isEnabled = {false};

    // Расчётное время запуска задачи
    QDateTime runDateTime = {QDateTime::currentDateTime()};

    // Время следующего запуска задачи
    QDateTime nextDateTime = {QDateTime::currentDateTime()};

    // Количество попыток повторных запусков после первого неудачного старта
    qint16 attemptLimit = {0};

    // Интервал между повторными запусками задачи (в минутах)
    qint16 attemptInterval = {15};

    // Статус выполнения задачи (не редактируется)
    TaskExecStatus taskExecStatus = {TaskExecStatus::NotRun};

    J_SERIALIZE_BEGIN
        J_SERIALIZE_OPT ( taskId          )
        J_SERIALIZE_ITEM( isEnabled       )
        J_SERIALIZE_ITEM( runDateTime     )
        J_SERIALIZE_ITEM( nextDateTime    )
        J_SERIALIZE_ITEM( attemptLimit    )
        J_SERIALIZE_ITEM( attemptInterval )
        J_SERIALIZE_OPT ( taskExecStatus  )
    J_SERIALIZE_END
};

struct TaskSyncNsiInfoA : TaskSyncNsi, Data<&command::TaskSyncNsiInfo,
                                             Message::Type::Answer>
{};

struct TaskSyncNsiEdit : TaskSyncNsi, Data<&command::TaskSyncNsiEdit,
                                            Message::Type::Command>
{};

struct SyncNsiVidmpA : Data<&command::SyncNsiVidmp,
                             Message::Type::Answer>
{
    QVector<NsiVidmp> items;
    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( items )
    J_SERIALIZE_END
};

struct SyncNsiProfileA : Data<&command::SyncNsiProfile,
                               Message::Type::Answer>
{
    QVector<NsiProfile> items;
    J_SERIALIZE_ONE( items )
};

struct SyncNsiLpu : Data<&command::SyncNsiLpu,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    // Код ЛПУ, с которого нужно начать заполнение items-ов
    qint32 code = {0};

    // Количество запрашиваемых строк данных
    qint32 count = {5000};

    QVector<NsiLpu> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_OPT ( code  )
        J_SERIALIZE_ITEM( count )
        J_SERIALIZE_ITEM( items )
    J_SERIALIZE_END
};

struct SyncNsiMkb : Data<&command::SyncNsiMkb,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    // Идентификатор МКБ, с которого нужно начать заполнение items-ов
    QString mkbCode = QString();

    // Количество запрашиваемых строк данных
    qint32 count = {5000};

    QVector<NsiMkb> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_OPT ( mkbCode )
        J_SERIALIZE_ITEM( count   )
        J_SERIALIZE_ITEM( items   )
    J_SERIALIZE_END
};


} // namespace data
} // namespace communication


