/*****************************************************************************
  В модуле представлен список идентификаторов команд для коммуникации между
  клиентской и серверной частями приложения.
  В данном модуле представлен список команд персональный для этого приложения.

  Требование надежности коммуникаций: однажды назначенный идентификатор коман-
  ды не должен более меняться.
*****************************************************************************/

#pragma once

#include "paging.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/** WEB
  Создание отчета
*/
extern const QUuidEx TaskReportCreate;

/** WEB
  Редактировать отчет
*/
extern const QUuidEx ReportEdit;

/** WEB
  Удалить отчет
*/
extern const QUuidEx ReportDelete;

/** WEB
  Информация по отчету
*/
extern const QUuidEx ReportInfo;

/** WEB
  Список отчетов
*/
extern const QUuidEx ReportList;

/** WEB
  Данные по отчету
*/
extern const QUuidEx ReportData;

} // namespace command

// Тип (вид) формируемого отчета
enum class ReportType : quint32
{
    Undefined  = 0,
    Lpu        = 1,  // Медицинская организация (ЛПУ)
    Mkb        = 2,  // Диагноз (МКБ без последней цифры)
    Vidmp      = 3,  // Вид медицинской помощи (VIDMP)
    MedProfile = 4   // Медицинский профиль
};

// Ранжирование отчета
enum class ReportRanging : quint32
{
    Undefined  = 0,
    MeeScores  = 1,  // Количество очков (score) для МЭЭ
    MeeRisk    = 2,  // Риск по МЭЭ (стоимость услуги × score)
    EkmpScores = 3,  // Количество очков (score) для ЭКМП
    EkmpRisk   = 4   // Риск по ЭКМП (стоимость услуги × score)
};

// Визуальное представление отчета
enum class ReportPresent : quint32
{
    Undefined = 0,
    Circular  = 1,  // Круговая диаграмма
    Linear    = 2,  // Линейная диаграмма
    Tabular   = 3   // Табличное представление
};

// Сортировка результатов отчета
enum class ReportSort : quint32
{
    Descending = 0,  // По убыванию
    Ascending  = 1   // По возрастанию
};

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct Report
{
    // Текущий статус создания отчета
    enum class ExecStatus : quint32
    {
        Undefined = 0,
        Creating  = 1, // Отчет в состоянии создания
        Success   = 2, // Создание отчета успешно завершено
        Failed    = 3  // Не удалось создать отчет
    };

    // Идентификатор отчета
    QUuidEx id;

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя
    QUuidEx userId;

    // Идентификатор оценки на основе которой создается отчет
    QUuidEx scoreId;

    // Тип отчета
    ReportType type = {ReportType::Undefined};

    // Наименование отчета
    QString name;

    // Время создания отчета
    QDateTime createDate = {QDateTime::currentDateTime()};

    // Левая и правая граница интервала выборки данных для отчета. Значения
    // интервала не могут  выходить  за  границы  интервала  оценки (score)
    // на основе которой создается отчет
    TimeRange period;

    //--- Поля для фильтрации по мед. критериям ---
    QString lpu;         // Список кодов ЛПУ (через запятую)
    QString mkb;         // Список кодов МКБ (через запятую)
    QString vidmp;       // Список кодов мед. помощи (через запятую)
    QString medProfile;  // Список кодов мед. профилей (через запятую)

    // Порог по score – числовое значение от 0 до 1
    double threshold = {0.5};

    // Количество записей, которые должны попасть в отчет
    int recordCount = {10};

    // TODO range переименовать в ranging после показа 12-го числа
    ReportRanging ranging = {ReportRanging::Undefined};

    // TODO diagram переименовать в present после показа 12-го числа
    ReportPresent present = {ReportPresent::Undefined};

    // Направление сортировки данных отчета
    ReportSort sort = {ReportSort::Descending};

    // Статус создания отчета
    ExecStatus execStatus = {ExecStatus::Undefined};

    //--- Вспомогательные поля для WEB-интерфейса ---
    QString scoreName;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id          )
        J_SERIALIZE_ITEM( taskId      )
        J_SERIALIZE_ITEM( userId      )
        J_SERIALIZE_ITEM( scoreId     )
        J_SERIALIZE_ITEM( type        )
        J_SERIALIZE_ITEM( name        )
        J_SERIALIZE_ITEM( createDate  )
        J_SERIALIZE_ITEM( period      )
        J_SERIALIZE_ITEM( lpu         )
        J_SERIALIZE_ITEM( mkb         )
        J_SERIALIZE_ITEM( vidmp       )
        J_SERIALIZE_ITEM( medProfile  )
        J_SERIALIZE_ITEM( threshold   )
        J_SERIALIZE_ITEM( recordCount )
        J_SERIALIZE_ITEM( ranging     )
        J_SERIALIZE_ITEM( present     )
        J_SERIALIZE_ITEM( sort        )
        J_SERIALIZE_ITEM( execStatus  )
        J_SERIALIZE_ITEM( scoreName   )
    J_SERIALIZE_END
};

struct ReportFilter
{
    // Идентификатор отчета
    QUuidEx id;

    // Идентификатор задачи
    //QUuidEx taskId;

    // Идентификатор пользователя (владельца оценки)
    QUuidEx userId;

    // Полное или частичное имя отчета
    QString name;

    // Тип отчета
    ReportType type = {ReportType::Undefined};

    // Фильтр "Дата создания"
    TimeRange createDate;

    // Фильтр "Период данных"
    TimeRange dataPeriod;

    // Идентификатор оценки, которая использовалась при создании отчета
    QUuidEx scoreId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id         )
        J_SERIALIZE_ITEM( userId     )
        J_SERIALIZE_ITEM( name       )
        J_SERIALIZE_ITEM( type       )
        J_SERIALIZE_ITEM( createDate )
        J_SERIALIZE_ITEM( dataPeriod )
        J_SERIALIZE_ITEM( scoreId    )
    J_SERIALIZE_END
};

struct TaskReportCreate : Report, Data<&command::TaskReportCreate,
                                        Message::Type::Command,
                                        Message::Type::Answer>
{};

struct ReportEdit : Data<&command::ReportEdit,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;
    QString name;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
        J_SERIALIZE_ITEM( name   )
    J_SERIALIZE_END
};

struct ReportDelete : Data<&command::ReportDelete,
                            Message::Type::Command,
                            Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct ReportInfo : Data<&command::ReportInfo,
                          Message::Type::Command>
{
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct ReportInfoA : Report, Data<&command::ReportInfo,
                                   Message::Type::Answer>
{};

struct ReportList : Data<&command::ReportList,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    // Фильтр выбора данных
    ReportFilter filter;

    // Фильтр для постраничной выборки данных
    PagingInfo paging;

    QVector<Report> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( paging )
        J_SERIALIZE_ITEM( items  )
    J_SERIALIZE_END
};

struct ReportDataItem
{
    QString name;
    QString code;
    double  value;

    QString nameLpu;
    QString codeLpu;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( name    )
        J_SERIALIZE_ITEM( code    )
        J_SERIALIZE_ITEM( value   )
        J_SERIALIZE_ITEM( nameLpu )
        J_SERIALIZE_ITEM( codeLpu )
    J_SERIALIZE_END
};

struct ReportData : Data<&command::ReportData,
                          Message::Type::Command,
                          Message::Type::Answer>
{
    QUuidEx reportId;
    QUuidEx userId;
    ReportSort reportSort = {ReportSort::Descending};

    QVector<ReportDataItem> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( reportId   )
        J_SERIALIZE_ITEM( userId     )
        J_SERIALIZE_ITEM( reportSort )
        J_SERIALIZE_ITEM( items      )
    J_SERIALIZE_END
};

} // namespace data
} // namespace communication
