/*****************************************************************************
  В модуле представлен список идентификаторов команд для коммуникации между
  клиентской и серверной частями приложения.
  В данном модуле представлен список команд персональный для этого приложения.

  Требование надежности коммуникаций: однажды назначенный идентификатор коман-
  ды не должен более меняться.
*****************************************************************************/

#pragma once

#include "paging.h"
#include "time_range.h"

#include "shared/qt/quuidex.h"
#include "shared/qt/communication/commands_base.h"
#include "shared/qt/communication/serialization/json.h"

namespace communication {
namespace command {

//----------------------------- Список команд --------------------------------

/** WEB
  Создание отчета (создание разовой задачи)
*/
extern const QUuidEx TaskReportFedCreate;

/** WEB
  Редактировать отчет
*/
extern const QUuidEx ReportFedEdit;

/** WEB
  Удалить отчет
*/
extern const QUuidEx ReportFedDelete;

/** WEB
  Информация по отчету
*/
extern const QUuidEx ReportFedInfo;

/** WEB
  Список отчетов
*/
extern const QUuidEx ReportFedList;

/** WEB
  Имя отчета (основного/корректирующего), который будет создан
*/
extern const QUuidEx ReportFedName;

/** WEB
  Запустить периодическую задачу создающую отчет немедленно
*/
extern const QUuidEx TaskReportFedNow;

/** WEB
  Возвращает настройки для периодической задачи
*/
extern const QUuidEx TaskReportFedInfo;

/** WEB
  Устанавливает настройки для периодической задачи
*/
extern const QUuidEx TaskReportFedEdit;


} // namespace command

//---------------- Структуры данных используемые в сообщениях ----------------

namespace data {

struct ReportFed
{
    // Текущий статус создания отчета
    enum class ExecStatus : quint32
    {
        Undefined = 0,
        Creating  = 1, // Отчет в состоянии создания
        Success   = 2, // Создание отчета успешно завершено
        Failed    = 3  // Не удалось создать отчет
    };

    // Идентификатор отчета
    QUuidEx id;

    // Идентификатор задачи
    QUuidEx taskId;

    // Идентификатор пользователя
    QUuidEx userId;

    // Идентификатор оценки на основе которой создается отчет
    QUuidEx scoreId;

    // Наименование отчета. Генерируется автоматически при создании отчета,
    // и более не меняется
    QString name;

    // Время создания отчета
    QDateTime createDate = {QDateTime::currentDateTime()};

    // Левая и правая граница интервала выборки данных для отчета. Значения
    // интервала берутся из оценки (score)  при  создании  отчета,  и более
    // не изменяются.
    TimeRange period;

    // Поле для хранения отчета в XML виде
    QString xmlData;

    // Признак, что отчет был создан  периодической  задачей.  Устанавливается
    // при создании задачи, и более не меняется
    bool isPeriodic = {false};

    // Признак, что отчет является корректирующим. Устанавливается при создании
    // отчета и более не меняется
    bool isCorrection = {false};

    // Порядковый номер отчета. Генерируется автоматически при создании отчета,
    // может быть скорректирован пользователем
    qint32 orderNum = {0};

    // Статус создания отчета
    ExecStatus execStatus = {ExecStatus::Undefined};

    // Текущее значение прогресса
    qint32 progressCurrent = {-1};

    // Максимальное значение прогресса для текущей задачи
    qint32 progressTotal = {-1};

    //--- Вспомогательные поля для WEB-интерфейса ---
    QString scoreName;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_OPT ( id              )
        J_SERIALIZE_OPT ( taskId          )
        J_SERIALIZE_ITEM( userId          )
        J_SERIALIZE_ITEM( scoreId         )
        J_SERIALIZE_OPT ( name            )
        J_SERIALIZE_OPT ( createDate      )
        J_SERIALIZE_OPT ( period          )
        J_SERIALIZE_OPT ( xmlData         )
        J_SERIALIZE_OPT ( isPeriodic      )
        J_SERIALIZE_ITEM( isCorrection    )
        J_SERIALIZE_OPT ( orderNum        )
        J_SERIALIZE_OPT ( execStatus      )
        J_SERIALIZE_OPT ( progressCurrent )
        J_SERIALIZE_OPT ( progressTotal   )
        J_SERIALIZE_OPT ( scoreName       )
    J_SERIALIZE_END
};

struct ReportFedFilter
{
    // Идентификатор отчета
    QUuidEx id;

    // Идентификатор пользователя
    QUuidEx userId;

    // Полное или частичное имя отчета
    QString name;

    // Фильтр "Дата создания"
    TimeRange createDate;

    // Фильтр "Период данных"
    TimeRange dataPeriod;

    // Идентификатор оценки, которая использовалась при создании отчета
    QUuidEx scoreId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id         )
        J_SERIALIZE_OPT ( userId     )
        J_SERIALIZE_OPT ( name       )
        J_SERIALIZE_OPT ( createDate )
        J_SERIALIZE_OPT ( dataPeriod )
        J_SERIALIZE_OPT ( scoreId    )
    J_SERIALIZE_END
};

struct TaskReportFedCreate : ReportFed,
                             Data<&command::TaskReportFedCreate,
                                   Message::Type::Command,
                                   Message::Type::Answer>
{};

struct ReportFedEdit : Data<&command::ReportFedEdit,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;
    QString name;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
        J_SERIALIZE_ITEM( name   )
    J_SERIALIZE_END
};

struct ReportFedDelete : Data<&command::ReportFedDelete,
                               Message::Type::Command,
                               Message::Type::Answer>
{
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct ReportFedInfo : Data<&command::ReportFedInfo,
                             Message::Type::Command>
{
    QUuidEx id;
    QUuidEx userId;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( id     )
        J_SERIALIZE_ITEM( userId )
    J_SERIALIZE_END
};

struct ReportFedInfoA : ReportFed, Data<&command::ReportFedInfo,
                                         Message::Type::Answer>
{};

struct ReportFedList : Data<&command::ReportFedList,
                             Message::Type::Command,
                             Message::Type::Answer>
{
    // Фильтр выбора данных
    ReportFedFilter filter;

    // Фильтр для постраничной выборки данных
    PagingInfo paging;

    QVector<ReportFed> items;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( filter )
        J_SERIALIZE_ITEM( paging )
        J_SERIALIZE_OPT ( items  )
    J_SERIALIZE_END
};

struct ReportFedNameA : Data<&command::ReportFedName,
                              Message::Type::Answer>
{
    QString rpeName;  // Имя основного отчета
    QString rpesName; // Имя корректирующего отчета
    int     rpeIndex  = {0}; // Индекс основного отчета
    int     rpesIndex = {0}; // Индекс корректирующего отчета
    QString fomsCode;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( rpeName   )
        J_SERIALIZE_ITEM( rpesName  )
        J_SERIALIZE_OPT ( rpeIndex  )
        J_SERIALIZE_OPT ( rpesIndex )
        J_SERIALIZE_OPT ( fomsCode  )
    J_SERIALIZE_END
};

struct TaskReportFed
{
    // Признак, что задача будет запущена (включение/отключение задачи)
    bool isEnabled = {false};

    // Идентификатор связной задачи, от которой зависит текущая задача
    QUuidEx parentId;

    // Расчётное время запуска задачи
    QDateTime runDateTime = {QDateTime::currentDateTime()};

    // Время следующего запуска задачи
    QDateTime nextDateTime = {QDateTime::currentDateTime()};

    // Код территориального фонда обязательного медицинского страхования
    QString fomsCode;

    // Директория к которую сохраняются отчеты сгенерированные автоматически
    QString storeDir;

    //--- Вспомогательные поля для WEB-интерфейса ---
    QString parentName;

    J_SERIALIZE_BEGIN
        J_SERIALIZE_ITEM( isEnabled    )
        J_SERIALIZE_ITEM( parentId     )
        J_SERIALIZE_ITEM( runDateTime  )
        J_SERIALIZE_ITEM( nextDateTime )
        J_SERIALIZE_OPT ( fomsCode     )
        J_SERIALIZE_OPT ( storeDir     )
        J_SERIALIZE_OPT ( parentName   )
    J_SERIALIZE_END
};

struct TaskReportFedInfoA : TaskReportFed, Data<&command::TaskReportFedInfo,
                                                 Message::Type::Answer>
{};

struct TaskReportFedEdit : TaskReportFed, Data<&command::TaskReportFedEdit,
                                                Message::Type::Command>
{};

} // namespace data
} // namespace communication
